/* Copyright (c) 1991-2002 Doshita Lab. Speech Group, Kyoto University */
/* Copyright (c) 2000-2002 Speech and Acoustics Processing Lab., NAIST */
/*   All rights reserved   */

/* adin_sndfile.c --- A/D-in functions wave file using libsndfile */

/* $Id: adin_sndfile.c,v 1.4 2002/09/11 22:01:50 ri Exp $ */

/* requires libsndfile (tested on 0.0.23)
   http://www.zip.com.au/~erikd/libsndfile/ */

#include <sent/stddefs.h>
#include <sent/speech.h>
#include <sent/adin.h>

#ifdef HAVE_LIBSNDFILE

#include <sndfile.h>

static int sfreq;		/* required sampling frequency (Hz) */
static SF_INFO sinfo;		/* wavefile info */
static SNDFILE *sp;			/* file handler */

/* check whether the format is 16kHz, 16bit, monoral */
static boolean
check_format(SF_INFO *s)
{
  if ((s->format & SF_FORMAT_TYPEMASK) != SF_FORMAT_RAW) {
    if (s->samplerate != sfreq) {
      j_printerr("adin_sndfile: sample rate != %d, it's %d Hz data\n", sfreq, s->samplerate);
      return FALSE;
    }
  }
  if (s->channels != 1) {
    j_printerr("adin_sndfile: channel num != 1, it has %d channels\n", s->channels);
    return FALSE;
  }
  if (s->pcmbitwidth != 16) {
    j_printerr("adin_sndfile: not 16-bit data, it's %d bit\n", s->pcmbitwidth);
    return FALSE;
  }
  return TRUE;
}

/* output format (compliant to libsnd-0.0.23) */
static void
print_format(SF_INFO *s)
{
  printf("file format: ");
  switch(s->format & SF_FORMAT_TYPEMASK) {
  case SF_FORMAT_WAV:    j_printf("Microsoft WAV"); break;
  case SF_FORMAT_AIFF:   j_printf("Apple/SGI AIFF"); break;
  case SF_FORMAT_AU:     j_printf("Sun/NeXT AU"); break;
  case SF_FORMAT_AULE:   j_printf("DEC AU"); break;
  case SF_FORMAT_RAW:    j_printf("RAW"); break;
  case SF_FORMAT_PAF:    j_printf("Ensoniq PARIS"); break;
  case SF_FORMAT_SVX:    j_printf("Amiga IFF / SVX8 / SV16"); break;
  case SF_FORMAT_NIST:   j_printf("Sphere NIST"); break;
  case SF_FORMAT_WMA:    j_printf("Windows Media Audio"); break;
  case SF_FORMAT_SMPLTD: j_printf("Sekd Samplitude"); break;
  }
  switch(s->format & SF_FORMAT_SUBMASK) {
  case SF_FORMAT_PCM:       j_printf(", PCM"); break;
  case SF_FORMAT_FLOAT:     j_printf(", floats"); break;
  case SF_FORMAT_ULAW:      j_printf(", U-Law"); break;
  case SF_FORMAT_ALAW:      j_printf(", A-Law"); break;
  case SF_FORMAT_IMA_ADPCM: j_printf(", IMA ADPCM"); break;
  case SF_FORMAT_MS_ADPCM:  j_printf(", Microsoft ADPCM"); break;
  case SF_FORMAT_PCM_BE:    j_printf(", Big endian PCM"); break;
  case SF_FORMAT_PCM_LE:    j_printf(", Little endian PCM"); break;
  case SF_FORMAT_PCM_S8:    j_printf(", Signed 8 bit PCM"); break;
  case SF_FORMAT_PCM_U8:    j_printf(", Unsigned 8 bit PCM"); break;
  case SF_FORMAT_SVX_FIB:   j_printf(", SVX Fibonacci Delta"); break;
  case SF_FORMAT_SVX_EXP:   j_printf(", SVX Exponential Delta"); break;
  case SF_FORMAT_GSM610:    j_printf(", GSM 6.10, "); break;
  case SF_FORMAT_G721_32:   j_printf(", 32kbs G721 ADPCM"); break;
  case SF_FORMAT_G723_24:   j_printf(", 24kbs G723 ADPCM"); break;
  }
  j_printf(", %d bit, %d Hz, %d channels\n", s->pcmbitwidth, s->samplerate, s->channels);
}



/***** global functions *****/

static boolean from_file;	/* TRUE is reading filename from listfile */
static FILE *fp_list;		/* used for the listfile */

boolean
adin_sndfile_standby(int freq, void *arg) /* arg: listfilename (NULL if not) */
{
  char *fname = arg;
  if (fname != NULL) {
    /* read input filename from file */
    if ((fp_list = fopen(fname, "r")) == NULL) {
      j_printerr("failed to open %s\n", fname);
      return(FALSE);
    }
    from_file = TRUE;
  } else {
    /* read filename from stdin */
    from_file = FALSE;
  }
  /* store sampling frequency */
  sfreq = freq;
  
  return(TRUE);
}
    
boolean
adin_sndfile_begin()
{
  char *speechfilename;
  boolean readp;

  /* ready to read next input */
  readp = FALSE;
  while(readp == FALSE) {
    if (from_file) {
      /* read file name from listfile */
      speechfilename = (char *)mymalloc(500);
      do {
	if (getl(speechfilename, 500, fp_list) == NULL) { /* end of input */
	  free(speechfilename);
	  fclose(fp_list);
	  return(FALSE); /* end of input */
	}
      } while (speechfilename[0] == '#'); /* skip comment */
    } else {
      /* read file name from stdin */
      speechfilename = get_line("enter filename->");
      if (speechfilename == NULL) return (FALSE);	/* end of input */
    }
    /* open input file */
    sinfo.samplerate = sfreq;
    sinfo.pcmbitwidth = 16;
    sinfo.channels = 1;
    sinfo.format = 0x0;
    if ((sp = sf_open_read(speechfilename, &sinfo)) == NULL) {
      /* retry assuming raw format */
      sinfo.samplerate = sfreq;
      sinfo.pcmbitwidth = 16;
      sinfo.channels = 1;
      sinfo.format = SF_FORMAT_RAW | SF_FORMAT_PCM_BE;
      if ((sp = sf_open_read(speechfilename, &sinfo)) == NULL) {
	sf_perror(sp);
	j_printerr("Error in opening speech data: \"%s\"\n",speechfilename);
      }
    }
    if (sp != NULL) {		/* open success */
      if (! check_format(&sinfo)) {
	j_printerr("Error: invalid format: \"%s\"\n",speechfilename);
      } else {
	j_printf("\ninput speechfile: %s\n",speechfilename);
	print_format(&sinfo);
	readp = TRUE;
      }
    }
    free(speechfilename);
  }
  return TRUE;
}

/* try to read `sampnum' samples and returns actual sample num recorded */
/* return -1 on error or end of file */
int
adin_sndfile_read(SP16 *buf, int sampnum)
{
  int cnt;

  cnt = sf_read_short(sp, buf, sampnum);
  if (cnt == 0) {		/* EOF */
    return -1;
  } else if (cnt < 0) {		/* error */
    sf_perror(sp);
    sf_close(sp);
    return -2;		/* error */
  }
  return cnt;
}

/* end recording */
boolean
adin_sndfile_end()
{
  /* close files */
  if (sf_close(sp) != 0) {
    sf_perror(sp);
    j_printerr("adin_sndfile: failed to close\n");
    return FALSE;
  }
  return TRUE;
}

#endif /* ~HAVE_LIBSNDFILE */
