// $Id$

//=============================================================================
/**
 *  @file    HSEventManager.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "HSServer.h"
#include "HSDevice.h"
#include "jyugem/gem/JGTaskTrigger.h"
#include "jyugem/gem/JGLogTime.h"
#include "BS2ACKMessage.h"
#include "BS2ErrorMessage.h"
#include "BS2ListItem.h"
#include "BS2DeclAtoms.h"
#include "HSEventManager.h"
#include "HSEventTraverser.h"
#include "HSMainManager.h"
#include "HSInfoManager.h"

static HSEventManager * _manager = NULL;

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
HSEventManager::HSEventManager() : JGEventManager()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSEventManager::HSEventManager");
}

//-----------------------------------------------------------------------------
HSEventManager::~HSEventManager()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSEventManager::~HSEventManager");

}

//-----------------------------------------------------------------------------
// Return own.
//-----------------------------------------------------------------------------
HSEventManager * HSEventManager::instance()
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::instance");
    if (_manager == NULL)
    {
        _manager = new HSEventManager;
        JGEventManager::instance(_manager);
    }
    return _manager;
}

//-----------------------------------------------------------------------------
// Setup alarm.
//-----------------------------------------------------------------------------
int HSEventManager::setup(int tid, bool ceed,
                          const std::vector<std::string>& ceids,
                          const std::vector<std::string>& vids,
                          const std::string& dir)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::setup");

    BS2Message * msg;
    HSInfoManager * info_mngr = (HSInfoManager *)this->equipment()->getInfoManager();
    if (ceids.size() == 0)
    {
        msg = this->allEnabled(ceed);
        JGEventTable * evtable = this->eventTable();
        JGEventTable::iterator iter = evtable->begin();
        for ( ; iter != evtable->end(); iter++)
        {
            (iter->second).enabled(ceed);
            (iter->second).outDir(dir);
        }
    }
    else
    {
        std::vector<JGid> event_ids;
        for (size_t i = 0; i < ceids.size(); i++)
        {
            JGid ceid;
            ceid.set(ceids[i], info_mngr->ceidFormat());
            JGEvent * event = this->find(ceid);
            if (event == NULL)
            {
                continue;
            }
            event->enabled(ceed);
            event_ids.push_back(ceid);

            event->outDir(dir);
        }
        msg = this->setupEvent(ceed, event_ids);
    }
    int result = this->send(msg, tid);
    return result;
}

//-----------------------------------------------------------------------------
// Link Event Report (S2F35)
//-----------------------------------------------------------------------------
BS2Message * HSEventManager::linkEventReport()
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::linkEventReport");

    BS2Message * msg = BS2Message::factory(SFCODE(2,35));
    BS2ListItem * bodylist = new BS2ListItem();
    BS2Item * item = this->getDataID();
    bodylist->add(item);

    BS2ListItem * rootlist = new BS2ListItem();
    JGEventTable::iterator iter = m_evtable.begin();
    for ( ; iter != m_evtable.end(); iter++)
    {
        JGEvent * event = &((*iter).second);
        BS2ListItem * list = new BS2ListItem();
        item = BS2Item::factory(_TX("CEID"), event->ceid());
        list->add(item);
        item = event->linkedReports();
        list->add(item);

        rootlist->add(list);
    }
    bodylist->add(rootlist);
    msg->add(bodylist);
    return msg;
}

//-----------------------------------------------------------------------------
// Link Event Report (S2F35)
//-----------------------------------------------------------------------------
BS2Message * HSEventManager::resetEventReport()
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::resetEventReport");

    BS2Message * msg = BS2Message::factory(SFCODE(2,35));
    BS2ListItem * bodylist = new BS2ListItem();
    BS2Item * item = this->getDataID();
    bodylist->add(item);

    BS2ListItem * rootlist = new BS2ListItem();
    JGEventTable::iterator iter = m_evtable.begin();
    for ( ; iter == m_evtable.end(); iter++)
    {
        JGEvent * event = &((*iter).second);
        BS2ListItem * list = new BS2ListItem();
        item = BS2Item::factory(_TX("CEID"), event->ceid());
        list->add(item);
        BS2ListItem * emptylist = new BS2ListItem();
        list->add(emptylist);
        rootlist->add(list);
    }
    bodylist->add(rootlist);
    msg->add(bodylist);
    return msg;
}

//-----------------------------------------------------------------------------
// Reset Reports (S2F33)
//-----------------------------------------------------------------------------
BS2Message * HSEventManager::resetReports()
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::resetReports");

    BS2Message * msg = BS2Message::factory(SFCODE(2,33));
    BS2ListItem * bodylist = new BS2ListItem();
    BS2Item * item = this->getDataID();
    bodylist->add(item);
    BS2ListItem * rootlist = new BS2ListItem();  // empty list
    bodylist->add(rootlist);
    msg->add(bodylist);
    return msg;
}

//-----------------------------------------------------------------------------
// Define Reports (S2F33)
//-----------------------------------------------------------------------------
BS2Message * HSEventManager::defineReports()
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::defineReports");

    BS2Message * msg = BS2Message::factory(SFCODE(2,33));
    BS2ListItem * bodylist = new BS2ListItem();
    BS2Item * item = this->getDataID();
    bodylist->add(item);

    BS2ListItem * rootlist = new BS2ListItem();
    JGReportTable::iterator rpt_it = m_rptable.begin();
    for ( ; rpt_it != m_rptable.end(); rpt_it++)
    {
        JGReport * report = &((*rpt_it).second);
        item = report->definedReports();
        rootlist->add(item);
    }
    bodylist->add(rootlist);
    msg->add(bodylist);
    return msg;
}

//-----------------------------------------------------------------------------
// Enable Event Report (S2F37)
//-----------------------------------------------------------------------------
BS2Message * HSEventManager::setupEvent(bool tf, vector<JGid>& ceids)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::setupEvent");

    BS2Message * msg = BS2Message::factory(SFCODE(2,37));
    BS2ListItem * bodylist = new BS2ListItem();

    BS2Item * item = BS2Item::factory(_TX("CEED"), JGvalue(tf));
    bodylist->add(item);

    BS2ListItem * rootlist = new BS2ListItem();
    for (size_t i = 0; i < ceids.size(); i++)
    {
        JGEvent * event = this->find(ceids[i]);
        if (event == NULL)
        {
            TRACE_ERROR((_TX("Event (%s) is not found.\n"),
                        ceids[i].toString().c_str()));
        }
        else
        {
            event->enabled(tf);
        }
        item = BS2Item::factory(_TX("CEID"), ceids[i]);
        rootlist->add(item);
    }
    bodylist->add(rootlist);
    msg->add(bodylist);
    return msg;
}

//-----------------------------------------------------------------------------
// Enabled/Disabled Event Report (S2F37)
//-----------------------------------------------------------------------------
BS2Message * HSEventManager::allEnabled(bool tf)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::allEnabled");

    BS2Message * msg = BS2Message::factory(SFCODE(2,37));
    BS2ListItem * bodylist = new BS2ListItem();

    BS2Item * item = BS2Item::factory(_TX("CEED"), JGvalue(tf));
    bodylist->add(item);

    BS2ListItem * rootlist = new BS2ListItem();
    JGEventTable::iterator iter = m_evtable.begin();
    for ( ; iter == m_evtable.end(); iter++)
    {
        JGEvent * event = &((*iter).second);
        event->enabled(tf);
    }
    bodylist->add(rootlist);
    msg->add(bodylist);
    return msg;
}


//-----------------------------------------------------------------------------
// Link Event Request
//-----------------------------------------------------------------------------
bool HSEventManager::isReportFile(const string& ceid, size_t total)
{
    return false;
}

string HSEventManager::getReportFile(const string& ceid)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::getReportFile");
    string strTime;
    b_time::makeDateName(ACE_OS::time(), strTime);

    BCHAR buf[PATH_MAX + 1];
    ACE_OS::getcwd(buf, PATH_MAX);
    buf[PATH_MAX] = '\0';
    string machine_id = m_equipment->name();
    string path(buf);
    JGVariable * dir = m_equipment->variable(_T("DIR"));
    if (dir != NULL)
    {
        path = dir->getv().toString();
        if (path.at(0) != DIR_SEPARATOR_CHAR)
        {
            path = buf;
        }
    }
    else
    {
        TRACE_ERROR((_TX("System variable (%s) is not found.\n"), _T("DIR")));
    }

    JGVariable * mid = m_equipment->variable(_T("MID"));
    if (mid != NULL)
    {
        if (mid->getv().size() > 0)
        {
            machine_id = mid->getv().toString();
        }
    }
    else
    {
        TRACE_ERROR((_TX("System variable (%s) is not found.\n"), _T("MID")));
    }

    if (path.at(path.size() - 1) != DIR_SEPARATOR_CHAR)
    {
        path += DIR_SEPARATOR_CHAR;
    }
    path += ceid;
    path += '.';
    path += machine_id;
    path += strTime;
    return path;
}

//-----------------------------------------------------------------------------
// Link Event Request (S6F11 -> S6F12)
//-----------------------------------------------------------------------------
BS2Message * HSEventManager::eventReport(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventManager::eventReport");

    BS2Message * replymsg;
    BYTE ack;
    HSServer * eqsvr = (HSServer *)this->equipment();

    HSEventTraverser  evtinfo(this, SFCODE(6,11));
    int result = msg->traverse(&evtinfo);    // parse S6F11
    if (result < 0)
    {
        //
        // Make error event report to host
        //
        //eqsvr->event_report(0, evtinfo.m_event->ceid().getInt(), _TX(""),
        //                    NULL, result, _TX(""));

        // ack = -result;
        ack = ACK_OK;
        replymsg = BS2Message::response(msg, ack, _TX("ACKC6"));
        return replymsg;
    }
    if (evtinfo.m_event == NULL)
    {
        //
        // Make error event report to host
        //
        //eqsvr->event_report(0, 0, _TX(""), NULL, 2, _TX("Unknown event"));
        //

        //ack = 2;            // Illegal format
        ack = ACK_OK;
        replymsg = BS2Message::response(msg, ack, _TX("ACKC6"));
        return replymsg;
    }

    evtinfo.parseRemains();

    // make host report
    ack = ACK_OK;
    HSMainManager * hostsrv = HSMainManager::instance();

    evtinfo.m_event->callback();   // Callbak user routine, as signal.

    size_t total_size = evtinfo.m_xmlmsg.size() + evtinfo.m_xml.size();
    int ceid = evtinfo.m_event->ceid().getInt();
    string bodyName = _TX("");
    if (this->isReportFile(evtinfo.m_event->name(), total_size))
    {
        bodyName = this->getReportFile(evtinfo.m_event->name());
        result = hostsrv->storeReport(bodyName, evtinfo.m_xml);
        if (result < 0)
        {
            TRACE_ERROR((_TX("Error vfei message write: (%s)\n"),
                evtinfo.m_xml.c_str()));
            return NULL;
        }
    }

    eqsvr->event_report(evtinfo.m_event->name(), bodyName, evtinfo.m_xml);

    replymsg = BS2Message::response(msg, ack, _TX("ACKC6"));
    return replymsg;
}


//-----------------------------------------------------------------------------
// Thread of received message event.
//-----------------------------------------------------------------------------
BS2Message * HSEventManager::msg_svc(JGMessageTrigger * trigger,
                                     BS2Message * msg)
{
    BS2Message * replymsg = NULL;

    if (msg->sf() == SFCODE(6,11))
    {
        replymsg = this->eventReport(msg);
    }
    else
    {   // Unexpected message
        replymsg = this->unrecognized(msg);
    }

    return replymsg;
}

