// $Id$

//=============================================================================
/**
 *  @file    b_varinfo.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "b_varinfo.h"
#include "jyugem/gem/JGEquipment.h"
#include "jyugem/gem/JGVariable.h"

int user_varinfo::get_value(string& itemval, int id)
{
    char * bufptr;
    int    buftype;
    int result = m_func(id, &bufptr, &buftype);
    if (result < 0 || bufptr == null)
    {
        itemval = "";
        return -1;
    }
    itemval = bufptr;
    if (buftype == USERBUF_DYNAMIC)
    {
        free(bufptr);
    }
    return 0;
}


int b_varinfo::add(user_variable * uvptr)
{
    TRACE_FUNCTION(TRL_LOW, "b_varinfo::add");

    if (uvptr->m_vid != 0)
    {
        b_id vid(uvptr->m_vid);
        string varname(uvptr->m_name);
        if (this->find(vid) != null || this->find(varname) != null)
        {
            TRACE_ERROR((_TX("Duplicate variable info.: %d %s\n"),
                        uvptr->m_vid, uvptr->m_name));
            return -1;
        }
        user_varinfo * varinfo = new user_varinfo(uvptr);
        m_id_vars.insert(std::make_pair(varinfo->m_vid, varinfo));
        m_name_vars.insert(std::make_pair(varinfo->m_name, varinfo));
    }
    else
    {
        string varname(uvptr->m_name);
        if (this->find(varname) != null)
        {
            TRACE_ERROR((_TX("Duplicate variable info.: %s\n"),
                        uvptr->m_name));
            return -1;
        }
        user_varinfo * varinfo = new user_varinfo(uvptr);
        m_name_vars.insert(std::make_pair(varinfo->m_name, varinfo));
    }
    return 0;
}

user_varinfo * b_varinfo::find(const string& name)
{
    namemap::iterator iter = m_name_vars.find(name);
    if (iter == m_name_vars.end())
    {
        return null;
    }
    return iter->second;
}

user_varinfo * b_varinfo::find(const b_id& id)
{
    id_map::iterator iter = m_id_vars.find(const_cast<b_id&>(id));
    if (iter == m_id_vars.end())
    {
        return null;
    }
    return iter->second;
}

void b_varinfo::dump() const
{
    TRACE_FUNCTION(TRL_LOW, "b_varinfo::dump");

    id_map::const_iterator iter = m_id_vars.begin();
    for ( ; iter == m_id_vars.end(); ++iter)
    {
        user_varinfo * vi = iter->second;
        TRACE_DEBUG((_TX("variable info.: %d %s 0x%x\n"),
            vi->m_vid.getInt(), vi->m_name.c_str(), vi->m_func));
    }
}


//
// Entry variable info. for C language interface
//
int gem_variable_add(user_variable * varinfo)
{
    return USER_Variables::instance()->add(varinfo);
}

int gem_set_value_i(user_vid uvid, int value)
{
    JGid vid(uvid);
    JGVariable * var = JGEquipment::instance()->variable(vid);
    if (var != NULL)
    {
        var->getv().setInt(value);
    }
    return 0;
}

int gem_set_value_f(user_vid uvid, double value)
{
    TRACE_FUNCTION(TRL_LOW, "gem_set_value_f");

    JGid vid(uvid);
    JGVariable * var = JGEquipment::instance()->variable(vid);
    if (var != NULL)
    {
        switch (var->format())
        {
        case ATOM_ASCII:
            {
                char buf[128];
                sprintf(buf, "%f", value);
                var->getv().set(buf);
            }
            break;
        case ATOM_BINARY:
            {
                var->getv().set((BYTE *)&value, sizeof(value));
            }
            break;
        case ATOM_BOOLEAN:
            {
                bool v = (value != 0.0) ? true : false;
                var->getv().set(v);
            }
            break;
        case ATOM_INT1:
            {
                char v = (char)value;
                var->getv().set(v);
            }
            break;
        case ATOM_INT2:
            {
                short v = (short)value;
                var->getv().set(v);
            }
            break;
        case ATOM_INT4:
            {
                int v = (int)value;
                var->getv().set(v);
            }
            break;
        case ATOM_INT8:
            {
                LONGLONG v = (LONGLONG)value;
                var->getv().set(v);
            }
            break;
        case ATOM_UINT1:
            {
                unsigned char v = (unsigned char)value;
                var->getv().set(v);
            }
            break;
        case ATOM_UINT2:
            {
                unsigned short v = (unsigned short)value;
                var->getv().set(v);
            }
            break;
        case ATOM_UINT4:
            {
                unsigned int v = (unsigned int)value;
                var->getv().set(v);
            }
            break;
        case ATOM_UINT8:
            {
                ULONGLONG v = (ULONGLONG)value;
                var->getv().set(v);
            }
            break;
        case ATOM_FLOAT4:
            {
                float v = (float)value;
                var->getv().set(v);
            }
        case ATOM_FLOAT8:
            {
                var->getv().set(value);
            }
            break;
        default:
            TRACE_ERROR((_TX("Can't convert value: double to %o"),
                        var->format()));
            return -1;
        }
    }
    return 0;
}

int gem_set_value_a(user_vid uvid, const char * value)
{
    JGid vid(uvid);
    string v(value);
    JGVariable * var = JGEquipment::instance()->variable(vid);
    if (var != NULL)
    {
        var->getv().set(v, var->format());
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Format code to string (static)
//-----------------------------------------------------------------------------
static BCHAR * _sml_const[64] =
{
    _TX("L"),            //  0: ATOM_LIST
    NULL,                //  1:
    NULL,                //  2:
    NULL,                //  3:
    NULL,                //  4:
    NULL,                //  5:
    NULL,                //  6:
    NULL,                //  7:
    _TX("B"),            //  8: ATOM_BINARY
    _TX("BL"),           //  9: ATOM_BOOLEAN
    NULL,                // 10:
    NULL,                // 11:
    NULL,                // 12:
    NULL,                // 13:
    NULL,                // 14:
    NULL,                // 15:
    _TX("A"),            // 16: ATOM_ASCII
    _TX("J"),            // 17: ATOM_JIS
    _TX("W"),            // 18: ATOM_UNICODE
    NULL,                // 19:
    NULL,                // 20:
    NULL,                // 21:
    NULL,                // 22:
    NULL,                // 23:
    _TX("I8"),           // 24: ATOM_INT8
    _TX("I1"),           // 25: ATOM_INT1
    _TX("I2"),           // 26: ATOM_INT2
    NULL,                // 27:
    _TX("I4"),           // 28: ATOM_INT4
    NULL,                // 29:
    NULL,                // 30:
    NULL,                // 31:
    _TX("F8"),           // 32: ATOM_FLOAT8
    NULL,                // 33:
    NULL,                // 34:
    NULL,                // 35:
    _TX("F4"),           // 36: ATOM_FLOAT4
    NULL,                // 37:
    NULL,                // 38:
    NULL,                // 39:
    _TX("U8"),           // 40: ATOM_UINT8
    _TX("U1"),           // 41: ATOM_UINT1
    _TX("U2"),           // 42: ATOM_UINT2
    NULL,                // 43:
    _TX("U4"),           // 44: ATOM_UINT4
    NULL,                // 45:
    NULL,                // 46:
    NULL,                // 47:
    NULL,                // 48:
    NULL,                // 49:
    NULL,                // 50:
    NULL,                // 51:
    NULL,                // 52:
    NULL,                // 53:
    NULL,                // 54:
    NULL,                // 55:
    NULL,                // 56:
    NULL,                // 57:
    NULL,                // 58:
    NULL,                // 59:
    NULL,                // 60:
    NULL,                // 61:
    NULL,                // 62:
    NULL,                // 63:
};


const BCHAR * b_varinfo::formatToSml(int format)
{
    if (format >= 64)
        return NULL;

    BCHAR * name = _sml_const[format];
    return name;
}


