// $Id$

//=============================================================================
/**
 *  @file    host_message.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================
#define BEE_BUILD_DLL

#include "jyugem/cell/host_message.h"

namespace jyugem { namespace cell
{

struct host_message_type
{
    host_message_type() : m_idx(0), m_name(null), m_reply(false), m_dir(UNKNOWN),
                          m_algotype(0), m_algorithm(""), m_enable(true) {}
    host_message_type(int idx, const char * name, bool reply, int dir);
    virtual ~host_message_type() {}

    virtual host_message * factory(main_manager * mngr) = 0;

    static host_message_type * instance(const std::string& cmd);
    static host_message_type * instance(int cmd);

    int   m_idx;
    const BCHAR * m_name;
    bool  m_reply;
    int   m_dir;
    int   m_algotype;
    std::string m_algorithm;
    bool   m_enable;
};

template< typename HOSTMSG, int NUM, bool REPLY = true, int DIR = HtoE >
struct hmsg_type : public host_message_type
{
    hmsg_type(const BCHAR * name) : host_message_type(NUM, name, REPLY, DIR) {}
    virtual ~hmsg_type() {}

    virtual host_message * factory(main_manager * mngr)
    {
        HOSTMSG * msgptr = new HOSTMSG(mngr);
        msgptr->m_rcmd = m_name;
        return msgptr;
    }
};

typedef std::map< std::string, host_message_type *, std::less<std::string> >
        hmsg_types;

static hmsg_types _hmsg_types;
static host_message_type * _hmsg_type_table[40] = {0};

host_message_type::host_message_type(int idx, const BCHAR * name, bool reply,
                                     int dir)
    : m_idx(idx), m_name(name), m_reply(reply), m_dir(dir), m_algotype(0),
      m_algorithm(""), m_enable(true)
{
    _hmsg_types.insert(std::make_pair(name, this));
    _hmsg_type_table[idx] = this;
}

//
// (static)
//
host_message_type * host_message_type::instance(const std::string& cmd)
{
    hmsg_types::iterator iter = _hmsg_types.find(cmd);
    if (iter == _hmsg_types.end())
        return null;
    return iter->second;
}

//
host_message_type * host_message_type::instance(int cmdnum)
{
    return _hmsg_type_table[cmdnum];
}

//-----------------------------------------------------------------------------
//
// vfei types entity
//
//-----------------------------------------------------------------------------
static hmsg_type<abort_message, VFEI_ABORT>
       _abort_msg_type("ABORT");
static hmsg_type<alarm_report_message, VFEI_ALARM_REPORT, true, EtoH>
       _alarm_report_type("ALARM_REPORT");
static hmsg_type<alarm_setup_message, VFEI_ALARM_SETUP>
       _alarm_setup_type("ALARM_SETUP");
static hmsg_type<cmd_ack_message, VFEI_CMD_ACK, false, EtoH>
       _cmd_ack_type("CMD_ACK");
static hmsg_type<display_message, VFEI_DISPLAY_MSG>
       _disp_message_type("DISPLAY_MSG");
static hmsg_type<event_report_message, VFEI_EVENT_REPORT, true, EtoH>
       _event_report_type("EVENT_REPORT");
static hmsg_type<event_setup_message, VFEI_EVENT_SETUP>
       _event_setup_type("EVENT_SETUP");
static hmsg_type<initialize_message, VFEI_INITIALIZE>
       _initialize_type("INITIALIZE");
static hmsg_type<mach_cmd_message, VFEI_MACH_CMD>
       _mach_cmd_type("MACH_CMD");
static hmsg_type<move_message, VFEI_MOVE>
       _move_type("MOVE");
static hmsg_type<pause_message, VFEI_PAUSE>
       _pause_type("PAUSE");
static hmsg_type<res_activate_message, VFEI_RES_ACTIVATE>
       _res_active_type("RES_ACTIVATE");
static hmsg_type<res_deactivate_message, VFEI_RES_DEACTIVATE>
       _res_deactive_type("RES_DEACTIVATE");
static hmsg_type<res_delete_message, VFEI_RES_DELETE>
       _res_delete_type("RES_DELETE");
static hmsg_type<res_list_message, VFEI_RES_LIST>
       _res_list_type("RES_LIST");
static hmsg_type<res_query_message, VFEI_RES_QUERY>
       _res_query_type("RES_QUERY");
static hmsg_type<res_request_message, VFEI_RES_REQUEST>
       _res_request_type("RES_REQUEST");
static hmsg_type<res_transfer_message, VFEI_RES_TRANSFER>
       _res_transfer_type("RES_TRANSFER");
static hmsg_type<restart_message, VFEI_RESTART>
       _restart_type("RESTART");
static hmsg_type<resume_message, VFEI_RESUME>
       _resume_type("RESUME");
static hmsg_type<set_param_message, VFEI_SET_PARAM>
       _set_param_type("SET_PARAM");
static hmsg_type<start_message, VFEI_START>
       _start_type("START");
static hmsg_type<status_list_message, VFEI_STATUS_LIST>
       _status_list_type("STATUS_LIST");
static hmsg_type<status_query_message, VFEI_STATUS_QUERY>
       _status_query_type("STATUS_QUERY");
static hmsg_type<trace_report_message, VFEI_TRACE_REPORT>
       _trace_report_type("TRACE_REPORT");
static hmsg_type<trace_setup_message, VFEI_TRACE_SETUP>
       _trace_setup_type("TRACE_SETUP");

//-----------------------------------------------------------------------------
//
// VFEI Message Class
//
//-----------------------------------------------------------------------------
host_message * host_message::factory(main_manager * mngr, std::string& cmdname)
{
    host_message_type * msgtype = host_message_type::instance(cmdname);
    if (msgtype == null)
    {
        return null;
    }
    return msgtype->factory(mngr);
}

//-----------------------------------------------------------------------------
host_message * host_message::factory(main_manager * mngr, int cmdid)
{
    host_message_type * msgtype = host_message_type::instance(cmdid);
    if (msgtype == null)
    {
        return null;
    }
    return msgtype->factory(mngr);
}

//
int host_message::toVfei(std::string& outbuf)
{
    outbuf = _TX("");
    return 0;
}

int host_message::toXml(std::string& outbuf)
{
    outbuf = _TX("");
    return 0;
}

//
// Convert item data to boolean
//
bool host_message::toBoolean(const any_item& item)
{
    bool result = item.m_value.getBool();
    return result;
}

} }  // *jyugem::cell*
