// $Id$

//=============================================================================
/**
 *  @file    vfei_parser.h
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef JYUGEM_CELL_VFEI_PARSER_H
#define JYUGEM_CELL_VFEI_PARSER_H

///////////////////////////////////////////////////////////////////////////////
#include <boost/spirit/core.hpp>
#include <boost/spirit/symbols/symbols.hpp>
#include <boost/spirit/utility.hpp>
#include <map>
#include "jyugem/cell/vfei.h"

//
// VFEI Syntax
//
//    <host_message> ::= <data_item>+
//    <data_item> ::= <data_item_name> "/" <format_code> "=" <data_item_value>
//                  | <array_item>
//                  | <list_item>
//    <array_item> ::= <data_item_name> "/" <format_code> "[" <array_size> "]" "="
//                    "[" <data_item_value>+ "]"
//    <array_size> ::= UNSIGNED_INT
//    <list_item> ::= <data_item_name> "/" "L" "[" <list_size> "]"
//                    "=" "[" <data_item>+ "]"
//    <list_size> ::= UNSIGNED_INT
//    <data_item_name> ::= STRING
//    <format_code> ::= "U1"
//                    | "U2"
//                    | "U4"
//                    | "U8"
//                    | "I1"
//                    | "I2"
//                    | "I4"
//                    | "I8"
//                    | "F4"
//                    | "F8"
//                    | "A"
//                    | "B"
//                    | "BL"
//    <data_item_value> ::= STRING
//                        | UNSIGNED_INT
//                        | SIGNED_INT
//                        | REAL_NUMBER
//                        | BOOLEAN
//
//    STRING ::= A string of single byte ASCII characters following the syntax
//            conventions of the ANSI C language string literal. This type is
//            associated with the format_code of gAh"
//    SIGNED_INT ::= The ASCII representation of a signed integer. This type is
//            associated with the format_codes of "I1", "I2", "I4" and "I8."
//    UNSIGNED_INT ::= The ASCII representation of a unsigned integer. This type is
//            associated with the format_codes of "U1", "U2", "U4", "U8" and "B."
//        Both SIGNED_INT and UNSIGNED_INT following the syntax
//        conventions of the ANSI C language integer constants.
//    REAL_NUMBER ::= The ASCII representation of a real number following the syntax
//            conventions of the ANSI C language float constant. This type is
//            associated with the format_codes of "F4" and "F8."
//    BOOLEAN ::= The ASCII representation of a unsigned integer following the
//            syntax conventions of the ANSI C language integer constants.
//            This type is associated with the format_code of "BL." Additionally
//            a value of 0 indicates FALSE while a non 0 value indicates TRUE.
//

////////////////////////////////////////////////////////////////////////////////
// VFEI Data Item
////////////////////////////////////////////////////////////////////////////////
// Data Item            data_item_name  format  value_item_val  message where used
// --------------------+---------------+-------+---------------+--------------------------
// ALARM_CATEGORY       ALARM_CATEGORY  U1                      ALARM REPORT
// ALARM_ID             ALARM_ID        I4                      ALARM SETUP
//                                                              ALARM REPORT
// ALARM_STATE          ALARM_STATE     U1      0               ALARM REPORT
//                                              1
// ALARM_TEXT           ALARM_TEXT      A                       ALARM REPORT
// COMMAND_ID           CMD             A       ABORT           All
//                                              ALARM_REPORT
//                                              ALARM_SETUP
//                                              CMD_ACK
//                                              DISPLAY_MSG
//                                              EVENT_REPORT
//                                              EVENT_SETUP
//                                              INITIALIZE
//                                              MACH_CMD
//                                              PAUSE
//                                              RES_ACTIVATE
//                                              RES_DEACT
//                                              RES_DELETE
//                                              RES_LIST
//                                              RES_QUERY
//                                              RES_REQUEST
//                                              RES_TRANS
//                                              RESTART
//                                              RESUME
//                                              SET_PARAM
//                                              START
//                                              STATUS_LIST
//                                              STATUS_QUERY
//                                              TRACE_REPORT
//                                              TRACE_SETUP
// COMMAND_TYPE CMD_TYPE                A                       MACHINE
//                                                              COMMAND
// DEST_LOCATION DEST_LOCATION          A                       MOVE
// DEST_PORT    DEST_PORT               A                       MOVE
// DESTINATION  DESTINATION             A                       RESOURCE
//                                                              TRANSFER
// DIR          DIR                     A                       EVENT SETUP
//                                                              TRACE SETUP
// ENABLE       ENABLE                  U1      0               ALARM SETUP
//                                              1               EVENT SETUP
//                                                              TRACE SETUP
// ERROR_CODE   ECD                     U4      See section 7   All reply and unsolicited messages
//                                              VFEI Predefined
//                                              Error Codes
// ERROR_TEXT   ETX                     A                       All reply and unsolicited messages
// EVENT_ID     EVENT_ID                A       See section 5   EVENT REPORT
//                                              VFEI Events     EVENT SETUP
// FILE         FILE                    A                       EVENT REPORT
//                                                              RESOURCE LIST
//                                                              RESOURCE QUERY
//                                                              STATUS LIST
//                                                              STATUS QUERY
//                                                              TRACE REPORT
// FREQUENCY    FREQUENCY               U4                      TRACE SETUP
// LEVEL        LEVEL                   U2                      ABORT
// MACHINE_ID   MID                     A                       All
// MESSAGE      MESSAGE                 A                       DISPLAY MESSAGE
// MSG_TYPE     MTY                     A       C               All
//                                              R
//                                              E
// REPORT_SIZE  REPORT_SIZE             U2                      TRACE REPORT
//                                                              TRACE SETUP
// RESOURCE_BODY RES_BODY               *1                      RESOURCE TRANSFER
// RESOURCE_FILE RES_FILE               A                       RESOURCE REQUEST
//                                                              RESOURCE TRANSFER
// RESOURCE_ID  RES_ID                  A                       RESOURCE ACTIVATE
//                                                              RESOURCE DEACTIVATE
//                                                              RESOURCE DELETE
//                                                              RESOURCE LIST
//                                                              RESOURCE QUERY
//                                                              RESOURCE REQUEST
//                                                              RESOURCE TRANSFER
// RESOURCE_LENGTH RES_LENGTH           U4                      RESOURCE TRANSFER
// SAMPLES      SAMPLES                 U2                      TRACE SETUP
// SOURCE       SOURCE                  A                       RESOURCE TRANSFER
// SRC_LOCATION SRC_LOCATION            A                       MOVE
// SRC_PORT     SRC_PORT                A                       MOVE
// TERMINAL     TERMINAL                B                       DISPLAY MESSAGE
// TG_ID        TG_ID                   A                       MOVE
// TG_TYPE      TG_TYPE                 A                       MOVE
// TID          TID                     U4                      All command and reply messages
// TRACE_ID     TRACE_ID                U2                      TRACE REPORT
//                                                              TRACE SETUP
// TYPE         TYPE                    A                       RECIPE
//                                                              TOOL
//                                                              COMPONENT
//                                                              CONSUMABLE
//                                                              MACHINE
//                                                              VFEI
//                                                              RESOURCE ACTIVATE
//                                                              RESOURCE DEACTIVATE
//                                                              RESOURCEb DELETE
//                                                              RESOURCE LIST
//                                                              RESOURCE QUERY
//                                                              RESOURCE REQUEST
//                                                              RESOURCE TRANSFER
//                                                              SET PARAM
// VARIABLE_ID  VAR_ID                  A       See section 6   EVENT SETUP
//                                              VFEI Variables  STATUS QUERY
//                                                              TRACE SETUP
//

////////////////////////////////////////////////////////////////////////////////
// VFEI Driver Error Codes
////////////////////////////////////////////////////////////////////////////////
// Error Designation             Code   Error Explanation
//-----------------------------+-------+------------------------------------------------
// VFEI_MSG_SYNTAX_ERROR        10026   Error receiving VFEI Command message.
// VFEI_PROTOCOL_ERROR          10027   Error sending VFEI Command message.
// VFEI_TIMEOUT                 10028   Error receiving VFEI Command reply message.
// VFEI_FILE_SYNTAX_ERROR       10029   Error sending VFEI Command reply message.
// VFEI_UNKNOWN_ERROR           30000   Unknown error has occurred.
// VFEI_INVALID_FORMAT          30007   The parameter or message format is incorrect.
// VFEI_VID_NOT_EXISTS          30009   A VID has been specified that does not exist.
// VFEI_EVENT_NOT_EXISTS        30011   An Event has been specified that does not exist.
// VFEI_UNEXPECTED_EVENT        30013   An Event that was not expected was received from
//                                      the machine.
// VFEI_MISSING_PARAM           30014   Parameter required to perform the command is missing.
// VFEI_REQUEST_DENIED          30018   Command request has been denied.
// VFEI_CMD_NOT_SUPPORTED       30019   This command is recognized but not supported by
//                                      the equipment.
// VFEI_UNEXPECTED_VALUE        30025   The value of one of the parameters was unexpected
//                                      or out of range.
// VFEI_FILE_IO_ERROR           30030   VFEI File I/O Error.
// COMM_MSG_SYNTAX_ERROR        40026   Syntax of the message to the equipment is incorrect.
// COMM_PROTOCOL_ERROR          40027   A low level communication error has occurred.
// COMM_TIMEOUT                 40028   A low level communication timeout has occurred.
// EQUIP_UNKNOWN_ERROR          50000   Unknown error has occurred.
// EQUIP_INVALID_COMMAND        50001   Equipment cannot perform command
// EQUIP_CANNOT_DO_NOW          50002   Command cannot be performed at this time.
// EQUIP_INVALID_PARAM          50003   Invalid parameter specified with the command.
// EQUIP_INVALID_C_STATE        50004   Equipment is in an invalid control state for accepting
//                                      the specified command. Equipment must be in a control
//                                      state that accepts the remote command.
// EQUIP_INVALID_P_STATE        50005   Equipment is in an invalid process state for accepting
//                                      the specified command. The current process state
//                                      does not accept the remote command.
// EQUIP_NOT_ENOUGH_SPACE       50006   Not enough room to store the resource.
// EQUIP_INVALID_FORMAT         50007   Message format is not correct.
// EQUIP_REPORT_EXISTS          50008   The equipment already has a report defined.
// EQUIP_VID_NOT_EXISTS         50009   At least one VID specified does not exist.
// EQUIP_LINK_EXISTS            50010   At least one event link is already defined on the equipment.
// EQUIP_EVENT_NOT_EXISTS       50011   At least one CEID specified does not exist.
// EQUIP_REPORT_NOT_EXISTS      50012   At least one report id specified does not exist.
// EQUIP_MISSING_PARAM          50014   Parameter required to perform the command is missing.
// EQUIP_ALREADY_HAVE           50015   Current command has already been specified.
// EQUIP_WILL_NOT_ACCEPT        50016   The equipment will not accept the current command.
// EQUIP_INVALID_TYPE           50017   Type specified is not valid for the equipment.
// EQUIP_REQUEST_DENIED         50018   Command request has been denied.
// EQUIP_INCOMPLETE_RESPONSE    50020   Used to indicate incomplete data reported in
//                                      STATUS LIST or RESOURCE LIST.
// EQUIP_RESOURCE_NOT_FND       50023   Resource specified was not found.
// EQUIP_UNEXPECTED_VALUE       50025   The value of one of the parameters was unexpected
//                                      or out of range.
//

////////////////////////////////////////////////////////////////////////////////
// VFEI Events
////////////////////////////////////////////////////////////////////////////////
// Event Designation    Event Description                                       Used By
// --------------------+-------------------------------------------------------+--------
// ABORT                Processing aborted                                      B
// COM_DISABLE          Communications with the equipment has been terminated
// COM_ENABLE           Communications with the equipment has been established
// MB_COMPLETE          Processing completed for a machine batch                B
// MB_START             Machine batch processing started                        B
// MBC_COMPLETE         Processing completed for a machine batch collection     B
// MBC_START            Processing started for a machine batch collection       B
// MESSAGE_RECEIVE      Terminal message sent is acknowledged by operator       B, G
// PAUSE                Processing suspended                                    B
// RESOURCE_ACTIVATE    Resource activate complete. Same as the GEM required    B, G
//                      PP_SELECTED
// RESUME               Processing resumed                                      B
// ALARM_DETECT         Alarm has been set                                      G
// ALARM_CLEAR          Alarm has been cleared                                  G
// COMMAND_ISSUE        Indicates operator activity while in the remote state   G
// COMPLETE             Processing has completed normally                       G
// CTRL_STATE_LOCAL     Equipment is in the local control state                 G
// CTRL_STATE_REMOTE    Equipment is in the remote control state                G
// EC_CHANGE            An equipment constant has been changed by the operator  G
// EQUIPMENT_OFFLINE    The equipment has transitioned to the offline state     G
// ID_RECEIVE           The ID of the Transport Group has been obtained
// MATERIAL_RECEIVE     Material was received                                   G
// MATERIAL_REMOVE      Material was removed                                    G
// PP_CHANGE            Indicates the operator has performed a process program  G
//                      change
// PROC_STATE_CHANGE    A processing state change has occurred on the equipment G
// RESOURCE_VERIFY      Equipment has verified the downloaded resource
// START                Equipment has entered the EXECUTING state               G
// STATE_CHANGE         A state change has occurred on the equipment            G
// STOP                 Processing on the equipment has stopped in response     G
//                      to a STOP command
// TG_ARRIVE            The Transport Group has arrived at its destination
//                      location
// TG_SENSE             Transport Group is sensed at a port
//

////////////////////////////////////////////////////////////////////////////////
//  VFEI Variables
////////////////////////////////////////////////////////////////////////////////
// Variable ID      Variable Description                                          Used By
// ----------------+-------------------------------------------------------------+----------
// CLOCK            Equipment's date/time of day clock                            B, G
// COM_STATE        Communications state (communicating or not-communicating)
// CONTROL_STATE    Current control state (local or remote) of the equipment      G
// EQ_MODEL_ID      Equipment model ID
// EQ_SOFT_VER      Equipment software version number
// MB_ID            Machine batch ID
// MBC_ID           Machine batch collection ID
// MBC_SIZE         Machine batch collection size
// PP_EXEC_NAME     Process program in the execute state                          B, G
// PROCESS_STATE    Current process state of the equipment                        B, G
// VFEI_CMDS        List of VFEI commands supported
// VFEI_REL         VFEI driver release
// VFEI_VER         VFEI specification version
// ABORT_LEVEL      Level of the most recent abort
// ALARM_ID         ID of the most recent alarm that went ON or OFF               G
// ALARMS_ENABLED   Contains the list of alarms enabled for reporting             G
// ALARMS_SET       A list of alarms currently in the set state                   G
// COM_TIMER        Time interval, in seconds, between attempts by the            G
//                  equipment to establish communications. Same as the GEM
//                  specified EstablishCommunicationsTimeout variable
// DIR_UPLOAD       Where to put (i.e. directory path) an unsolicited resource
//                  transfer
// EVENTS_ENABLED   The list of events enabled for reporting
// HEART_BEAT       Time interval, in seconds, between attempts by the VFEI
//                  driver to verify the communications link is operating
// INIT_COMM_STATE  Equipment's initial communication state (enabled or
//                  disabled)
// INIT_CTRL_STATE  Equipment's initial control state (local or remote)
// LAST_EVENT       Event ID of the most recent event
// LOCATION         Name of the physical position on a piece of equipment at
//                  which a transfer object may reside
// PORT             A point on equipment at which a change of ownership of a
//                  transfer object occurs.
// PP_CHANGE_NAME   Process program ID of the most recently created, edited, or   G
//                  deleted process program
// PP_CHANGE_STATUS How the process program "PP_CHANGE_NAME" was                  G
//                  changed (created, edited, or deleted)
// PREV_PROC_STATE  Previous process state of the equipment G
// STATION          An addressable sub-assembly of a machine ID
// TG_ID            Transport group ID. Identifier for material that is being
//                  moved from a source to destination location.
// TG_TYPE          Transport group type. Type of material that is being moved
//                  from a source to destination location.
// UPLOAD_TYPE      How to store (i.e., ASCII, binary.) an unsolicited resource
//                  transfer
// VERBOSITY        Debug level. Effects verbosity of reporting or logging
//

////////////////////////////////////////////////////////////////////////////////
// VFEI COMMANDS
////////////////////////////////////////////////////////////////////////////////
// Command Name         Stream/Function
//---------------------+---------------+----------------------------------------
// ABORT                S2F49           Abort Propcess
// ALARM_REPORT         S5F1            Alarm Report
// ALARM_SETUP          S5F3            Enable/Disable Alarm Reporting
// CMD_ACK              SxFx            Command Acknowledgment
// DISPLAY_MSG          S10F3           Display Message on Equipment
// EVENT_REPORT         S6F11           Event Occurs on Equipment
// EVENT_SETUP          S2F37           Enable/Disable Data Collection by Event
// INITIALIZE           SxFx            Initialize Logical Communications
// MACH_CMD             S2F49           Remote Equipment Command
// MOVE                 S2F49           Request Transfer of a Transfer Group
// PAUSE                S2F49           Pause Processing
// RES_ACTIVATE         S15F21          Resource Activate
// RES_DEACTIVATE       S15F21          Resource Deactivate
// RES_DELETE           S7F17           Resource Delete
// RES_LIST             S7F19           Resource List
// RES_QUERY            S7F5            Resource Query
// RES_REQUEST          S7F1            Resource Transfer Request
// RES_TRANSFER         S7F3            Resource Transfer
// RESTART              S2F49           Restart Logical Communications
// RESUME               S2F49           Resume Processing
// SET_PARAM            S15F21          Set Resource Parameters
// START                S2F49           Start Processing
// STATUS_LIST          S1F11           Status Variable List
// STATUS_QUERY         S1F3            Query for Current Status
// TRACE_REPORT         S6F1            Data Collection Based On Time
// TRACE_SETUP          S2F23           Enable Data Collection on Time
//

//



///////////////////////////////////////////////////////////////////////////////
namespace jyugem { namespace cell
{
using namespace boost::spirit;
using namespace std;

typedef map<std::string, std::string, less<std::string> > dictionary;

struct cmd_parser : boost::spirit::symbols<int>
{
    cmd_parser()
    {
        add ("abort",          VFEI_ABORT)
            ("alarm_report",   VFEI_ALARM_REPORT)
            ("alarm_setup",    VFEI_ALARM_SETUP)
            ("cmd_ack",        VFEI_CMD_ACK)
            ("display_msg",    VFEI_DISPLAY_MSG)
            ("event_report",   VFEI_EVENT_REPORT)
            ("event_setup",    VFEI_EVENT_SETUP)
            ("initialize",     VFEI_INITIALIZE)
            ("mach_cmd",       VFEI_MACH_CMD)
            ("move",           VFEI_MOVE)
            ("pause",          VFEI_PAUSE)
            ("res_activate",   VFEI_RES_ACTIVATE)
            ("res_deactivate", VFEI_RES_DEACTIVATE)
            ("res_delete",     VFEI_RES_DELETE)
            ("res_list",       VFEI_RES_LIST)
            ("res_query",      VFEI_RES_QUERY)
            ("res_request",    VFEI_RES_REQUEST)
            ("res_transfer",   VFEI_RES_TRANSFER)
            ("restart",        VFEI_RESTART)
            ("resume",         VFEI_RESUME)
            ("set_param",      VFEI_SET_PARAM)
            ("start",          VFEI_START)
            ("status_list",    VFEI_STATUS_LIST)
            ("status_query",   VFEI_STATUS_QUERY)
            ("trace_report",   VFEI_TRACE_REPORT)
            ("trace_setup",    VFEI_TRACE_SETUP);
    }
};
const cmd_parser  command_p;

struct format_parser : boost::spirit::symbols<int>
{
    format_parser()
    {
        add ("a", 020)("bl", 011)("b", 010)
            ("u1", 051)("u2", 052)("u4", 054)("u8", 050)
            ("i1", 031)("i2", 032)("i4", 034)("i8", 030)
            ("f4", 044)("f8", 040);
    }
};
const format_parser  format_p;

///////////////////////////////////////////////////////////////////////////
//
//  vfei grammar
//
///////////////////////////////////////////////////////////////////////////
template <typename ActionsT>
struct vfei_grammar : public grammar<vfei_grammar<ActionsT> >
{
    vfei_grammar(ActionsT& actions_) : actions(actions_) {}

    template <typename ScannerT>
    struct definition
    {
        definition(vfei_grammar const& self)
        {
          host_message
            = (cmd_item >> mid_item >> mty_item >> !tid_item
            >> *(data_item [self.actions.link]) >> *space_p)
               [self.actions.vfei]
            ;

          cmd_item
            = (str_p("CMD") >> '/' >> 'A' >> '=' >> cmd_string
            >> *space_p) [self.actions.cmd]
            ;

          mid_item
            = (str_p("MID") >> '/' >> 'A' >> '=' >> item_string
            >> *space_p) [self.actions.mid]
            ;

          mty_item
            = str_p("MTY") >> '/' >> 'A' >> '=' >> ch_p('"')
            >> (ch_p('C') | 'R' | 'E') [self.actions.mty]
            >> ch_p('"') >> *space_p
            ;

          tid_item
            = str_p("TID") >> '/' >> str_p("U4") >> '='
            >> uint_p [self.actions.tid]
            >> *space_p
            ;

          objspec_item
            = (str_p("OBJTYPE") >> '/' >> 'A' >> '=' >> string_tk
            >> *space_p) [self.actions.objspec]
            ;

          data_item
            = ( list_item
            | array_item
            | (identifier >> '/' >> format_code
            >> ch_p('=') [self.actions.ibegin]
            >> data_item_value >> *space_p) [self.actions.value_]
            ) [self.actions.iend]
            ;

          array_item
            = identifier >> '/' >> format_code >> '['
            >> uint_p [self.actions.memberq]
            >> ']' >> ch_p('=') [self.actions.abegin]
            >> '[' >> *(data_item_value [self.actions.array_])
            >> ']' >> *space_p
            ;

          list_item
            = identifier >> '/' >> 'L' >> '['
            >> uint_p [self.actions.memberq] >> ']'
            >> ch_p('=') [self.actions.lbegin]
            >> '[' >> *(data_item [self.actions.list_])
            >> ']' >> *space_p
            ;

          space
            = *(blank_p | comment)
            ;

          comment
            = +(comment_p("//") | comment_p("/*", "*/")) >> *space_p
            ;

          cmd_string
            = confix_p('"', cmd_code, '"')
            >> *space_p
            ;

          cmd_code
            = as_lower_d
              [
                command_p [self.actions.cmdid]
              ]
            ;

          format_code
            = as_lower_d
              [
                format_p [self.actions.fmt]
              ]
            ;

          item_string
            = confix_p('"', identifier, '"')
            >> *space_p
            ;

          identifier
            = ((alpha_p | '_') >> *(alnum_p | '_'))
              [self.actions.indentifier]
            ;

          data_item_value
            = (string_tk | boolean | number  )
            ;

          string_tk
            = confix_p('"', *c_escape_ch_p, '"') [self.actions.strval]
            >> *space_p
            ;

          boolean
            = as_lower_d["true"]    [self.actions.trueval]
            | as_lower_d["false"]   [self.actions.falseval]
            >> *space_p
            ;

          number
            = ( (as_lower_d["0x"] >> hex_p  [self.actions.intval])
            | strict_real_p                 [self.actions.realval]
            | epsilon_p('0') >> oct_p       [self.actions.intval]
            | uint_p                        [self.actions.uintval]
            | int_p                         [self.actions.intval]
            ) [self.actions.numstr] >> *space_p
            ;
        }

        rule<ScannerT>
            host_message, cmd_item, mid_item, mty_item, tid_item,
            objspec_item, data_item, cmd_string, item_string, identifier,
            cmd_code, format_code, data_item_value, array_item,
            list_item, space, comment, number, boolean, string_tk
            ;

        rule<ScannerT> const&
        start() const { return host_message; }
    };

    ActionsT&   actions;
};

} } // *namespace* jyugem::cell

#endif // JYUGEM_CELL_VFEI_PARSER_H

