// $Id$

//=============================================================================
/**
 *  @file    HSEventTraverser.h
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "HSEventTraverser.h"
#include "HSEventManager.h"
#include "HSServer.h"
#include "jyugem/gem/JGSpecification.h"
#include "jyugem/gem/JGObjectManager.h"
#include "jyugem/gem/JGClass.h"
#include "jyugem/gem/JGAttribute.h"
#include "jyugem/gem/JGVariable.h"
#include "BS2Message.h"
#include "BS2ListItem.h"
#include "BS2Item.h"
#include "BS2DeclAtoms.h"
#include "b_varinfo.h"
#include "xercesc/util/XMLString.hpp"
#include "xercesc/util/Base64.hpp"
XERCES_CPP_NAMESPACE_USE

static const BCHAR * _formatToSchema(int format);
static int _toBase64(BYTE * bin, size_t size, std::string& buf64);

static BCHAR * _unknown_state = _TX("\"UNKNOWN\"");

//-----------------------------------------------------------------------------
// Constructor/Destoructor
//-----------------------------------------------------------------------------
HSEventTraverser::HSEventTraverser(HSEventManager * mngr, int sf)
        : BS2Traverser(), m_manager(mngr), m_sf(sf), m_xml(""), m_xmlmsg(""),
          m_event(NULL), m_report(NULL), m_variable(NULL),
          m_rptnum(0), m_varnum(0), m_first(true), m_nestq(0), m_objtype(NULL),
          m_attribute(NULL), m_attrnum(0), m_baseclmn("")
{
    JGObjectManager * objmngr = mngr->equipment()->getObjectManager();
    m_spec = objmngr->specification();
}

//-----------------------------------------------------------------------------
// Add value string to xml string
//-----------------------------------------------------------------------------
int HSEventTraverser::addItemValue(JGvalue& secsval, const string& itemName,
                                   int reqFormat, string& xmlbuf)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::addItemValue");

    //TRACE_DEBUG((_TX("Value: [%s] item = %s, format = %o(%o)\n"),
    //            secsval.toString().c_str(), itemName.c_str(), itemFormat,
    //            secsval.format()));

    const char * formatStr;
    int itemFormat = secsval.format();

    xmlbuf += m_baseclmn;
    xmlbuf += "    <";
    xmlbuf += itemName;
    xmlbuf += " xsi:type=\"";

    if (itemFormat & ATOM_ARRAY)
    {
        BCHAR sizebuf[64];
        sprintf(sizebuf, _TX("%d"), secsval.m_q);

        formatStr = _formatToSchema(itemFormat & 0x3F);
        if (formatStr == NULL)
        {
            TRACE_ERROR((_TX("Invalid format: item = %s, format = %o\n"),
                        itemName.c_str(), itemFormat));
            return -1;
        }
        xmlbuf += "ns2:Array\" ns2:arrayType=\"";
        xmlbuf += formatStr;
        xmlbuf += "[";
        xmlbuf += sizebuf;
        xmlbuf += "]\">";
        xmlbuf += secsval.toString();  // must convert to soap value
        xmlbuf += "</";
        xmlbuf += itemName;
        xmlbuf += ">\n";
    }
    else if (itemFormat == ATOM_BINARY)
    {
        BCHAR sizebuf[64];
        sprintf(sizebuf, _TX("%d"),  secsval.size());
        if (secsval.size() <= 1)
        {
            xmlbuf += "xsd:unsignedByte\">";
            xmlbuf += secsval.toString();
        }
        else
        {
            std::string buf64;
            _toBase64(secsval.m._bin, secsval.size(), buf64);
            xmlbuf += "xsd:base64\">";
            xmlbuf += buf64;
        }
        xmlbuf += "</";
        xmlbuf += itemName;
        xmlbuf += ">\n";
    }
    else
    {
        formatStr = _formatToSchema(itemFormat);
        if (formatStr == NULL)
        {
            TRACE_ERROR((_TX("Invalid format: item = %s, format = %o\n"),
                        itemName.c_str(), itemFormat));
            return -1;
        }
        xmlbuf += formatStr;
        xmlbuf += "\">";

        //if (itemFormat == ATOM_ASCII || itemFormat == ATOM_JIS)
        //{
        //    // xmlbuf += "\"";
        //    xmlbuf += secsval.toString();
        //   // xmlbuf += "\"";
        //}
        //else
        //{
            xmlbuf += secsval.toString();
        //}
        xmlbuf += "</";
        xmlbuf += itemName;
        xmlbuf += ">\n";
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Add enum value to xml string
//-----------------------------------------------------------------------------
int HSEventTraverser::addEnumValue(JGvalue& secsval, const string& itemName,
                                   b_enumerator * enumptr, string& xmlbuf)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::addEnumValue");

    //TRACE_DEBUG((_TX("Value: [%s] item = %s, format = %o\n"),
    //            secsval.toString().c_str(), itemName.c_str(),
    //            secsval.format()));

    xmlbuf += m_baseclmn;
    xmlbuf += "    <";
    xmlbuf += itemName;
    xmlbuf += " xsi:type=\"";

    int itemFormat = secsval.format();

    if (itemFormat & ATOM_ARRAY)
    {
        BCHAR sizebuf[64];
        sprintf(sizebuf, _TX("%d"),  secsval.m_q);

        xmlbuf += "ns2:Array\" ns2:arrayType=\"xsd:string[";
        xmlbuf += sizebuf;
        xmlbuf += "]\">\n";
        for (size_t j = 0; j < secsval.m_q; j++)
        {
            xmlbuf += m_baseclmn;
            xmlbuf += "      ";
            xmlbuf += "<item xsi:type=\"xsd:string\">";
            int num = secsval.getIntAt(j);

            string enumstr("");
            bool retval = enumptr->numToString(num, enumstr);
            if (retval == false)
            {
                TRACE_ERROR((_TX("Illegal enumerate data: [%s] item = %s, value = %d\n"),
                            enumptr->name_p(), itemName.c_str(), num));
                xmlbuf += _unknown_state;
            }
            else
            {
                xmlbuf += enumstr;
            }
            xmlbuf += "</item>\n";
        }
        xmlbuf += "</";
        xmlbuf += itemName;
        xmlbuf += ">\n";
    }
    else
    {
        xmlbuf += "xsd:string\">";
        int num = secsval.getInt();
        string enumstr("");
        bool retval = enumptr->numToString(num, enumstr);
        if (retval == false)
        {
            TRACE_ERROR((_TX("Illegal enumerate data: [%s] item = %s, value = %d\n"),
                        enumptr->name_p(), itemName.c_str(), num));
            enumstr = _unknown_state;
        }
        xmlbuf += enumstr;
        xmlbuf += "</";
        xmlbuf += itemName;
        xmlbuf += ">\n";
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Add value string to xml string (only value)
//-----------------------------------------------------------------------------
int HSEventTraverser::addValue(JGvalue& secsval, int reqFormat,
                               string& xmlbuf)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::addValue");

    //TRACE_DEBUG((_TX("Value: [%s], format = %o(%o)\n"),
    //            secsval.toString().c_str(), reqFormat, secsval.format()));

    int itemFormat = secsval.format();

    if (itemFormat & ATOM_ARRAY)
    {
        xmlbuf += secsval.toString();
    }
    else if (itemFormat == ATOM_BINARY)
    {
        xmlbuf += secsval.toString();
    }
    else
    {
        if (itemFormat == ATOM_ASCII || itemFormat == ATOM_JIS)
        {
            xmlbuf += secsval.toString();
        }
        else
        {
            xmlbuf += secsval.toString();
        }
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Write remain variables to xml message
//-----------------------------------------------------------------------------
int HSEventTraverser::parseRemains()
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::parseRemains");

    if (m_report != NULL && m_report->size() > m_varnum)
    {
        for (int i = m_varnum; i < m_report->size(); i++)
        {
            if (i > 0)
            {
                m_xml += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
            }
            JGVariable * var = m_report->variable(i);
            if (var->enumerator() != NULL)
            {
                this->addEnumValue(var->getv(), var->name(), var->enumerator(),
                                   m_xml);
            }
            else
            {
                this->addItemValue(var->getv(), var->name(), var->format(),
                               m_xml);
            }
        }
    }

    if (m_event->isOutputFile())
    {
        m_xml += _TX("\n");
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Parse item in message (override)
//-----------------------------------------------------------------------------
int HSEventTraverser::parseItem(BS2Item * item)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::parseItem");

    int result;
    BS2Atom * atom = item->atom();
    if (item->name() ==  _TX("V"))
    {
        //TRACE_DEBUG((_TX("Parse V(alue): num = %d.\n"), m_varnum));

        JGvalue val;

        if (m_first)
        {
            m_first = false;    // don't add space
        }
        else
        {
            m_xml += "  <parameters>\n";
        }
        JGVariable * var;
        if (m_sf == SFCODE(6,11))
        {
            var = m_report->variable(m_varnum++);
        }
        else if (m_sf == SFCODE(6,13))
        {   // variable is found by vid.
            var = m_variable;
        }
        else
        {
            TRACE_ERROR((_TX("Illegal stream/function number.\n")));
            return -3;
        }

        if (var == NULL)
        {
            TRACE_ERROR((_TX("Variable is null (item count over).\n")));
            return -3;
        }

        if (atom->format() != var->format())
        {
            TRACE_ERROR((_TX("Warning: %s differnt value format (%o/%o).\n"),
                        var->charName(), atom->format(), var->format()));
        }
        if (var->enumerator() != NULL)
        {
            result = this->addEnumValue(*atom, var->name(), var->enumerator(), m_xml);
        }
        else
        {
            result = this->addItemValue(*atom, var->name(), var->format(), m_xml);
        }
        if (result < 0)
        {
            return result;
        }
    }
    else if (item->name() == _TX("RPTID"))
    {
        JGid rptid;
        atom->get(rptid);

        //TRACE_DEBUG((_TX("Parse RPTID: %s.\n"), rptid.toString().c_str()));

        m_report = m_manager->findReport(rptid);
        if (m_report == NULL)
        {
            TRACE_ERROR((_TX("RPTID(%s) not found.\n"), rptid.toString().c_str()));
            return -5;            // report is not found
        }
        m_xml += "  <gem:report>\n";
        m_varnum = 0;
    }
    else if (item->name() == _TX("VID"))  // for S6F13
    {
        JGid vid;
        atom->get(vid);

        //TRACE_DEBUG((_TX("Parse VID: %s.\n"), vid.toString().c_str()));

        m_variable = m_manager->equipment()->variable(vid);
        if (m_variable == NULL)
        {
            TRACE_ERROR((_TX("VID(%s) not found.\n"), vid.toString().c_str()));
            return -5;            // vid is not found
        }
    }
    else if (item->name() == _TX("CEID"))
    {
        JGid ceid;
        atom->get(ceid);

        //TRACE_DEBUG((_TX("Parse CEID: %s.\n"), ceid.toString().c_str()));

        m_event = m_manager->find(ceid);
        if (m_event == NULL)
        {
            TRACE_ERROR((_TX("CEID(%s) not found.\n"), ceid.toString().c_str()));
            return -4;             // ceid is not exist
        }
        m_xmlmsg = m_baseclmn;
        m_xmlmsg += "<gem:event xmlns:gem=\"urn:www.jyugem.com\" ";
        m_xmlmsg += "SOAP-ENV:encodeingStyle=\"http://www.w3.org/2001/09/soap-encoding\">\n";
        m_xmlmsg += "  <gem:ceid xsi:type=\"xsd:string\">";
        m_xmlmsg += m_event->name();
        m_xmlmsg += "</gem:ceid>\n";
    }
    else if (item->name() == _TX("DATAID"))
    {
        atom->get(m_dataid);

        //TRACE_DEBUG((_TX("Parse DATAID: %s.\n"), m_dataid.toString().c_str()));
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Parse list item in message (override)
//-----------------------------------------------------------------------------
int HSEventTraverser::beginList(BS2ListItem * listitem)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::beginList");

    this->BS2Traverser::beginList(listitem);

    int result = 0;
    int memberNum;
    bool is_listitem = false;
    BS2List * listatom;
    char * typedisp;

    if (listitem->isListItem())
    {
        memberNum = listitem->memberq();
        is_listitem = true;
        typedisp = "Item";
    }
    else if (listitem->isList())
    {
        listatom = (BS2List *)((BS2Item *)listitem)->atom();
        is_listitem = false;
        typedisp = "Atom";
    }
    else
    {
        TRACE_DEBUG((_TX("Not List Item or Atom.\n")));
        return -1;
    }


    //TRACE_DEBUG((_TX("List %s [%d]: nest = %d.\n"), typedisp, memberNum, m_nest));

    if (! is_listitem)
    {
        memberNum = listatom->getList().size();         // Need to hide member data;

        if (m_nest == VLIST_DEPTH)
        {   // list of v
            m_variable = m_report->variable(m_varnum++);
            if (m_variable == NULL)
            {
                TRACE_ERROR((_TX("Variable count over (%d: %d).\n"),
                        memberNum, m_varnum));
                return -1;
            }
            result = parseListAtom(m_variable, listatom, m_xml);
        }
        else
        {
            TRACE_ERROR((_TX("Illegal nested level (%d).\n"), m_nest));
            return -1;
        }
    }

    return result;
}

//-----------------------------------------------------------------------------
// Parse list item in message (override)
//-----------------------------------------------------------------------------
int HSEventTraverser::endList(BS2ListItem * listitem)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::endList");

    if (m_nest == VLIST_END)
    {   // list of value
        if (! m_first)
        {
            m_xml += m_baseclmn;
            m_xml += "  </gem:report>\n";
        }
        m_xml += m_baseclmn;
        m_xml += "</gem:event>\n";
    }
    else if (m_nest == VLIST_DEPTH)
    {   // list of value
    }
    else if (m_nest > VLIST_DEPTH)
    {   // list of object
    }

    this->BS2Traverser::endList(listitem);
    return 0;
}

//-----------------------------------------------------------------------------
// Parse list item in message (override)
//-----------------------------------------------------------------------------
int HSEventTraverser::beginValueList(BS2Item * item)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::beginValueList");
    this->BS2Traverser::beginValueList(item);

    int result = 0;
    // Check List type  that is BS2ListItem or BS2Item(BS2List)
    int  memberNum = 0;
    if (item->isList())
    {   // List of atom(value)
        BS2List * listatom = (BS2List *)item->atom();
        memberNum = listatom->getList().size();     // Need to hide member data;

        if (m_nest == VLIST_DEPTH)
        {   // list of v
            m_variable = m_report->variable(m_varnum++);
            if (m_variable == NULL)
            {
                TRACE_ERROR((_TX("Variable count over (%d: %d).\n"),
                        memberNum, m_varnum));
                return -1;
            }
            result = parseListAtom(m_variable, listatom, m_xml);
        }
        else
        {
            TRACE_ERROR((_TX("Illegal nested level (%d).\n"), m_nest));
            return -1;
        }
    }
    else
    {   // Not list ?
        TRACE_ERROR((_TX("Argument item is not list.\n")));
        return -1;
    }
    return result;
}

//-----------------------------------------------------------------------------
// Top of Parse Variable
//-----------------------------------------------------------------------------
int HSEventTraverser::parseListAtom(JGVariable * var, BS2List * listatom,
                                    string& xmlbuf)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::parseListAtom");

    int result;
    BS2Atoms& atoms = listatom->getList();
    int memberNum = atoms.size();
    if (memberNum == 0)
    {
        return 0;                   // ignore empty list
    }
    JGClass * clazz = NULL;

    if (var->substance() == "vector")
    {
        if (var->objtype() != NULL)
        {
            if (m_first)
            {
                m_first = false;    // don't add space
            }
            else
            {
                xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
            }
            addListHeader(var->name(), ATOM_LIST, memberNum, xmlbuf);
            clazz = var->objtype();
            for (int i = 0; i < memberNum; i++)
            {
                BS2Atom * mbr_atom = atoms[i];
                if (! mbr_atom->isList())
                {
                    TRACE_ERROR((_TX("Item must be list type (recieved primitive).\n")));
                    return -1;
                }
                BS2Atoms& objv = ((BS2List *)mbr_atom)->getList();

                if (i > 0)
                {
                    xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
                }
                addListHeader(clazz->userName(), ATOM_LIST, objv.size(), xmlbuf);
                result = parseClassAtom(clazz, objv, xmlbuf);
                if (result < 0)
                {
                    return result;
                }
                addListTail(xmlbuf);
            }
            addListTail(xmlbuf);
        }
        else                    // Can't support vector of vector
        {   // primitive
            if (m_first)
            {
                m_first = false;    // don't add space
            }
            else
            {
                xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
            }
            addListHeader(var->name(), var->format(), memberNum, xmlbuf);
            bool restrain = false;
            for (int i = 0; i < memberNum; i++)
            {
                BS2Atom * mbr_atom = atoms[i];
                if (mbr_atom->isList())
                {
                    TRACE_ERROR((_TX("Item must be primitive type (recieved list).\n")));
                    return -1;
                }
                if ((! restrain) && mbr_atom->format() != var->format())
                {
                    TRACE_ERROR((_TX("Warning: %s differnt value format (%o/%o).\n"),
                        var->charName(), mbr_atom->format(), var->format()));
                    restrain = true;
                }

                if (i > 0)
                {
                    xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
                }

                result = this->addValue(*mbr_atom, var->format(), xmlbuf);
                if (result < 0)
                {
                    return result;
                }
            }
            addListTail(xmlbuf);
        }
    }
    else if (var->substance() == "object")
    {
        if (m_first)
        {
            m_first = false;    // don't add space
        }
        else
        {
            xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
        }
        addListHeader(var->name(), ATOM_LIST, memberNum, xmlbuf);
        clazz = var->objtype();
        if (clazz == NULL)
        {
            TRACE_ERROR((_TX("Variable(%s) has no objtype.\n"),
                    var->name().c_str()));
            return -1;
        }
        result = parseClassAtom(clazz, atoms, xmlbuf);

        addListTail(xmlbuf);
    }
    else
    {
        TRACE_ERROR((_TX("Illegal variable type(%s).\n"),
                    var->substance().c_str()));
        return -1;
    }
    return result;
}

//-----------------------------------------------------------------------------
// Parse Class/Item
//-----------------------------------------------------------------------------
int HSEventTraverser::parseClassAtom(JGClass * clazz, BS2Atoms& atoms,
                                     string& xmlbuf)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::parseClassAtom");

    int result;

    int member_num1 = clazz->attrSize();
    int member_num2 = atoms.size();
    if (member_num1 != member_num2)
    {
         // Log: warning
        TRACE_ERROR((_TX("Warning: Un-matched attribute count(%d / %d).\n"),
                    member_num1, member_num2));
    }
    int memberNum = (member_num2 > member_num1) ? member_num1 : member_num2;

    for (int i = 0; i < memberNum; i++)
    {
        if (i > 0)
        {
            xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
        }
        JGAttribute * mbr_attr = (JGAttribute *)clazz->at(i);
        BS2Atom * atom = atoms[i];
        result = parseAttrAtom(mbr_attr, atom, xmlbuf);
        if (result < 0)
        {
            break;
        }
    }
    if (memberNum < member_num1)
    {
        for (int l = memberNum; l < member_num1; l++)
        {
            if (l > 0)
            {
                xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
            }
            JGAttribute * mbr_attr = (JGAttribute *)clazz->at(l);
            result = this->addItemValue(mbr_attr->getInitValue(),
                         mbr_attr->userName(), mbr_attr->type(), xmlbuf);
            if (result < 0)
            {
                break;
            }
        }
    }

    return result;
}

//-----------------------------------------------------------------------------
// Parse Attribute/Item
//-----------------------------------------------------------------------------
int HSEventTraverser::parseAttrAtom(JGAttribute * attr, BS2Atom * atom,
                                    string& xmlbuf)
{
    TRACE_FUNCTION(TRL_LOW, "HSEventTraverser::parseAttrAtom");

    int result;
    int memberNum;

    if (attr->isVector())
    {
        if (! atom->isList())
        {
            TRACE_ERROR((_TX("Received data is not list type.\n")));
            return -1;
        }
        BS2Atoms& atoms = ((BS2List *)atom)->getList();
        memberNum = atoms.size();
        if (memberNum == 0)
        {
            return 0;                   // ignore empty list
        }

        JGAttribute * nested = ((JGVector *)attr)->member();
        if (nested->isStruct() || nested->isVector())
        {
            addListHeader(attr->userName(), ATOM_LIST, memberNum, xmlbuf);
            for (size_t i = 0; i < atoms.size(); i++)
            {
                if (i > 0)
                {
                    xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
                }
                BS2Atom * atom = atoms[i];
                result = this->parseAttrAtom(nested, atom, xmlbuf);
                if (result < 0)
                {
                    break;
                }
            }
            this->addListTail(xmlbuf);
        }
        else
        {   // primitive
            addListHeader(attr->userName(), nested->type(), memberNum, xmlbuf);
            bool restrain = false;
            for (size_t j = 0; j < atoms.size(); j++)
            {
                if (j > 0)
                {
                    xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
                }
                BS2Atom * atom = atoms[j];

                if ((! restrain) && atom->format() != nested->type())
                {
                    TRACE_ERROR((_TX("Warning: %s differnt value format (%o/%o).\n"),
                        nested->userName().c_str(), atom->format(), nested->type()));
                    restrain = true;
                }
#if 0
                result = this->addValue(*atom, nested->type(), xmlbuf);
#else
                result = this->addItemValue(*atom, attr->userName(), nested->type(), xmlbuf);
#endif
                if (result < 0)
                {
                    break;
                }
            }
            addListTail(xmlbuf);
        }
    }
    else if (attr->isStruct())
    {
        if (! atom->isList())
        {
            TRACE_ERROR((_TX("Received data is not list type.\n")));
            return -1;
        }
        BS2Atoms& atoms = ((BS2List *)atom)->getList();
        memberNum = atoms.size();
        JGStruct * ztruct = (JGStruct *)attr;
        int member_num1 = ztruct->length();
        int member_num2 = atoms.size();
        if (member_num1 != member_num2)
        {
            TRACE_ERROR((_TX("Warning: Un-matched attribute count(%d / %d).\n"),
                    member_num1, member_num2));
        }
        int memberNum = (member_num2 > member_num1) ? member_num1 :
                                                      member_num2;
        addListHeader(ztruct->userName(), ATOM_LIST, memberNum, xmlbuf);
        for (int k = 0; k < memberNum; k++)
        {
            if (k > 0)
            {
                xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
            }
            JGAttribute * mbr_attr = ztruct->at(k);
            BS2Atom * atom = atoms[k];
            result = this->parseAttrAtom(mbr_attr, atom, xmlbuf);
            if (result < 0)
            {
                break;
            }
        }
        if (memberNum < member_num1)
        {
            for (int l = memberNum; l < member_num1; l++)
            {
                if (l > 0)
                {
                    xmlbuf += (m_event->isOutputFile()) ? _TX("\n") : _TX(" ");
                }
                JGAttribute * mbr_attr = ztruct->at(l);
                result = this->addItemValue(mbr_attr->getInitValue(),
                             mbr_attr->userName(), mbr_attr->type(), xmlbuf);
                if (result < 0)
                {
                    break;
                }
            }
        }
        addListTail(xmlbuf);
    }
    else
    {
        if (atom->isList())
        {
            TRACE_ERROR((_TX("Data must be primitive type(recieved list).\n")));
            return -1;
        }
        if (atom->format() != attr->type())
        {
            TRACE_ERROR((_TX("Warning: %s differnt value format (%o/%o).\n"),
                attr->charName(), atom->format(), attr->type()));
        }
        string itemName = attr->userName();
        result = this->addItemValue(*atom, itemName, attr->type(), xmlbuf);
    }
    return result;
}


//-----------------------------------------------------------------------------
// Format code to string (static)
//-----------------------------------------------------------------------------
static BCHAR * _schema_const[64] =
{
    _TX("gem:Array"),            //  0: ATOM_LIST
    NULL,                        //  1:
    NULL,                        //  2:
    NULL,                        //  3:
    NULL,                        //  4:
    NULL,                        //  5:
    NULL,                        //  6:
    NULL,                        //  7:
    _TX("xsd:base64"),           //  8: ATOM_BINARY
    _TX("xsd:boolean"),          //  9: ATOM_BOOLEAN
    NULL,                        // 10:
    NULL,                        // 11:
    NULL,                        // 12:
    NULL,                        // 13:
    NULL,                        // 14:
    NULL,                        // 15:
    _TX("xsd:string"),           // 16: ATOM_ASCII
    _TX("xsd:string"),           // 17: ATOM_JIS
    _TX("xsd:string"),           // 18: ATOM_UNICODE
    NULL,                        // 19:
    NULL,                        // 20:
    NULL,                        // 21:
    NULL,                        // 22:
    NULL,                        // 23:
    _TX("xsd:long"),             // 24: ATOM_INT8
    _TX("xsd:byte"),             // 25: ATOM_INT1
    _TX("xsd:short"),            // 26: ATOM_INT2
    NULL,                        // 27:
    _TX("xsd:integer"),          // 28: ATOM_INT4
    NULL,                        // 29:
    NULL,                        // 30:
    NULL,                        // 31:
    _TX("xsd:double"),           // 32: ATOM_FLOAT8
    NULL,                        // 33:
    NULL,                        // 34:
    NULL,                        // 35:
    _TX("xsd:float"),            // 36: ATOM_FLOAT4
    NULL,                        // 37:
    NULL,                        // 38:
    NULL,                        // 39:
    _TX("xsd:unsignedLong"),     // 40: ATOM_UINT8
    _TX("xsd:unsignedByte"),     // 41: ATOM_UINT1
    _TX("xsd:unsignedShort"),    // 42: ATOM_UINT2
    NULL,                        // 43:
    _TX("xsd:unsignedInt"),      // 44: ATOM_UINT4
    NULL,                        // 45:
    NULL,                        // 46:
    NULL,                        // 47:
    NULL,                        // 48:
    NULL,                        // 49:
    NULL,                        // 50:
    NULL,                        // 51:
    NULL,                        // 52:
    NULL,                        // 53:
    NULL,                        // 54:
    NULL,                        // 55:
    NULL,                        // 56:
    NULL,                        // 57:
    NULL,                        // 58:
    NULL,                        // 59:
    NULL,                        // 60:
    NULL,                        // 61:
    NULL,                        // 62:
    NULL,                        // 63:
};


const BCHAR * _formatToSchema(int format)
{
    if (format >= 64)
        return NULL;

    BCHAR * name = _schema_const[format];
    return name;
}

int _toBase64(BYTE * bin, size_t size, std::string& buf64)
{
    size_t outSize;
    XMLByte* base64p = Base64::encode(bin, size, &outSize);

    buf64 = (char *)base64p;
    XMLString::release(&base64p);
    return (int)outSize;
}
