// $Id$

//=============================================================================
/**
 *  @file    HSMainManager.cpp
 *
 *  @author Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "HSMainManager.h"
#include "HSDevice.h"
#include "HSInfoManager.h"
#include "HSAlarmManager.h"
#include "HSEventManager.h"
#include "HSServer.h"
#include "jyugem/gem/JGTimeModule.h"
#include "jyugem/gem/JGObjectManager.h"
#include "jyugem/gem/JGFile.h"
#include "jyugem/gem/rcmd.h"
#include "c_wrapper.h"
#include "b_varinfo.h"
using namespace jyugem::cell;

//-----------------------------------------------------------------------------
//
// Host Simulator Manager
//
//-----------------------------------------------------------------------------
static HSMainManager * _manager = NULL;

//-----------------------------------------------------------------------------
// Return own.
//-----------------------------------------------------------------------------
HSMainManager * HSMainManager::instance()
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::instance");
    if (_manager == NULL)
    {
        _manager = new HSMainManager();
        JGEquipmentManager::instance(_manager);
    }
    return _manager;
}

//-----------------------------------------------------------------------------
HSMainManager * HSMainManager::instance(const BCHAR * name, const BCHAR * logname)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::instance");
    if (_manager == NULL)
    {
        _manager = new HSMainManager(name, logname);
        JGEquipmentManager::instance(_manager);
    }
    return _manager;
}

//-----------------------------------------------------------------------------
HSMainManager * HSMainManager::instance(HSMainManager * app)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::instance");
    if (_manager == NULL)
    {
        if (app != NULL)
        {
            _manager = new HSMainManager;
        }
        else
        {
            _manager = app;
        }
        JGEquipmentManager::instance(_manager);
    }
    return _manager;
}

//-----------------------------------------------------------------------------
// Initialize
//-----------------------------------------------------------------------------
int HSMainManager::init(void * param)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::init");

    this->start_log();

    vector<EquipmentParam> * equiparam =
            reinterpret_cast< vector<EquipmentParam> * >(param);
    //
    // Create equipments
    //
    if (equiparam == NULL)
    {
        m_represent = new HSServer();  //*
        m_represent->setDSNName(_TX("gem.edb"));
        if (m_root.size() == 0)
        {
            char * rootdir = ACE_OS::getenv("GEM_ROOT");
            if ((rootdir == NULL) || (_tcslen(rootdir) == 0))
            {
                rootdir = ".";
            }
            m_root = rootdir;
            m_root += DIR_SEPARATOR_STRING;
            m_root += _TX("jyugem");
        }
        m_represent->setRootDirectory(m_root);
        if (m_represent->init() < 0)
        {
            return -1;
        }
    }
    else
    {
        m_represent = NULL;
        for (size_t i = 0; i < equiparam->size(); i++)
        {
            HSServer * equip = new HSServer((*equiparam)[i].m_devname); //*
            string dbname = (*equiparam)[i].m_devname;
            dbname += _TX(".edb");
            equip->setDSNName(dbname);
            string topdir = _TX("");
            if (m_root.size() > 0)
            {
                topdir = m_root;
            }
            topdir += DIR_SEPARATOR_STRING;
            topdir += (*equiparam)[i].m_root;
            equip->setRootDirectory(topdir);

            if (equip->init((*equiparam)[i].m_config) < 0)
            {
                return -1;
            }
            if (equiparam->size() == 1)
            {
                m_represent = equip;
            }
            else
            {
                m_equipments.insert(std::make_pair((*equiparam)[i].m_devname, equip));
            }
        }
    }

    // this->act_equipments();

    ACE_OS::sleep(1);

    return BEE_SUCCESS;
}

//------------------------------------------------------------------------------
// Put vfei message to mailbox.
//------------------------------------------------------------------------------
int HSMainManager::begin(void * param)
{
    this->act_equipments(param);
    this->open();
    return 0;
}

//------------------------------------------------------------------------------
// Store reporting message in file
//------------------------------------------------------------------------------
int HSMainManager::storeReport(const string& filename, const string& vfeibody,
                             UINT top)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::storeReport");

    int result = 0;
    JGFile rptFile(filename);

    if ((result = rptFile.open()) < 0)
    {
        return result;
    }
    if (top == 0)
    {
        result = rptFile.write(vfeibody);
    }
    else
    {
        result = rptFile.write((BYTE *)(vfeibody.c_str() + top),
                               vfeibody.size() - top);
    }
    rptFile.close();
    return result;
}


//------------------------------------------------------------------------------
// Put vfei message to mailbox.
//------------------------------------------------------------------------------
const char * HSMainManager::getErrorText(int errcode)
{
    return 0;
}
//------------------------------------------------------------------------------
// Parse xml message.
//------------------------------------------------------------------------------
int HSMainManager::parse_xml(const char * xmlcmd, size_t cmdsize)
{
    return 0;
}

//------------------------------------------------------------------------------
// Put vfei message.
//------------------------------------------------------------------------------
extern rcmd * parse_vfei_string(HSMainManager * vfeimngr, const char * cmdstr);

int HSMainManager::parse_vfei(const char * vfeicmd, size_t cmdsize)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::parse_vfei");

    rcmd * vfei_rcmd = parse_vfei_string(this, vfeicmd);
    if (vfei_rcmd == NULL)
    {
        return -1;
    }
    this->request(vfei_rcmd);
    return 0;
}

//------------------------------------------------------------------------------
// Request host command to equipment.
//------------------------------------------------------------------------------
int HSMainManager::request(host_cmd * cmd)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::request");
    HSServer * equip = dev_server(cmd->m_devid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), cmd->m_devid.c_str()));
        return -1;
    }
    equip->request(*cmd);
    delete cmd;               // *
    return 0;
}

//------------------------------------------------------------------------------
// Put vfei message to mailbox.
//------------------------------------------------------------------------------
int HSMainManager::put(ACE_Message_Block * mb, ACE_Time_Value * tv)
{
    return this->putq(mb, tv);
}

//------------------------------------------------------------------------------
// VFEI Command
//------------------------------------------------------------------------------
int HSMainManager::abort(const std::string& mid, int tid, const std::string& port,
                         int level)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::abort");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->abort(tid, port, level);
}

//------------------------------------------------------------------------------
int HSMainManager::alarm_report(const std::string& mid, int alid, BYTE alstate,
                                const std::string& altx, BYTE alcd,
                                int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::alarm_report");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->alarm_report(alid, alstate, altx, alcd, errcode, errtext);
}

//------------------------------------------------------------------------------
int HSMainManager::alarm_setup(const std::string& mid, int tid, bool aled,
                               const std::vector<std::string>& alids)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::alarm_setup");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->alarm_setup(tid, aled, alids);
}

//------------------------------------------------------------------------------
int HSMainManager::command_ack(const std::string& mid, int tid, int errcode,
                               const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::command_ack");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->command_ack(tid, errcode, errtext);
}

//------------------------------------------------------------------------------
int HSMainManager::display(const std::string& mid, int tid,
                           BYTE terminal, const std::string& text)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::display");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->display(tid, terminal, text);
}

//------------------------------------------------------------------------------
int HSMainManager::event_report(const std::string& mid, const std::string& ceid,
                                const std::string& userp, const std::string& file,
                                int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::event_report");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->event_report(ceid, userp, file, errcode, errtext);
}

//------------------------------------------------------------------------------
int HSMainManager::event_setup(const std::string& mid, int tid, bool ceed,
                               const std::vector<std::string>& ceids,
                               const std::vector<std::string>& vids,
                               const std::string& dir)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::event_setup");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->event_setup(tid, ceed, ceids, vids, dir);

}

//------------------------------------------------------------------------------
int HSMainManager::initialize(const std::string& mid, int tid)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::initialize");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->initialize(tid);
}

//------------------------------------------------------------------------------
int HSMainManager::machine_command(const std::string& mid, int tid,
                                   const std::string& cmdtype,
                                   const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::machine_command");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->machine_command(tid, cmdtype, params);
}

//------------------------------------------------------------------------------
int HSMainManager::move_command(const std::string& mid, int tid,
                                const std::string& tgtype,
                                const std::string& tgid,
                                const std::string& src_loc,
                                const std::string& dest_loc,
                                const std::string& src_port,
                                const std::string& dest_port)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::move_command");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->move_command(tid, tgtype, tgid, src_loc, dest_loc, src_port,
                               dest_port);
}

//------------------------------------------------------------------------------
int HSMainManager::pause(const std::string& mid, int tid,
                         const std::string& port)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::pause");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->pause(tid, port);
}

//------------------------------------------------------------------------------
int HSMainManager::resource_activate(const std::string& mid, int tid,
                                     const std::string& type,
                                     const std::vector<std::string>& resids,
                                     const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::resource_activate");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->resource_activate(tid, type, resids, params);
}

//------------------------------------------------------------------------------
int HSMainManager::resource_deactivate(const std::string& mid, int tid,
                                       const std::string& type,
                                       const std::vector<std::string>& resids)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::resource_deactivate");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->resource_deactivate(tid, type, resids);
}

//------------------------------------------------------------------------------
int HSMainManager::resource_delete(const std::string& mid, int tid,
                                   const std::string& type,
                                   const std::vector<std::string>& resids)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::resource_delete");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->resource_delete(tid, type, resids);
}

//------------------------------------------------------------------------------
int HSMainManager::resource_list(const std::string& mid, int tid,
                                 const std::string& type,
                                 const std::vector<std::string>& resids,
                                 const std::string& file,
                                 int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::resource_list");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->resource_list(tid, type, resids, file, errcode, errtext);
}

//------------------------------------------------------------------------------
int HSMainManager::resource_query(const std::string& mid, int tid,
                                  const std::string& type,
                                  const std::vector<std::string>& resids,
                                  const std::string& file)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::abort");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->resource_query(tid, type, resids, file);
}

//------------------------------------------------------------------------------
int HSMainManager::resource_request(const std::string& mid, int tid,
                                    const std::string& type,
                                    const std::vector<std::string>& resids,
                                    const std::string& file)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::resource_request");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->resource_request(tid, type, resids, file);
}

//------------------------------------------------------------------------------
int HSMainManager::resource_transfer(const std::string& mid, int tid,
                                     const std::string& type,
                                     const std::string& resid, size_t length,
                                     const char * body)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::resource_transfer");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->resource_transfer(tid, type, resid, length, body);
}

//------------------------------------------------------------------------------
int HSMainManager::restart(const std::string& mid, int errcode,
                           const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::restart");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->restart(errcode, errtext);
}

//------------------------------------------------------------------------------
int HSMainManager::resume(const std::string& mid, int tid,
                          const std::string& port,
                          const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::resume");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->resume(tid, port, params);
}

//------------------------------------------------------------------------------
int HSMainManager::set_param(const std::string& mid, int tid,
                             const std::string& res_type,
                             const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::set_param");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->set_param(tid, res_type, params);
}

//------------------------------------------------------------------------------
int HSMainManager::start(const std::string& mid, int tid,
                         const std::string& port, const std::string& lotid,
                         int mbc_size, const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::start");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->start(tid, port, lotid, mbc_size, params);
}

//------------------------------------------------------------------------------
int HSMainManager::status_list(const std::string& mid, int tid,
                               const std::string& rpt, const std::string& file,
                               int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::status_list");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->status_list(tid, rpt, file, errcode, errtext);
}

//------------------------------------------------------------------------------
int HSMainManager::status_query(const std::string& mid, int tid,
                                const std::vector<std::string>& vids,
                                const std::string& file)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::status_query");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->status_query(tid, vids, file);
}

//------------------------------------------------------------------------------
int HSMainManager::trace_report(const std::string& mid, int tid, int trace_id,
                                size_t report_length, const std::string& rpt,
                                const std::string& file,
                                int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::trace_report");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->trace_report(tid, trace_id, report_length, rpt, file,
                               errcode, errtext);
}

//------------------------------------------------------------------------------
int HSMainManager::trace_setup(const std::string& mid, int tid, bool enable,
                               int trace_id, size_t frequency, size_t samples,
                               size_t report_size,
                               const std::vector<std::string>& vids,
                               const std::string& dir)
{
    TRACE_FUNCTION(TRL_LOW, "HSMainManager::trace_setup");
    HSServer * equip = dev_server(mid);
    if (equip == NULL)
    {
        TRACE_ERROR((_TX("Equipment not found (%s)\n"), mid.c_str()));
        return -1;
    }
    return equip->trace_setup(tid, enable, trace_id, frequency, samples,
                              report_size, vids, dir);
}


//------------------------------------------------------------------------------
// Start communication in service
//------------------------------------------------------------------------------
int HSMainManager::svc()
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::svc");
    for (;;)
    {
        ACE_Message_Block * mb;

        ACE_Time_Value tv(1, 250 * 1000);
        ACE_Time_Value timeout(ACE_OS::time(0));
        timeout += tv;
        if (this->getq(mb, &timeout) < 0)
        {   // No vfei message
            continue;
        }
        int length = (int)mb->length();
        if (length > 0)
        {
            ACE_ERROR((LM_ERROR, _TX("*** Puts message to host ***\n%s\n"),
                         mb->rd_ptr()));
            //if (m_hostcom->send(mb->rd_ptr(), _tcslen(mb->rd_ptr())) < 0)
            //{
            //    ACE_ERROR((LM_ERROR, _TX("Can't send message to host(%s)\n")));
            //}
        }
        // Free up the buffer memory and the Message_Block.
        mb->release ();

        if (length == 0)
            break;
    }

    return 0;
}
