// $Id$

//=============================================================================
/**
 *  @file    pyhostsim.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL


#include <boost/python.hpp>
#include <boost/python/suite/indexing/vector_indexing_suite.hpp>
namespace python = boost::python;

#include "HSMainManager.h"

static MUTEX_Int _transaction_id(1);
int get_transaction_id()
{
    int result = _transaction_id++;
    return result;
}


namespace jyugem { namespace cell
{

struct Id
{
    std::string m_id;

    typedef std::string::value_type value_type;
    typedef std::string::size_type  size_type;
    typedef std::string::difference_type  difference_type;

    Id(): m_id("0") {}
    Id(std::string s): m_id(s) {}

    std::string repr() const { return m_id; }
    bool operator==(Id const& x) const { return m_id == x.m_id; }
    bool operator!=(Id const& x) const { return m_id != x.m_id; }
};

struct hs_manager_wrapper
{
    HSMainManager * m_manager;

    hs_manager_wrapper()
    {
        m_manager = HSMainManager::instance();
    }
    ~hs_manager_wrapper() {}

    int init()  { return m_manager->init(); }
    int begin() { return m_manager->begin(); }
    int quit()  { return m_manager->quit(); }

    int xml(std::string xml)
    {
        return m_manager->parse_xml(xml.c_str(), xml.size());
    }
    int vfei(std::string vfei)
    {
        return m_manager->parse_vfei(vfei.c_str(), vfei.size());
    }

    int abort(std::string mid, std::string port, int level)
    {
        return m_manager->abort(mid, get_transaction_id(), port, level);
    }
    int alarm_setup(std::string mid, bool aled, std::vector<Id>& ids)
    {
        std::vector<std::string> alids;
        for (size_t i = 0; i < ids.size(); i++)
        {
            alids.push_back(ids[i].m_id);
        }
        return m_manager->alarm_setup(mid, get_transaction_id(), aled, alids);
    }
    int display(std::string mid, int terminal, std::string text)
    {
        return m_manager->display(mid, get_transaction_id(), (BYTE)terminal,
                                  text);
    }
    int event_setup(std::string mid, bool ceed, std::vector<Id>& event_ids,
                            std::vector<Id>& var_ids, std::string dir)
    {
        std::vector<std::string> ceids;
        for (size_t i = 0; i < event_ids.size(); i++)
        {
            ceids.push_back(event_ids[i].m_id);
        }
        std::vector<std::string> vids;
        for (size_t i = 0; i < var_ids.size(); i++)
        {
            vids.push_back(var_ids[i].m_id);
        }
        return m_manager->event_setup(mid, get_transaction_id(), ceed, ceids,
                                      vids, dir);
    }
    int initialize(std::string mid)
    {
        //return m_manager->initialize(mid, get_transaction_id());
        return m_manager->initialize("", get_transaction_id());
    }
    int machine_command(std::string mid, std::string cmdtype,
                        std::vector<any_item>& params)
    {
        return m_manager->machine_command(mid, get_transaction_id(),
                                          cmdtype, params);
    }

    int move_command(std::string mid, std::string tgtype, std::string tgid,
                     std::string src_loc, std::string dest_loc,
                     std::string src_port, std::string dest_port)
    {
        return m_manager->move_command(mid, get_transaction_id(), tgtype,
                                       tgid, src_loc, dest_loc, src_port,
                                       dest_port);
    }
    int pause(std::string mid, std::string port)
    {
        return m_manager->pause(mid, get_transaction_id(), port);
    }
    int resource_activate(std::string mid, std::string type,
                          std::vector<Id>& ids,
                          std::vector<any_item>& params)
    {
        std::vector<std::string> resids;
        for (size_t i = 0; i < ids.size(); i++)
        {
            resids.push_back(ids[i].m_id);
        }
        return m_manager->resource_activate(mid, get_transaction_id(), type,
                                            resids, params);
    }
    int resource_deactivate(std::string mid, std::string type,
                            std::vector<Id>& ids)
    {
        std::vector<std::string> resids;
        for (size_t i = 0; i < ids.size(); i++)
        {
            resids.push_back(ids[i].m_id);
        }
        return m_manager->resource_deactivate(mid, get_transaction_id(), type,
                                              resids);
    }
    int resource_delete(std::string mid, std::string type, std::vector<Id>& ids)
    {
        std::vector<std::string> resids;
        for (size_t i = 0; i < ids.size(); i++)
        {
            resids.push_back(ids[i].m_id);
        }
        return m_manager->resource_delete(mid, get_transaction_id(), type,
                                          resids);
    }
    int resource_query(std::string mid, std::string type,
                       std::vector<Id>& ids, std::string file)
    {
        std::vector<std::string> resids;
        for (size_t i = 0; i < ids.size(); i++)
        {
            resids.push_back(ids[i].m_id);
        }
        return m_manager->resource_query(mid, get_transaction_id(), type,
                                         resids, file);
    }
    int resource_request(std::string mid, std::string type,
                         std::vector<Id>& ids, std::string file)
    {
        std::vector<std::string> resids;
        for (size_t i = 0; i < ids.size(); i++)
        {
            resids.push_back(ids[i].m_id);
        }
        return m_manager->resource_request(mid, get_transaction_id(), type,
                                           resids, file);
    }

    int resource_transfer(std::string mid, std::string type,
                          std::string resid, std::string body)
    {
        return m_manager->resource_transfer(mid, get_transaction_id(), type,
                                            resid, body.size(), body.c_str());
    }

    int resume(std::string mid, std::string port, std::vector<any_item>& params)
    {
        return m_manager->resume(mid, get_transaction_id(), port, params);
    }
    int set_param(std::string mid, std::string port, std::vector<any_item>& params)
    {
        return m_manager->set_param(mid, get_transaction_id(), port, params);
    }
    int start_message(std::string mid, std::string port, std::string lotid,
                      int mbc_size, std::vector<any_item> params)
    {
        //std::vector<any_item> params;
        return m_manager->start(mid, get_transaction_id(), port, lotid,
                                mbc_size, params);
    }
    int status_query(std::string mid, std::vector<Id>& ids, std::string file)
    {
        std::vector<std::string> vids;
        for (size_t i = 0; i < ids.size(); i++)
        {
            vids.push_back(ids[i].m_id);
        }
        return m_manager->status_query(mid, get_transaction_id(), vids, file);
    }
    int trace_setup(std::string mid, bool enable, int trace_id, size_t frequency,
                    size_t samples, size_t report_size, std::vector<Id>& ids,
                    std::string dir)
    {
        std::vector<std::string> vids;
        for (size_t i = 0; i < ids.size(); i++)
        {
            vids.push_back(ids[i].m_id);
        }
        return m_manager->trace_setup(mid, get_transaction_id(), enable, trace_id,
                                      frequency, samples, report_size, vids, dir);
    }

    static void wrap()
    {
        python::scope outer
            = python::class_< hs_manager_wrapper >("EquipmentManager")
            .def("init", &hs_manager_wrapper::init)
            .def("begin", &hs_manager_wrapper::begin)
            .def("quit", &hs_manager_wrapper::quit)
            .def("xml", &hs_manager_wrapper::xml)
            .def("vfei", &hs_manager_wrapper::vfei)
            .def("abort", &hs_manager_wrapper::abort)
            .def("alarm_setup", &hs_manager_wrapper::alarm_setup)
            .def("display", &hs_manager_wrapper::display)
            .def("event_setup", &hs_manager_wrapper::event_setup)
            .def("initialize", &hs_manager_wrapper::initialize)
            .def("machine_command", &hs_manager_wrapper::machine_command)
            .def("move_command", &hs_manager_wrapper::move_command)
            .def("pause", &hs_manager_wrapper::pause)
            .def("res_activate", &hs_manager_wrapper::resource_activate)
            .def("res_deactivate", &hs_manager_wrapper::resource_deactivate)
            .def("res_delete", &hs_manager_wrapper::resource_delete)
            .def("res_query", &hs_manager_wrapper::resource_query)
            .def("res_request", &hs_manager_wrapper::resource_request)
            .def("res_transfer", &hs_manager_wrapper::resource_transfer)
            .def("resume", &hs_manager_wrapper::resume)
            .def("set_param", &hs_manager_wrapper::set_param)
            .def("start", &hs_manager_wrapper::start_message)
            .def("status_query", &hs_manager_wrapper::status_query)
            .def("trace_setup", &hs_manager_wrapper::trace_setup)
        ;
    }
};


//
//
//
void init_module()
{
    hs_manager_wrapper::wrap();

    python::class_<any_item>("Parameter",
        python::init<std::string, int, std::string>())
        .def(python::init<any_item>())
    ;
    // Compile check only...
    python::class_<std::vector<any_item> >("Parameters")
        .def(python::vector_indexing_suite<std::vector<any_item> >())
    ;

    python::class_<Id>("Id")
        .def(python::init<>())
        .def(python::init<Id>())
        .def(python::init<std::string>())
        .def("__repr__", &Id::repr)
    ;

    python::class_<std::vector<Id> >("Ids")
        .def(python::vector_indexing_suite<std::vector<Id> >())
    ;
}

} } // *namespace* jyugem::cell

BOOST_PYTHON_MODULE(hostsim)
{
    jyugem::cell::init_module();
}
