// $Id$

//=============================================================================
/**
 *  @file    VFEIEventTraverser.h
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "VFEIEventTraverser.h"
#include "HSServer.h"
#include "HSEventManager.h"
#include "jyugem/gem/JGVariable.h"
#include "BS2Message.h"
#include "BS2ListItem.h"
#include "BS2Item.h"
#include "BS2DeclAtoms.h"

static BCHAR * _unknown_state = _TX("\"UNKNOWN\"");

//-----------------------------------------------------------------------------
// Constructor/Destoructor
//-----------------------------------------------------------------------------
VFEIEventTraverser::VFEIEventTraverser(JGEventManager * mngr, int sf)
        : BS2Traverser(), m_manager(mngr), m_sf(sf), m_vfei(""), m_vfeimsg(""),
          m_event(NULL), m_report(NULL), m_variable(NULL),
          m_rptnum(0), m_varnum(0), m_first(true), m_objtype(NULL),
          m_attribute(NULL), m_attrnum(0)
{
    m_spec = (JGSpecification *)ObjSpec::instance();
}

//-----------------------------------------------------------------------------
// Add value string to vfei string
//-----------------------------------------------------------------------------
int VFEIEventTraverser::addItemValue(JGvalue& secsval, const string& itemName,
                                   int reqFormat, string& vfeibuf)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::addItemValue");

    //TRACE_DEBUG((_TX("Value: [%s] item = %s, format = %o(%o)\n"),
    //            secsval.toString().c_str(), itemName.c_str(), itemFormat,
    //            secsval.format()));

    const char * formatStr;
    int itemFormat = secsval.format();

    //JGvalue val;
    //val.set("", itemFormat);
    //val = secsval;

    vfeibuf += itemName;
    vfeibuf += "/";

    if (itemFormat & ATOM_ARRAY)
    {
        BCHAR sizebuf[64];
        sprintf(sizebuf, _TX("%d"), secsval.m_q);

        formatStr = b_value::smlStr(itemFormat & 0x3F);
        if (formatStr == NULL)
        {
            TRACE_ERROR((_TX("Invalid format: item = %s, format = %o\n"),
                        itemName.c_str(), itemFormat));
            return -1;
        }
        vfeibuf += formatStr;
        vfeibuf += "[";
        vfeibuf += sizebuf;
        vfeibuf += "]=[";
        vfeibuf += secsval.toString();
        vfeibuf += "]";
    }
    else if (itemFormat == ATOM_BINARY)
    {
        BCHAR sizebuf[64];
        sprintf(sizebuf, _TX("%d"),  secsval.size());

        vfeibuf += "B[";
        vfeibuf += sizebuf;
        vfeibuf += "]=[";
        vfeibuf += secsval.toString();
        vfeibuf += "]";
    }
    else
    {
        formatStr = b_value::smlStr(itemFormat);
        if (formatStr == NULL)
        {
            TRACE_ERROR((_TX("Invalid format: item = %s, format = %o\n"),
                        itemName.c_str(), itemFormat));
            return -1;
        }
        vfeibuf += formatStr;
        vfeibuf += "=";

        if (itemFormat == ATOM_ASCII || itemFormat == ATOM_JIS)
        {
            vfeibuf += "\"";
            vfeibuf += secsval.toString();
            vfeibuf += "\"";
        }
        else
        {
            vfeibuf += secsval.toString();
        }
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Add enum value to vfei string
//-----------------------------------------------------------------------------
int VFEIEventTraverser::addEnumValue(JGvalue& secsval, const string& itemName,
                                   b_enumerator * enumptr, string& vfeibuf)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::addEnumValue");

    //TRACE_DEBUG((_TX("Value: [%s] item = %s, format = %o\n"),
    //            secsval.toString().c_str(), itemName.c_str(),
    //            secsval.format()));

    vfeibuf += itemName;
    vfeibuf += "/A";

    int itemFormat = secsval.format();

    if (itemFormat & ATOM_ARRAY)
    {
        BCHAR sizebuf[64];
        sprintf(sizebuf, _TX("%d"),  secsval.m_q);

        vfeibuf += "[";
        vfeibuf += sizebuf;
        vfeibuf += "]=[";
        for (size_t j = 0; j < secsval.m_q; j++)
        {
            int num = secsval.getIntAt(j);

            if (j > 0)
            {
                vfeibuf += _TX(" ");
            }
            string enumstr("");
            bool retval = enumptr->numToString(num, enumstr);
            if (retval == false)
            {
                TRACE_ERROR((_TX("Illegal enumerate data: [%s] item = %s, value = %d\n"),
                            enumptr->name_p(), itemName.c_str(), num));
                vfeibuf += _unknown_state;
            }
            else
            {
                vfeibuf += "\"";
                vfeibuf += enumstr;
                vfeibuf += "\"";
            }
        }
        vfeibuf += "]";
    }
    else
    {
        vfeibuf += "=";
        int num = secsval.getInt();
        string enumstr("");
        bool retval = enumptr->numToString(num, enumstr);
        if (retval == false)
        {
            TRACE_ERROR((_TX("Illegal enumerate data: [%s] item = %s, value = %d\n"),
                        enumptr->name_p(), itemName.c_str(), num));
            vfeibuf += _unknown_state;
            return 0;
        }
        vfeibuf += "\"";
        vfeibuf += enumstr;
        vfeibuf += "\"";
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Add value string to vfei string (only value)
//-----------------------------------------------------------------------------
int VFEIEventTraverser::addValue(JGvalue& secsval, int reqFormat,
                               string& vfeibuf)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::addValue");

    //TRACE_DEBUG((_TX("Value: [%s], format = %o(%o)\n"),
    //            secsval.toString().c_str(), reqFormat, secsval.format()));

    int itemFormat = secsval.format();

    if (itemFormat & ATOM_ARRAY)
    {
        vfeibuf += "[";
        vfeibuf += secsval.toString();
        vfeibuf += "]";
    }
    else if (itemFormat == ATOM_BINARY)
    {
        vfeibuf += "[";
        vfeibuf += secsval.toString();
        vfeibuf += "]";
    }
    else
    {
        if (itemFormat == ATOM_ASCII || itemFormat == ATOM_JIS)
        {
            vfeibuf += "\"";
            vfeibuf += secsval.toString();
            vfeibuf += "\"";
        }
        else
        {
            vfeibuf += secsval.toString();
        }
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Write remain variables to vfei message
//-----------------------------------------------------------------------------
int VFEIEventTraverser::parseRemains()
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::parseRemains");

    if (m_report != NULL && m_report->size() > m_varnum)
    {
        for (int i = m_varnum; i < m_report->size(); i++)
        {
            if (i > 0)
            {
                m_vfei += _TX(" ");
            }
            JGVariable * var = m_report->variable(i);
            if (var->enumerator() != NULL)
            {
                this->addEnumValue(var->getv(), var->name(), var->enumerator(),
                                   m_vfei);
            }
            else
            {
                this->addItemValue(var->getv(), var->name(), var->format(),
                               m_vfei);
            }
        }
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Parse item in message (override)
//-----------------------------------------------------------------------------
int VFEIEventTraverser::parseItem(BS2Item * item)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::parseItem");

    int result;
    BS2Atom * atom = item->atom();
    if (item->name() ==  _TX("V"))
    {
        //TRACE_DEBUG((_TX("Parse V(alue): num = %d.\n"), m_varnum));

        JGvalue val;

        if (m_first)
            m_first = false;    // don't add space
        else
            m_vfei += " ";

        JGVariable * var;
        if (m_sf == SFCODE(6,11))
        {
            var = m_report->variable(m_varnum++);
        }
        else if (m_sf == SFCODE(6,13))
        {   // variable is found by vid.
            var = m_variable;
        }
        else
        {
            TRACE_ERROR((_TX("Illegal stream/function number.\n")));
            return -3;
        }

        if (var == NULL)
        {
            TRACE_ERROR((_TX("Variable is null (item count over).\n")));
            return -3;
        }

        if (atom->format() != var->format())
        {
            TRACE_ERROR((_TX("Warning: %s differnt value format (%o/%o).\n"),
                        var->charName(), atom->format(), var->format()));
        }
        if (var->enumerator() != NULL)
        {
            result = this->addEnumValue(*atom, var->name(), var->enumerator(), m_vfei);
        }
        else
        {
            result = this->addItemValue(*atom, var->name(), var->format(), m_vfei);
        }
        if (result < 0)
        {
            return result;
        }
    }
    else if (item->name() == _TX("RPTID"))
    {

        JGid rptid;
        atom->get(rptid);

        //TRACE_DEBUG((_TX("Parse RPTID: %s.\n"), rptid.toString().c_str()));

        m_report = JGEventManager::instance()->findReport(rptid);
        if (m_report == NULL)
        {
            TRACE_ERROR((_TX("RPTID(%s) not found.\n"), rptid.toString().c_str()));
            return -5;            // report is not found
        }

        m_varnum = 0;
    }
    else if (item->name() == _TX("VID"))  // for S6F13
    {
        JGid vid;
        atom->get(vid);

        //TRACE_DEBUG((_TX("Parse VID: %s.\n"), vid.toString().c_str()));

        m_variable = JGEquipment::instance()->variable(vid);
        if (m_variable == NULL)
        {
            TRACE_ERROR((_TX("VID(%s) not found.\n"), vid.toString().c_str()));
            return -5;            // vid is not found
        }
    }
    else if (item->name() == _TX("CEID"))
    {
        JGid ceid;
        atom->get(ceid);

        //TRACE_DEBUG((_TX("Parse CEID: %s.\n"), ceid.toString().c_str()));

        m_event = JGEventManager::instance()->find(ceid);
        if (m_event == NULL)
        {
            TRACE_ERROR((_TX("CEID(%s) not found.\n"), ceid.toString().c_str()));
            return -4;             // ceid is not exist
        }
        m_vfeimsg = "EVENT_ID/A=\"";
        m_vfeimsg += m_event->name();
        m_vfeimsg += "\" ";
    }
    else if (item->name() == _TX("DATAID"))
    {
        atom->get(m_dataid);

        //TRACE_DEBUG((_TX("Parse DATAID: %s.\n"), m_dataid.toString().c_str()));
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Parse list item in message (override)
//-----------------------------------------------------------------------------
int VFEIEventTraverser::beginList(BS2ListItem * listitem)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::beginList");

    this->BS2Traverser::beginList(listitem);

    int result = 0;
    int memberNum;
    bool is_listitem = false;
    BS2List * listatom;
    char * typedisp;

    if (listitem->isListItem())
    {
        memberNum = listitem->memberq();
        is_listitem = true;
        typedisp = "Item";
    }
    else if (listitem->isList())
    {
        listatom = (BS2List *)((BS2Item *)listitem)->atom();
        is_listitem = false;
        typedisp = "Atom";
    }
    else
    {
        TRACE_DEBUG((_TX("Not List Item or Atom.\n")));
        return -1;
    }


    //TRACE_DEBUG((_TX("List %s [%d]: nest = %d.\n"), typedisp, memberNum, m_nest));

    if (! is_listitem)
    {
        memberNum = listatom->getList().size();         // Need to hide member data;

        if (m_nest == VLIST_DEPTH)
        {   // list of v
            m_variable = m_report->variable(m_varnum++);
            if (m_variable == NULL)
            {
                TRACE_ERROR((_TX("Variable count over (%d: %d).\n"),
                        memberNum, m_varnum));
                return -1;
            }
            result = parseListAtom(m_variable, listatom, m_vfei);
        }
        else
        {
            TRACE_ERROR((_TX("Illegal nested level (%d).\n"), m_nest));
            return -1;
        }
    }

    return result;
}

//-----------------------------------------------------------------------------
// Parse list item in message (override)
//-----------------------------------------------------------------------------
int VFEIEventTraverser::endList(BS2ListItem * listitem)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::endList");

    if (m_nest == VLIST_DEPTH)
    {   // list of value
    }
    else if (m_nest > VLIST_DEPTH)
    {   // list of object
    }

    this->BS2Traverser::endList(listitem);
    return 0;
}

//-----------------------------------------------------------------------------
// Parse list item in message (override)
//-----------------------------------------------------------------------------
int VFEIEventTraverser::beginValueList(BS2Item * item)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::beginValueList");
    this->BS2Traverser::beginValueList(item);

    int result = 0;
    // Check List type  that is BS2ListItem or BS2Item(BS2List)
    int  memberNum = 0;
    if (item->isList())
    {   // List of atom(value)
        BS2List * listatom = (BS2List *)item->atom();
        memberNum = listatom->getList().size();         // Need to hide member data;

        if (m_nest == VLIST_DEPTH)
        {   // list of v
            m_variable = m_report->variable(m_varnum++);
            if (m_variable == NULL)
            {
                TRACE_ERROR((_TX("Variable count over (%d: %d).\n"),
                        memberNum, m_varnum));
                return -1;
            }
            result = parseListAtom(m_variable, listatom, m_vfei);
        }
        else
        {
            TRACE_ERROR((_TX("Illegal nested level (%d).\n"), m_nest));
            return -1;
        }
    }
    else
    {   // Not list ?
        TRACE_ERROR((_TX("Argumet item is not list.\n")));
        return -1;
    }
    return result;
}

//-----------------------------------------------------------------------------
// Top of Parse Variable
//-----------------------------------------------------------------------------
int VFEIEventTraverser::parseListAtom(JGVariable * var, BS2List * listatom,
                                    string& vfeibuf)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::parseListAtom");

    int result;
    BS2Atoms& atoms = listatom->getList();
    int memberNum = atoms.size();
    if (memberNum == 0)
    {
        return 0;                   // ignore empty list
    }
    JGClass * clazz = NULL;

    if (var->func() == "vector")
    {
        if (var->objtype() != NULL)
        {
            if (m_first)
                m_first = false;    // don't add space
            else
                vfeibuf += _TX(" ");
            addListHeader(var->name(), ATOM_LIST, memberNum, vfeibuf);
            clazz = var->objtype();
            for (int i = 0; i < memberNum; i++)
            {
                BS2Atom * mbr_atom = atoms[i];
                if (! mbr_atom->isList())
                {
                    TRACE_ERROR((_TX("Item must be list type(recieved primitive).\n")));
                    return -1;
                }
                BS2Atoms& objv = ((BS2List *)mbr_atom)->getList();

                if (i > 0)
                    vfeibuf += _TX(" ");

                addListHeader(clazz->userName(), ATOM_LIST, objv.size(), vfeibuf);
                result = parseClassAtom(clazz, objv, vfeibuf);
                if (result < 0)
                {
                    return result;
                }
                addListTail(vfeibuf);
            }
            addListTail(vfeibuf);
        }
        else                    // Can't support vector of vector
        {   // primitive
            if (m_first)
                m_first = false;    // don't add space
            else
                vfeibuf += _TX(" ");
            addListHeader(var->name(), var->format(), memberNum, vfeibuf);
            bool restrain = false;
            for (int i = 0; i < memberNum; i++)
            {
                BS2Atom * mbr_atom = atoms[i];
                if (mbr_atom->isList())
                {
                    TRACE_ERROR((_TX("Item must be primitive type(recieved list).\n")));
                    return -1;
                }
                if ((! restrain) && mbr_atom->format() != var->format())
                {
                    TRACE_ERROR((_TX("Warning: %s differnt value format (%o/%o).\n"),
                        var->charName(), mbr_atom->format(), var->format()));
                    restrain = true;
                }

                if (i > 0)
                    vfeibuf += _TX(" ");

                result = this->addValue(*mbr_atom, var->format(), vfeibuf);
                if (result < 0)
                {
                    return result;
                }
            }
            addListTail(vfeibuf);
        }
    }
    else if (var->func() == "object")
    {
        if (m_first)
            m_first = false;    // don't add space
        else
            vfeibuf += _TX(" ");
        addListHeader(var->name(), ATOM_LIST, memberNum, vfeibuf);
        clazz = var->objtype();
        if (clazz == NULL)
        {
            TRACE_ERROR((_TX("Variable(%s) has no objtype.\n"),
                    var->name().c_str()));
            return -1;
        }
        result = parseClassAtom(clazz, atoms, vfeibuf);

        addListTail(vfeibuf);
    }
    else
    {
        TRACE_ERROR((_TX("Illegal variable type(%s).\n"), var->func().c_str()));
        return -1;
    }
    return result;
}

//-----------------------------------------------------------------------------
// Parse Class/Item
//-----------------------------------------------------------------------------
int VFEIEventTraverser::parseClassAtom(JGClass * clazz, BS2Atoms& atoms,
                                     string& vfeibuf)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::parseClassAtom");

    int result;

    int member_num1 = clazz->attrSize();
    int member_num2 = atoms.size();
    if (member_num1 != member_num2)
    {
         // Log: warning
        TRACE_ERROR((_TX("Warning: Un-matched attribute count(%d / %d).\n"),
                    member_num1, member_num2));
    }
    int memberNum = (member_num2 > member_num1) ? member_num1 : member_num2;

    for (int i = 0; i < memberNum; i++)
    {
        if (i > 0)
            vfeibuf += _TX(" ");

        JGAttribute * mbr_attr = (JGAttribute *)clazz->at(i);
        BS2Atom * atom = atoms[i];
        result = parseAttrAtom(mbr_attr, atom, vfeibuf);
        if (result < 0)
        {
            break;
        }
    }
    if (memberNum < member_num1)
    {
        for (int l = memberNum; l < member_num1; l++)
        {
            if (l > 0)
                vfeibuf += _TX(" ");

            JGAttribute * mbr_attr = (JGAttribute *)clazz->at(l);
            result = this->addItemValue(mbr_attr->getInitValue(),
                         mbr_attr->userName(), mbr_attr->type(), vfeibuf);
            if (result < 0)
            {
                break;
            }
        }
    }

    return result;
}

//-----------------------------------------------------------------------------
// Parse Attribute/Item
//-----------------------------------------------------------------------------
int VFEIEventTraverser::parseAttrAtom(JGAttribute * attr, BS2Atom * atom,
                                    string& vfeibuf)
{
    TRACE_FUNCTION(TRL_LOW, "VFEIEventTraverser::parseAttrAtom");

    int result;
    int memberNum;

    if (attr->isVector())
    {
        if (! atom->isList())
        {
            TRACE_ERROR((_TX("Received data is not list type.\n")));
            return -1;
        }
        BS2Atoms& atoms = ((BS2List *)atom)->getList();
        memberNum = atoms.size();
        if (memberNum == 0)
        {
            return 0;                   // ignore empty list
        }

        JGAttribute * nested = ((JGVector *)attr)->member();
        if (nested->isStruct() || nested->isVector())
        {
            addListHeader(attr->userName(), ATOM_LIST, memberNum, vfeibuf);
            for (size_t i = 0; i < atoms.size(); i++)
            {
                if (i > 0)
                    vfeibuf += _TX(" ");

                BS2Atom * atom = atoms[i];
                result = this->parseAttrAtom(nested, atom, vfeibuf);
                if (result < 0)
                {
                    break;
                }
            }
            this->addListTail(vfeibuf);
        }
        else
        {   // primitive
            addListHeader(attr->userName(), nested->type(), memberNum, vfeibuf);
            bool restrain = false;
            for (size_t j = 0; j < atoms.size(); j++)
            {
                if (j > 0)
                    vfeibuf += _TX(" ");

                BS2Atom * atom = atoms[j];

                if ((! restrain) && atom->format() != nested->type())
                {
                    TRACE_ERROR((_TX("Warning: %s differnt value format (%o/%o).\n"),
                        nested->userName().c_str(), atom->format(), nested->type()));
                    restrain = true;
                }

                result = this->addValue(*atom, nested->type(), vfeibuf);
                if (result < 0)
                {
                    break;
                }
            }
            addListTail(vfeibuf);
        }
    }
    else if (attr->isStruct())
    {
        if (! atom->isList())
        {
            TRACE_ERROR((_TX("Received data is not list type.\n")));
            return -1;
        }
        BS2Atoms& atoms = ((BS2List *)atom)->getList();
        memberNum = atoms.size();
        JGStruct * ztruct = (JGStruct *)attr;
        int member_num1 = ztruct->length();
        int member_num2 = atoms.size();
        if (member_num1 != member_num2)
        {
            TRACE_ERROR((_TX("Warning: Un-matched attribute count(%d / %d).\n"),
                    member_num1, member_num2));
        }
        int memberNum = (member_num2 > member_num1) ? member_num1 :
                                                      member_num2;
        addListHeader(ztruct->userName(), ATOM_LIST, memberNum, vfeibuf);
        for (int k = 0; k < memberNum; k++)
        {
            if (k > 0)
                vfeibuf += _TX(" ");

            JGAttribute * mbr_attr = ztruct->at(k);
            BS2Atom * atom = atoms[k];
            result = this->parseAttrAtom(mbr_attr, atom, vfeibuf);
            if (result < 0)
            {
                break;
            }
        }
        if (memberNum < member_num1)
        {
            for (int l = memberNum; l < member_num1; l++)
            {
                if (l > 0)
                    vfeibuf += _TX(" ");

                JGAttribute * mbr_attr = ztruct->at(l);
                result = this->addItemValue(mbr_attr->getInitValue(),
                             mbr_attr->userName(), mbr_attr->type(), vfeibuf);
                if (result < 0)
                {
                    break;
                }
            }
        }
        addListTail(vfeibuf);
    }
    else
    {
        if (atom->isList())
        {
            TRACE_ERROR((_TX("Data must be primitive type(recieved list).\n")));
            return -1;
        }
        if (atom->format() != attr->type())
        {
            TRACE_ERROR((_TX("Warning: %s differnt value format (%o/%o).\n"),
                attr->charName(), atom->format(), attr->type()));
        }
        result = this->addItemValue(*atom, attr->userName(), attr->type(), vfeibuf);
    }
    return result;
}
