// $Id$

//=============================================================================
/**
 *  @file    rcmd.h
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#if !defined(JYUGEM_CELL_RCMD_HPP)
#define JYUGEM_CELL_RCMD_HPP

#include "jyugem/gem/gembase.h"

namespace jyugem { namespace cell
{

struct any_item
{
    any_item() : m_pname(_TX("")) {}
    any_item(const std::string& name) : m_pname(name) {}
    any_item(const std::string& name, int fmt, const std::string& v)
        : m_pname(name)
    {
        m_value.set(v.c_str(), fmt);
    }
    any_item(const any_item& rhs)
        : m_pname(rhs.m_pname), m_value(rhs.m_value), m_member(rhs.m_member) {}
    ~any_item() {}

    any_item& operator=(const any_item& rhs)
    {
        if (this != &rhs)
        {
            m_pname = rhs.m_pname;
            m_value = rhs.m_value;
            m_member = rhs.m_member;
        }
        return *this;
    }
    bool operator==(const any_item& rhs)
    {
        if (this == &rhs)
            return true;
        if (m_pname == rhs.m_pname && m_value == rhs.m_value)
        {
            if (m_member.size() == rhs.m_member.size() && m_member.size() == 0)
                return true;
            else if (m_member.size() == rhs.m_member.size() && m_member.size() > 0)
            {
                for (size_t q = 0; q < m_member.size(); q++)
                {
                    if (m_member[q] == rhs.m_member[q])
                        continue;
                    return false;  // not equal
                }
                return true;
            }
            return false;
        }
        return false;
    }

    void print(std::ostream& out, size_t indentq)
    {
        std::string spaces(indentq * 2, ' ');
        out << spaces << "<item name=\"" << m_pname
        << "\" format=\"" << m_value.format() << "\">" << m_value.toString()
        << "</item>" << std::endl;
    }
    any_item * clone()
    {
        return new any_item(*this);
    }

    void add(const JGvalue& val)
    {
        TRACE_FUNCTION(TRL_LOW, "any_item::add");
        if (m_value.isNil() || m_value.m_t != ATOM_LIST)
        {
            b_valvect vv;
            m_value = vv;
        }
        if (m_value.m_t == ATOM_LIST && m_value.m._vec != NULL)
        {
            JGvalue * vptr = new JGvalue(val);
            m_value.m._vec->push_back(vptr);
        }
        else
        {
            TRACE_ERROR((_TX("Value is not array format\n")));
        }
    }
    void add(const any_item& mbr)
    {
        BEEAssert(m_value.isNil());
        m_member.push_back(mbr);
    }

    const std::string& name() const { return m_pname; }
    int   format() const            { return m_value.format(); }

    std::string m_pname;
    JGvalue     m_value;
    std::vector<any_item> m_member;
};


//
struct rcmd
{
    rcmd() : m_rcmd(""), m_devid(""), m_tid(0), m_wait(false), m_objspec("") {}
    rcmd(const std::string& cmd, const std::string& mid = "", int tid = 0,
         bool wait = false)
        : m_rcmd(cmd), m_devid(mid), m_tid(tid), m_wait(wait), m_objspec("") {}
    rcmd(const rcmd& rhs) { this->copy(rhs); }
    ~rcmd() {}

    rcmd& operator=(const rcmd& rhs)
    {
        if (this != &rhs)
            this->copy(rhs);
        return *this;
    }

    const std::string& name() const { return m_rcmd; }
    const char * name_c() { return m_rcmd.c_str(); }
    bool  reply()         { return m_wait; }

    void print(std::ostream& out)
    {
        out << "<command name=\"" << m_rcmd << "\" device=\""
            << m_devid << "\" transaction=\"" << m_tid << "\" response=\""
            << ((m_wait) ? "TRUE" : "FALSE") << "\">" << std::endl;
        if (m_objspec.size() > 0)
        {
            out << "  <item name=\"OBJSPEC\" format=\"A\">" << m_objspec
                << "</item>" << std::endl;
        }
        for (size_t i = 0; i < m_items.size(); i++)
        {
            m_items[i].print(out, 1);
        }
        out << "</command>" << std::endl;
    }

    virtual void append(const any_item& mbr)
    {
        m_items.push_back(mbr);
    }

    void copy(const rcmd& rhs)
    {
        m_rcmd = rhs.m_rcmd;
        m_devid = rhs.m_devid;
        m_tid = rhs.m_tid;
        m_wait = rhs.m_wait;
        m_objspec = rhs.m_objspec;
        for (size_t i = 0; i < rhs.m_items.size(); i++)
        {
            m_items.push_back(rhs.m_items[i]);
        }
    }

    std::string m_rcmd;
    std::string m_devid;  // mid
    int         m_tid;
    bool        m_wait;
    std::string m_objspec;
    std::vector<any_item> m_items;
};


} }  // *jyugem::cell*

#endif  // JYUGEM_CELL_RCMD_HPP
