unit untHttp;

interface

uses
  Classes, SysUtils, IdHttp, IdComponent, untGZIP, SyncObjs,
  untStreamTool;

type
  THttpErrpr = (heNoError, heBrokenGzip, heSocketError, heMoved);

  TAsyncHttp = class
  private
    FIdHttp: TIdHttp;
    FWriteEvent: TMemoryStreamEx;
    FBufferReader: TStreamReader;
    FReadPosition: integer;
    FAddHeaders: string;
    FBuffer: TMemoryStream;
    FGzipStream: TGzipDecompressStream;
  protected
    procedure HttpReceived(const Buff; Count: int64);
    procedure HttpStatus(ASender: TObject; const AStatus: TIdStatus; const AStatusText: string);
  public
    URL:          string;
    LastModified: string;
    UserAgent:    string;
    UseGzip: boolean;
    ErrorCode: THttpErrpr;
    OnStatus:   TIdStatusEvent;
    OnReceived: TNotifyEvent;
    ContentLength: integer;
    StartRange:    integer;
    ResponseCode:  integer;
    ReceivedLines: TStringList;
    ProxyHost: string;
    ProxyPort: Integer;
    procedure Get;
    procedure Close;

    procedure Head;
    procedure AddHeader(HeaderName, Value: string);
    constructor Create;
    destructor Destroy; override;
  end;

implementation

uses
  untGlobal, IdException;

{ TAsyncHttp }

constructor TAsyncHttp.Create;
begin
  FIdHttp := TIdHttp.Create(nil);
  ReceivedLines := TStringList.Create;
end;

destructor TAsyncHttp.Destroy;
begin
  ReceivedLines.Clear;
  ReceivedLines.Free;
  FIdHttp.Free;
  inherited;
end;

procedure TAsyncHttp.Close;
begin
  FIdHttp.Disconnect;
end;

procedure TAsyncHttp.Head;
var
  i: integer;

begin
  with FIdHttp do
  begin
    Request.Connection := 'close';
    Request.UserAgent  := UserAgent;
    OnStatus           := HttpStatus;
    if ProxyHost <> '' then
    begin
      ProxyParams.ProxyServer := ProxyHost;
      ProxyParams.ProxyPort   := ProxyPort;
      Request.Pragma          := 'no-cache';
    end;
  end;

  FIdHttp.ConnectTimeout := gConfig.CONNECT_TIMEOUT;

  try
    FIdHttp.Head(URL);
  except on EIdHTTPProtocolException do;
  end;

  for i := 0 to FIdHttp.Response.RawHeaders.Count -1 do
  begin
    ReceivedLines.Add(FIdHttp.Response.RawHeaders[i]);
  end;

  ResponseCode := FIdHttp.Response.ResponseCode;
  if ResponseCode = 200 then
  begin
    // VƂ
    LastModified := FIdHttp.Response.RawHeaders.Values['Last-Modified'];
  end;
end;
//
// 񓯊IGET
//
procedure TAsyncHttp.Get;
var
  headers: TStringList;
  i: Integer;

begin
  ErrorCode := heNoError;

  headers := TStringList.Create;
  FWriteEvent := TMemoryStreamEx.Create;
  FBuffer := TMemoryStream.Create;
  FBufferReader := TStreamReader.Create(FBuffer);
  FGzipStream := TGzipDecompressStream.Create(FBuffer);

  try
    FWriteEvent.OnWrite := HttpReceived;
    FReadPosition := 0;

    with FIdHttp do
    begin
      Request.ContentRangeStart := StartRange;
      Request.Connection        := 'close';
      Request.UserAgent         := UserAgent;
      OnStatus                  := HttpStatus;
    end;

    if UseGzip and (StartRange = 0) then
    begin
      FIdHttp.Request.AcceptEncoding := 'gzip';
    end;

    if LastModified <> '' then
    begin
      AddHeader('If-Modified-Since', LastModified);
    end;

    headers.Text := FAddHeaders;
    for i := 0 to headers.Count - 1 do
    begin
      FIdHttp.Request.CustomHeaders.Add(headers[i]);
    end;

    if ProxyHost <> '' then
    begin
      with FIdHttp do
      begin
        ProxyParams.ProxyServer := ProxyHost;
        ProxyParams.ProxyPort   := ProxyPort;
        Request.Pragma          := 'no-cache';
      end;
    end;

    FIdHttp.ConnectTimeout := gConfig.CONNECT_TIMEOUT;

    try
         FIdHttp.Get(URL, FWriteEvent);
    except
      on EIdHTTPProtocolException do
        ;
      on EIdException do
        begin
          ErrorCode := heSocketError;
        end;
    end;

    ResponseCode := FIdHttp.Response.ResponseCode;
    if (ResponseCode = 200) or (ResponseCode = 206) then
    begin
      // VƂ
      LastModified := FIdHttp.Response.RawHeaders.Values['Last-Modified'];
    end;

    if (ResponseCode = 302) then
    begin
      ErrorCode := heMoved;
      if Assigned(OnReceived) then
      begin
        // DATꍇO\
        OnReceived(self);
      end;
    end;


    //̃G[͐Ȃ̂ŃRgAEg
    {
    if (ResponseCode <> 304) and (ResponseCode > 0) then
    begin
      if (ResponseCode < 200) or (ResponseCode > 299) then
      begin
        ErrorCode := heSocketError;
      end;
    end;
    }

    if FIdHttp.Response.ContentEncoding = 'gzip' then
    begin
      ContentLength := FBuffer.Size;
    end else
    begin
      ContentLength := FIdHttp.Response.ContentLength;
    end;
  finally
    headers.Free;
    FBufferReader.Free;
    FWriteEvent.Free;
    FGzipStream.Free;
    FBuffer.Clear;
    FBuffer.Free;
  end;
end;

procedure TAsyncHttp.AddHeader(HeaderName, Value: string);
begin
  FAddHeaders := FAddHeaders + HeaderName + ': ' + Value + #13#10;
end;

procedure TAsyncHttp.HttpReceived(const Buff; Count: int64);
var
  line: string;

begin
  FBuffer.Seek(0, soFromEnd);

  if FIdHttp.Response.ContentEncoding = 'gzip' then
  begin
    try
      FGzipStream.Write(Buff, Count);
    except on Exception do
      begin
        ErrorCode := heBrokenGzip;
        exit;
      end;
    end;
  end else
  begin
    FBuffer.Write(Buff, Count);
  end;

  line := '';

  FBuffer.Seek(FReadPosition, soFromBeginning);
  while FBufferReader.ReadLine(line) do
  begin
    FReadPosition := FBuffer.Position;
    ReceivedLines.Add(line);
  end;

  if Assigned(OnReceived) and (ReceivedLines.Count > 0) then
  begin
    OnReceived(self);
  end;
end;

procedure TAsyncHttp.HttpStatus(ASender: TObject;
                                const AStatus: TIdStatus;
                                const AStatusText: string);
begin
  if Assigned(OnStatus) then
  begin
    OnStatus(self, AStatus, AStatusText);
  end;
end;

end.
