unit untBoard;

interface

uses
  untTopic, untHttp, SysUtils, untTool, Classes, Forms;

type

  TTopicReceivedEvent = procedure(Sender : TObject; Topic : TTopic) of object;
  TBoard = class
  private
    FDownloading  : Boolean;
    FBoardid      : string;
    FCategory     : string;
    FDisplayName  : string;
    FBoardPath    : string;
    FCurrentIndex : Integer;
    FOnTopicReceived: TTopicReceivedEvent;
    procedure SetBoardid(const Value: string);
    procedure SetCategory(const Value: string);
    procedure SetDisplayName(const Value: string);
    procedure SetOnTopicReceived(const Value: TTopicReceivedEvent);
    procedure RaiseTopicReceivedEvent(Topic : TTopic);
  public
		property  Category    : string read FCategory;
		property  DisplayName : string read FDisplayName;
		property  Boardid     : string read FBoardid;
    property  OnTopicReceived : TTopicReceivedEvent read FOnTopicReceived write SetOnTopicReceived;
    property  IsDownloading : boolean read FDownloading;
    constructor Create(BoardId, DisplayName, Category : string);
    procedure Download(); virtual; abstract;
    property BoardPath : string read FBoardPath;
  end;

  TOnlineBoard = class(TBoard)
  private
    FHttpGet  : THttpGet;
    FServer   : string;
    FServerId : string;
    procedure SetServer(const Value: string);
    function GetBoardUrl()   : string;
    function GetSubjectTxtUrl: string;
    function GetBbsCgiUrl  : string;
    procedure HttpReceive(Sender: TObject; Complete: Boolean; ReceiveText: string);
    procedure HttpStateChange(Sender: TObject; HttpState : THttpState);
  public
    property  Server : string read FServer;
    property  BoardUrl      : string read GetBoardUrl;
    property  SubjectTxtUrl : string read GetSubjectTxtUrl;
    property  BbsCgiUrl   : string read GetBbsCgiUrl;
    constructor Create(BoardId, DisplayName, Category, Server : string);
    destructor  Destroy; override;
    procedure Download(); override;
  end;

  TOfflineBoard = class(TBoard)
  private
    FLoadedLog : Boolean;
    FIdxPath   : string;
    FTopicList : TList;
    procedure LoadLog();
  public
    constructor Create(idxpath : string; boardname : string);
    destructor  Destroy; override;
    procedure Download(); override;
    procedure AppendTopic(Topic : TTopic);
    procedure RemoveTopic(Topic : TTopic);
    procedure Save();
    procedure Patrol();
  end;

implementation

uses
  untGlobal;

{ TBoard }

//  vpeB 

procedure TBoard.SetBoardid(const Value: string);
begin
  FBoardid := Value;
end;

procedure TBoard.SetCategory(const Value: string);
begin
  FCategory := Value;
end;

procedure TBoard.SetDisplayName(const Value: string);
begin
  FDisplayName := Value;
end;

procedure TBoard.SetOnTopicReceived(const Value: TTopicReceivedEvent);
begin
  FOnTopicReceived := Value;
end;

//  \bh 

{ --------------------------------------------------------
  ֐: Create
  pr  : RXgN^
    : BoardId
  @@@: DisplayName
  @@@; Category
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
constructor TBoard.Create(BoardId, DisplayName, Category: string);
begin
  FBoardId     := BoardId;
  FDisplayName := DisplayName;
  FCategory    := Category;
  FDownloading := false;
end;

{ --------------------------------------------------------
  ֐: RaiseTopicReceivedEvent
  pr  : TopicM|`Cxg𔭍s
    : Topic
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TBoard.RaiseTopicReceivedEvent(Topic: TTopic);
begin

  if Assigned(FOnTopicReceived) then
    FOnTopicReceived(self, Topic);

end;

{ TOnlineBoard }

//  vpeB 

procedure TOnlineBoard.SetServer(const Value: string);
begin
  FServer := Value;
end;

function TOnlineBoard.GetBoardUrl: string;
begin
  result := 'http://' + FServer + '/' + FBoardId + '/';
end;

function TOnlineBoard.GetSubjectTxtUrl: string;
begin
  result := GetBoardUrl() + 'subject.txt';
end;

function TOnlineBoard.GetBbsCgiUrl: string;
begin
  result := 'http://' + FServer + '/test/bbs.cgi';;
end;

//  \bh 

{ --------------------------------------------------------
  ֐: Create
  pr  : RXgN^
    : BoardId
  @@@: DisplayName
  @@@: Category
  @@@: Server
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
constructor TOnlineBoard.Create(BoardId, DisplayName, Category,
  Server: string);
var
  intPos : Integer;
begin
  inherited Create(BoardId, DisplayName, Category);

  FServer := Server;
  intPos := Pos('.', Server);
  if intPos > 0 then
    FServerId := CopyAfter(Server, intPos + 1)
  else
    FServerId := Server;

  FBoardPath := FServerId + '/' + FBoardId;
end;

{ --------------------------------------------------------
  ֐: Download
  pr  : gsbN̎MJn
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOnlineBoard.Download;
begin

  if FDownloading then exit;
  FDownloading := true;

  // HTTPNX̗グ
  FCurrentIndex := 0;
  FHttpGet.Free;
  FHttpGet := THttpGet.Create;
  FHttpGet.ReceiveEventTiming := OnOneLine;
  FHttpGet.OnReceive          := HttpReceive;
  FHttpGet.OnStateChange      := HttpStateChange;

  FHttpGet.Connect(SubjectTxtUrl);

end;

{ --------------------------------------------------------
  ֐: Destroy
  pr  : fXgN^
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
destructor TOnlineBoard.Destroy;
begin
  FHttpGet.Free;

  inherited;
end;


//  Cxg 

// f[^M
procedure TOnlineBoard.HttpReceive(Sender: TObject; Complete: Boolean;
  ReceiveText: string);
var
  P1, P2, P3    : Integer;
  TopicId       : string;
  TopicTitle    : string;
  TopicMsgCount : Integer;
  TopicPath     : string;
  Topic         : TTopic;
begin

  // TopicId
  P1 := Pos('.dat', ReceiveText);
  if P1 = 0 then exit;
  TopicId := Copy(ReceiveText, 1, P1 - 1);

  // TopicTitle
  P2 := LastDelimiter('(', ReceiveText);
  if P2 = 0 then exit;
  TopicTitle := Copy(ReceiveText, P1 + 6, P2 - P1 - 7);

  // TopicMsgCount
  P3 := LastDelimiter(')', ReceiveText);
  if P3 = 0 then exit;
  TopicMsgCount := StrToIntNeo(Copy(ReceiveText, P2 + 1, P3 - P2 - 1));

  // XbhNX
  TopicPath := FBoardPath + '/' + TopicId;
  Topic     := gBBSCore.GetTopic(TopicPath);

  Inc(FCurrentIndex);
  Topic.Index := FCurrentIndex;
  Topic.Title        := TopicTitle;
  Topic.MessageCount := TopicMsgCount;

  // Cxgs
  RaiseTopicReceivedEvent(Topic);

end;

// 󋵕ω
procedure TOnlineBoard.HttpStateChange(Sender: TObject; HttpState: THttpState);
var
  contentsize : Integer;
begin

  if HttpState = hsDisconnected then
  begin

    FDownloading := false;

  end;

end;

{ TOfflineBoard }

{ --------------------------------------------------------
  ֐: Create
  pr  : RXgN^
    : idxpath
  @@@: BoardName
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
constructor TOfflineBoard.Create(idxpath, boardname: string);
begin
  inherited Create(boardname, boardname, '}CtH_');

  FIdxPath     := idxpath;

  FBoardPath := 'offline/' + boardname;

  FTopicList := TList.Create;

end;

{ --------------------------------------------------------
  ֐: Download
  pr  : gsbN̎MJn
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.Download;
var
  I      : Integer;
begin
  inherited;

  LoadLog();

  for I := 0 to FTopicList.Count - 1 do
    RaiseTopicReceivedEvent(FTopicList[i]);

end;

{ --------------------------------------------------------
  ֐: AppendTopic
  pr  : gsbN̒ǉ
    : Topic
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.AppendTopic(Topic: TTopic);
begin

  LoadLog();

  FTopicList.Add(Topic);
  Topic.IsFavorite := true;
  Topic.SaveIdx;
  Save();

end;

{ --------------------------------------------------------
  ֐: RemoveTopic
  pr  : gsbN̍폜
    : Topic
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.RemoveTopic(Topic: TTopic);
begin

  LoadLog();

  FTopicList.Remove(Topic);
  Topic.IsFavorite := false;
  Topic.SaveIdx;
  Save();

end;

{ --------------------------------------------------------
  ֐: Destroy
  pr  : fXgN^
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
destructor TOfflineBoard.Destroy;
begin
  FTopicList.Free;

  inherited;
end;

{ --------------------------------------------------------
  ֐: Save
  pr  : ۑ
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.Save;
var
  I       : Integer;
  Topic   : TTopic;
  stlSave : TStringList;
begin

  LoadLog();

  stlSave := TStringList.Create;
  for I := 0 to FTopicList.Count - 1 do
  begin
    Topic := TTopic(FTopicList[I]);
    stlSave.Add(Topic.TopicPath);
  end;

  stlSave.SaveToFile(FIdxPath); 
  stlSave.Free;

end;

{ --------------------------------------------------------
  ֐: Patrol
  pr  : 
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.Patrol;
var
  I : Integer;
  Topic : TTopic;
begin

  LoadLog();

  for I := 0 to FTopicList.Count - 1 do
  begin
    Application.ProcessMessages;
    Topic := TTopic(FTopicList[I]);
    gBBSCore.DownloadTopic(Topic); 
  end;

end;


//  vCx[g֐ 

{ --------------------------------------------------------
  ֐: LoadLog
  pr  : O̓ǂݍ
    : Ȃ
  ߂l: Ȃ
  l  : Ȃ
  ------------------------------------------------------ }
procedure TOfflineBoard.LoadLog;
var
  stlLog : TStringList;
  Topic  : TTopic;
  I      : Integer;
begin

  if FLoadedLog = false then
  begin
    if FileExists(FIdxPath) then
    begin

      stlLog := TStringList.Create;
      stlLog.LoadFromFile(FIdxPath);

      // gsbN̓ǂݍ
      for I := 0 to stlLog.Count - 1 do
        if stlLog[I] <> '' then
        begin
          Topic := gBBSCore.GetTopic(stlLog[I]);
          FTopicList.Add(Topic);
        end;

      stlLog.Free;
    end;
    FLoadedLog := true;
  end;


end;

end.
