/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-actions-bookmark.c,v 1.17 2004/03/20 11:49:50 makeinu Exp $
 */

#include "kz-actions-bookmark.h"

#include "egg-action.h"
#include "egg-toggle-action.h"
#include "egg-radio-action.h"
#include "intl.h"
#include "kz-icons.h"
#include "kz-bookmark-edit-win.h"


static void
act_new_window (EggAction *action, KzBookmarkEditor *editor)
{
	GtkWidget *widget;

	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	widget = kz_bookmark_editor_new(editor->root_folder);
	gtk_widget_show(widget);
}


static void
act_close_window (EggAction *action, KzBookmarkEditor *editor)
{
	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	kz_bookmark_editor_store_state(editor);
	gtk_widget_destroy(GTK_WIDGET(editor));
}


static void
act_edit_bookmark (EggAction *action, KzBookmarkEditor *editor)
{
	GList *node;

	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	for (node = kz_bookmark_editor_get_selected_list(editor);
	     node;
	     node = g_list_next(node))

	{
		GtkWidget *widget = kz_bookmark_edit_win_new(node->data);

		gtk_widget_show(widget);
	}
}


static void
insert_bookmark_item (KzBookmarkEditor *editor, KzBookmark *bookmark)
{
	KzBookmark *parent, *sibling = NULL;
	GList *list;

	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));

	list = kz_bookmark_editor_get_selected_list(editor);

	if (list)
	{
		sibling = list->data;
		parent = kz_bookmark_get_parent(sibling);
	}
	else
	{
		parent = editor->current_folder;
		g_return_if_fail(KZ_IS_BOOKMARK(parent));
		sibling = NULL;
	}

	kz_bookmark_insert_before(parent, bookmark, sibling);

	g_list_free(list);
}


static void
act_insert_bookmark (EggAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *bookmark;

	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	bookmark = kz_bookmark_new_with_attrs(_("New bookmark"), NULL, NULL);
	insert_bookmark_item(editor, bookmark);
	g_object_unref(G_OBJECT(bookmark));
}


static void
act_insert_folder (EggAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *folder;

	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	folder = kz_bookmark_pure_folder_new();
	kz_bookmark_set_title(folder, _("New folder"));
	insert_bookmark_item(editor, folder);
	g_object_unref(G_OBJECT(folder));
}


static void
act_insert_remote_bookmark (EggAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *file;

	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	file = kz_bookmark_file_new(NULL, NULL, NULL);
	kz_bookmark_set_title(file, _("New Remote Bookmark"));
	insert_bookmark_item(editor, file);
	g_object_unref(G_OBJECT(file));
}


static void
act_insert_separator (EggAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *sep;

	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	sep = kz_bookmark_separator_new();
	insert_bookmark_item(editor, sep);
	g_object_unref(G_OBJECT(sep));
}


static void
act_remove_selected_item (EggAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *parent, *bookmark;
	GList *list, *node;

	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	list = kz_bookmark_editor_get_selected_list(editor);

	parent = editor->current_folder;
	g_return_if_fail(KZ_IS_BOOKMARK(parent));

	/* FIXME!: show dialog */

	for (node = list; node; node = g_list_next(node))
	{
		bookmark = node->data;
		if (!KZ_IS_BOOKMARK(bookmark)) continue;
		kz_bookmark_remove(parent, bookmark);
	}

	g_list_free(list);
}


static void
act_show_hide_folder_view (EggAction *action, KzBookmarkEditor *editor)
{
	g_return_if_fail(EGG_IS_TOGGLE_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));
	g_return_if_fail(GTK_IS_WIDGET(editor->scrolledwin[0]));

	if (egg_toggle_action_get_active (EGG_TOGGLE_ACTION(action)))
		gtk_widget_show(editor->scrolledwin[0]);
	else
		gtk_widget_hide(editor->scrolledwin[0]);
}


static void
act_show_hide_content_view (EggAction *action, KzBookmarkEditor *editor)
{
	g_return_if_fail(EGG_IS_TOGGLE_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));
	g_return_if_fail(GTK_IS_WIDGET(editor->content_view));

	if (egg_toggle_action_get_active (EGG_TOGGLE_ACTION(action)))
		gtk_widget_show(editor->content_view);
	else
		gtk_widget_hide(editor->content_view);
}


static void
act_list_mode (EggAction *action, KzBookmarkEditor *editor)
{
	g_return_if_fail(EGG_IS_RADIO_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	kz_bookmark_editor_set_tree_mode(editor, FALSE);
}


static void
act_tree_mode (EggAction *action, KzBookmarkEditor *editor)
{
	g_return_if_fail(EGG_IS_RADIO_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	kz_bookmark_editor_set_tree_mode(editor, TRUE);
}


static void
act_go_up (EggAction *action, KzBookmarkEditor *editor)
{
	KzBookmark *parent;

	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_BOOKMARK_EDITOR(editor));

	if (!editor->current_folder) return;

	parent = kz_bookmark_get_parent(editor->current_folder);
	if (!parent) return;

	kz_bookmark_editor_set_current(editor, parent);
}


static void
cb_popup_menu_hide (void)
{
	gtk_main_quit();
}


EggActionGroupEntry kz_actions_bookmark[] =
{
  {"StockFileMenu",     N_("_File"),       NULL, NULL, NULL, NULL, NULL},
  {"StockEditMenu",     N_("_Edit"),       NULL, NULL, NULL, NULL, NULL},
  {"StockViewMenu",     N_("_View"),       NULL, NULL, NULL, NULL, NULL},
  {"StockGoMenu",       N_("_Go"),         NULL, NULL, NULL, NULL, NULL},
  {"StockModeMenu",     N_("_Mode"),       NULL, NULL, NULL, NULL, NULL},
  {"FakeToplevel",      (""),              NULL, NULL, NULL, NULL, NULL},

  {"NewWindow",            N_("_New Window"),             GTK_STOCK_NEW,      "<control>N",  NULL, G_CALLBACK(act_new_window),   NULL},
  {"CloseWindow",          N_("_Close Window"),           GTK_STOCK_CLOSE,    "<control>W",  NULL, G_CALLBACK(act_close_window), NULL},

  {"EditBookmarkItem",     N_("_Edit selected item"),     NULL,               NULL,         NULL, G_CALLBACK(act_edit_bookmark),    NULL},
  {"InsertBookmark",       N_("Insert _bookmark"),        GTK_STOCK_NEW,      "<control>B", NULL, G_CALLBACK(act_insert_bookmark),  NULL},
  {"InsertFolder",         N_("Insert _folder"),          KZ_STOCK_FOLDER,    "<control>F", NULL, G_CALLBACK(act_insert_folder),    NULL},
  {"InsertRemoteBookmark", N_("Insert _remote bookmark"), GTK_STOCK_SAVE,     "<control>R", NULL, G_CALLBACK(act_insert_remote_bookmark), NULL},
  {"InsertSeparator",      N_("Insert _separator"),       KZ_STOCK_SEPARATOR, "<control>P", NULL, G_CALLBACK(act_insert_separator), NULL},
  {"RemoveBookmarkItem",   N_("_Remove selected item"),   GTK_STOCK_DELETE,   "Delete",     NULL, G_CALLBACK(act_remove_selected_item), NULL},

  {"ShowHideFolderView",  N_("Folder view"),  NULL, NULL, NULL, G_CALLBACK(act_show_hide_folder_view),  NULL, TOGGLE_ACTION},
  {"ShowHideContentView", N_("Content view"), NULL, NULL, NULL, G_CALLBACK(act_show_hide_content_view), NULL, TOGGLE_ACTION},

  {"ListMode", N_("_List Mode"), NULL, NULL, NULL, G_CALLBACK(act_list_mode), NULL, RADIO_ACTION},
  {"TreeMode", N_("_Tree Mode"), NULL, NULL, NULL, G_CALLBACK(act_tree_mode), NULL, RADIO_ACTION, "ListMode"},

  {"GoUp",  N_("_Up"), GTK_STOCK_GO_UP, NULL, NULL, G_CALLBACK(act_go_up), NULL},
};
const gint kz_actions_bookmark_len = G_N_ELEMENTS(kz_actions_bookmark);


EggActionGroup *
kz_actions_bookmark_create_group (KzBookmarkEditor *editor,
				  GtkAccelGroup *accel_group)
{
	EggActionGroup *action_group;
	gint i;

	action_group = egg_action_group_new("KzBookmarkEditor");
	egg_action_group_set_accel_group(action_group, accel_group);

	for (i = 0; i < kz_actions_bookmark_len; i++)
		kz_actions_bookmark[i].user_data = editor;

	egg_action_group_add_actions (action_group,
				      kz_actions_bookmark,
				      kz_actions_bookmark_len);

	for (i = 0; i < kz_actions_bookmark_len; i++)
		kz_actions_bookmark[i].user_data = NULL;

	return action_group;
}


void
kz_actions_bookmark_popup_menu_modal (KzBookmarkEditor *editor,
				      guint button, guint time)
{
	GtkWidget *popup_menu = NULL;

	popup_menu = egg_menu_merge_get_widget(editor->menu_merge,
					       "/popups/BookmarksViewPopup");
	if (!popup_menu) return;

	g_signal_connect(G_OBJECT(popup_menu), "hide",
			 G_CALLBACK(cb_popup_menu_hide), editor);
	gtk_menu_popup(GTK_MENU(popup_menu), NULL, NULL,
		       NULL, NULL, button, time);
	gtk_main();

	g_signal_handlers_disconnect_by_func(G_OBJECT(popup_menu),
					     G_CALLBACK(cb_popup_menu_hide),
					     editor);
}
