/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "favoritelistview.h"

#include "libkita/favoritethreads.h"
#include "libkita/kita_misc.h"
#include "libkita/threadinfo.h"
#include "libkita/signalcollection.h"
#include "libkita/parsemisc.h"

#include <qmap.h>
#include <qapplication.h>
#include <qclipboard.h>
#include <qdatetime.h>
#include <qtoolbutton.h>
#include <qlabel.h>

#include <klocale.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kdebug.h>
#include <klistview.h>

FavoriteListView::FavoriteListView( QWidget* parent, const char* name )
        : Kita::ThreadListView( parent, name )
{
    subjectList->addColumn( i18n( "Board" ) );

    KindLabel->hide();

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( subjectList, SIGNAL( returnPressed( QListViewItem* ) ),
             SLOT( loadThread( QListViewItem* ) ) );
    connect( subjectList, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
             SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
    connect( ReloadButton, SIGNAL( clicked() ),
             SLOT( slotReloadButton() ) );
    connect( this, SIGNAL( openBoardRequested( const QString&, bool ) ),
             signalCollection, SIGNAL( openBoardRequested( const QString&, bool ) ) );
    connect( signalCollection, SIGNAL( favoritesUpdated() ),
             SLOT( update() ) );

    connect( signalCollection, SIGNAL( updateSubjectTab( const KURL& ) ),
             SLOT( slotUpdateSubject( const KURL& ) ) );  
}

FavoriteListView::~FavoriteListView()
{}

void FavoriteListView::update()
{
    // FIXME: KitaSubjectView::loadBoard()˥ԡ
    {
        m_hitList.clear();
        m_nextHitIndex = 0;
        m_prevquery = "";
    }

    const QValueList<QString> threadList = FavoriteThreads::getInstance() -> threadList();

    subjectList->clear();

    KitaThreadInfo* cache = KitaThreadInfo::getInstance();

    QValueList<QString>::const_iterator it;
    for ( it = threadList.begin(); it != threadList.end(); ++it ) {
        const Kita::Thread* thread = Kita::Thread::getByURL( *it );

        QDateTime since;
        since.setTime_t( KURL( thread->datURL() ).fileName().section( '.', 0, 0 ).toInt() );

        int readNum = cache->readNum( thread->datURL() );
        int resNum = KitaThreadInfo::resNum( thread->datURL() );

        KListViewItem* item = new KListViewItem( subjectList );
        item->setText( Col_Board, thread->boardName() );
        item->setText( Col_Subject, Kita::unescape( thread->name() ) );
        item->setText( Col_Read, QString( "%1" ).arg( readNum, 4 ) );
        if ( resNum > 0 ) {
            item->setText( Col_ResNum, QString( "%1" ).arg( resNum, 4 ) );
            if ( resNum != readNum ) {
                item->setText( Col_Unread, QString( "%1" ).arg( resNum - readNum, 4 ) );
            }
        }
        item->setText( Col_Since, since.toString( "yy/MM/dd hh:mm" ) );
        item->setText( Col_DatURL, thread->datURL() );
    }
}

void FavoriteListView::loadThread( QListViewItem* item )
{
    if ( ! item ) return ;

    const QValueList<QString> threadList = FavoriteThreads::getInstance() -> threadList();

    QValueList<QString>::const_iterator it;
    for ( it = threadList.begin(); it != threadList.end(); ++it ) {
        const Kita::Thread* thread = Kita::Thread::getByURL( *it );

        if ( thread->datURL() == item->text( Col_DatURL ) ) {
            emit showThreadRequested( thread->datURL(), false );
        }
    }
}

void FavoriteListView::slotUpdateSubject( const KURL& url )
{
    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    Kita::Thread* updated_thread = Kita::Thread::getByURLNew( datURL );
    for ( QListViewItem * item = subjectList->firstChild(); item; item = item->nextSibling() ) {
        if ( item->text( Col_DatURL ) == datURL.prettyURL() ) {
            int resNum = updated_thread->resNum();
            int readNum = KitaThreadInfo::readNum( updated_thread->datURL() );
            item->setText( Col_ResNum, QString( "%1" ).arg( resNum, 4 ) );
            item->setText( Col_Read, QString( "%1" ).arg( readNum, 4 ) );
        }
    }
}

// TODO: KitaSubjectView饹Ʊ᥽åɤȤۤȤƱ
void FavoriteListView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
    if ( item == 0 ) {
        return ;
    }

    KPopupMenu popup( 0 );
    popup.insertItem( i18n( "Open with Web Browser" ), 0 );
    popup.insertItem( i18n( "Open with new tab" ), 1 );
    popup.insertItem( i18n( "Copy title and URL" ), 2 );
    popup.insertItem( i18n( "Remove from Favorites" ), 3 );

    Kita::Thread* thread = Kita::Thread::getByURL( item->text( Col_DatURL ) );

    QClipboard* clipboard = QApplication::clipboard();

    switch ( popup.exec( point ) ) {
    case 0:
        KRun::runURL( thread->url(), "text/html" );
        break;
    case 1:
        emit showThreadRequested( thread->datURL(), true );
        break;
    case 2:
        clipboard->setText( thread->name() + "\n" + thread->url() );
        break;
    case 3:
        emit bookmarked( thread->datURL(), false );
        break;
    default:
        break;
    }
}

void FavoriteListView::slotReloadButton()
{
    QValueList<QString> boardList;
    
    const QValueList<QString> threadList = FavoriteThreads::getInstance() -> threadList();
    QValueList<QString>::const_iterator it;
    for ( it = threadList.begin(); it != threadList.end(); ++it ) {
        const Kita::Thread* thread = Kita::Thread::getByURL( *it );
        QString boardURL = Kita::datToBoard( thread->datURL() );
        if ( boardList.contains( boardURL ) == 0 ) {
            boardList.append( boardURL );
        }
    }

    QValueList<QString>::const_iterator it2;
    for ( it2 = boardList.begin(); it2 != boardList.end(); ++it2 ) {
        // Ĥsubject.txt
        Kita::Board::getThreadList( (*it2 ) );
    }
}

