/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kita.h"
#include "pref.h"
#include "kitathreadview.h"
#include "kitasubjectview.h"
#include "kitaboardview.h"
#include "libkita/threadinfo.h"
#include "libkita/favoriteboards.h"
#include "kitathreadtabwidget.h"
#include "kitasubjecttabwidget.h"
#include "kitanavi.h"

#include "libkita/favoritethreads.h"
#include "libkita/kita_misc.h"
#include "libkita/kitaconfig.h"
#include "libkita/signalcollection.h"
#include "libkita/parsemisc.h"

#include <qdragobject.h>
#include <qtextcodec.h>
#include <qevent.h>
#include <qregexp.h>

#include <klineedit.h>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <kaccel.h>
#include <kio/netaccess.h>
#include <kuserprofile.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kurl.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kdebug.h>
#include <ksimpleconfig.h>
#include <kedittoolbar.h>
#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kshortcut.h>
#include <kaboutdata.h>
#include <kparts/partmanager.h>

KitaMainWindow::KitaMainWindow()
        : KParts::DockMainWindow( 0, "Kita" ),
        m_printer( 0 )
{
    // accept dnd
    setAcceptDrops( true );

    // setup view, dock
    setupView();

    // then, setup our actions
    setupActions();

    // load cache
    loadCache();

    // load favorites
    loadFavorites();

    // load favorite boards;
    loadFavoriteBoards();

    // load completion
    loadCompletion();

    // load ascii art
    loadAsciiArt();

    // load abone lists
    loadAboneIDList();
    loadAboneNameList();
    loadAboneWordList();

    // and a status bar
    statusBar() ->show();

    readDockConfig( KGlobal::config() );

    readConfig( KGlobal::config() );

    // apply the saved mainwindow settings, if any, and ask the mainwindow
    // to automatically save settings if changed: window size, toolbar
    // position, icon size, etc.
    setAutoSaveSettings();

    // allow the view to change the statusbar and caption
    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( setMainCaption( const QString& ) ),
             SLOT( slotSetMainCaption( const QString& ) ) );

    connect( signalCollection, SIGNAL( setMainStatusbar( const QString& ) ),
             SLOT( slotSetMainStatusbar( const QString& ) ) );

    connect( signalCollection, SIGNAL( setMainURLLine( const KURL& ) ),
             SLOT( setUrl( const KURL& ) ) );

    connect( signalCollection, SIGNAL( switchSubjectView() ),
             SLOT( slotSwitchSubjectView() ));

    connect( this, SIGNAL( switchSubjectView() ),
	     signalCollection, SIGNAL( switchSubjectView() ));
    
    connect( m_subjectTab, SIGNAL( showThreadRequested( const QString&, bool ) ),
             SLOT( switchThreadView( const QString&, bool ) ) );

    connect( m_subjectTab, SIGNAL( showThreadRequested( const QString&, bool ) ),
             m_threadTab, SLOT( showThread( const QString&, bool ) ) );

    connect( signalCollection, SIGNAL( writeSucceeded() ),
             SLOT( slotWriteSucceeded() ) );

    connect( m_threadTab->manager(), SIGNAL( activePartChanged( KParts::Part* ) ),
             SLOT( createGUI( KParts::Part* ) ) );

    connect( m_subjectTab, SIGNAL( loadBoardCompleted( const KURL& ) ),
             SLOT( setUrl( const KURL& ) ) );

    connect( signalCollection, SIGNAL( bookmarked( const QString&, bool ) ),
             SLOT( bookmark( const QString&, bool ) ) );

    connect( m_urlLine, SIGNAL( returnPressed() ),
             SLOT( slotURLLine() ) );

    connect( this, SIGNAL( favoritesUpdated() ),
             signalCollection, SIGNAL( favoritesUpdated() ) );
    //QPopupMenu* popup = static_cast<QPopupMenu *>( factory() ->container( "window", this ) );
    //connect( popup, SIGNAL( aboutToShow() ),
    //         SLOT( windowMenuAboutToShow() ) );

    QPopupMenu* settingsPopup = static_cast<QPopupMenu *>( factory() ->container( "settings", this ) );
    connect( settingsPopup, SIGNAL( aboutToShow() ),
             SLOT( settingsMenuAboutToShow() ) );

    m_boardView->loadBoardList();
    m_boardView->setFocus();
    m_boardListAction->activate();
    m_subjectListAction->activate(); // dummy code
}

KitaMainWindow::~KitaMainWindow()
{
    if ( ! m_subjectListAction->isChecked() )
        makeDockVisible( m_subjectDock );

    saveAboneIDList();
    saveAboneNameList();
    saveAboneWordList();

    saveAsciiArt();

    saveCompletion();

    saveFavoriteBoards();

    saveFavorites();

    saveCache();

    saveMainWindowSettings( KGlobal::config(), "MainWindow" );

    writeConfig( KGlobal::config() );

    writeDockConfig( KGlobal::config() );
}

void KitaMainWindow::load( const KURL& url )
{
    QString target;
    // the below code is what you should normally do.  in this
    // example case, we want the url to our own.  you probably
    // want to use this code instead for your app

#if 0
    // download the contents
    if ( KIO::NetAccess::download( url, target ) )
    {
        // set our caption
        setCaption( url );

        // load in the file (target is always local)
        loadFile( target );

        // and remove the temp file
        KIO::NetAccess::removeTempFile( target );
    }
#endif

    setCaption( url.url() );
    //    m_view->openURL(url);
}

void KitaMainWindow::setupActions()
{
    KStdAction::quit( this, SLOT( close() ), actionCollection() );
    KStdAction::copy( this, SLOT( slotEditCopy() ), actionCollection() );

    m_toolbarAction = KStdAction::showToolbar( this,
                      SLOT( optionsShowToolbar() ),
                      actionCollection() );

    m_statusbarAction = KStdAction::showStatusbar( this,
                        SLOT( optionsShowStatusbar() ),
                        actionCollection() );

    m_boardListAction = new KToggleAction( i18n( "Show board list" ),
                                           0,
                                           this,
                                           SLOT( windowShowBoardlist() ),
                                           actionCollection(),
                                           "window_show_board_list" );

    m_subjectListAction = new KToggleAction( i18n( "Show subject list" ),
                          0,
                          this,
                          SLOT( windowShowSubjectlist() ),
                          actionCollection(),
                          "window_show_subject_list" );

    m_urlLine = new KLineEdit( "", 0 );
    new KWidgetAction( m_urlLine,
                       i18n( "URL Line" ),
                       0,
                       this,
                       SLOT( slotURLLine() )         /* dummy */,
                       actionCollection(), "url_line_action" );

    KStdAction::keyBindings( this, SLOT( optionsConfigureKeys() ), actionCollection() );
    KStdAction::configureToolbars( this, SLOT( optionsConfigureToolbars() ), actionCollection() );
    KStdAction::preferences( this, SLOT( optionsPreferences() ), actionCollection() );

    new KAction( i18n( "Load BoardList" ),
                 0,
                 m_boardView,
                 SLOT( loadBoardList() ),
                 actionCollection(),
                 "load_board_list" );

    new KAction( i18n( "reset windows" ),
                 0,
                 this,
                 SLOT( resetWindows() ),
                 actionCollection(),
                 "reset_win" );

    m_toggleViewAction = new KAction(
                             i18n( "&Toggle subject view and thread view" ),
                             KShortcut(Qt::Key_F7),
                             this,
                             SLOT( windowToggleView() ),
                             actionCollection(),
                             "window_toggle_view" );

    // action for subjecttabwidget
    new KAction( i18n( "Close Current Thread Tab" ),
                 KStdAccel::close(),
                 m_threadTab,
                 SLOT( closeCurrentTab() ),
                 actionCollection(),
                 "threadtab_closetab" );
    new KAction( i18n( "&Reload subject list"),
                 KShortcut(Qt::CTRL | Qt::Key_F5),
                 m_subjectTab,
                 SLOT( reloadSubjectList() ),
                 actionCollection(),
                 "threadlist_reload" );

    m_threadTab->manager()->addManagedTopLevelWidget( m_mainDock );
    setXMLFile( "kitaui.rc" );
    createGUI( m_threadTab->manager()->activePart() );
}

void KitaMainWindow::slotURLLine()
{
    QString text = m_urlLine->text();
    QString datURL = Kita::threadToDat( text );
    m_threadTab->showThread( datURL, KitaConfig::alwaysUseTab() );
}

void KitaMainWindow::writeConfig( KConfig* config )
{
    config->setGroup( "Global" );
    config->writeEntry( "Font", KitaConfig::font() );
    config->writeEntry( "ThreadFont", KitaConfig::threadFont() );
    config->writeEntry( "ShowMailAddress", KitaConfig::showMailAddress() );
    config->writeEntry( "MarkTime", KitaConfig::MarkTime() );
    config->writeEntry( "SortOrder", KitaConfig::sortOrder() );
    config->writeEntry( "AlwaysUseTab", KitaConfig::alwaysUseTab() );
    config->writeEntry( "ShowAA", KitaConfig::showAA() );
    config->writeEntry( "AfterShowNum", KitaConfig::afterShowNum() );
    config->writeEntry( "PreShowNum", KitaConfig::preShowNum() );
    config->writeEntry( "UsePart", KitaConfig::usePart() );
    config->writeEntry( "PartMimeList", KitaConfig::partMimeList() );

    config->setGroup( "Color" );
    config->writeEntry( "Popup", KitaConfig::popupColor() );
    config->writeEntry( "PopupBackground", KitaConfig::popupBackgroundColor() );
}

void KitaMainWindow::saveProperties( KConfig* config )
{
    // the 'config' object points to the session managed
    // config file.  anything you write here will be available
    // later when this app is restored

    //    if (m_view->currentURL() != QString::null)
    //        config->writeEntry("lastURL", m_view->currentURL());
    writeConfig( config );
}

void KitaMainWindow::readConfig( KConfig* config )
{
    config->setGroup( "Global" );
    KitaConfig::setFont( config->readFontEntry( "Font" ) );
    KitaConfig::setThreadFont( config->readFontEntry( "ThreadFont" ) );
    KitaConfig::setShowMailAddresss( config->readBoolEntry( "ShowMailAddress" ) );
    KitaConfig::setMarkTime( config->readUnsignedNumEntry( "MarkTime" , 24 ) );
    KitaConfig::setViewMode( config->readNumEntry( "ViewMode", KitaConfig::View_2pane ) );
    KitaConfig::setSortOrder( config->readNumEntry( "SortOrder", KitaConfig::Order_Mark ) );
    KitaConfig::setAlwaysUseTab( config->readBoolEntry( "AlwaysUseTab", false ) );
    KitaConfig::setShowAA( config->readBoolEntry( "ShowAA", false ) );
    KitaConfig::setAfterShowNum( config->readNumEntry( "AfterShowNum", 100 ) );
    KitaConfig::setPreShowNum( config->readNumEntry( "PreShowNum", 50 ) );
    KitaConfig::setUsePart( config->readBoolEntry( "UsePart", true ) );
    if ( config->hasKey( "PartMimeList" ) )
        KitaConfig::setPartMimeList( config->readListEntry( "PartMimeList" ) );
    else {
        // set partMimeList default
        QStringList list;
        list << "image/gif" << "image/jpeg" << "image/png" << "image/x-bmp";
        KitaConfig::setPartMimeList( list );
    }

    setFont( KitaConfig::font() );

    config->setGroup( "Color" );

    QColor popupColor = config->readColorEntry( "Popup" );
    if ( popupColor.isValid() )
    {
        KitaConfig::setPopupColor( popupColor );
    }

    QColor popupBackgroundColor = config->readColorEntry( "PopupBackground" );
    if ( popupColor.isValid() )
    {
        KitaConfig::setPopupBackgroundColor( popupBackgroundColor );
    }
}

void KitaMainWindow::readProperties( KConfig* config )
{
    // the 'config' object points to the session managed
    // config file.  this function is automatically called whenever
    // the app is being restored.  read in here whatever you wrote
    // in 'saveProperties'

    //    QString url = config->readEntry("lastURL");

    //    if (url != QString::null)
    //        m_view->openURL(KURL(url));
    readConfig( config );
}

void KitaMainWindow::dragEnterEvent( QDragEnterEvent* event )
{
    // accept uri drops only
    event->accept( QUriDrag::canDecode( event ) );
}

void KitaMainWindow::dropEvent( QDropEvent* event )
{
    // this is a very simplistic implementation of a drop event.  we
    // will only accept a dropped URL.  the Qt dnd code can do *much*
    // much more, so please read the docs there
    QStrList uri;

    // see if we can decode a URI.. if not, just ignore it
    if ( QUriDrag::decode( event, uri ) )
    {
        // okay, we have a URI.. process it
        QString url, target;
        url = uri.first();

        // load in the file
        load( KURL( url ) );
    }
}

void KitaMainWindow::fileNew()
{
    // this slot is called whenever the File->New menu is selected,
    // the New shortcut is pressed (usually CTRL+N) or the New toolbar
    // button is clicked

    // create a new window
    ( new KitaMainWindow ) ->show();
}

void KitaMainWindow::optionsShowToolbar()
{
    // this is all very cut and paste code for showing/hiding the
    // toolbar
    if ( m_toolbarAction->isChecked() )
    {
        toolBar() ->show();
    }
    else
    {
        toolBar() ->hide();
    }
}

void KitaMainWindow::optionsShowStatusbar()
{
    // this is all very cut and paste code for showing/hiding the
    // statusbar
    if ( m_statusbarAction->isChecked() )
    {
        statusBar() ->show();
    }
    else
    {
        statusBar() ->hide();
    }
}

void KitaMainWindow::optionsConfigureKeys()
{
    KKeyDialog dlg( TRUE, this);
    QPtrList<KXMLGUIClient> actList = guiFactory()->clients();
    for ( QPtrListIterator<KXMLGUIClient> act( actList ) ; act.current() ; ++act )
    {
        const KAboutData *dat = (*act)->instance()->aboutData();
        dlg.insert( (*act)->actionCollection(),dat->programName() );
    }
    dlg.configure();
}

void KitaMainWindow::optionsConfigureToolbars()
{
    // use the standard toolbar editor
    KEditToolbar dlg( factory() );
    connect( &dlg, SIGNAL( newToolbarConfig() ),
             SLOT( newToolbarConfig() ) );
    dlg.exec();
}

void KitaMainWindow::newToolbarConfig()
{
    // this slot is called when user clicks "Ok" or "Apply" in the toolbar editor.
    // recreate our GUI, and re-apply the settings (e.g. "text under icons", etc.)
    applyMainWindowSettings( KGlobal::config(), "MainWindow" );
}

void KitaMainWindow::windowShowBoardlist()
{
    if ( m_boardListAction->isChecked() )
    {
        makeDockVisible( m_boardDock );
    }
    else
    {
        makeDockInvisible( m_boardDock );
    }
}

void KitaMainWindow::windowShowSubjectlist()
{
    if ( m_subjectListAction->isChecked() )
    {
        makeDockVisible( m_subjectDock );
    }
    else
    {
        makeDockInvisible( m_subjectDock );
    }
}

void KitaMainWindow::resetWindows()
{
    m_boardDock->manualDock( m_mainDock, KDockWidget::DockLeft, 20 );
    m_subjectDock->manualDock( m_mainDock, KDockWidget::DockTop, 30 );
}


void KitaMainWindow::windowToggleView()
{
    //FIXME: m_subjectDock->dockSite() & KDockWidget::DockCenter != 0 ʳΤȤϤɤ롩
    kdDebug() << "Do toggle" << endl;
    if ( !m_subjectListAction->isChecked() ) return;
    if( !m_subjectDock->isVisible()
	|| !m_subjectDock->isActiveWindow() ) emit switchSubjectView();
    else{
    KDockWidget *w = m_mainDock;
    kdDebug() << QString( "toggle target: %1" ).arg( w->name() ) << endl;
    w->makeDockVisible();
    w->getWidget()->setFocus();
    }
}

void KitaMainWindow::windowMenuAboutToShow()
{
    //m_boardListAction->setChecked( m_boardDock->isVisible() );
    //m_subjectListAction->setChecked( m_subjectDock->isVisible() );
    //m_toggleViewAction->setEnabled(m_subjectDock->dockSite() & KDockWidget::DockCenter);
}

void KitaMainWindow::windowCloseTab()
{
    m_threadTab->closeCurrentTab();
}

void KitaMainWindow::settingsMenuAboutToShow()
{
    m_toolbarAction->setChecked( toolBar() ->isVisible() );
    m_statusbarAction->setChecked( statusBar() ->isVisible() );
}

void KitaMainWindow::optionsPreferences()
{
    // popup some sort of preference dialog, here
    KitaPreferences dlg;

    connect( &dlg, SIGNAL( fontChanged( const QFont& ) ),
             SLOT( setFont( const QFont& ) ) );

    // XXX
    dlg.setInitialSize( QSize( 600, 400 ) );
    if ( dlg.exec() )
    {
        // redo your settings
    }
}

void KitaMainWindow::slotSetMainStatusbar( const QString& statusStr )
{
    // display the text on the statusbar
    statusBar() ->message( statusStr );
}

void KitaMainWindow::setupView()
{
    m_mainDock = createDockWidget( "main", 0L, 0L, i18n( "main" ), i18n( "main" ) );
    m_threadTab = new KitaThreadTabWidget( m_mainDock );
    m_mainDock->setWidget( m_threadTab );
    m_mainDock->setDockSite( KDockWidget::DockFullSite );
    m_mainDock->setEnableDocking( KDockWidget::DockNone );
    connect( Kita::SignalCollection::getInstance(), SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ),
             SLOT( slotOpenURLRequest( const KURL&, const KParts::URLArgs& ) ) );

    /* emit when this widget is deactivated */
    connect( this, SIGNAL( windowDeactivated() ),
             Kita::SignalCollection::getInstance(), SIGNAL( windowDeactivated() ));

    /* If this widget is active and receives
       signal isKitaActive, then emit signal kitaIsActive. */
    connect( Kita::SignalCollection::getInstance(), SIGNAL( isKitaActive() ),
             this, SLOT( slotIsKitaActive() ) );

    connect( this, SIGNAL( kitaIsActive() ),
             Kita::SignalCollection::getInstance(), SIGNAL( kitaIsActive() ) );

    setView( m_mainDock );
    setMainDockWidget( m_mainDock );

    m_subjectDock = createDockWidget( "subject", 0L, 0L, i18n( "subject" ), i18n( "subject" ) );
    m_subjectTab = new KitaSubjectTabWidget( m_subjectDock );
    m_subjectDock->setWidget( m_subjectTab );
    m_subjectDock->setDockSite( KDockWidget::DockNone );
    connect( m_subjectDock, SIGNAL( headerCloseButtonClicked() ), SLOT( closeSubjectView() ) );
    connect( m_subjectDock, SIGNAL( iMBeingClosed() ), SLOT( closeSubjectView() ) );

    m_boardDock = createDockWidget( "board", 0L, 0L, i18n( "board" ), i18n( "board" ) );
    m_boardView = new KitaBoardView( m_boardDock );
    m_boardDock->setWidget( m_boardView );
    m_boardDock->setDockSite( KDockWidget::DockNone );
    connect( m_boardDock, SIGNAL( headerCloseButtonClicked() ), SLOT( closeBoardView() ) );
    connect( m_boardDock, SIGNAL( iMBeingClosed() ), SLOT( closeBoardView() ) );

    m_boardDock->manualDock( m_mainDock, KDockWidget::DockLeft, 20 );
    m_subjectDock->manualDock( m_mainDock, KDockWidget::DockTop, 30 );

    /* setup kitanavi */
    KitaNavi::createKitaNavi(this,m_threadTab);
}

void KitaMainWindow::closeSubjectView()
{
    m_subjectListAction->setChecked( FALSE );
}

void KitaMainWindow::closeBoardView()
{
    m_boardListAction->setChecked( FALSE );
}

void KitaMainWindow::slotOpenURLRequest( const KURL& url, const KParts::URLArgs& args )
{
    kdDebug() << QString( "Requested OpenURL: \"%1\"" ).arg( url.url() ) << endl;
    kdDebug() << QString( "content type: \"%1\"" ).arg( args.contentType() ) << endl;
    kdDebug() << QString( "service type: \"%1\"" ).arg( args.serviceType ) << endl;
    kdDebug() << QString( "NewTab: %1, reload: %2" )
    .arg( args.newTab() ? "yes" : "no" )
    .arg( args.reload ? "yes" : "no" ) << endl;
    KURL datURL = Kita::ParseMisc::parseURLonly( url );

    QRegExp regexp( "http://([^/]*)/([^/]*)/dat/(.*)\\.dat" );
    if ( regexp.search( datURL.url() ) == -1 ) {
        if ( KitaConfig::usePart() ) {
            QString mimetype = KIO::NetAccess::mimetype( url );
            kdDebug() << QString( "queried MIME type: \"%1\"" ).arg( mimetype ) << endl;
            if ( KitaConfig::partMimeList().find( mimetype ) != KitaConfig::partMimeList().end() ) {
                KService::Ptr service = KServiceTypeProfile::preferredService( mimetype, "KParts/ReadOnlyPart" );
                if ( service ) {
                    QString libName = QFile::encodeName( service->library() );
                    m_threadTab->showAlternativeView( url, libName, mimetype );
                    return;
                }
            }
        }
        new KRun( url );
    } else {
        m_threadTab->showThread( url.prettyURL(), true );
    }
}

void KitaMainWindow::loadCache()
{
    KitaThreadInfo * cache = KitaThreadInfo::getInstance();
    QString cacheConfigPath = locateLocal( "appdata", "cache" );
    QFile file( cacheConfigPath );
    if ( file.open( IO_ReadOnly ) )
    {
        QDataStream stream( &file );
        stream >> *cache;
    }
}

void KitaMainWindow::saveCache()
{
    KitaThreadInfo * cache = KitaThreadInfo::getInstance();
    QString cacheConfigPath = locateLocal( "appdata", "cache" );
    QFile file( cacheConfigPath );
    if ( file.open( IO_WriteOnly ) )
    {
        QDataStream stream( &file );
        stream << *cache;
    }
}

/*!
    \fn KitaMainWindow::setFont(QFont& font)
 */

void KitaMainWindow::setFont( const QFont& font )
{
    m_subjectTab->setFont( font );
    m_boardView->setFont( font );
}


void KitaMainWindow::slotWriteSucceeded()
{
    statusBar() ->message( i18n( "posting succeeded." ) );
}

void KitaMainWindow::setUrl( const KURL& url )
{
    m_urlLine->setText( url.url() );
}

void KitaMainWindow::bookmark( const QString& datURL, bool on )
{
    FavoriteThreads * favorite = FavoriteThreads::getInstance();

    if ( on )
    {
        favorite->insert( datURL );
    }
    else
    {
        favorite->remove( datURL );
    }
    emit favoritesUpdated();
}

void KitaMainWindow::loadFavorites()
{
    QString favoritesConfigPath = locateLocal( "appdata", "favorites.xml" );
    QFile file( favoritesConfigPath );
    if ( file.open( IO_ReadOnly ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QString xml = stream.read();
        FavoriteThreads::readFromXML( xml );
    }
}

void KitaMainWindow::saveFavorites()
{
    QString favoritesConfigPath = locateLocal( "appdata", "favorites.xml" );
    QFile file( favoritesConfigPath );
    if ( file.open( IO_WriteOnly ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );
        stream << FavoriteThreads::getInstance() ->toXML();
    }
}

void KitaMainWindow::loadCompletion()
{
    QString configPath = locateLocal( "appdata", "completion" );
    KSimpleConfig config( configPath );

    KitaConfig::setNameCompletionItems( config.readListEntry( "name" ) );
}

void KitaMainWindow::saveCompletion()
{
    QString configPath = locateLocal( "appdata", "completion" );
    KSimpleConfig config( configPath );

    config.writeEntry( "name", KitaConfig::nameCompletion() ->items() );
    config.sync();
}

void KitaMainWindow::loadAsciiArt()
{
    QString configPath = locateLocal( "appdata", "asciiart" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        while ( ( str = stream.readLine() ) != QString::null )
        {
            if ( ! str.isEmpty() )
            {
                list << str;
            }
        }
        KitaConfig::setAsciiArtList( list );
    }
}

void KitaMainWindow::saveAsciiArt()
{
    QString configPath = locateLocal( "appdata", "asciiart" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) )
    {
        QTextStream stream( &file ) ;
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list = KitaConfig::asciiArtList();
        QStringList::iterator it;
        for ( it = list.begin(); it != list.end(); ++it )
        {
            stream << ( *it ) << endl;
        }
    }
}

void KitaMainWindow::loadFavoriteBoards()
{
    QString configPath = locateLocal( "appdata", "favorite_boards.xml" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QString xml = stream.read();
        Kita::FavoriteBoards::readFromXML( xml );
    }
}

void KitaMainWindow::saveFavoriteBoards()
{
    QString configPath = locateLocal( "appdata", "favorite_boards.xml" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );
        stream << Kita::FavoriteBoards::toXML();
    }
}

void KitaMainWindow::loadAboneIDList()
{
    QString configPath = locateLocal( "appdata", "abone_id" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        while ( ( str = stream.readLine() ) != QString::null )
        {
            if ( ! str.isEmpty() )
            {
                list << str;
            }
        }
        KitaConfig::setAboneIDList( list );
    }
}

void KitaMainWindow::saveAboneIDList()
{
    QString configPath = locateLocal( "appdata", "abone_id" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) )
    {
        QTextStream stream( &file ) ;
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list = KitaConfig::aboneIDList();
        QStringList::iterator it;
        for ( it = list.begin(); it != list.end(); ++it )
        {
            stream << ( *it ) << endl;
        }
    }
}

void KitaMainWindow::loadAboneNameList()
{
    QString configPath = locateLocal( "appdata", "abone_name" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        while ( ( str = stream.readLine() ) != QString::null )
        {
            if ( ! str.isEmpty() )
            {
                list << str;
            }
        }
        KitaConfig::setAboneNameList( list );
    }
}

void KitaMainWindow::saveAboneNameList()
{
    QString configPath = locateLocal( "appdata", "abone_name" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) )
    {
        QTextStream stream( &file ) ;
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list = KitaConfig::aboneNameList();
        QStringList::iterator it;
        for ( it = list.begin(); it != list.end(); ++it )
        {
            stream << ( *it ) << endl;
        }
    }
}

void KitaMainWindow::loadAboneWordList()
{
    QString configPath = locateLocal( "appdata", "abone_word" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        while ( ( str = stream.readLine() ) != QString::null )
        {
            if ( ! str.isEmpty() )
            {
                list << str;
            }
        }
        KitaConfig::setAboneWordList( list );
    }
}

void KitaMainWindow::saveAboneWordList()
{
    QString configPath = locateLocal( "appdata", "abone_word" );
    QFile file( configPath );
    if ( file.open( IO_WriteOnly ) )
    {
        QTextStream stream( &file ) ;
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list = KitaConfig::aboneWordList();
        QStringList::iterator it;
        for ( it = list.begin(); it != list.end(); ++it )
        {
            stream << ( *it ) << endl;
        }
    }
}

void KitaMainWindow::slotSwitchSubjectView()
{
    if ( m_subjectListAction->isChecked() )
        m_subjectDock->makeDockVisible();
}

void KitaMainWindow::switchThreadView( const QString&, bool )
{
    m_mainDock->makeDockVisible();
}

void KitaMainWindow::slotSetMainCaption( const QString& captionStr )
{
    setCaption( captionStr );
}

void KitaMainWindow::slotEditCopy()
{
    QWidget * widget = kapp->focusWidget();

    if ( widget )
    {
        QKeyEvent e( QEvent::KeyPress, Key_C, 'C', ControlButton );
        QApplication::sendEvent( widget, &e );
    }
}

/* when window is deactivated, emit signal to popup */ /* protected */
void KitaMainWindow::windowActivationChange ( bool  )
{
    if(!isActiveWindow()) emit windowDeactivated();
}

/* private slot */
void KitaMainWindow::slotIsKitaActive()
{
    if(isActiveWindow()) emit kitaIsActive();
}


#include "kita.moc"
