/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include <kio/netaccess.h>

#include <qfile.h>
#include <qtextcodec.h>
#include <qstringlist.h>
#include <qregexp.h>
#include <klistview.h>
#include <kpopupmenu.h>
#include <klocale.h>
#include <krun.h>
#include <kdebug.h>
#include <qvaluelist.h>
#include <qheader.h>

#include "kitaboardview.h"
#include "kita.h"
#include "part/kita2ch.h"
#include "libkita/qcp932codec.h"
#include "libkita/favoriteboards.h"
#include "libkita/board.h"
#include "libkita/category.h"
#include "libkita/kitaconfig.h"
#include "libkita/signalcollection.h"

QCp932Codec* KitaBoardView::m_cp932Codec = NULL;

KitaBoardView::KitaBoardView( QWidget *parent, const char *name )
        : KitaBoardViewBase( parent, name )
        , m_favorites( 0 )
{
    m_boardList->setSorting( -1 );
    m_boardList->addColumn( i18n( "board name" ) );
    m_boardList->header()->setClickEnabled( FALSE );
    
    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( m_boardList, SIGNAL( mouseButtonClicked( int, QListViewItem*, const QPoint&, int ) ),
             SLOT( slotMouseButtonClicked( int, QListViewItem* ) ) );
    connect( m_boardList, SIGNAL( returnPressed( QListViewItem* ) ), SLOT( loadBoard( QListViewItem* ) ) );
    connect( m_boardList, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
             SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
    connect( Kita::FavoriteBoards::getInstance(), SIGNAL( changed() ), SLOT( refreshFavoriteBoards() ) );
    connect( this, SIGNAL( openBoardRequested( const QString&, bool ) ),
             signalCollection, SIGNAL( openBoardRequested( const QString&, bool ) ) );
}

KitaBoardView::~KitaBoardView()
{}

QString KitaBoardView::getCategory( const QString& line ) const
{
    QRegExp regexp( "<BR><BR><B>(.*)</B><BR>", false );
    if ( regexp.search( line ) != -1 ) {
        return regexp.cap( 1 );
    } else {
        return QString::null;
    }
}

QString KitaBoardView::getBoardURL( const QString& line ) const
{
    QRegExp regexp( "<A HREF=([^ ]*).*>(.*)</A>", false );
    if ( regexp.search( line ) != -1 ) {
        QString board_url = regexp.cap( 1 );
        QString board_title = regexp.cap( 2 );
        Kita::Board::setName( board_url, board_title );
        return board_url;
    } else {
        return QString::null;
    }
}

bool KitaBoardView::isBoardURL( const QString& url ) const
{
    QRegExp url_2ch( "http://.*\\.2ch\\.net/.*" );
    QRegExp url_bbspink( "http://.*\\.bbspink\\.com/.*" );
    QRegExp url_www_2ch( "http://www\\.2ch\\.net/.*" );
    QRegExp url_machibbs( "http://.*\\.machi\\.to/.*" );

    if ( url.isEmpty() ) return false;

    if ( url_2ch.search( url ) == -1 && url_bbspink.search( url ) == -1
            && url_machibbs.search( url ) == -1 ) return false;
    if ( url_www_2ch.search( url ) != -1 ) return false;

    return true;
}

void KitaBoardView::loadBoardList()
{
    QString tmpFile;
    QString url = KitaConfig::boardListURL();
    if ( ! KIO::NetAccess::download( url, tmpFile ) ) {
        return ;
    }

    QFile file( tmpFile );
    if ( ! file.open( IO_ReadOnly ) ) {
        return ;
    }

    QTextStream stream( &file );
    if( !m_cp932Codec ) m_cp932Codec = new QCp932Codec();
    stream.setCodec( m_cp932Codec );
    QValueList<Kita::Category> categoryList = getCategoryList( stream.read() );

    // clear list
    m_boardList->clear();
    m_favorites = NULL;

    KListViewItem* previous = 0;
    QValueList<Kita::Category>::iterator it;
    for ( it = categoryList.begin(); it != categoryList.end(); ++it ) {
        Kita::Category category = (*it);
        QValueList<QString> board_url_list = category.getBoardURLList();

        if ( board_url_list.isEmpty() ) {
            continue;
        }

        KListViewItem* categoryItem = new KListViewItem( m_boardList, previous, category.name() );

        KListViewItem* previousBoard = 0;
        QValueList<QString>::iterator it;
        for ( it = board_url_list.begin(); it != board_url_list.end(); ++it ) {
            QString boardURL = *it;
            QString boardName = Kita::Board::getName( *it );
            previousBoard = new KListViewItem( categoryItem, previousBoard, boardName, boardURL );
        }
        previous = categoryItem;
    }

    refreshFavoriteBoards();
}

QValueList<Kita::Category> KitaBoardView::getCategoryList( const QString& html ) const
{
    QValueList<Kita::Category> result;

    QStringList lines = QStringList::split( "\n", html );
    QStringList::iterator it;

    Kita::Category current_category;
    for ( it = lines.begin(); it != lines.end(); ++it ) {
        QString category_name = getCategory( *it );
        if ( category_name != QString::null ) {
            if ( ! current_category.name().isEmpty() ) {
                result.append( current_category );
            }
            current_category = Kita::Category( category_name );
        } else {
            QString boardURL = getBoardURL( *it );

            if ( isBoardURL( boardURL ) ) {
                if ( ! current_category.name().isEmpty() ) {
                    current_category.append( boardURL );
                }
            }
        }
    }
    return result;
}

void KitaBoardView::refreshFavoriteBoards()
{
    if ( ! m_favorites ) {
        m_favorites = new KListViewItem( m_boardList, 0, i18n( "Favorites" ) );
    }

    do {
        QListViewItem* child = m_favorites->firstChild();
        delete child;
    } while ( m_favorites->childCount() != 0 );

    QValueList<KURL> boards = Kita::FavoriteBoards::boards();
    QValueList<KURL>::iterator it;

    for ( it = boards.begin(); it != boards.end(); ++it ) {
        QString boardURL = ( *it ).url();
        QString boardName = Kita::Board::getName( boardURL );
        new KListViewItem( m_favorites, 0, boardName, boardURL );
    }
}

void KitaBoardView::loadBoard( QListViewItem* item )
{
    if ( ! item ) return ;

    QString boardName = item->text( 0 );
    QString boardURL = item->text( 1 );

    if ( boardURL.isEmpty() ) {
        return ;
    }

    if ( KitaConfig::alwaysUseTab() ) {
        emit openBoardRequested( boardURL, true );
    } else {
        emit openBoardRequested( boardURL, false );
    }
}

void KitaBoardView::setFont( const QFont& font )
{
    m_boardList->setFont( font );
}

void KitaBoardView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
    if ( item == 0 ) {
        return ;
    }

    KPopupMenu popup( 0 );
    popup.insertItem( i18n( "Open with Web Browser" ), 0 );
    popup.insertItem( i18n( "Open with New Tab" ), 1 );
    if ( item->parent() == m_favorites ) {
        popup.insertItem( i18n( "Remove from Favorites" ), 3 );
    } else {
        popup.insertItem( i18n( "Add to Favorites" ), 2 );
    }

    KURL url = item->text( 1 );

    switch ( popup.exec( point ) ) {
    case 0:
        KRun::runURL( url, "text/html" );
        break;
    case 1:
        emit openBoardRequested( url.url(), true );
        break;
    case 2:
        Kita::FavoriteBoards::append( url );
        break;
    case 3:
        Kita::FavoriteBoards::remove( url );
        break;
    default:
        break;
    }
}

void KitaBoardView::slotMouseButtonClicked( int button, QListViewItem* item )
{
    if ( ! item ) return ;
    
    QString boardName = item->text( 0 );
    QString boardURL = item->text( 1 );
    if ( boardURL.isEmpty() ) {
        m_boardList->setOpen( item, !item->isOpen() );
        return ;
    }

    emit sigShowSubject();
    
    switch ( button ) {
    case MidButton:
        emit openBoardRequested( boardURL, true );
        break;
    case LeftButton:
        if ( KitaConfig::alwaysUseTab() ) {
            emit openBoardRequested( boardURL, true );
        } else {
            emit openBoardRequested( boardURL, false );
        }
        break;
    }
}
