/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitasubjectview.h"

// include files for Qt
#include <qtoolbutton.h>
#include <qtextbrowser.h>
#include <qtextcodec.h>
#include <qregexp.h>
#include <qfile.h>
#include <qclipboard.h>
#include <qapplication.h>
#include <qdatetime.h>
#include <qlabel.h>
#include <qmessagebox.h>

// kdelibs/kio
#include <kio/slaveconfig.h>
#include <kio/netaccess.h>

// include files for KDE
#include <kwin.h>
#include <kfilterdev.h>
#include <klistview.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kdebug.h>

#include "libkita/threadinfo.h"
#include "libkita/thread.h"
#include "libkita/kita_misc.h"
#include "libkita/kitaconfig.h"
#include "libkita/favoritethreads.h"
#include "libkita/datmanager.h"
#include "libkita/parsemisc.h"
#include "libkita/signalcollection.h"
#include "part/kita2ch.h"

KitaSubjectView::KitaSubjectView( QWidget* parent, const char* name )
        : Kita::ThreadListView( parent, name )
        , m_unreadNum( 0 ), m_readNum( 0 ), m_newNum( 0 )
{
    connect( subjectList, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
             SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
    connect( subjectList, SIGNAL( returnPressed( QListViewItem* ) ),
             SLOT( loadThread( QListViewItem* ) ) );
    connect( ReloadButton, SIGNAL( clicked() ),
             SLOT( reloadSubject() ) );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( updateSubjectTab( const KURL& ) ),
             SLOT( slotUpdateSubject( const KURL& ) ) );
    connect( this, SIGNAL( closeThreadTab( const KURL& ) ),
	     signalCollection, SIGNAL( closeThreadTab( const KURL& ) ));
}

KitaSubjectView::~KitaSubjectView()
{}

void KitaSubjectView::reloadSubject()
{
    if ( ! m_boardURL.isEmpty() ) {
        loadBoard( m_boardURL );
    }
}

const QString KitaSubjectView::boardURL()
{
    return m_boardURL;
}

const QString KitaSubjectView::boardName()
{
    return Kita::Board::getName( m_boardURL );
}

void KitaSubjectView::loadThread( QListViewItem* item )
{
    if ( ! item ) return ;

    KURL datURL = item->text( Col_DatURL );

    Kita::Thread* thread = Kita::Thread::getByURL( datURL.url() );

    if ( KitaConfig::alwaysUseTab() ) {
        emit showThreadRequested( thread->datURL(), true );
    } else {
        emit showThreadRequested( thread->datURL(), false );
    }
}
/*
ͥ
ts_readed   : ɤ1000ĶƤ
ts_normal   : ̤1ʾФäƤ
ts_new      : ̤1Τ
ts_read   : ɤ1000ʲΤ
ts_hasunread: ̤ɤ
*/

enum {
    ts_readed,
    ts_normal,
    ts_new,
    ts_read,
    ts_hasunread
};

void KitaSubjectView::loadBoard( const QString& boardURL )
{
    setActiveWindow();
    topLevelWidget() ->raise();
    QDateTime currentdat = QDateTime::currentDateTime();

    // reset member variables.
    // FIXME: FavoriteListView::update()˥ԡ
    {
        m_hitList.clear();
        m_nextHitIndex = 0;
        m_prevquery = "";
    }

    m_boardURL = boardURL;
    m_unreadNum = 0;
    m_readNum = 0;
    m_newNum = 0;

    KitaThreadInfo* cache = KitaThreadInfo::getInstance();
    QPtrList<Kita::Thread> threadList = Kita::Board::getThreadList( boardURL );

    // clear list
    subjectList->clear();

    for ( unsigned i = 0; i < threadList.count(); i++ ) {
        Kita::Thread* thread = threadList.at( i );

        QDateTime since;
        since.setTime_t( Kita::datToSince( thread->datURL() ) );

        int readNum = cache->readNum( thread->datURL() );
        int resNum = thread->resNum();

        Kita::ThreadListViewItem* item = new Kita::ThreadListViewItem( subjectList );
        item->setText( Col_ID, QString( "%1" ).arg( i + 1, 4 ) );
        item->setText( Col_Subject, Kita::unescape( thread->name() ) );
        item->setText( Col_ResNum, QString( "%1" ).arg( resNum, 4 ) );
        item->setText( Col_Read, ( readNum > 0 ) ? QString( "%1" ).arg( readNum, 4 ) : QString( "" ) );
        item->setText( Col_Unread, ( readNum > 0 && resNum != readNum ) ? QString( "%1" ).arg( resNum - readNum, 4 ) : QString( "" ) );
        item->setText( Col_Board, thread->boardName() );
        item->setText( Col_Since, since.toString( "yy/MM/dd hh:mm" ) );
        item->setText( Col_DatURL, thread->datURL() );

        // FIXME: updateThread˥ԡƤ
        // TODO: 'N'Υ
        if ( readNum > 0 && resNum != readNum ) {
            // ̤ɤ
            item->setPixmap( Col_Mark, SmallIcon( "unread" ) );
            m_unreadNum++;
            item->setText( Col_MarkOrder, QString::number( ts_hasunread ) );
        } else if ( readNum > 0 ) {
            // 
            item->setPixmap( Col_Mark, SmallIcon( "read" ) );
            m_readNum++;
            //if( readNum > 1000) item->setText( Col_MarkOrder, "0" );
            //else item->setText( Col_MarkOrder, "3" );
            item->setText( Col_MarkOrder, QString::number( ( readNum > 1000 ) ? ts_readed : ts_read ) );
        } else if ( since.secsTo( currentdat ) < 3600 * KitaConfig::MarkTime() ) {
            item->setPixmap( Col_Mark, SmallIcon( "newthread" ) );
            m_newNum++;
            item->setText( Col_MarkOrder, QString::number( ts_new ) );
        } else {
            item->setText( Col_MarkOrder, QString::number( ts_normal ) );
        }
    }

    if ( HideButton->isOn() ) {
        HideButton->toggle();
    }
    emit loadBoardCompleted( boardURL );

    switch ( KitaConfig::sortOrder() ) {
    case KitaConfig::Order_Mark:
        subjectList->setSorting( Col_Mark );
        break;
    case KitaConfig::Order_ID:
        subjectList->setSorting( Col_ID );
        break;
    default:
        // do nothing
        break;
    }

    subjectList->setFocus();
    UpdateKindLabel();
}

/* public */
void KitaSubjectView::setFocusList()
{
    setActiveWindow();
    topLevelWidget()->raise();
    subjectList->setFocus();
}


void KitaSubjectView::UpdateKindLabel()
{
    QString fmtstr;
    fmtstr += QString( "<font color=#C50000>%1</font>" ).arg( m_unreadNum );
    fmtstr += QString( "/<font color=#00C200>%1</font>" ).arg( m_readNum );
    fmtstr += QString( "/<font color=#5AAAFF>%1</font>" ).arg( m_newNum );
    KindLabel->setText( fmtstr );
}

void KitaSubjectView::setFont( const QFont& font )
{
    subjectList->setFont( font );
}

void KitaSubjectView::slotUpdateSubject( const KURL& url )
{
    QDateTime currentdat = QDateTime::currentDateTime();

    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    Kita::Thread* updated_thread = Kita::Thread::getByURLNew( datURL );
    for ( QListViewItem * item = subjectList->firstChild(); item; item = item->nextSibling() ) {
        if ( item->text( Col_DatURL ) == datURL.prettyURL() ) {
	    QDateTime since;
	    since.setTime_t( Kita::datToSince( datURL.prettyURL() ) );
	    
            int resNum = updated_thread->resNum();
            int readNum = KitaThreadInfo::readNum( updated_thread->datURL() );
            item->setText( Col_ResNum, QString( "%1" ).arg( resNum, 4 ) );
	    item->setText( Col_Read, ( readNum > 0 ) ? QString( "%1" ).arg( readNum, 4 ) : QString( "" ) );
            item->setText( Col_Unread, readNum > 0 && resNum != readNum ? QString( "%1" ).arg( resNum - readNum, 4 ) : QString( "" ) );

            switch ( item->text( Col_MarkOrder ).toInt() ) {
                case ts_readed    :
                case ts_read      : m_readNum--; break;
                case ts_new       : m_newNum--; break;
                case ts_hasunread : m_unreadNum--; break;
            }

            if ( readNum > 0 && resNum != readNum ) {
                // ̤ɤ
                item->setPixmap( Col_Mark, SmallIcon( "unread" ) );
                m_unreadNum++;
                item->setText( Col_MarkOrder, QString::number( ts_hasunread ) );
            } else if ( readNum > 0 ) {
                // 
                item->setPixmap( Col_Mark, SmallIcon( "read" ) );
                m_readNum++;
                //if( readNum > 1000) item->setText( Col_MarkOrder, "0" );
                //else item->setText( Col_MarkOrder, "3" );
                item->setText( Col_MarkOrder, QString::number( ( readNum > 1000 ) ? ts_readed : ts_read ) );
            } else if ( since.secsTo( currentdat ) < 3600 * KitaConfig::MarkTime() ) {
                item->setPixmap( Col_Mark, SmallIcon( "newthread" ) );
                m_newNum++;
                item->setText( Col_MarkOrder, QString::number( ts_new ) );
            } else {
                item->setPixmap( Col_Mark, NULL );	    
                item->setText( Col_MarkOrder, QString::number( ts_normal ) );
            }
            UpdateKindLabel();
        }
    }
}

void KitaSubjectView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
    if ( item == 0 ) {
        return ;
    }

    Kita::Thread* thread = Kita::Thread::getByURL( item->text( Col_DatURL ) );
    bool isFavorites = FavoriteThreads::getInstance()->contains( thread->datURL() );
    
    KPopupMenu popup( 0 );
    popup.insertItem( i18n( "Open with Web Browser" ), 0 );
    popup.insertItem( i18n( "Open with new tab" ), 1 );
    popup.insertItem( i18n( "Copy title and URL" ), 2 );
    if ( isFavorites ) {
        popup.insertItem( i18n( "Remove from Favorites" ), 3 );
    } else {
        popup.insertItem( i18n( "Add to Favorites" ), 3 );
    }
    popup.insertSeparator();
    popup.insertItem( i18n( "Delete Log" ), 4 );
    
    QClipboard* clipboard = QApplication::clipboard();

    switch ( popup.exec( point ) ) {
    case 0:
        KRun::runURL( thread->url(), "text/html" );
        break;
    case 1:
        emit showThreadRequested( thread->datURL(), true );
        break;
    case 2:
        clipboard->setText( thread->name() + "\n" + thread->url() );
        break;
    case 3:
        if ( isFavorites ) {
            emit bookmarked( thread->datURL(), false );
        } else {
            emit bookmarked( thread->datURL(), true );
        }
        break;
    case 4:
        deleteLog( Kita::DatManager::threadURL( thread->datURL() ) );
        break;
    default:
        break;
    }
}

void KitaSubjectView::deleteLog( const KURL& url )
{
    if ( QMessageBox::warning( this,
                               "Kita",
                               i18n( "Do you want to delete Log ?" ),
                               QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
            == QMessageBox::Ok )
    {
        Kita::DatManager::deleteCache( url, this );
	  emit closeThreadTab( url );
	  slotUpdateSubject( url );
    }
}
