/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitathreadtabwidget.h"

#include "kitathreadpart.h"
#include "kitathreadview.h"

#include <klibloader.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kdebug.h>

#include <qclipboard.h>
#include <qapplication.h>
#include <qmessagebox.h>
#include <qregexp.h>
#include <qptrlist.h>

#include "libkita/kita_misc.h"
#include "libkita/parsemisc.h"
#include "libkita/signalcollection.h"
#include "libkita/datmanager.h"
#include "libkita/thread.h"

#define MAX_TABLABEL_LEN 20

enum {
    ID_Browser_Open,
    ID_Copy_Title_URL,
    ID_Close
};

KitaThreadTabWidget::KitaThreadTabWidget( QWidget* parent, const char* name, WFlags f )
        : QTabWidget( parent, name, f )
{
    setTabBar( new KitaThreadTabBar( this ) );
    connect( tabBar(), SIGNAL( deleteMenuSelected( QWidget* ) ),
             SLOT( deleteView( QWidget* ) ) );

    m_manager = new KParts::PartManager( topLevelWidget(), this, "KitaPartManager" );

    KitaThreadView* view = createView();

    if ( view ) {
        m_viewList.append( view );
        addTab( view, "thread" );
    }

    connectSignals();
}

KitaThreadTabWidget::~KitaThreadTabWidget()
{
    KParts::Part * p;
    while ( ( p = m_manager->parts()->getFirst() ) != NULL ) {
        m_manager->removePart( p );
        delete p;
    }
    delete m_manager;
}

void KitaThreadTabWidget::showThread( const QString& url, bool withNewTab )
{
    QString refstr;
    KURL datURL = Kita::ParseMisc::parseURL( url, refstr);
    QString threadName = Kita::DatManager::threadName( datURL );
    int maxNum = Kita::DatManager::getReadNum( datURL );
    int num = 0;
    if( refstr != QString::null ){
	int i = refstr.find( "-" );
	if( i != -1 ) num = refstr.left( i ).toInt();
	else num = refstr.toInt();
    }
    if( num == 0 ) num = Kita::DatManager::getKokoyonNum( datURL );

    KitaThreadView * view = findView( Kita::DatManager::threadURL( datURL ) );
    if ( view ) {
        setCurrentPage( indexOf( view ) );
        if ( view->threadURL().isEmpty() ) {
            view->showThread( datURL, num );
        } else {
            view->slotReloadButton();
        }
    } else if ( withNewTab ) {
        KitaThreadView * newView = createView();

        if ( newView ) {
            addTab( newView, getTabLabel( threadName ) );
            newView->showThread( datURL, num );
            showPage( newView );
            m_viewList.append( newView );
        }
    } else {
        static_cast<KitaThreadView *>( currentPage() ) ->showThread( datURL, num );
    }

    setTabLabel( currentPage(), QString( "%1 (%2)" ).arg( getTabLabel( threadName ) ).arg( maxNum ) );
    setTabToolTip( currentPage(), threadName );
}

KitaThreadView* KitaThreadTabWidget::createView()
{
    KLibFactory * factory = KLibLoader::self() ->factory( "libkitapart" );
    if ( ! factory ) {
        QMessageBox::critical( parentWidget(),
                               i18n( "Load Error" ),
                               i18n( "can't load libkitapart." ) );
        return 0;
    }
    KitaThreadPart* part = static_cast<KitaThreadPart *>( factory->create( this, "thread", "KitaThreadPart" ) );
    KitaThreadView* view = static_cast<KitaThreadView *>( part->widget() );

    m_manager->addPart( part );

    return view;
}

void KitaThreadTabWidget::showAlternativeView( const KURL& url, const QString& libName, const QString& mimetype )
{
    KLibFactory *factory = KLibLoader::self()->factory( libName );
    if ( !factory ) {
        QMessageBox::critical( parentWidget(), i18n( " Load Error" ),
                               QString( i18n( "can't load %1.") ).arg( libName ) );
        return;
    }
    if ( factory->inherits( "KParts::Factory" ) ) {
        KParts::Part *part = static_cast<KParts::Factory*>(factory)->createPart( this );
        m_manager->addPart( part );
        addTab( part->widget(), getTabLabel( url.url() ) );
        showPage( part->widget() );
        setTabToolTip( currentPage(), url.url() );
        KParts::BrowserExtension *ext = KParts::BrowserExtension::childObject( part );
        if ( ext ) {
            KParts::URLArgs arg( false, 0, 0, mimetype );
            ext->setURLArgs( arg );
        }
        static_cast<KParts::ReadOnlyPart*>(part)->openURL( url );
    }
}

void KitaThreadTabWidget::connectSignals()
{
    connect( this, SIGNAL( currentChanged ( QWidget * ) ),
             SLOT( slotCurrentChanged ( QWidget * ) ) );
    
    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( this, SIGNAL( setMainStatusbar( const QString& ) ),
             signalCollection, SIGNAL( setMainStatusbar( const QString& ) ) );
    connect( this, SIGNAL( setMainCaption( const QString& ) ),
             signalCollection, SIGNAL( setMainCaption( const QString& ) ) );
    connect( this, SIGNAL( switchSubjectView() ),
             signalCollection, SIGNAL( switchSubjectView() ));
    connect( signalCollection, SIGNAL( updateThreadTab( const KURL& ) ),
             this, SLOT( slotUpdateThreadTab( const KURL& ) ) );
    connect( signalCollection, SIGNAL( closeCurrentThreadTab() ),
             SLOT( closeCurrentTab() ));
    connect( signalCollection, SIGNAL( closeThreadTab( const KURL& ) ),
             SLOT( slotCloseThreadTab( const KURL& )));
}

KitaThreadView* KitaThreadTabWidget::findView( const QString& threadURL )
{
    KitaThreadView * view;
    for ( view = m_viewList.first(); view; view = m_viewList.next() ) {
        if ( view->threadURL().url() == threadURL ) {
            return view;
        }
        if ( view->threadURL().isEmpty() ) {
            return view; // default view
        }
    }
    return 0;
}

KParts::Part* KitaThreadTabWidget::findPartFromView( QWidget* view )
{
    KParts::Part *p;
    QPtrListIterator<KParts::Part> it( *( m_manager->parts() ) );
    while ( ( p = (*it) ) != NULL )
    {
        if ( p->widget() == view ) return p;
        ++it;
    }
    return NULL;
}

void KitaThreadTabWidget::deleteView( QWidget* v )
{
    kdDebug() << "deleteView(" << static_cast<void*>( v ) << ")" << endl;

    removePage( v );
    m_viewList.remove( static_cast<KitaThreadView*>(v) );

    m_manager->removePart( findPartFromView( v ) );
    delete v;

    if ( m_viewList.count() == 0 ) {
        KitaThreadView * view = createView();

        if ( view ) {
            m_viewList.append( view );
            addTab( view, "thread" );

            showPage( view );
            emit setMainStatusbar( "" );
            emit setMainCaption( "" );
	    emit switchSubjectView();
        }
    }
}

void KitaThreadTabWidget::focusSearchCombo()
{
    static_cast<KitaThreadView *>( currentPage() ) ->focusSearchCombo();
}

void KitaThreadTabWidget::reloadThread()
{
    static_cast<KitaThreadView *>( currentPage() ) ->slotReloadButton();
}

void KitaThreadTabWidget::closeCurrentTab()
{
    QWidget * view = currentPage();
    CloseCurrentThreadTabEvent* e = new CloseCurrentThreadTabEvent( indexOf( view ) );
    QApplication::postEvent( this, e );  // Qt will delete it when done
}

/* public slot */
void KitaThreadTabWidget::slotCloseThreadTab( const KURL& url )
{
    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    QWidget * view = findView( Kita::DatManager::threadURL( datURL ) );
    if( view ){
	CloseCurrentThreadTabEvent* e = new CloseCurrentThreadTabEvent( indexOf( view ) );
	QApplication::postEvent( this, e );  // Qt will delete it when done
    }
}

/* Calling deleteView from KitaThreadView will be the
   cause of crash.  So you need to call deleteView
   via custom event.                                   */ /* protected */
void KitaThreadTabWidget::customEvent( QCustomEvent * e )
{
    if ( e->type() == EVENT_CloseCurrentThreadTab ) {
	deleteView ( page( static_cast< CloseCurrentThreadTabEvent* >(e)->getIndex() ) );
    }
}

KitaThreadTabBar::KitaThreadTabBar( QWidget* parent, const char* name )
        : QTabBar( parent, name )
{}

KitaThreadTabBar::~KitaThreadTabBar()
{}

void KitaThreadTabBar::contextMenuEvent( QContextMenuEvent* e )
{
    KPopupMenu popup( 0 );
    popup.insertItem( i18n( "Open with Web Browser" ), ID_Browser_Open );
    popup.insertItem( i18n( "Copy title and URL" ), ID_Copy_Title_URL );
    popup.insertItem( i18n( "Close this tab" ), ID_Close );

    KitaThreadTabWidget* parent = static_cast<KitaThreadTabWidget*>( parentWidget() );
    KParts::ReadOnlyPart* view = static_cast<KParts::ReadOnlyPart*>( parent->findPartFromView( parent->currentPage() ) );
    QClipboard* clipboard = QApplication::clipboard();
    KitaThreadView *threadView = NULL;

    if ( view->className() != "KitaThreadPart" ) {
        popup.setItemEnabled( ID_Copy_Title_URL, FALSE );
        threadView = static_cast<KitaThreadView*>( parent->currentPage() );
    }

    switch ( popup.exec( e->globalPos() ) ) {
    case ID_Browser_Open:
        KRun( view->url() );
        break;
    case ID_Copy_Title_URL:
        if ( threadView )
            clipboard->setText( threadView->threadName() + "\n" + threadView->threadURL().url() );
        break;
    case ID_Close:
        emit deleteMenuSelected( view->widget() );
        break;
    default:
        break;
    }

    kdDebug() << "currentPage = " << tab( currentTab() ) ->text() << endl;
}

void KitaThreadTabWidget::slotCurrentChanged( QWidget * w )
{
    m_manager->setActivePart( findPartFromView( w ) );
}

const QString KitaThreadTabWidget::getTabLabel( const QString &name )
{
        QString threadTitle = Kita::unescape( name );
        QString title = ( threadTitle.length() > MAX_TABLABEL_LEN ) ? threadTitle.left( MAX_TABLABEL_LEN ) + "..." : threadTitle;
        return title;
}

void KitaThreadTabWidget::slotUpdateThreadTab( const KURL& url )
{
    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    Kita::Thread* thread = Kita::Thread::getByURL( datURL.prettyURL() );
    KitaThreadView * view = findView( thread->url() );
    if ( view )
        setTabLabel( currentPage(), QString( "%1 (%2)" ).arg( getTabLabel( thread->name() ) ).arg( thread->resNum() ) );
}
