/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "board.h"

#include <config.h>

#include <kio/slaveconfig.h>
#include <kio/netaccess.h>
#include <kdebug.h>

#include <kfilterdev.h>

#include <qregexp.h>
#include <qdict.h>

#include "kita_misc.h"
#include "threadinfo.h"

#include "qcp932codec.h"

using namespace Kita;

QDict<Board>* Board::m_boardDict = 0;
QCp932Codec* Board::m_cp932Codec = NULL;
QEucJpCodec* Board::m_eucJpCodec = NULL;

Board::Board()
{}

Board::Board( const QString& boardURL, const QString boardName ) : m_boardURL( boardURL ), m_boardName( boardName )
{}

Board::~Board()
{}

const QString Board::getID( const QString& boardURL )
{
    return KURL( boardURL ).fileName();
}

const QPtrList<Thread> Board::getThreadList( const QString& boardURL )
{
    if ( boardURL.isEmpty() ) {
        kdWarning() << "Waring: board url is empty!" << endl;
        return QPtrList<Thread>();
    }
    QPtrList<Thread> threadList;

    KIO::SlaveConfig::self() ->setConfigData( "http",
            KURL( boardURL ).host() ,
            "UserAgent",
            QString( "Monazilla/1.00 (Kita/%1)" ).arg( VERSION ) );
    QString tmpFile;
    if ( KIO::NetAccess::download( Kita::boardToSubject( boardURL ), tmpFile ) ) {
        QIODevice * tmpDevice = KFilterDev::deviceForFile( tmpFile, "application/x-gzip" );
        tmpDevice->open( IO_ReadOnly );

        if( !m_cp932Codec ) m_cp932Codec = new QCp932Codec();
        if( !m_eucJpCodec ) m_eucJpCodec = new QEucJpCodec();
        QTextStream stream( tmpDevice );
        
        if ( Kita::boardType( boardURL ) == Board_JBBS ) {
            stream.setCodec( m_eucJpCodec );
        } else {
            stream.setCodec( m_cp932Codec );
        }

        // parse subject.txt(only one format...)
        // FIXME: need to refactoring
        QRegExp regexp;
        switch ( Kita::boardType( boardURL ) ) {
        case Board_2ch:
            regexp.setPattern( "(\\d+\\.dat)<>(.*)\\((\\d+)\\)" );
            break;
        case Board_MachiBBS:
        case Board_JBBS:
        default:
            regexp.setPattern( "(\\d+\\.cgi),(.*)\\((\\d+)\\)" );
            break;
        }
        QString line;

        while ( ( line = stream.readLine() ) != QString::null ) {
            int pos = regexp.search( line );
            if ( pos != -1 ) {
                QString fname = regexp.cap( 1 );
                QString subject = regexp.cap( 2 );
                QString num = regexp.cap( 3 );

                KURL datURL = boardURL;
                datURL.addPath( "/dat/" + fname );
                Kita::Thread::setName( datURL.url(), subject );
                Kita::Thread* thread = Kita::Thread::getByURL( datURL.url() );
                KitaThreadInfo::setResNum( datURL.url(), num.toInt() );
                threadList.append( thread );
            }
        }

        KIO::NetAccess::removeTempFile( tmpFile );
    }
    return threadList;
}

Board* Board::getByURL( const QString& boardURL )
{
    // FIXME: null֥Ȥ׵ΤӤ˺äƤ롣
    if ( boardURL.isNull() ) {
        return new NullBoard();
    }

    if ( m_boardDict == 0 ) {
        // FIXME: setAutoDeleteäưȤΤ뤳
        m_boardDict = new QDict<Board>();
    }

    if ( m_boardDict->find( boardURL ) ) {
        return m_boardDict->find( boardURL );
    }

    Board* newBoard = new Board( boardURL );
    m_boardDict->insert( boardURL, newBoard );

    return newBoard;
}

void Board::setName( const QString& boardURL, const QString& boardName )
{
    if ( boardURL.isNull() ) {
        return ;
    }

    if ( m_boardDict == 0 ) {
        // FIXME: setAutoDeleteäưȤΤ뤳
        m_boardDict = new QDict<Board>();
    }

    if ( m_boardDict->find( boardURL ) ) {
        m_boardDict->find( boardURL ) ->m_boardName = boardName;
    } else {
        Board* newBoard = new Board( boardURL, boardName );
        m_boardDict->insert( boardURL, newBoard );
    }
    return ;
}

const QString& Board::getName( const QString& boardURL )
{
    return Board::getByURL( boardURL ) ->m_boardName;
}

void Board::replace( const QString& fromURL, const QString& toURL )
{
    QDictIterator<Kita::Board> it( *m_boardDict );
    for ( ; it.current(); ++it ) {
        QString url = it.currentKey();
        Kita::Board* board = it.current();
        if ( url.find( fromURL ) == 0 ) {
            m_boardDict->remove( url );
            url = url.replace( 0, fromURL.length(), toURL );
            board->m_boardURL = url;
            m_boardDict->insert( url, board );
            it.toFirst();
        }
    }
}

bool Board::isWritingSupported( const KURL& boardURL )
{
    if ( Kita::boardType( boardURL.url() ) == Board_2ch ) {
        return true;
    } else {
        return false;
    }
}

NullBoard::NullBoard()
{}

NullBoard::~ NullBoard()
{}
