/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto , 2004 by 421                    *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include <kpopupmenu.h>
#include <klocale.h>
#include <khtml_events.h>
#include <kdebug.h>
#include <krun.h>

#include <dom/html_element.h>

#include <qcursor.h>
#include <qapplication.h>
#include <qclipboard.h>
#include <qmessagebox.h>
#include <qregexp.h>

#include "kitahtmlpart.h"
#include "kitahtmlview.h"
#include "kitadomtree.h"
#include "kitanavi.h"

#include "libkita/kitaconfig.h"
#include "libkita/datmanager.h"
#include "libkita/datinfo.h"
#include "libkita/parsemisc.h"
#include "libkita/signalcollection.h"


/*-------------------------------------------*/


/*-------------------------------------*/
/* Don't forget to call setup() later. */

KitaHTMLPart::KitaHTMLPart( QWidget* parent, const char* name )
        : KHTMLPart( new KitaHTMLView( this, parent, name ) )
{
    m_mode = HTMLPART_MODE_KHTML;
    m_popup = NULL;
    m_domtree = NULL;
    m_datURL = QString::null;
    m_updatedKokoyon = FALSE;

    clearPart();
    createHTMLDocument();
    connectSignals();
}


KitaHTMLPart::~KitaHTMLPart()
{
    clearPart();
}


/* private */
void KitaHTMLPart::clearPart()
{

    deletePopup();

    /* delete KitaDomTree */
    if ( m_domtree ) delete m_domtree;
    m_domtree = NULL;

    /* update Kokomade Yonda */
    if( m_mode == HTMLPART_MODE_MAINPART && !m_updatedKokoyon && !m_datURL.isEmpty())
    {
        int readNum = Kita::DatManager::getReadNum( m_datURL );
        if( readNum ) Kita::DatManager::setKokoyonNum( m_datURL, readNum );
    }
    m_updatedKokoyon = FALSE;

    /* don't forget to unlock previous datURL here. */
    if( !m_datURL.isEmpty() ) Kita::DatManager::unlock( m_datURL );

    m_anchorStack.clear();
    m_centerNum = 0;
    findTextInit();
}



/* public */
bool KitaHTMLPart::setup( int mode, const KURL& url )
{
    if( url.isEmpty() ) return FALSE;

    clearPart();

    /* Lock datURL. Don't forget to unlock it later ! */
    m_datURL = Kita::ParseMisc::parseURLonly( url );
    Kita::DatManager::lock( m_datURL );

    /* reset abone */
    Kita::DatManager::resetAbone( m_datURL );

    /* create HTML Document */
    createHTMLDocument();

    /* create DOM manager */
    m_mode = mode;
    if( m_mode == HTMLPART_MODE_MAINPART || m_mode == HTMLPART_MODE_NAVI )
    {
        m_domtree = new KitaDomTree( htmlDocument(), m_datURL );
    }

    return TRUE;
}


/* private */
void KitaHTMLPart::connectSignals()
{

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();


    /* rendering */
    connect( this, SIGNAL( redrawHTMLPart( const KURL&, bool ) ), signalCollection, SIGNAL( redrawHTMLPart( const KURL&, bool ) ));
    connect( signalCollection, SIGNAL( redrawHTMLPart( const KURL&, bool ) ), SLOT( slotRedrawHTMLPart( const KURL& , bool) ));
    connect( signalCollection, SIGNAL( redrawAllHTMLPart( bool ) ), SLOT( slotRedrawAllHTMLPart( bool ) ));
    connect( signalCollection, SIGNAL( setFontOfHTMLPart() ), SLOT( slotSetFontOfHTMLPart() ));

    /* popup */
    connect( this, SIGNAL( onURL( const QString& ) ), SLOT( slotOnURL( const QString& ) ));
    connect( this, SIGNAL( isKitaActive() ), signalCollection, SIGNAL( isKitaActive() ));

    connect( view(), SIGNAL( leave() ), SLOT( slotLeave() ));
    connect( view(), SIGNAL( verticalSliderReleased() ), SLOT( slotVSliderReleased() ));
    connect( view(), SIGNAL( horizontalSliderReleased() ), SLOT( slotHSliderReleased() ));

    connect( signalCollection, SIGNAL( kitaIsActive() ), SLOT( slotKitaIsActive() ));
    connect( signalCollection, SIGNAL( windowDeactivated() ), SLOT( slotHideChildPopup() ));


    /* click */
    connect( this, SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ),
             signalCollection, SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ) );


    /* goto anchor */
    connect( view(), SIGNAL( pushDown() ), SLOT( slotClickTugi100() ));
}



/* private */
void KitaHTMLPart::createHTMLDocument()
{
    /* style */
    QString style = QString( "style=\"font-size: %1pt; "
                             "font-family: %2; "
                             "color: %3; "
                             "background-color: %4; border-width: 0; \"" )
                    .arg( KitaConfig::threadFont().pointSize() )
                    .arg( KitaConfig::threadFont().family() )
                    .arg( KitaConfig::threadColor().name() )
                    .arg( KitaConfig::threadBackgroundColor().name() );

    QString text = "<html><head></head>";
    text += "<body " + style + "></body></html>";

    setJScriptEnabled( false );
    setJavaEnabled( false );

    /* Use dummy URL here, and protocol should be "file:".
       If protocol is "http:", local image files are not shown
       (for security reasons ?).
     */
    begin( "file:/dummy.htm" );
    write( text );
    end();
}



/* public */
const int KitaHTMLPart::getMode() const { return m_mode; }



/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* direct rendering functions */


/*-----------------*/
/* show  responses */

/* !!! don't forget to call updateScreen() later !!! */   /* public */
void KitaHTMLPart::showResponses( int startnum, int endnum )
{
    if( !m_domtree ) return;

    for ( int i = startnum ; i <= endnum; i++ ) m_domtree->appendRes( i, FALSE );
}


/* do parsing only.            */
/* call showResponses() later  */ /* public */
void KitaHTMLPart::parseResponses( int startnum, int endnum )
{
    if( !m_domtree ) return;

    for ( int i = startnum ; i <= endnum; i++ ) m_domtree->parseRes( i, i );
}


/*------------------------------------------------*/
/* Show all responses ,header, footer, and etc,
   if these are not shown.                        */

/* note that updateScreen() is called internally. */
/* So, you need not call it later.                */  /* public slot */
void KitaHTMLPart::showAll()
{
    if( !m_domtree ) return;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );
    if ( top != 1 || bottom != readNum )
    {
        showResponses( 1, readNum );
        updateScreen( TRUE, TRUE );
    }
}


/*------------------------*/
/* insert belt node       */

/* !!! don't forget to call updateScreen() later !!! */   /* public */
void KitaHTMLPart::insertBeltNode( const QString& idstr )
{
    if( !m_domtree ) return;

    m_domtree->createBeltNode( idstr );
}


/*--------------------------*/
/* insert header node       */

/* !!! don't forget to call updateScreen() later !!! */   /* public */
void KitaHTMLPart::insertHeaderNode( const QString& str )
{
    if( !m_domtree ) return;

    m_domtree->createCommentNode( str, "header", 0, 2, TRUE );
}


/*--------------------------*/
/* insert footer node       */

/* !!! don't forget to call updateScreen() later !!! */   /* public */
void KitaHTMLPart::insertFooterNode( const QString& str )
{
    if( !m_domtree ) return;

    m_domtree->createCommentNode( str, "footer", 0, 0, FALSE );
}



/*-------------------*/
/* update screen     */ /* public */
void KitaHTMLPart::updateScreen( bool showHeaderEtc, bool clock )
{
    if( !m_domtree )
    {
        view()->setFocus();
        return;
    }

    /* show clock cursor */
    if( clock )
    {
        QCursor qc; qc.setShape( Qt::WaitCursor );
        QApplication::setOverrideCursor( qc );
    }

    /* show header, footer, and kokomadeyonda, etc. */
    if( showHeaderEtc )
    {

        int readNum = Kita::DatManager::getReadNum( m_datURL);

        m_domtree->appendMae100();
        m_domtree->appendTugi100();
        m_domtree->appendKokoyon();
        m_domtree->appendFooter( readNum );
    }

    /*----------------------------*/

    /* update display */
    htmlDocument().applyChanges();
    view()->layout();
    view()->setVScrollBarMode( QScrollView::AlwaysOn );
    view()->setFocus();

    if(clock) QApplication::restoreOverrideCursor();
}


/* public */
void KitaHTMLPart::setInnerHTML( const QString& innerHTML )
{
    createHTMLDocument();
    htmlDocument().body().setInnerHTML( innerHTML );
}



/* redraw screen  */
/* These slots are connected to signalCollection.  */ /* public slot */
void KitaHTMLPart::slotRedrawHTMLPart( const KURL& datURL, bool force )
{

    if( m_domtree == NULL ) return;
    if( m_datURL != datURL ) return;

    m_domtree->redraw( force );
}

/* public slot */
void KitaHTMLPart::slotRedrawAllHTMLPart( bool force )
{
    if( m_domtree == NULL ) return;

    m_domtree->redraw( force );
}

/* public slot */
void KitaHTMLPart::slotSetFontOfHTMLPart()
{

    QFont font = KitaConfig::threadFont();

    DOM::CSSStyleDeclaration style = htmlDocument().body().style();
    style.setProperty( "font-family", font.family(), "" );
    style.setProperty( "font-size", QString( "%1pt" ).arg( font.pointSize() ), "" );
    htmlDocument().applyChanges();
}


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* cache functions */


/*----------------------------------*/
/* load thread from cache & show it */ /* public */

/*  top = centerNum - preShowNum
    bottom = centerNum + afterShowNum
    readNum = Kita::DatManager::getReadNum
 
    No.1 <- show -> No.20 <- not show -> No.(top) <- show -> No.(bottom) <- not show -> No.(readNum) */

bool KitaHTMLPart::load( int centerNum )
{

    /* config */

    int preShowNum = KitaConfig::preShowNum();
    if( preShowNum == 0 ) preShowNum = 5000;
    int afterShowNum = KitaConfig::afterShowNum();
    if( afterShowNum == 0 ) afterShowNum = 5000;  /* TODO: fix it. */

    /*--------------------------------*/
    m_centerNum = centerNum;

    if( m_mode != HTMLPART_MODE_MAINPART ) return FALSE;
    if( !m_domtree ) return FALSE;
    if( Kita::DatManager::getReadNum( m_datURL ) == 0 ) return FALSE;

    m_domtree->appendTemplate();
    showResponses( m_centerNum - preShowNum, m_centerNum + afterShowNum );
    updateScreen( TRUE , FALSE );
    gotoAnchor( QString().setNum( m_centerNum ), FALSE );
    view()->setFocus();

    return TRUE;
}



/*----------------------------*/
/* start reloading            */

/* see also slotReceiveData()
   and slotFinishLoad().      */ /* public */
bool KitaHTMLPart::reload()
{
    /* config */

    const bool online = TRUE;

    /*--------------------------------*/

    if( !online ) return FALSE;
    if( !m_domtree ) return FALSE;
    if( m_mode != HTMLPART_MODE_MAINPART )
    {
        /* If this is not MainPart, then open MainPart.  */
        KParts::URLArgs argdummy;
        emit openURLRequest( m_datURL, argdummy );
        return FALSE;
    }

    m_domtree->StopParseThread();
    m_firstReceive = TRUE;
    if( m_centerNum == 0 ) m_centerNum = m_domtree->getBottomResNumber();

    /* DatManager will call back slotReceiveData and slotFinishLoad. */
    Kita::DatManager::updateCache( m_datURL , this );
    view()->setFocus();

    return TRUE;
}



/*---------------------------------------------*/
/* This slot is called after Kita::DatManager
   received new data, and emits receiveData()  */ /* !!! "public" slot !!! */
void KitaHTMLPart::slotReceiveData()
{
    /* config */

    const int delta = 20;
    const int oneAfterAnother = TRUE;
    int afterShowNum = KitaConfig::afterShowNum();
    if( afterShowNum == 0 ) afterShowNum = 5000;  /* TODO: fix it. */

    /*--------------------------------*/

    if( m_mode != HTMLPART_MODE_MAINPART ) return;
    if( !m_domtree ) return;

    int readNum = Kita::DatManager::getReadNum( m_datURL );
    int bottom = m_domtree->getBottomResNumber();
    int showNum = m_centerNum + afterShowNum;

    /* parsing */
    if( oneAfterAnother ) parseResponses( bottom+1, readNum );

    /* rendering */
    if( m_firstReceive
            || ( bottom + delta < readNum && readNum <= showNum + delta -1 ) )
    {

        if( oneAfterAnother )
        {
            showResponses( bottom+1, QMIN( readNum, showNum ) );
            updateScreen( TRUE, FALSE );
        }
        else parseResponses( bottom+1, QMIN( readNum, showNum ) );
    }

    if( m_firstReceive  && m_centerNum < readNum )
    {
        gotoAnchor( QString().setNum( m_centerNum ), FALSE );
        m_firstReceive = FALSE;
    }

    emit receiveData();
}


/*--------------------------------------------*/
/* This slot is called after Kita::DatManager
   finished updating new data,
   and emits finishReload()                   */ /* !!! "public" slot !!! */
void KitaHTMLPart::slotFinishLoad()
{
    /* config */
    int afterShowNum = KitaConfig::afterShowNum();
    if( afterShowNum == 0 ) afterShowNum = 5000;  /* TODO: fix it. */

    /*--------------------------------*/

    if( m_mode != HTMLPART_MODE_MAINPART ) return;
    if( !m_domtree ) return;

    int bottom = m_domtree->getBottomResNumber();
    int shownNum = m_centerNum + afterShowNum;

    showResponses(bottom+1, shownNum );
    updateScreen( TRUE, FALSE );
    m_domtree->parseAllRes();
    m_centerNum = 0;

    emit finishReload();
}




/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* goto anchor */


/* public */
bool KitaHTMLPart::gotoAnchor( const QString& anc, bool pushPosition )
{
    if( anc == QString::null ) return FALSE;
    if( !m_domtree || m_mode == HTMLPART_MODE_KHTML || m_mode == HTMLPART_MODE_POPUP )
        return KHTMLPart::gotoAnchor( anc );

    hidePopup();

    QString ancstr = anc;
    int res = ancstr.toInt();

    if ( res > 1 )
    {

        /* is target valid ? */
        if ( !Kita::DatManager::isResValid( m_datURL, res ) ) return FALSE;

        /* show res if it is not shown */
        if ( !m_domtree->isResShown( res ) )
        {

            if( m_mode != HTMLPART_MODE_MAINPART ) return FALSE;

            int top = m_domtree->getTopResNumber();
            int bottom = m_domtree->getBottomResNumber();

            if ( res > bottom )
            {
                showResponses( bottom + 1, res );
                updateScreen( TRUE, TRUE );
            }
            else if ( res < top )
            {
                m_domtree->appendTemplate();
                showResponses( res, bottom );
                updateScreen( TRUE , TRUE );
            }
        }

        /* Target is not shown. Maybe it is aboned. */
        while ( res > 1 && !m_domtree->isResShown( res ) ) res--;

        ancstr = QString().setNum( res );
    }

    if ( res == 1 ) ancstr = "header";
    if ( pushPosition ) pushCurrentPosition();
    return KHTMLPart::gotoAnchor( ancstr );
}



/* jump to kokomade yonda */ /* public slot */
void KitaHTMLPart::slotGotoKokoyon()
{
    if( !m_domtree ) return;
    if( m_mode != HTMLPART_MODE_MAINPART ) return;

    int kokoyon = Kita::DatManager::getKokoyonNum( m_datURL );
    gotoAnchor( QString().setNum( kokoyon ), FALSE );
}



/* public slot  */
void KitaHTMLPart::slotGobackAnchor()
{
    if ( m_anchorStack.empty() ) return;

    QString anc = m_anchorStack.last();
    m_anchorStack.pop_back();
    gotoAnchor( anc , FALSE );
}




/* private */
void KitaHTMLPart::pushCurrentPosition()
{
    QString anchor, ancundermouse;
    DOM::Node node;

    /* find the id of current node */
    node = nodeUnderMouse();
    while ( node != NULL && node.nodeName().string() != "div" ) node = node.parentNode();
    if ( node == NULL ) return ;
    anchor = static_cast<DOM::Element>( node ).getAttribute( "id" ).string();

    m_anchorStack += anchor;
}


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* Tugi100, Mae100 etc.  */


/* public slot */
void KitaHTMLPart::slotClickTugi100()
{
    if( !m_domtree ) return;
    if( m_mode != HTMLPART_MODE_MAINPART ) return;

    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );

    if ( readNum != bottom )
    {
        showResponses( bottom + 1, bottom + 100 );
        updateScreen( TRUE, TRUE );
    }
}



/* public slot */
void KitaHTMLPart::slotClickNokori()
{
    if( !m_domtree ) return;
    if( m_mode != HTMLPART_MODE_MAINPART ) return;

    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );

    if ( readNum != bottom )
    {
        showResponses( bottom + 1, readNum );
        updateScreen( TRUE, TRUE );
    }
}


/* public slot */
void KitaHTMLPart::slotClickGotoFooter()
{
    if( !m_domtree
            || m_mode != HTMLPART_MODE_MAINPART
      )
    {
        gotoAnchor( "footer", FALSE );
        return;
    }

    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );

    if ( readNum != bottom )
    {
        showResponses( bottom + 1, readNum );
        updateScreen( TRUE, TRUE );
    }

    gotoAnchor( "footer", FALSE );
}



/* public slot */
void KitaHTMLPart::slotClickMae100()
{
    if( !m_domtree ) return;
    if( m_mode != HTMLPART_MODE_MAINPART ) return;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();

    if ( top != 1 )
    {
        m_domtree->appendTemplate();
        showResponses( top - 100, bottom );
        updateScreen( TRUE, TRUE );

        gotoAnchor( QString().setNum( top ), FALSE );
    }
}



/* public slot */
void KitaHTMLPart::slotClickMaeZenbu()
{
    if( !m_domtree ) return;
    if( m_mode != HTMLPART_MODE_MAINPART ) return;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();

    if ( top != 1 )
    {
        showResponses( 1, bottom );
        updateScreen( TRUE, TRUE);

        gotoAnchor( QString().setNum( top ), FALSE );
    }
}


/* public slot */
void KitaHTMLPart::slotClickTmpNext100()
{
    if( !m_domtree ) return;
    if( m_mode != HTMLPART_MODE_MAINPART ) return;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();
    int tmpnum = m_domtree->getTemplateNumber();

    if ( tmpnum < top )
    {
        m_domtree->setTemplateNumber( tmpnum + 100 );
        m_domtree->appendTemplate();
        showResponses( top, bottom );
        updateScreen( TRUE, TRUE );

        gotoAnchor( QString().setNum( tmpnum ), FALSE );
    }
}


/* public slot */
void KitaHTMLPart::slotClickShowAll()
{
    if( !m_domtree ) return;
    if( m_mode != HTMLPART_MODE_MAINPART ) return;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );

    if ( top != 1 || bottom != readNum )
    {
        showResponses( 1, readNum );
        updateScreen( TRUE, TRUE );

        gotoAnchor( "header", FALSE );
    }
}






/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* search */


/* private */
void KitaHTMLPart::findTextInit()
{
    m_findNode = NULL;
    m_findPos = -1;
    m_find_y = 0;
}



/* public */
bool KitaHTMLPart::findText( const QString &query, bool reverse )
{
    if( m_mode != HTMLPART_MODE_MAINPART && m_mode != HTMLPART_MODE_NAVI ) return FALSE;

    QRegExp regexp( query );
    regexp.setCaseSensitive( FALSE );

    /* init */
    if ( m_findNode.isNull() )
    {

        m_findNode = htmlDocument().body();
        m_find_y = 0;

        /* move to the last child node */
        if ( reverse )
        {
            while ( !m_findNode.lastChild().isNull() ) m_findNode = m_findNode.lastChild();
            m_find_y = view()->contentsHeight();
        }
    }

    while ( 1 )
    {

        if ( m_findNode.nodeType() == DOM::Node::TEXT_NODE
                || m_findNode.nodeType() == DOM::Node::CDATA_SECTION_NODE )
        {

            /* find the word in the current node */
            DOM::DOMString nodeText = m_findNode.nodeValue();
            QString nodestr = nodeText.string();
            if ( reverse && m_findPos != -1 ) nodestr.setLength( m_findPos );

            if ( reverse ) m_findPos = nodestr.findRev( regexp, m_findPos );
            else m_findPos = nodestr.find( regexp, m_findPos + 1 );

            /* scroll & select & return */
            if ( m_findPos != -1 )
            {

                int matchLen = regexp.matchedLength();

                QRect qr = m_findNode.getRect();
                view()->setContentsPos( qr.left() - 50, m_find_y - 100 );
                DOM::Range rg( m_findNode, m_findPos, m_findNode, m_findPos + matchLen );
                setSelection( rg );

                return TRUE;
            }

        }

        /*------------------------*/
        else if ( m_findNode.nodeName().string() == "dt"
                  || m_findNode.nodeName().string() == "dd" )
        {

            QRect qr = m_findNode.getRect();

            if ( reverse ) m_find_y = qr.bottom();
            else m_find_y = qr.top();
        }

        /*------------------------*/
        else if ( m_findNode.nodeName().string() == "br" )
        {

            DOM::Node tmpnode = m_findNode.previousSibling();

            if ( tmpnode != NULL )
            {

                QRect qr = tmpnode.getRect();
                if ( reverse ) m_find_y -= qr.bottom() - qr.top();
                else m_find_y += qr.bottom() - qr.top();
            }
        }

        /*------------------------*/

        m_findPos = -1;
        DOM::Node next;

        /* move to the next node */
        if ( !reverse )
        {

            next = m_findNode.firstChild();
            if ( next.isNull() ) next = m_findNode.nextSibling();

            while ( !m_findNode.isNull() && next.isNull() )
            {
                m_findNode = m_findNode.parentNode();
                if ( !m_findNode.isNull() ) next = m_findNode.nextSibling();
            }
        }
        /* revearse */
        else
        {

            next = m_findNode.lastChild();
            if ( next.isNull() ) next = m_findNode.previousSibling();

            while ( !m_findNode.isNull() && next.isNull() )
            {
                m_findNode = m_findNode.parentNode();
                if ( !m_findNode.isNull() ) next = m_findNode.previousSibling();
            }
        }

        m_findNode = next;
        if ( m_findNode.isNull() )
        {
            m_findNode = NULL;
            return FALSE;
        }
    }

    return FALSE;
}






/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* popup menu */


/* private */
void KitaHTMLPart::showPopupMenu( const KURL& kurl )
{
    enum {
        ID_Home_Link,
        ID_Temp_Link,
        ID_Koko_Link,
        ID_End_Link,

        ID_COPY_Link,
        ID_Copy_Str,
        ID_Abone_Word,
        ID_Extract,
        ID_Open_Browser,

        /*-----------------*/
        ID_Back_Link /* stay bottom */
    };

    QPoint point = QCursor::pos();
    QString url = kurl.prettyURL();
    bool showppm = FALSE;
    const int ID_Goto_Mark = ID_Back_Link + m_anchorStack.size();

    QString str;

    /* If selected Text is composed of only digits, then show res popup. */
    if( !m_pushctrl && showSelectedDigitPopup() ) return;

    /*-----------------------------------*/
    /* create menu items                 */

    KPopupMenu* popupMenu = new KPopupMenu( view() );
    KPopupMenu* backSubMenu = NULL;
    KPopupMenu* markSubMenu = NULL;

    popupMenu->clear();


    /*------*/
    /* jump */
    if( m_domtree &&
            (m_mode == HTMLPART_MODE_MAINPART || m_mode == HTMLPART_MODE_NAVI ) )
    {

        showppm = TRUE;

        /* back */
        if ( !m_anchorStack.empty() )
        {
            backSubMenu = new KPopupMenu( view() );
            backSubMenu->clear();

            int i = m_anchorStack.size();
            QStringList::iterator it;
            for ( it = m_anchorStack.begin(); it != m_anchorStack.end(); it++,i-- )
            {
                str = (*it) + "   " + Kita::DatManager::getPlainBody( m_datURL, (*it).toInt() ).left( 10 );
                backSubMenu->insertItem( str, ID_Back_Link +(i-1), 0 );
            }

            popupMenu->insertItem( i18n( "Back" ), backSubMenu );
            popupMenu->insertSeparator();
        }

        /* mark */
        /* !! very dangerous to call DatManager::getDatInfoPointer without locking !! */
        Kita::DatManager::lock( m_datURL );
        Kita::DatInfo* datInfo = Kita::DatManager::getDatInfoPointer( m_datURL );
        for( int i = 1; i <= datInfo->getReadNum() ; i++ )
        {
            if( datInfo->isMarked( i ) )
            {
                if( !markSubMenu )
                {
                    markSubMenu = new KPopupMenu( view() );
                    markSubMenu->clear();
                    popupMenu->insertItem( i18n( "Mark" ), markSubMenu );
                    popupMenu->insertSeparator();
                }

                str = QString().setNum( i ) + "   " + datInfo->getPlainBody( i ).left( 10 );
                markSubMenu->insertItem( str, ID_Goto_Mark + i );
            }
        }
        Kita::DatManager::unlock( m_datURL );

        /* home */
        popupMenu->insertItem( i18n( "Start" ), ID_Home_Link );


        /* template */
        if ( m_mode == HTMLPART_MODE_MAINPART )
        {

            if( m_domtree->isMae100Shown() ) popupMenu->insertItem( i18n( "template" ), ID_Temp_Link );

            int kokoyon = Kita::DatManager::getKokoyonNum( m_datURL );
            if( kokoyon )
            {
                str = i18n( "Kokomade Yonda (%1)" ).arg( kokoyon );
                popupMenu->insertItem( str, ID_Koko_Link );
            }
        }


        /* end */
        popupMenu->insertItem( i18n("End"), ID_End_Link );
    }


    /*--------------*/
    /* copy & abone */
    if ( hasSelection() )
    {
        if( showppm ) popupMenu->insertSeparator();
        showppm = TRUE;

        popupMenu->insertItem( "Copy", ID_Copy_Str );

        popupMenu->insertItem( i18n( "Extract by '%1'" ).arg( selectedText() ), ID_Extract );
        popupMenu->insertItem( i18n( "Add '%1' to abone list" ).arg( selectedText() ), ID_Abone_Word );
    }


    /*-----------*/
    /* copy link */
    if ( url != QString::null )
    {
        if( showppm ) popupMenu->insertSeparator();
        showppm = TRUE;

        popupMenu->insertItem( i18n( "Open with Web Browser" ), ID_Open_Browser );
        popupMenu->insertSeparator();
        popupMenu->insertItem( i18n( "Copy Link Location" ), ID_COPY_Link );
    }


    /*-----------------------------------*/

    /* show menu */
    if( showppm )
    {

        QClipboard* clipboard = QApplication::clipboard();
        KParts::URLArgs argdummy;

        int ret = popupMenu->exec( point );
        switch ( ret  )
        {

        case ID_COPY_Link:
            clipboard->setText( url , QClipboard::Clipboard );
            clipboard->setText( url , QClipboard::Selection );
            break;

        case ID_Open_Browser:
            emit openURLRequest( url, argdummy );
            break;

        case ID_Home_Link: gotoAnchor( "header", FALSE ); break;

        case ID_Temp_Link: gotoAnchor( "mae100", FALSE ); break;

        case ID_Koko_Link: slotGotoKokoyon(); break;

        case ID_End_Link:  slotClickGotoFooter(); break;

        case ID_Copy_Str:
            clipboard->setText( selectedText(), QClipboard::Clipboard );
            break;

        case ID_Abone_Word:

            if ( QMessageBox::information( view(),"Kita",
                                           i18n( "Do you want to add '%1' to abone list ?" ).arg( selectedText() ),
                                           QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                    == QMessageBox::Ok )
            {

                KitaConfig::addAboneWord( selectedText() );
                emit redrawHTMLPart( m_datURL, FALSE );
            }

            break;

        case ID_Extract:
            KitaNavi::showKitaNaviByWord( m_datURL, selectedText() );
            break;

        default:

            /* mark */
            if( ret >= ID_Goto_Mark )
            {
                gotoAnchor( QString().setNum( ret - ID_Goto_Mark ), FALSE );
            }

            /* back */
            else if( ret >= ID_Back_Link )
            {
                for(int i = 0; i < ret - ID_Back_Link; i++ ) m_anchorStack.pop_back();
                slotGobackAnchor();
            }

            break;
        }

    }

    if( popupMenu ) delete popupMenu;
    if( backSubMenu ) delete backSubMenu;
    if( markSubMenu ) delete markSubMenu;
}



/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* mouse event */


/* protected */
void KitaHTMLPart::khtmlMousePressEvent( khtml::MousePressEvent* e )
{
    KURL kurl;
    if( e->url().string() != QString::null )
        kurl = KURL( Kita::DatManager::boardURL( m_datURL ), e->url().string() );

    m_pushctrl = m_pushmidbt = m_pushrightbt = FALSE;
    if(e->qmouseEvent()->button() & Qt::RightButton) m_pushrightbt = TRUE;
    if ( e->qmouseEvent() ->state() & Qt::ControlButton ) m_pushctrl = TRUE;
    if ( e->qmouseEvent() ->button() & Qt::MidButton ) m_pushmidbt = TRUE;

    if ( e->url() != NULL )
    {

        if ( e->url().string().at( 0 ) == '#' )
        { /* anchor */
            kurl =  m_datURL;
            kurl.setRef( e->url().string().mid(1) ) ;
        }

        clickAnchor( kurl );
        m_pushctrl = m_pushmidbt = m_pushrightbt = FALSE;
        return;
    }

    /* popup menu */
    if( m_pushrightbt )
    {
        showPopupMenu( kurl );
        m_pushctrl = m_pushmidbt = m_pushrightbt = FALSE;
        return;
    }

    KHTMLPart::khtmlMousePressEvent( e );
}





/*-------------------------------------------------------*/
/*-------------------------------------------------------*/
/* click */



/* private slot */
void KitaHTMLPart::slotOpenURLRequest( const KURL& urlin, const KParts::URLArgs& )
{
    clickAnchor(urlin);
}


/*------------------------------------------------------*/
/* This function is called when user clicked res anchor */ /* private */
void KitaHTMLPart::clickAnchor( const KURL& urlin )
{
    QString refstr;
    KURL datURL = Kita::ParseMisc::parseURL( urlin ,refstr );

    /*--------------------*/
    /* Ctrl + right click */
    if( m_pushctrl && m_pushrightbt )
    {
        showPopupMenu( urlin );
        return;
    }

    /*--------------------------------*/
    /* If this is not anchor, then    */
    /* emit openURLRequest and return */

    if ( datURL.host() != m_datURL.host() || datURL.path() != m_datURL.path() )
    {

        if( m_pushrightbt )
        { /* right click */

            /* start multi-popup mode or show popup menu */
            if( !startMultiPopup() ) showPopupMenu( urlin );

            return;
        }

        KParts::URLArgs argdummy;
        emit openURLRequest( urlin, argdummy );
        return ;
    }

    if ( refstr == QString::null ) return ;

    /*---------------------------*/
    /* show popupmenu for #write */

    if ( refstr.left( 5 ) == "write" )
    {
        showWritePopupMenu( refstr );
        return;
    }

    /*----------------------------*/
    /* extract responses by ID    */

    if( refstr.left(5) == "idpop")
    {
        showIDPopup( refstr );
        return;
    }

    /*-------------------------*/
    /* start multi-popup mdde  */
    if( m_pushrightbt && startMultiPopup() ) return;


    /*----------------------------*/
    /* next 100 ,before 100 ,etc. */
    if( showNext100Etc( refstr ) ) return;


    /*-------------------------------*/
    /* open Kita Navi or goto anchor */

    int refNum,refNum2;

    int i = refstr.find( "-" );
    if ( i != -1 )
    {
        refNum = refstr.left( i ).toInt();
        refNum2 = refstr.mid(i+1).toInt();
        if( refNum2 < refNum ) refNum2 = refNum;
    }
    else refNum = refNum2 = refstr.toInt();

    if ( !refNum ) return ;

    if( m_pushctrl || m_pushmidbt ) KitaNavi::showKitaNavi( m_datURL, refNum, refNum2 );
    else if( ( m_mode == HTMLPART_MODE_KHTML || m_mode == HTMLPART_MODE_POPUP )
             || ( m_mode == HTMLPART_MODE_NAVI && !m_domtree->isResShown( refNum ) ) )
    {
        KParts::URLArgs argdummy;
        emit openURLRequest( urlin, argdummy );
    }
    else gotoAnchor( QString().setNum( refNum ), TRUE );
}



/*---------------------------------------------------------*/
/* popup menu that is opened when user clicked res number. */
/* This funtcion is called in only clickAnchor().          */ /* private */
void KitaHTMLPart::showWritePopupMenu( const QString& refstr)
{
    enum{
        WRITEMENU_RES,
        WRITEMENU_QUOTE,
        WRITEMENU_SHOWBROWSER,
        WRITEMENU_SHOWNAVI,
        WRITEMENU_RESTREE,
        WRITEMENU_REVERSERESTREE,
        WRITEMENU_EXTRACTNAME,
        WRITEMENU_COPY,
        WRITEMENU_COPYURL,
        WRITEMENU_COPYTHREADNAME,
        WRITEMENU_SETKOKOYON,
        WRITEMENU_MARK,
        WRITEMENU_ABONENAME
    };

    QClipboard * clipboard = QApplication::clipboard();
    QString str, resstr;
    int resNum = refstr.mid( 5 ).toInt();
    QString namestr = Kita::DatManager::getPlainName( m_datURL, resNum );

    /* show res tree */
    if( m_pushrightbt )
    {
        int num;
        QString htmlstr = Kita::DatManager::getTreeByRes( m_datURL, resNum, num );
        if ( !num ) return ;
        QString tmpstr = QString("<DIV>No.%1 : [%2]<BR>").arg( resNum ).arg( num );
        tmpstr += htmlstr + "<BR><BR></DIV>";
        showPopup( m_datURL, tmpstr );
        startMultiPopup();
        return;
    }

    /* open kitanavi */
    else if( m_pushctrl | m_pushmidbt )
    {
        KitaNavi::showKitaNaviResTree(m_datURL,resNum);
        return;
    }

    /*---------------------*/
    /* create popup menu */
    QString plainStr;

    KPopupMenu *popupMenu = new KPopupMenu( view() );
    popupMenu->clear();

    /* write */
    if( m_mode == HTMLPART_MODE_MAINPART || m_mode == HTMLPART_MODE_NAVI )
    {

        popupMenu->insertItem( i18n( "write response" ), WRITEMENU_RES );
        popupMenu->insertItem( i18n( "quote this" ), WRITEMENU_QUOTE );
        popupMenu->insertSeparator();

    }

    /* mark */
    popupMenu->insertItem( i18n( "Mark" ), WRITEMENU_MARK );
    popupMenu->setItemChecked( WRITEMENU_MARK, Kita::DatManager::isMarked( m_datURL, resNum ) );
    popupMenu->insertSeparator();

    /* open */
    popupMenu->insertItem( i18n( "Open with KitaNavi" ), WRITEMENU_SHOWNAVI );
    popupMenu->insertItem( i18n( "Open with Web Browser" ), WRITEMENU_SHOWBROWSER );
    popupMenu->insertSeparator();

    /* util */
    popupMenu->insertItem( i18n( "res tree" ), WRITEMENU_RESTREE );
    popupMenu->insertItem( i18n( "reverse res tree" ), WRITEMENU_REVERSERESTREE );
    popupMenu->insertItem( i18n( "extract by name" ), WRITEMENU_EXTRACTNAME );
    popupMenu->insertSeparator();

    /* copy */
    popupMenu->insertItem( i18n( "copy URL" ), WRITEMENU_COPYURL );
    popupMenu->insertItem( i18n( "Copy title and URL" ), WRITEMENU_COPYTHREADNAME );
    popupMenu->insertItem( i18n( "copy" ), WRITEMENU_COPY );

    /* kokkoma de yonda */
    if( m_domtree && m_mode == HTMLPART_MODE_MAINPART )
    {
        popupMenu->insertSeparator();
        popupMenu->insertItem( i18n( "set Kokomade Yonda" ), WRITEMENU_SETKOKOYON );
    }

    /* abone */
    popupMenu->insertSeparator();
    popupMenu->insertItem( i18n( "add name to abone list" ), WRITEMENU_ABONENAME );


    /*--------------------------------------*/
    /* show popup menu */

    int ret = popupMenu->exec( QCursor::pos() );
    delete popupMenu;
    switch ( ret )
    {

    case WRITEMENU_RES:
        resstr = ">>" + QString().setNum( resNum ) + "\n";
        emit openWriteDialog( resstr );
        break;

    case WRITEMENU_QUOTE:
        resstr = ">>" + QString().setNum( resNum ) + "\n"
                 + "> " + Kita::DatManager::getPlainTitle( m_datURL, resNum ) + "\n"
                 + "> " + Kita::DatManager::getPlainBody( m_datURL, resNum ).replace("\n","\n> ") + "\n";
        emit openWriteDialog( resstr );
        break;

    case WRITEMENU_MARK:
        Kita::DatManager::setMark( m_datURL, resNum, ! Kita::DatManager::isMarked( m_datURL, resNum ) );
        break;

    case WRITEMENU_COPY:
    case WRITEMENU_COPYURL:
    case WRITEMENU_COPYTHREADNAME:
        str = QString::null;

        /* title */
        if( ret == WRITEMENU_COPYTHREADNAME || ret == WRITEMENU_COPY )
        {
            str = Kita::DatManager::threadName( m_datURL );
        }

        /* url */
        if( str != QString::null ) str += "\n";
        str += Kita::DatManager::threadURL( m_datURL ) + QString().setNum( resNum ) + "\n";

        /* body */
        if( ret == WRITEMENU_COPY )
        {
            str += "\n"
                   + Kita::DatManager::getPlainTitle( m_datURL, resNum ) + "\n"
                   + Kita::DatManager::getPlainBody( m_datURL, resNum ) + "\n";
        }

        /* copy */
        clipboard->setText( str , QClipboard::Clipboard );
        clipboard->setText( str , QClipboard::Selection );
        break;


    case WRITEMENU_SETKOKOYON:
        Kita::DatManager::setKokoyonNum( m_datURL, resNum );
        m_updatedKokoyon = TRUE;
        updateScreen( TRUE, TRUE );
        gotoAnchor( QString().setNum( resNum ), FALSE );
        break;

    case WRITEMENU_SHOWBROWSER:
        str = Kita::DatManager::threadURL( m_datURL )
              + QString().setNum( resNum );

        KRun::runURL( str, "text/html" );
        break;


    case WRITEMENU_SHOWNAVI:
        KitaNavi::showKitaNavi( m_datURL, resNum, resNum);
        break;


    case WRITEMENU_RESTREE:
        KitaNavi::showKitaNaviResTree( m_datURL, resNum);
        break;

    case WRITEMENU_REVERSERESTREE:
        KitaNavi::showKitaNaviRevResTree( m_datURL, resNum );
        break;

    case WRITEMENU_EXTRACTNAME:
        KitaNavi::showKitaNaviByName( m_datURL, namestr );
        break;

    case WRITEMENU_ABONENAME:
        if ( QMessageBox::information( view(),"Kita",
                                       i18n( "Do you want to add '%1' to abone list ?" ).arg( namestr ),
                                       QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                == QMessageBox::Ok )
        {

            KitaConfig::addAboneName( namestr );
            emit redrawHTMLPart( m_datURL, FALSE );
        }
        break;

    default:
        break;
    }

}



/*--------------------------------------------------*/
/* popup that is opened when user clicked ID        */
/* This funtcion is called in only clickAnchor().   */ /* private */
void KitaHTMLPart::showIDPopup( const QString& refstr )
{
    QString strid = refstr.mid(5)
                    .replace("%2B","+")   /* decode %2B -> + */
                    .replace("%2F", "/"); /* decode %2F -> / */

    /* popup */
    if( m_pushrightbt )
    {
        int num;
        QString htmlstr
        = Kita::DatManager::getHtmlByID( m_datURL, strid, num );
        if ( num <= 1 ) return ;
        QString tmpstr = QString("<DIV>ID:%1:[%2]<BR>").arg( strid ).arg( num );
        tmpstr += htmlstr + "<BR><BR></DIV>";
        showPopup( m_datURL, tmpstr );
        startMultiPopup();
    }

    /* open kitanavi when user pushed Ctrl+Left or Mid button. */
    else if( m_pushctrl | m_pushmidbt ) KitaNavi::showKitaNaviByID( m_datURL, strid );

    else
    {

        enum{
            IDMENU_EXTRACT,
            IDMENU_ABONE
        };

        KPopupMenu *popupMenu = new KPopupMenu( view() );
        popupMenu->clear();
        if( Kita::DatManager::getNumByID( m_datURL, strid ) > 1 )
            popupMenu->insertItem( i18n( "extract by ID" ), IDMENU_EXTRACT );
        popupMenu->insertItem( i18n( "add id to abone list" ), IDMENU_ABONE );
        int ret = popupMenu->exec( QCursor::pos() );
        delete popupMenu;
        switch ( ret  )
        {

        case IDMENU_EXTRACT:
            KitaNavi::showKitaNaviByID( m_datURL, strid );
            break;

        case IDMENU_ABONE:
            /* add ID to abone list */
            if ( QMessageBox::information( view(),"Kita",
                                           i18n( "Do you want to add '%1' to abone list ?" ).arg( strid ),
                                           QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                    == QMessageBox::Ok )
            {

                KitaConfig::addAboneID( strid );
                emit redrawHTMLPart( m_datURL, FALSE );
            }
            break;

        default:
            break;
        }
    }
}



/*-------------------------------------------------------*/
/* funtcion when user clicked next 100 ,before 100 ,etc. */
/* This funtcion is called in only clickAnchor().        */ /* private */
bool KitaHTMLPart::showNext100Etc( const QString& refstr )
{
    if( m_mode != HTMLPART_MODE_MAINPART ) return FALSE;

    if ( refstr.left( 7 ) == "tugi100" )
    {
        slotClickTugi100();
        return TRUE;

    }
    else if ( refstr.left( 6 ) == "nokori" )
    {
        slotClickNokori();
        return TRUE;

    }
    else if ( refstr.left( 7 ) == "tosaigo" )
    {
        slotClickGotoFooter();
        return TRUE;

    }
    else if ( refstr.left( 6 ) == "mae100" )
    {
        slotClickMae100();
        return TRUE;

    }
    else if ( refstr.left( 8 ) == "maezenbu" )
    {
        slotClickMaeZenbu();
        return TRUE;

    }
    else if ( refstr.left( 6 ) == "tmp100" )
    {
        slotClickTmpNext100();
        return TRUE;

    }
    else if ( refstr.left( 5 ) == "zenbu" )
    {
        slotClickShowAll();
        return TRUE;
    }

    return FALSE;
}



/*-------------------------------------------------------*/
/*-------------------------------------------------------*/
/* popup */


/*---------------------*/
/* show popup window   */ /* private */
void KitaHTMLPart::showPopup( const KURL& url, const QString& innerHTML )
{
    deletePopup();
    m_multiPopup = FALSE;

    m_popup = new Kita::ResPopup( view() , url );

    connect( m_popup, SIGNAL( hideChildPopup() ),SLOT( slotHideChildPopup() ) );

    m_popup->setText( innerHTML );
    m_popup->adjustSize();
    m_popup->adjustPos();
    m_popup->show();
}


/*------------------------*/
/* start multi-popup mode */ /* private */
bool KitaHTMLPart::startMultiPopup()
{

    if( m_popup && m_popup->isVisible() )
    {
        m_multiPopup = TRUE;
        m_popup->moveMouseAbove();
    }
    else m_multiPopup = FALSE;

    return m_multiPopup;
}


/* Is it multi-popup mode now ? */ /* private */
bool KitaHTMLPart::isMultiPopupMode()
{
    if( !m_popup ) m_multiPopup = FALSE;
    else if( m_popup->isHidden() ) m_multiPopup = FALSE;

    return m_multiPopup;
}


/* private */
void KitaHTMLPart::hidePopup()
{
    if( m_popup ) m_popup->hide();
    m_multiPopup = FALSE;
}



/* private */
void KitaHTMLPart::deletePopup()
{
    if ( m_popup ) delete m_popup;
    m_popup = NULL;
    m_multiPopup = FALSE;
}


/* return TRUE if this view is under mouse. */ /* private */
bool KitaHTMLPart::isUnderMouse( int mrgwd, int mrght )
{
    QPoint pos = QCursor::pos();
    int cx = pos.x(), cy = pos.y();

    QPoint viewpos = view()->mapToGlobal( QPoint(0,0) );
    int px = viewpos.x(), py = viewpos.y();
    int wd = view()->visibleWidth(), ht = view()->visibleHeight();

    if( ( px < cx && cx < px + wd + mrgwd )
            && ( py  < cy && cy < py + ht + mrght  ) )
    {
        return TRUE;
    }

    return FALSE;
}


/* private slot */
void KitaHTMLPart::slotLeave()
{
    if ( isMultiPopupMode() ) return;
    if( view()->isHorizontalSliderPressed() ) return;
    if( view()->isVerticalSliderPressed () ) return;

    hidePopup();

    /* emit signal to have parent hide this if this is popup . */
    if( m_mode == HTMLPART_MODE_POPUP && !isUnderMouse( 0, 0 ) ) emit hideChildPopup();
}


/* private slot */
void KitaHTMLPart::slotVSliderReleased()
{

    QScrollBar* bar = view()->verticalScrollBar();
    QRect rt = bar->sliderRect();
    int mrg = rt.right() - rt.left();

    hidePopup();

    /* emit signal to have parent hide this if this is popup . */
    if( m_mode == HTMLPART_MODE_POPUP && !isUnderMouse( mrg, 0 ) ) emit hideChildPopup();
}


/* private slot */
void KitaHTMLPart::slotHSliderReleased()
{

    QScrollBar* bar = view()->horizontalScrollBar();
    QRect rt = bar->sliderRect();
    int mrg = rt.bottom() - rt.top();

    hidePopup();

    /* emit signal to have parent hide this if this is popup . */
    if( m_mode == HTMLPART_MODE_POPUP && !isUnderMouse( 0, mrg ) ) emit hideChildPopup();
}



/* private slot */
void KitaHTMLPart::slotHideChildPopup()
{
    hidePopup();

    /* emit signal to have parent hide this if this is popup . */
    if( m_mode == HTMLPART_MODE_POPUP && !isUnderMouse( 0, 0 ) ) emit hideChildPopup();
}



/*------------------------------------------*/
/* called back when kita is active .
   see also an explanation in slotOnURL.    */ /* private slot */
void KitaHTMLPart::slotKitaIsActive()
{
    m_kitaIsActive = TRUE;
}



/*---------------------------------------------------*/
/* This slot is called when mouse moves onto the URL */ /* private slot */
void KitaHTMLPart::slotOnURL( const QString& url )
{
    /* config */

    const int maxpopup = 10;  /* max number of responses shown in the popup window */

    /*----------------------------*/

    if( isMultiPopupMode() ) return;

    deletePopup();

    if ( url.isEmpty() ) return ;
    if ( url.left( 7 ) == "mailto:" ) return;

    /* Is Kita active now ?

       emit SIGNAL( isKitaActive() ) to KitaMainWindow, KitaNavi, etc. ,
       and if one of them is active, then slotKitaIsActive() is called
       back, and m_kitaIsActive is set to TRUE.   */
    m_kitaIsActive = FALSE;
    emit isKitaActive();
    if( !m_kitaIsActive ) return;

    /*------------------------*/
    /* id popup               */

    if ( url.left( 6 ) == "#idpop" )
    {
        int num = Kita::DatManager::getNumByID( m_datURL, url.mid( 6 ) );
        if ( num <= 1 ) return ;
        QString tmpstr = QString( "<DIV>ID:%1:[%2]</DIV>" ).arg( url.mid( 6 ) ).arg( num );
        showPopup( m_datURL, tmpstr );
        return ;
    }


    /*-------------------------*/
    /* popup for anchor        */

    QString innerHTML = QString::null;
    QString refstr;
    KURL datURL = m_datURL;
    int refNum;
    int refNum2;

    /* get reference */
    if ( url.at( 0 ) == '#' ) refstr = url.mid( 1 );
    else datURL = Kita::ParseMisc::parseURL( KURL( m_datURL, url ) ,refstr);

    int i = refstr.find( "-" );
    if ( i != -1 )
    { /* >>refNum-refNum2 */

        refNum = refstr.left( i ).toInt();
        refNum2 = refstr.mid( i + 1 ).toInt();

        if ( refNum )
        {
            if ( refNum2 < refNum ) refNum2 = refNum;
            if ( refNum2 - refNum > maxpopup - 1 ) refNum2 = refNum + maxpopup - 1;
        }

    }
    else
    { /* >>refNum */
        refNum = refstr.toInt();
        refNum2 = refNum;
    }

    /* another thread ? */
    if ( datURL.host() != m_datURL.host() || datURL.path() != m_datURL.path() )
    {

        /* show them with boadname & subject */
        QString boardName = Kita::DatManager::boardName( datURL );
        if ( boardName != QString::null ) innerHTML += "[" +boardName +"] ";
        QString subName = Kita::DatManager::threadName( datURL );
        if ( subName != QString::null ) innerHTML += subName +"<br><br>";

        if ( !refNum ) refNum = refNum2 = 1;
    }

    /* get HTML and show it */
    if ( !refNum ) return ;
    innerHTML += Kita::DatManager::getHtml( datURL, refNum, refNum2 );

    if ( innerHTML != QString::null ) showPopup( datURL,  innerHTML );
}



/* If selected Text is composed of only digits,
   then show res popup.                          */ /* private */
bool KitaHTMLPart::showSelectedDigitPopup()
{

    if( !hasSelection() ) return FALSE;

    QString linkstr;
    int refNum[2];
    unsigned int pos;
    QString selectText = selectedText();
    const QChar *chpt = selectText.unicode();
    unsigned int length = selectText.length();

    if( Kita::ParseMisc::parseResAnchor( chpt, length, linkstr, refNum, pos ) )
    {
        if( refNum[1] < refNum[0] ) refNum[1] = refNum[0];
        QString innerHTML = Kita::DatManager::getHtml( m_datURL, refNum[0], refNum[1] );
        if ( innerHTML != QString::null )
        {
            showPopup( m_datURL,  innerHTML );
            startMultiPopup();
            return TRUE;
        }
    }

    return FALSE;
}




/*--------------------------------------------------------------*/
/*--------------------------------------------------------------*/
/* class ResPopup */


namespace Kita
{
    ResPopup::ResPopup( KHTMLView* view, const KURL& datURL )
            : QFrame( view, "res_popup",
                      WStyle_Customize
                      | WStyle_NoBorder
                      | WStyle_Tool
                      | WType_TopLevel
                      | WX11BypassWM
                    )
    {
        m_textBrowser = new QTextBrowser( this );
        m_textBrowser->setWordWrap( QTextEdit::NoWrap );
        m_textBrowser->setResizePolicy( QScrollView::AutoOne );
        m_textBrowser->setFont( KitaConfig::threadFont() );

        m_htmlPart = new KitaHTMLPart(this);
        m_htmlPart->setup( HTMLPART_MODE_POPUP , datURL );
        connect( m_htmlPart, SIGNAL( hideChildPopup() ), SIGNAL( hideChildPopup() ) );
    }


    ResPopup::~ResPopup()
    {
        if(m_textBrowser) delete m_textBrowser;
        m_textBrowser = NULL;
        if(m_htmlPart) delete m_htmlPart;
        m_htmlPart = NULL;
    }


    /* public */
    void ResPopup::setText( const QString& str )
    {

        QString style = QString( "style=\"font-size: %1pt; "
                                 "font-family: %2; "
                                 "color: %3; "
                                 "background-color: %4; border-width: 0; \"" )
                        .arg( KitaConfig::threadFont().pointSize() )
                        .arg( KitaConfig::threadFont().family() )
                        .arg( KitaConfig::popupColor().name() )
                        .arg( KitaConfig::popupBackgroundColor().name() );

        QString text = "<html><head>";
        text += "</head><body " + style + ">";
        text += str + "</body></html>";

        if( m_textBrowser ) m_textBrowser->setText( text );

        if( m_htmlPart )
        {
            m_htmlPart->setJScriptEnabled(false);
            m_htmlPart->setJavaEnabled(false);
            m_htmlPart->begin("file:/dummy.htm");
            m_htmlPart->write(text);
            m_htmlPart->end();
            m_htmlPart->view()->setVScrollBarMode(QScrollView::AlwaysOff);
        }
    }


    /* public */
    void ResPopup::adjustSize()
    {
        /* config */

        const int mrg = 50;
        const int mrg2 = 10;

        /*---------------------------*/

        if( !m_textBrowser || !m_htmlPart ) return;

        /* get frame size from qtextbrowser because I could not
           get size by resizing the view of KHTMLPart...
           ( It is very ugly... Is there any good idea? ) */
        m_textBrowser->resize( m_textBrowser->contentsWidth() + mrg2,
                               m_textBrowser->contentsHeight() );
        int wd = m_textBrowser->contentsWidth()  +mrg;
        int ht = m_textBrowser->contentsHeight() +mrg;
        delete m_textBrowser;
        m_textBrowser = NULL;

        m_htmlPart->view()->resize( wd,ht );
        QFrame::adjustSize();
    }


    /* public */
    void ResPopup::adjustPos()
    {
        /* config */

        const int mrg = 10;

        /*----------------------------*/

        if( !m_htmlPart ) return;

        QRect qr = QApplication::desktop() ->rect();
        int sw = qr.width(), sh = qr.height();
        int wd = width(), ht = height();

        QPoint pos = QCursor::pos();
        int x = pos.x(), y = pos.y();

        if ( x + mrg + wd < sw )
        {
            x = x + mrg;
            y = QMAX( 0, y - ( ht + mrg ) );

        }
        else if ( y - ( mrg + ht ) >= 0 )
        {
            x = QMAX( 0, sw - wd );
            y = y - ( ht + mrg );

        }
        else if ( y + ( mrg + ht ) <= sh )
        {
            x = QMAX( 0, sw - wd );
            y = y + mrg;

        }
        else if ( x - ( mrg + wd ) >= 0 )
        {
            x = x - ( mrg + wd );
            y = QMAX( 0, y - ( ht + mrg ) );

        }
        else
        {
            x = x + mrg;
            y = QMAX( 0, y - ( ht + mrg ) );
        }

        pos.setX( x );
        pos.setY( y );
        move( pos );

        if( x + wd >= sw )
        {
            m_htmlPart->view()->setVScrollBarMode( QScrollView::AlwaysOn );
            wd = sw - x;
        }
        if( y + ht >= sh )
        {
            m_htmlPart->view()->setVScrollBarMode( QScrollView::AlwaysOn );
            ht = sh - y;
        }
        m_htmlPart->view()->resize( wd, ht );
        resize( wd, ht );
    }


    /* move mouse pointer above the popup frame */  /* public */
    void ResPopup::moveMouseAbove()
    {
        /* config */

        const int mrg = 10;

        /*-------------------------------*/

        QPoint pos = QCursor::pos();
        int cx = pos.x(), cy = pos.y();
        int px =  x();
        int py =  y();
        int wd = width();
        int ht = height();

        if( cx <= px ) cx = px+mrg;
        else if(cx >= px+wd ) cx = px+wd-mrg;

        if( cy <= py ) cy = py+mrg;
        else if( cy >= py+ht ) cy = py+ht-mrg;

        QCursor::setPos( cx, cy );
    }

}

