/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitawritedialog.h"
#include "libkita/qcp932codec.h"
#include "libkita/kitaconfig.h"
#include "kita2ch.h"
#include "kitahtmlpart.h"
#include "kitahtmlview.h"
#include "libkita/datmanager.h"
#include "libkita/parsemisc.h"
#include "libkita/signalcollection.h"

#include <ksqueezedtextlabel.h>
#include <klocale.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <klineedit.h>

#include <qapplication.h>
#include <qtextedit.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qeucjpcodec.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qtabwidget.h>
#include <qcheckbox.h>

#include <kio/job.h>

KitaWriteDialogPrivate* KitaWriteDialog::s_private = 0;

KitaWriteDialog::KitaWriteDialog( const KURL& datURL, QWidget* parent, const char* name )
        : KitaWriteDialogBase( parent, name, WStyle_Customize |  WType_TopLevel | WStyle_DialogBorder | WStyle_Title | WStyle_SysMenu | WStyle_MinMax | Qt::WDestructiveClose )
{
    QFont font = KitaConfig::font();
    bodyText->setFont( font );
    nameLine->setFont( font );
    nameLabel->setFont( font );
    mailLine->setFont( font );
    mailLabel->setFont( font );

    faceCombo->clear();
    faceCombo->insertItem( "" );
    QStringList list = KitaConfig::asciiArtList();
    QStringList::iterator it;
    for ( it = list.begin(); it != list.end(); ++it )
    {
        faceCombo->insertItem( *it );
    }

    /* emit when this widget is deactivated */
    connect( this, SIGNAL( windowDeactivated() ),
             Kita::SignalCollection::getInstance(), SIGNAL( windowDeactivated() ));

    /* If this widget is active and receives
       signal isKitaActive, then emit signal kitaIsActive. */
    connect( Kita::SignalCollection::getInstance(), SIGNAL( isKitaActive() ),
             this, SLOT( slotIsKitaActive() ) );

    connect( this, SIGNAL( kitaIsActive() ),
             Kita::SignalCollection::getInstance(), SIGNAL( kitaIsActive() ) );

    connect( faceCombo, SIGNAL( activated( const QString& ) ),
             bodyText, SLOT( insert( const QString& ) ) );
    /*--------------------*/
    /* init preview       */

    m_datURL = datURL;

    connect( qtw, SIGNAL( currentChanged ( QWidget * ) ),
             this, SLOT( slotCurrentChanged ( QWidget * ) ) );

    m_preview = new KitaHTMLPart( NULL );
    qtw->addTab( m_preview->view(), i18n( "preview" ) );

    m_preview->setup( HTMLPART_MODE_KHTML, m_datURL );

    /* move to center */
    QWidget * desk = QApplication::desktop();
    QPoint p = QPoint( (desk->width() - width() )/2, (desk->height()-height())/2 );
    move(p);
}

KitaWriteDialog::~KitaWriteDialog()
{}

const QString KitaWriteDialog::name() const
{
    return nameLine->text();
}

const QString KitaWriteDialog::mail() const
{
    return mailLine->text();
}

const QString KitaWriteDialog::body() const
{
    return bodyText->text();
}

void KitaWriteDialog::fetchPostInfo( Kita::PostInfo* storage ) const
{
    storage->name = name();
    storage->mail = mail();
    storage->body = body();
}

// TODO: Thread饹Ȥ褦˽񤭴
KitaWriteDialog* KitaWriteDialog::open( const Kita::PostInfo& info,
                                        const KURL& datURL,
                                        QWidget* parent )
{
    if ( ! info.check() ) return 0;
    if ( ! s_private )
    {
        s_private = new KitaWriteDialogPrivate;
        if ( ! s_private ) return 0;
    }

    KitaWriteDialog* new_dlg = new KitaWriteDialog( datURL, parent );
    if ( ! new_dlg ) return 0;

    new_dlg->m_postInfo = info;
    s_private->m_heaplist.append( new_dlg );

    new_dlg->connect( new_dlg->buttonOk, SIGNAL( clicked() ),
                      new_dlg, SLOT( postMessage() ) );

    new_dlg->connect( new_dlg->buttonCancel, SIGNAL( clicked() ),
                      new_dlg, SLOT( reject() ) );

    // set fields
    new_dlg->nameLine->setText( info.name );
    new_dlg->mailLine->setText( info.mail );
    if ( info.mail == "sage" )
    {
        new_dlg->sageBox->setChecked( true );
        new_dlg->m_mailswap = "";
    }
    new_dlg->bodyText->setText( info.body );

    new_dlg->boardNameLabel->setText( Kita::DatManager::thread_boardName( datURL ) );
    new_dlg->threadNameLabel->setText( Kita::DatManager::thread_name( datURL ) );
    new_dlg->threadURLLabel->setText( Kita::DatManager::thread_url( datURL ) );

    KCompletion* comp = KitaConfig::nameCompletion();
    new_dlg->nameLine->setCompletionObject( comp );

    new_dlg->setCaption( i18n( "Write in %1" ).arg( Kita::DatManager::thread_name( datURL ) ) );
    return new_dlg;
}

static void logPostMessage( const Kita::PostInfo& postInfo,
                            const QString& boardName,
                            const QString& threadName,
                            const QString& threadURL )
{
    QDateTime now = QDateTime::currentDateTime();
    QString logPath = locateLocal( "appdata", "log.txt" );

    QFile file( logPath );
    if ( file.open( IO_WriteOnly | IO_Append ) )
    {
        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        stream << "Date   : " << now.toString( "yyyy/MM/dd hh:mm:ss" ) << endl; // ߻
        stream << "Board  : " << boardName << endl;
        stream << "Thread : " << threadName << endl;
        stream << "URL    : " << threadURL << endl;
        stream << "Name   : " << postInfo.name << endl;
        stream << "Mail   : " << postInfo.mail << endl << endl;
        stream << postInfo.body << endl;
        stream << "----------------------------------------" << endl;

        file.close();
    }
}

void KitaWriteDialog::postMessage()
{
    fetchPostInfo( &m_postInfo );
    if ( ! m_postInfo.check() )
        return ;


    KitaConfig::addNameCompletionItem( m_postInfo.name );

    KURL bbscgi;
    bbscgi.setProtocol( "http" );
    bbscgi.setHost( m_postInfo.host );
    bbscgi.setPath( m_postInfo.bbscgi );

    //int mib = QTextCodec::codecForName("CP932")->mibEnum();
    QString postStr = m_postInfo.postStr();

    kdDebug() << "query: " << postStr << endl;
    m_array.resize( 0 );

    KIO::TransferJob* job = KIO::http_post( bbscgi, postStr.utf8(), true );
    job->addMetaData( "content-type", "Content-type: application/x-www-form-urlencoded" );
    job->addMetaData( "referrer", QString( "http://%1/%2/" ).arg( m_postInfo.host ).arg( m_postInfo.bbs ) );

    if ( ! m_postInfo.cookie.isEmpty() )
        job->addMetaData( "cookie", m_postInfo.cookie.prepend( "Cookie: " ) );

    // TODO: rewrite!
    logPostMessage( m_postInfo, boardNameLabel->text(), threadNameLabel->text(), threadURLLabel->text() );

    emit postStarted( job, m_postInfo );

    connect( job, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             this, SLOT( slotRecieveData( KIO::Job*, const QByteArray& ) ) );
    connect( job, SIGNAL( result( KIO::Job* ) ),
             this, SLOT( slotPostFinished( KIO::Job* ) ) );
}

void KitaWriteDialog::slotRecieveData( KIO::Job*, const QByteArray& data )
{
    m_array.append( data.data() );
}

void KitaWriteDialog::slotPostFinished( KIO::Job* job )
{
    QString response = QTextCodec::codecForName( "sjis" ) ->toUnicode( m_array );
    QCString output = QTextCodec::codecForName( "eucJP" ) ->fromUnicode( response );
    kdDebug() << "response: '" << output.data() << "'" << endl;
    emit postFinished( job );
    KURL bbscgi;
    bbscgi.setProtocol( "http" );
    bbscgi.setHost( m_postInfo.host );
    bbscgi.setPath( m_postInfo.bbscgi );
    emit postResponse( response, bbscgi );

    m_array.resize( 0 );

    int idx = s_private->m_heaplist.find( this );
    if ( idx != -1 )
        close( TRUE );
}

void KitaWriteDialog::reject()
{
    switch ( QMessageBox::warning( this, "Kita",
                                   i18n( "If you close this dialog, you will lose text.\n"
                                         "Do you want to close?" ),
                                   QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default ) )
    {
    case QMessageBox::Ok:
        close( TRUE );
        break;
    case QMessageBox::Cancel:
        // do nothing
        break;
    }
}

// vim:sw=2:


/* update preview screen */
void KitaWriteDialog::slotCurrentChanged ( QWidget * w )
{
    if ( w == bodyText ) return ;

    int resnum = Kita::DatManager::getReadNum( m_datURL ) + 1 ;

    QDateTime now = QDateTime::currentDateTime();
    QString bodystr = body();
    bodystr.replace( "<", "&lt;" ).replace( ">", "&gt;" ).replace( "\n", " <br> " );
    QString namestr = name();
    if ( namestr == NULL || namestr.length() == 0 ) namestr = "(default name)";
    QString rawData = namestr + "<>" + mail() + "<>"
                      + now.toString( "yy/MM/dd hh:mm" )
                      + "<> " + bodystr + " <>";

    QString htmlstr = Kita::ParseMisc::DatToHtml(rawData,resnum,KitaConfig::showMailAddress());
    m_preview->setInnerHTML( htmlstr );
}


/* when window is deactivated, emit signal to popup */ /* protected */
void KitaWriteDialog::windowActivationChange ( bool  )
{
    if(!isActiveWindow()) emit windowDeactivated();
}

/* private slot */
void KitaWriteDialog::slotIsKitaActive()
{
    if(isActiveWindow()) emit kitaIsActive();
}
