/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "pref.h"

#include <klocale.h>
#include <kfontdialog.h>
#include <kcolordialog.h>
#include <kpushbutton.h>
#include <ktextedit.h>
#include <kcolorbutton.h>

#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qmessagebox.h>
#include <qcheckbox.h>
#include <qspinbox.h>
#include <qbuttongroup.h>
#include <qvbox.h>

#include "libkita/kitaconfig.h"
#include "libkita/signalcollection.h"
#include "kitafontprefbase.h"
#include "debugprefbase.h"
#include "aboneprefpage.h"

using namespace Kita;

KitaPreferences::KitaPreferences()
        : KDialogBase( TreeList, i18n( "Kita Preferences" ),
                       Help | Default | Ok | Apply | Cancel, Ok )
{
    enableButtonApply( false );
    enableButton( Help, false );
    // this is the base class for your preferences dialog.  it is now
    // a Treelist dialog.. but there are a number of other
    // possibilities (including Tab, Swallow, and just Plain)
    QVBox * fontFrame = addVBoxPage( i18n( "Font" ), i18n( "Font" ) );
    m_fontPage = new KitaFontPrefPage( fontFrame );

    connect( m_fontPage, SIGNAL( fontChanged( const QFont& ) ),
             SIGNAL( fontChanged( const QFont& ) ) );

    QVBox * colorFrame = addVBoxPage( i18n( "Color" ), i18n( "Color" ) );
    m_colorPage = new KitaColorPrefPage( colorFrame );

    QVBox * asciiArtFrame = addVBoxPage( i18n( "AsciiArt" ), i18n( "AsciiArt" ) );
    m_asciiArtPage = new Kita::AsciiArtPrefPage( asciiArtFrame );

    QVBox * uiFrame = addVBoxPage( i18n( "User Interface" ), i18n( "User Interface" ) );
    m_uiPage = new Kita::UIPrefPage( uiFrame );

    QVBox * aboneFrame = addVBoxPage( i18n( "Abone" ), i18n( "Abone" ) );
    m_abonePage = new Kita::AbonePrefPage( aboneFrame );

    QVBox * debugFrame = addVBoxPage( i18n( "Debug" ), i18n( "Debug" ) );
    m_debugPage = new DebugPrefPage( debugFrame );

    connect( m_fontPage, SIGNAL( changed() ), SLOT( slotChanged() ) );
    connect( m_colorPage, SIGNAL( changed() ), SLOT( slotChanged() ) );
    connect( m_asciiArtPage, SIGNAL( changed() ), SLOT( slotChanged() ) );
    connect( m_uiPage, SIGNAL( changed() ), SLOT( slotChanged() ) );
    connect( m_abonePage, SIGNAL( changed() ), SLOT( slotChanged() ) );
}

void KitaPreferences::slotApply()
{
    qDebug("activePageIndex = %d", activePageIndex() );
    switch( activePageIndex() )
    {
        // TODO: refactoring.
    case 0:
        // font
        m_fontPage->apply();
        break;
    case 1:
        // color
        m_colorPage->apply();
        break;
    case 2:
        // asciiart
        m_asciiArtPage->commit();
        break;
    case 3:
        // user interface
        m_uiPage->apply();
        break;
    case 4:
        // abone
        m_abonePage->apply();
        break;
    case 5:
        // debug
        break;
    }
    enableButtonApply( false );
}

void KitaPreferences::slotChanged()
{
    enableButtonApply( true );
}

KitaFontPrefPage::KitaFontPrefPage( QWidget* parent )
        : KitaFontPrefBase( parent )
{
    connect( fontButton, SIGNAL( clicked() ), SLOT( slotFontButtonClicked() ) );

    connect( threadFontButton, SIGNAL( clicked() ),
             SLOT( slotThreadFontButtonClicked() ) );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( this, SIGNAL( setFontOfHTMLPart() ),
             signalCollection, SIGNAL( setFontOfHTMLPart() ));

    updateButtons();

    m_threadFontchanged = FALSE;
}

void KitaPreferences::slotOk()
{
    m_fontPage->apply();
    m_colorPage->apply();
    m_asciiArtPage->commit();
    m_uiPage->apply();
    m_abonePage->apply();

    KDialogBase::slotOk();
}

void KitaFontPrefPage::apply()
{
    QFont font = fontButton->font();
    KitaConfig::setFont( font );
    emit fontChanged( font );

    if( m_threadFontchanged )
    {
        QFont threadFont = threadFontButton->font();
        KitaConfig::setThreadFont( threadFont );
        emit setFontOfHTMLPart(); /* to all HTMLPart */
    }

    m_threadFontchanged = FALSE;
}

void KitaFontPrefPage::slotFontButtonClicked()
{
    QFont font = fontButton->font();

    if ( KFontDialog::getFont( font, false, this ) == QDialog::Accepted )
    {
        fontButton->setText( font.family() + " " + QString::number( font.pointSize() ) );
        fontButton->setFont( font );
        emit changed();
    }
}

void KitaFontPrefPage::slotThreadFontButtonClicked()
{
    QFont threadFont = threadFontButton->font();

    if ( KFontDialog::getFont( threadFont, false, this ) == QDialog::Accepted )
    {
        threadFontButton->setText( threadFont.family() + " " + QString::number( threadFont.pointSize() ) );
        threadFontButton->setFont( threadFont );
        emit changed();
        m_threadFontchanged = TRUE;
    }
}

void KitaFontPrefPage::updateButtons()
{
    QFont font = KitaConfig::font();
    fontButton->setText( font.family() + " " + QString::number( font.pointSize() ) );
    fontButton->setFont( font );

    QFont threadFont = KitaConfig::threadFont();
    threadFontButton->setText( threadFont.family() + " " + QString::number( threadFont.pointSize() ) );
    threadFontButton->setFont( threadFont );
}

KitaColorPrefPage::KitaColorPrefPage( QWidget *parent )
        : KitaColorPrefBase( parent )
{
    popupColorButton->setColor( KitaConfig::popupColor() );
    popupBackgroundColorButton->setColor( KitaConfig::popupBackgroundColor() );

    connect( popupColorButton, SIGNAL( changed( const QColor& ) ), SIGNAL( changed() ) );
    connect( popupBackgroundColorButton, SIGNAL( changed( const QColor& ) ), SIGNAL( changed() ) );
}

void KitaColorPrefPage::apply()
{
    KitaConfig::setPopupColor( popupColorButton->color() );
    KitaConfig::setPopupBackgroundColor( popupBackgroundColorButton->color() );
}

DebugPrefPage::DebugPrefPage( QWidget* parent )
        : DebugPrefBase( parent )
{
    connect( replaceButton, SIGNAL( clicked() ), SLOT( replace() ) );
}

void DebugPrefPage::replace()
{
    QString fromURL = fromURLLine->text();
    QString toURL = toURLLine->text();

    if ( toURL.find( fromURL ) == 0 )
    {
        QMessageBox::critical( this, i18n( "replace" ),
                               i18n( "'To' URL must not contain 'From' URL." ) );;
        return ;
    }

    if ( QMessageBox::warning( this, i18n( "replace" ),
                               i18n( "This tool may possibly break thread's info.\n"
                                     "Please create backups of '~/.kde/share/apps/kita/cache'\n"
                                     "and '~/.kde/share/apps/kita/favorites.xml' before performing.\n"
                                     "Are you ready?" ), QMessageBox::Ok,
                               QMessageBox::Cancel | QMessageBox::Default | QMessageBox::Escape ) == QMessageBox::Ok )
    {
        FavoriteThreads::replace( fromURL, toURL );
        Kita::Thread::replace( fromURL, toURL );
        KitaThreadInfo::replace( fromURL, toURL );
        Kita::FavoriteBoards::replace( fromURL, toURL );

        QMessageBox::information( this, i18n( "replace" ), i18n( "Complete." ) );
    }
}

AsciiArtPrefPage::AsciiArtPrefPage( QWidget* parent )
        : AsciiArtPrefBase( parent )
{
    asciiArtText->setText( KitaConfig::asciiArtList().join( "\n" ) );
    showAACheckBox->setChecked( KitaConfig::showAA() );

    connect( asciiArtText, SIGNAL( textChanged() ), SIGNAL( changed() ) );
    connect( showAACheckBox, SIGNAL( clicked() ), SIGNAL( changed() ) );
}

void AsciiArtPrefPage::commit()
{
    QString text = asciiArtText->text();
    QStringList list = QStringList::split( '\n', text );

    KitaConfig::setAsciiArtList( list );
    KitaConfig::setShowAA( showAACheckBox->isChecked() );
}

UIPrefPage::UIPrefPage( QWidget* parent )
        : UIPrefBase( parent )
{
    useTabCheckBox->setChecked( KitaConfig::alwaysUseTab() );
    MarkTimespinBox->setValue( KitaConfig::MarkTime() );
    showMailCheckBox->setChecked( KitaConfig::showMailAddress() );
    viewGroup->setButton( KitaConfig::viewMode() );
    orderGroup->setButton( KitaConfig::sortOrder() );
    afterShowNumSpin->setValue( KitaConfig::afterShowNum() );
    preShowNumSpin->setValue( KitaConfig::preShowNum() );
    useExtPartCheck->setChecked( KitaConfig::usePart() );
    partMimeList->setText( KitaConfig::partMimeList().join( "," ) );

    connect( useTabCheckBox, SIGNAL( clicked() ), SIGNAL( changed() ) );
    connect( MarkTimespinBox, SIGNAL( valueChanged( int ) ), SIGNAL( changed() ) );
    connect( showMailCheckBox, SIGNAL( clicked() ), SIGNAL( changed() ) );
    connect( viewGroup, SIGNAL( clicked( int ) ), SIGNAL( changed() ) );
    connect( orderGroup, SIGNAL( clicked( int ) ), SIGNAL( changed() ) );
    connect( afterShowNumSpin, SIGNAL( valueChanged( int ) ), SIGNAL( changed() ) );
    connect( preShowNumSpin, SIGNAL( valueChanged( int ) ), SIGNAL( changed() ) );
#define NCC(ctl, signal) \
    connect( ctl, SIGNAL( signal ), SIGNAL( changed() ) )
//    NCC( extPartGroup, toggled( bool ) );
    NCC( useExtPartCheck, toggled( bool ) );
    NCC( partMimeList, textChanged( const QString& ) );
}

void UIPrefPage::apply()
{
    KitaConfig::setAlwaysUseTab( useTabCheckBox->isChecked() );
    KitaConfig::setMarkTime( MarkTimespinBox->value() );
    KitaConfig::setShowMailAddresss( showMailCheckBox->isChecked() );
    KitaConfig::setViewMode( viewGroup->id( viewGroup->selected() ) );
    KitaConfig::setSortOrder( orderGroup->id( orderGroup->selected() ) );
    KitaConfig::setAfterShowNum( afterShowNumSpin->value() );
    KitaConfig::setPreShowNum( preShowNumSpin->value() );
    KitaConfig::setUsePart( useExtPartCheck->isChecked() );
    KitaConfig::setPartMimeList( partMimeList->text() );
}

#include "pref.moc"
