/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include <kpopupmenu.h>
#include <klocale.h>
#include <khtml_events.h>
#include <kdebug.h>

#include <dom/html_element.h>
#include <dom/html_misc.h>

#include <qcursor.h>
#include <qapplication.h>
#include <qclipboard.h>
#include <qmessagebox.h>
#include <qregexp.h>

#include "htmlpart.h"
#include "domtree.h"

#include "kitaui/htmlview.h"

#include "image/imageview.h"

#include "libkita/kitaconfig.h"
#include "libkita/datmanager.h"
#include "libkita/boardmanager.h"
#include "libkita/datinfo.h"
#include "libkita/kita_misc.h"
#include "libkita/signalcollection.h"
#include "libkita/imgmanager.h"


/*-------------------------------------------*/


/*-------------------------------------*/
/* Don't forget to call setup() later. */

KitaHTMLPart::KitaHTMLPart( QWidget* parent, const char* name )
        : KHTMLPart( new KitaHTMLView( this, parent, name ) )
{
    m_mode = HTMLPART_MODE_KHTML;
    m_popup = NULL;
    m_domtree = NULL;
    m_datURL = QString::null;
    m_updatedKokoyon = FALSE;

    clearPart();
    createHTMLDocument();
    connectSignals();
}


KitaHTMLPart::~KitaHTMLPart()
{
    clearPart();
}


/* private */
void KitaHTMLPart::clearPart()
{

    slotDeletePopup();

    /* delete KitaDomTree */
    if ( m_domtree ) delete m_domtree;
    m_domtree = NULL;

    /* update ViewPos */
    if ( m_mode == HTMLPART_MODE_MAINPART && !m_updatedKokoyon && !m_datURL.isEmpty() ) {
        int readNum = Kita::DatManager::getReadNum( m_datURL );
        if ( readNum ) {
            Kita::DatManager::setViewPos( m_datURL, readNum );
        }
    }
    m_updatedKokoyon = FALSE;

    /* clear variables */
    m_anchorStack.clear();
    m_centerNum = 0;
    m_jumpNumAfterLoading = 0;
    findTextInit();

    if ( !m_datURL.isEmpty() ) {  /* This part is opened. */

        /* don't forget to unlock previous datURL here. */
        Kita::DatManager::unlock( m_datURL );

        if ( m_mode == HTMLPART_MODE_MAINPART ) { /* This part is on the main thread view. */

            /* tell Thread class that "thread is closed" */
            Kita::DatManager::setMainThreadOpened( m_datURL, FALSE );

            /* emit "deactivated all thread view" SIGNAL */
            emit activateThreadView( QString::null );

            /*  update subject tab. */
            emit updateSubjectTab( m_datURL );
        }
    }
    m_datURL = QString::null;
    m_mode = HTMLPART_MODE_KHTML;
}



/* public */
bool KitaHTMLPart::setup( int mode, const KURL& url )
{
    if ( url.isEmpty() ) return FALSE;

    clearPart();

    m_datURL = Kita::getDatURL( url );
    m_mode = mode;

    if ( m_mode == HTMLPART_MODE_MAINPART ) { /* This part is on the main thread view. */

        /* create DatInfo explicitly to open new thread.   */
        /* Usually, DatInfo is NOT created if ReadNum == 0.*/
        /* See also DatManager::createDatInfo() and        */
        /*          DatManager::getDatInfo().              */
        Kita::DatManager::createDatInfo( m_datURL );

        /* tell Thread class that "thread is opend" */
        Kita::DatManager::setMainThreadOpened( m_datURL, TRUE );

        /* reset abone */
        Kita::DatManager::resetAbone( m_datURL );
    }

    /* Lock datURL. Don't forget to unlock it later ! */
    Kita::DatManager::lock ( m_datURL );

    /* create HTML Document */
    createHTMLDocument();

    /* create DOM manager */
    if ( m_mode == HTMLPART_MODE_MAINPART || m_mode == HTMLPART_MODE_NAVI ) {
        m_domtree = new KitaDomTree( htmlDocument(), m_datURL );
    }

    return TRUE;
}


/* private */
void KitaHTMLPart::connectSignals()
{

    Kita::SignalCollection * signalCollection = Kita::SignalCollection::getInstance();


    /* rendering */
    connect( this, SIGNAL( redrawHTMLPart( const KURL&, bool ) ), signalCollection, SIGNAL( redrawHTMLPart( const KURL&, bool ) ) );
    connect( signalCollection, SIGNAL( redrawHTMLPart( const KURL&, bool ) ), SLOT( slotRedrawHTMLPart( const KURL& , bool ) ) );
    connect( signalCollection, SIGNAL( redrawAllHTMLPart( bool ) ), SLOT( slotRedrawAllHTMLPart( bool ) ) );
    connect( signalCollection, SIGNAL( threadFaceChanged() ), SLOT( slotSetFaceOfHTMLPart() ) );
    connect( signalCollection, SIGNAL( setStyleSheetOfHTMLPart() ), SLOT( slotSetStyleSheetOfHTMLPart() ) );

    /* popup */
    connect( this, SIGNAL( onURL( const QString& ) ), SLOT( slotOnURL( const QString& ) ) );
    connect( this, SIGNAL( isKitaActive() ), signalCollection, SIGNAL( isKitaActive() ) );

    connect( view(), SIGNAL( leave() ), SLOT( slotLeave() ) );
    connect( view(), SIGNAL( verticalSliderReleased() ), SLOT( slotVSliderReleased() ) );
    connect( view(), SIGNAL( horizontalSliderReleased() ), SLOT( slotHSliderReleased() ) );

    connect( signalCollection, SIGNAL( kitaIsActive() ), SLOT( slotKitaIsActive() ) );
    connect( signalCollection, SIGNAL( windowDeactivated() ), SLOT( slotHideChildPopup() ) );


    /* click */
    connect( this, SIGNAL( openURLRequestExt(
                               const KURL&, const KParts::URLArgs&, QString, int,
                               const KURL& ) ),
             signalCollection, SIGNAL( openURLRequestExt(
                                           const KURL& , const KParts::URLArgs&, QString, int,
                                           const KURL& ) ) );


    /* goto anchor */
    connect( view(), SIGNAL( pushDown() ), SLOT( slotClickTugi100() ) );

    /* kitanavi */
    connect( this, SIGNAL( showKitaNavi( const KURL& , int , int ) ),
             signalCollection, SIGNAL( showKitaNavi( const KURL& , int , int ) ) );
    connect( this, SIGNAL( showKitaNaviByID( const KURL& , QString ) ),
             signalCollection, SIGNAL( showKitaNaviByID( const KURL& , QString ) ) );
    connect( this, SIGNAL( showKitaNaviByWord( const KURL& , QString ) ),
             signalCollection, SIGNAL( showKitaNaviByWord( const KURL& , QString ) ) );
    connect( this, SIGNAL( showKitaNaviByName( const KURL& , QString ) ),
             signalCollection, SIGNAL( showKitaNaviByName( const KURL& , QString ) ) );
    connect( this, SIGNAL( showKitaNaviResTree( const KURL&, int ) ),
             signalCollection, SIGNAL( showKitaNaviResTree( const KURL&, int ) ) );
    connect( this, SIGNAL( showKitaNaviRevResTree( const KURL&, int ) ),
             signalCollection, SIGNAL( showKitaNaviRevResTree( const KURL&, int ) ) );

    /* write dock */
    connect( this, SIGNAL( activateThreadView( const KURL& ) ),
             signalCollection, SIGNAL( activateThreadView( const KURL& ) ) );
    connect( this, SIGNAL( closeWriteTab( const KURL& ) ),
             signalCollection, SIGNAL( closeWriteTab( const KURL& ) ) );

    /* image viewer */
    connect( this, SIGNAL( redrawImage( const KURL& ) ),
             signalCollection, SIGNAL( redrawImage( const KURL& ) ) );

    /* subject tab */
    connect( this, SIGNAL( updateSubjectTab( const KURL& ) ),
             signalCollection, SIGNAL( updateSubjectTab( const KURL& ) ) );
}



/* private */
void KitaHTMLPart::createHTMLDocument()
{
    /* style */
    QString style = QString( "body { font-size: %1pt; font-family: \"%2\"; color: %3; background-color: %4; }" )
                    .arg( KitaConfig::threadFont().pointSize() )
                    .arg( KitaConfig::threadFont().family() )
                    .arg( KitaConfig::threadColor().name() )
                    .arg( KitaConfig::threadBackgroundColor().name() );

    QString text = "<html><head><style>";
    text += KitaConfig::defaultStyleSheetText();
    text += style;
    if ( KitaConfig::useStyleSheet() ) {
        text += KitaConfig::styleSheetText();
    }
    text += "</style></head><body></body></html>";

    setJScriptEnabled( false );
    setJavaEnabled( false );

    /* Use dummy URL here, and protocol should be "file:".
       If protocol is "http:", local image files are not shown
       (for security reasons ?).
     */
    begin( "file:/dummy.htm" );
    write( text );
    end();
}



/* public */
const int KitaHTMLPart::getMode() const { return m_mode; }



/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* direct rendering functions */


/*-----------------*/
/* show  responses */

/* !!! don't forget to call updateScreen() later !!! */   /* public */
void KitaHTMLPart::showResponses( int startnum, int endnum )
{
    if ( !m_domtree ) return ;

    for ( int i = startnum ; i <= endnum; i++ ) m_domtree->appendRes( i );
}


/* do parsing only.            */
/* call showResponses() later  */ /* public */
void KitaHTMLPart::parseResponses( int startnum, int endnum )
{
    if ( !m_domtree ) return ;

    for ( int i = startnum ; i <= endnum; i++ ) m_domtree->createResElement( i );
}


/*------------------------------------------------*/
/* Show all responses ,header, footer, and etc,
   if these are not shown.                        */

/* note that updateScreen() is called internally. */
/* So, you need not call it later.                */  /* public slot */
void KitaHTMLPart::showAll()
{
    if ( !m_domtree ) return ;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );
    if ( top != 1 || bottom != readNum ) {

        QCursor qc; qc.setShape( Qt::WaitCursor );
        QApplication::setOverrideCursor( qc );

        showResponses( 1, readNum );
        updateScreen( TRUE, FALSE );

        QApplication::restoreOverrideCursor();
    }
}


/*------------------------*/
/* insert belt node       */

/* !!! don't forget to call updateScreen() later !!! */   /* public */
void KitaHTMLPart::insertBeltNode( const QString& idstr )
{
    if ( !m_domtree ) return ;

    m_domtree->createBeltNode( idstr );
}


/*--------------------------*/
/* insert header node       */

/* !!! don't forget to call updateScreen() later !!! */   /* public */
void KitaHTMLPart::insertHeaderNode( const QString& str )
{
    if ( !m_domtree ) return ;

    m_domtree->createCommentNode( str, "header", 0, 2, TRUE );
}


/*--------------------------*/
/* insert footer node       */

/* !!! don't forget to call updateScreen() later !!! */   /* public */
void KitaHTMLPart::insertFooterNode( const QString& str )
{
    if ( !m_domtree ) return ;

    m_domtree->createCommentNode( str, "footer", 0, 0, FALSE );
}



/*-------------------*/
/* update screen     */ /* public */
void KitaHTMLPart::updateScreen( bool showHeaderEtc, bool clock )
{
    if ( !m_domtree ) {
        view() ->setFocus();
        return ;
    }

    /* show clock cursor */
    if ( clock ) {
        QCursor qc; qc.setShape( Qt::WaitCursor );
        QApplication::setOverrideCursor( qc );
    }

    /* show header, footer, and kokomadeyonda, etc. */
    if ( showHeaderEtc ) {
        m_domtree->appendMae100();
        m_domtree->appendTugi100();
        m_domtree->appendKokoyon();
        m_domtree->appendFooterAndHeader();
    }

    /* change color of number of the res which is responsed. */
    if ( m_mode == HTMLPART_MODE_MAINPART || m_mode == HTMLPART_MODE_NAVI ) {

        if ( KitaConfig::checkResponsed() ) m_domtree->changeColorOfAllResponsedNumber();
    }

    /*----------------------------*/

    /* update display */
    htmlDocument().applyChanges();
    view() ->layout();
    view() ->setVScrollBarMode( QScrollView::AlwaysOn );
    view() ->setFocus();

    if ( clock ) QApplication::restoreOverrideCursor();
}


/* public */
void KitaHTMLPart::setInnerHTML( const QString& innerHTML )
{
    createHTMLDocument();
    htmlDocument().body().setInnerHTML( innerHTML );
}



/* redraw screen  */
/* These slots are connected to signalCollection.  */ /* public slot */
void KitaHTMLPart::slotRedrawHTMLPart( const KURL& datURL, bool force )
{

    if ( m_domtree == NULL ) return ;
    if ( m_datURL != datURL ) return ;

    m_domtree->redraw( force );
}

/* public slot */
void KitaHTMLPart::slotRedrawAllHTMLPart( bool force )
{
    if ( m_domtree == NULL ) return ;

    m_domtree->redraw( force );
}

/* public slot */
void KitaHTMLPart::slotSetFaceOfHTMLPart()
{
    QFont font = KitaConfig::threadFont();

    DOM::CSSStyleDeclaration style = htmlDocument().body().style();
    style.setProperty( "font-family", font.family(), "" );
    style.setProperty( "font-size", QString( "%1pt" ).arg( font.pointSize() ), "" );
    style.setProperty( "color", KitaConfig::threadColor().name(), "" );
    style.setProperty( "background-color", KitaConfig::threadBackgroundColor().name(), "" );
    htmlDocument().applyChanges();
}

/* public slot */
void KitaHTMLPart::slotSetStyleSheetOfHTMLPart()
{
    /* [0]<html> -> [1]<head> -> [2]<style> */
    DOM::HTMLCollection collection = htmlDocument().all();
    DOM::HTMLElement elm;
    unsigned int i;
    for ( i = 0 ; i < collection.length() ; i++ ) {
        elm = collection.item( i );
        if ( elm.tagName().upper() == "STYLE" ) {
            QString style = QString( "body { font-size: %1pt; font-family: %2; color: %3; background-color: %4; }" )
                            .arg( KitaConfig::threadFont().pointSize() )
                            .arg( KitaConfig::threadFont().family() )
                            .arg( KitaConfig::threadColor().name() )
                            .arg( KitaConfig::threadBackgroundColor().name() );

            QString style0 = KitaConfig::defaultStyleSheetText();
            style0 += style;
            if ( KitaConfig::useStyleSheet() ) {
                style0 += KitaConfig::styleSheetText();
            }

            elm.setInnerText( style0 );
            htmlDocument().applyChanges();
            break;
        }
    }
}

/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* cache functions */


/*----------------------------------*/
/* load thread from cache & show it */ /* public */

/*  top = centerNum - preShowNum
    bottom = centerNum + afterShowNum
    readNum = Kita::DatManager::getReadNum
 
    No.1 <- show -> No.20 <- not show -> No.(top) <- show -> No.(bottom) <- not show -> No.(readNum) */

bool KitaHTMLPart::load( int centerNum )
{

    /* config */

    int showNum = KitaConfig::showNum();
    if ( showNum == 0 ) showNum = 10000;

    /*--------------------------------*/
    m_centerNum = centerNum;
    m_jumpNumAfterLoading = 0;

    if ( m_mode != HTMLPART_MODE_MAINPART ) return FALSE;
    if ( !m_domtree ) return FALSE;
    if ( Kita::DatManager::getReadNum( m_datURL ) == 0 ) return FALSE;

    m_domtree->appendTemplate( 0 );
    int startNum = QMAX( m_centerNum - showNum, m_domtree->getTemplateNumber() );
    int endNum = QMIN( m_centerNum + showNum, Kita::DatManager::getReadNum( m_datURL ) );
    showResponses( startNum, endNum );
    updateScreen( TRUE , FALSE );
    gotoAnchor( QString().setNum( m_centerNum ), FALSE );
    view() ->setFocus();

    return TRUE;
}



/*----------------------------*/
/* start reloading            */

/* see also slotReceiveData()
   and slotFinishLoad().      */ /* public */
bool KitaHTMLPart::reload( int jumpNum )
{
    /* config */

    const bool online = TRUE;

    /*--------------------------------*/

    if ( !online ) return FALSE;
    if ( !m_domtree ) return FALSE;
    if ( m_mode != HTMLPART_MODE_MAINPART ) {
        /* If this is not MainPart, then open MainPart.  */
        emit openURLRequestExt( m_datURL );
        return FALSE;
    }

    m_firstReceive = TRUE;
    if ( m_centerNum == 0 ) m_centerNum = m_domtree->getBottomResNumber();
    m_jumpNumAfterLoading = jumpNum;

    /* DatManager will call back slotReceiveData and slotFinishLoad. */
    Kita::DatManager::updateCache( m_datURL , this );
    view() ->setFocus();

    return TRUE;
}



/*---------------------------------------------*/
/* This slot is called after Kita::DatManager
   received new data, and emits receiveData()  */ /* !!! "public" slot !!! */
void KitaHTMLPart::slotReceiveData()
{
    /* config */

    const int delta = 20;
    const int oneAfterAnother = TRUE;
    int showNum = KitaConfig::showNum();
    if ( showNum == 0 ) showNum = 5000;  /* TODO: fix it. */

    /*--------------------------------*/

    if ( m_mode != HTMLPART_MODE_MAINPART ) return ;
    if ( !m_domtree ) return ;

    int readNum = Kita::DatManager::getReadNum( m_datURL );
    int bottom = m_domtree->getBottomResNumber();
    showNum += m_centerNum;

    /* parsing */
    if ( oneAfterAnother ) parseResponses( bottom + 1, readNum );

    /* rendering */
    if ( m_firstReceive
            || ( bottom + delta < readNum && readNum <= showNum + delta - 1 ) ) {

        if ( oneAfterAnother ) {
            showResponses( bottom + 1, QMIN( readNum, showNum ) );
            updateScreen( TRUE, FALSE );
        } else parseResponses( bottom + 1, QMIN( readNum, showNum ) );
    }

    if ( m_firstReceive && m_centerNum < readNum ) {
        gotoAnchor( QString().setNum( m_centerNum ), FALSE );
        m_firstReceive = FALSE;
    }

    emit receiveData();
}


/*--------------------------------------------*/
/* This slot is called after Kita::DatManager
   finished updating new data,
   and emits finishReload()                   */ /* !!! "public" slot !!! */
void KitaHTMLPart::slotFinishLoad()
{
    /* config */
    int showNum = KitaConfig::showNum();
    if ( showNum == 0 ) showNum = 5000;  /* TODO: fix it. */

    /*--------------------------------*/

    if ( m_mode != HTMLPART_MODE_MAINPART ) return ;
    if ( !m_domtree ) return ;

    int bottom = m_domtree->getBottomResNumber();
    int shownNum = m_centerNum + showNum;

    showResponses( bottom + 1, shownNum );
    updateScreen( TRUE, FALSE );
    //    m_domtree->parseAllRes();
    m_centerNum = 0;

    if ( m_jumpNumAfterLoading ) gotoAnchor( QString().setNum( m_jumpNumAfterLoading ), FALSE );
    m_jumpNumAfterLoading = 0;

    emit finishReload();
}




/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* goto anchor */


/* public */
bool KitaHTMLPart::gotoAnchor( const QString& anc, bool pushPosition )
{
    if ( anc == QString::null ) return FALSE;
    if ( !m_domtree || m_mode == HTMLPART_MODE_KHTML || m_mode == HTMLPART_MODE_POPUP )
        return KHTMLPart::gotoAnchor( anc );

    hidePopup();

    QString ancstr = anc;
    int res = ancstr.toInt();

    if ( res > 1 ) {

        /* is target valid ? */
        if ( !Kita::DatManager::isResValid( m_datURL, res ) ) return FALSE;

        /* show res if it is not shown */
        if ( !m_domtree->isResShown( res ) ) {

            if ( m_mode != HTMLPART_MODE_MAINPART ) return FALSE;

            int top = m_domtree->getTopResNumber();
            int bottom = m_domtree->getBottomResNumber();

            if ( res > bottom ) {
                showResponses( bottom + 1, res );
                updateScreen( TRUE, TRUE );
            } else if ( res < top ) {
                m_domtree->appendTemplate( 0 );
                showResponses( res, bottom );
                updateScreen( TRUE , TRUE );
            }
        }

        /* Target is not shown. Maybe it is aboned. */
        while ( res > 1 && !m_domtree->isResShown( res ) ) res--;

        ancstr = QString().setNum( res );
    }

    if ( res == 1 ) ancstr = "header";
    if ( pushPosition ) pushCurrentPosition();

    /* KHTMLPart::gotoAnchor() will fail if the thread is not shown. */
    /* So KHTMLPart::gotoAnchor() should be called via custom event. */
    /* See also KitaHTMLPart::customEvent() */
    GotoAnchorEvent * e = new GotoAnchorEvent( ancstr );
    QApplication::postEvent( this, e );  // Qt will delete it when done

    return TRUE;
}



/* jump to kokomade yonda */ /* public slot */
void KitaHTMLPart::slotGotoKokoyon()
{
    if ( !m_domtree ) return ;
    if ( m_mode != HTMLPART_MODE_MAINPART ) return ;

    int kokoyon = Kita::DatManager::getViewPos( m_datURL );
    gotoAnchor( QString().setNum( kokoyon ), FALSE );
}



/* public slot  */
void KitaHTMLPart::slotGobackAnchor()
{
    if ( m_anchorStack.empty() ) return ;

    QString anc = m_anchorStack.last();
    m_anchorStack.pop_back();
    gotoAnchor( anc , FALSE );
}




/* private */
void KitaHTMLPart::pushCurrentPosition()
{
    m_anchorStack += getCurrentIDofNode();
}


/* find the id of current node */ /* private */
QString KitaHTMLPart::getCurrentIDofNode()
{
    DOM::Node node;
    node = nodeUnderMouse();
    while ( node != NULL && node.nodeName().string() != "div" ) node = node.parentNode();
    if ( node == NULL ) return QString::null;

    return static_cast<DOM::Element>( node ).getAttribute( "id" ).string();
}


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* Tugi100, Mae100 etc.  */


/* public slot */
void KitaHTMLPart::slotClickTugi100()
{
    if ( !m_domtree ) return ;
    if ( m_mode != HTMLPART_MODE_MAINPART ) return ;

    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );

    if ( readNum != bottom ) {
        showResponses( bottom + 1, bottom + 100 );
        updateScreen( TRUE, TRUE );
    }
}



/* public slot */
void KitaHTMLPart::slotClickNokori()
{
    if ( !m_domtree ) return ;
    if ( m_mode != HTMLPART_MODE_MAINPART ) return ;

    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );

    if ( readNum != bottom ) {
        showResponses( bottom + 1, readNum );
        updateScreen( TRUE, TRUE );
    }
}


/* public slot */
void KitaHTMLPart::slotClickGotoFooter()
{
    if ( !m_domtree
            || m_mode != HTMLPART_MODE_MAINPART
       ) {
        gotoAnchor( "footer", FALSE );
        return ;
    }

    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );

    if ( readNum != bottom ) {
        showResponses( bottom + 1, readNum );
        updateScreen( TRUE, TRUE );
    }

    gotoAnchor( "footer", FALSE );
}



/* public slot */
void KitaHTMLPart::slotClickMae100()
{
    if ( !m_domtree ) return ;
    if ( m_mode != HTMLPART_MODE_MAINPART ) return ;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();

    if ( top != 1 ) {
        m_domtree->appendTemplate( 0 );
        showResponses( top - 100, bottom );
        updateScreen( TRUE, TRUE );

        gotoAnchor( QString().setNum( top ), FALSE );
    }
}



/* public slot */
void KitaHTMLPart::slotClickMaeZenbu()
{
    if ( !m_domtree ) return ;
    if ( m_mode != HTMLPART_MODE_MAINPART ) return ;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();

    if ( top != 1 ) {
        showResponses( 1, bottom );
        updateScreen( TRUE, TRUE );

        gotoAnchor( QString().setNum( top ), FALSE );
    }
}


/* public slot */
void KitaHTMLPart::slotClickTmpNext100()
{
    if ( !m_domtree ) return ;
    if ( m_mode != HTMLPART_MODE_MAINPART ) return ;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();
    int tmpnum = m_domtree->getTemplateNumber();

    if ( tmpnum < top ) {
        m_domtree->appendTemplate( tmpnum + 100 );
        showResponses( top, bottom );
        updateScreen( TRUE, TRUE );

        gotoAnchor( QString().setNum( tmpnum ), FALSE );
    }
}


/* public slot */
void KitaHTMLPart::slotClickShowAll()
{
    if ( !m_domtree ) return ;
    if ( m_mode != HTMLPART_MODE_MAINPART ) return ;

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();
    int readNum = Kita::DatManager::getReadNum( m_datURL );

    if ( top != 1 || bottom != readNum ) {

        QCursor qc; qc.setShape( Qt::WaitCursor );
        QApplication::setOverrideCursor( qc );

        showResponses( 1, readNum );
        updateScreen( TRUE, FALSE );

        QApplication::restoreOverrideCursor();

        gotoAnchor( "header", FALSE );
    }
}






/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* search */


/* private */
void KitaHTMLPart::findTextInit()
{
    m_findNode = NULL;
    m_findPos = -1;
    m_find_y = 0;
}



/* public */
bool KitaHTMLPart::findText( const QString &query, bool reverse )
{
    if ( m_mode != HTMLPART_MODE_MAINPART && m_mode != HTMLPART_MODE_NAVI ) return FALSE;

    QRegExp regexp( query );
    regexp.setCaseSensitive( FALSE );

    /* init */
    if ( m_findNode.isNull() ) {

        m_findNode = htmlDocument().body();
        m_find_y = 0;

        /* move to the last child node */
        if ( reverse ) {
            while ( !m_findNode.lastChild().isNull() ) m_findNode = m_findNode.lastChild();
            m_find_y = view() ->contentsHeight();
        }
    }

    while ( 1 ) {

        if ( m_findNode.nodeType() == DOM::Node::TEXT_NODE
                || m_findNode.nodeType() == DOM::Node::CDATA_SECTION_NODE ) {

            /* find the word in the current node */
            DOM::DOMString nodeText = m_findNode.nodeValue();
            QString nodestr = nodeText.string();
            if ( reverse && m_findPos != -1 ) nodestr.setLength( m_findPos );

            if ( reverse ) m_findPos = nodestr.findRev( regexp, m_findPos );
            else m_findPos = nodestr.find( regexp, m_findPos + 1 );

            /* scroll & select & return */
            if ( m_findPos != -1 ) {

                int matchLen = regexp.matchedLength();

                QRect qr = m_findNode.getRect();
                view() ->setContentsPos( qr.left() - 50, m_find_y - 100 );
                DOM::Range rg( m_findNode, m_findPos, m_findNode, m_findPos + matchLen );
                setSelection( rg );

                return TRUE;
            }

        }

        /*------------------------*/
        else if ( m_findNode.nodeName().string() == "table" ) {

            QRect qr = m_findNode.getRect();

            m_find_y = qr.bottom();
        }

        /*------------------------*/
        else if ( m_findNode.nodeName().string() == "div" ) {

            QRect qr = m_findNode.getRect();

            if ( reverse ) m_find_y = qr.bottom();
            else m_find_y = qr.top();
        }

        /*------------------------*/
        else if ( m_findNode.nodeName().string() == "br" ) {

            DOM::Node tmpnode = m_findNode.previousSibling();

            if ( tmpnode != NULL ) {

                QRect qr = tmpnode.getRect();
                if ( reverse ) m_find_y -= qr.bottom() - qr.top();
                else m_find_y += qr.bottom() - qr.top();
            }
        }

        /*------------------------*/

        m_findPos = -1;
        DOM::Node next;

        /* move to the next node */
        if ( !reverse ) {

            next = m_findNode.firstChild();
            if ( next.isNull() ) next = m_findNode.nextSibling();

            while ( !m_findNode.isNull() && next.isNull() ) {
                m_findNode = m_findNode.parentNode();
                if ( !m_findNode.isNull() ) next = m_findNode.nextSibling();
            }
        }
        /* revearse */
        else {

            next = m_findNode.lastChild();
            if ( next.isNull() ) next = m_findNode.previousSibling();

            while ( !m_findNode.isNull() && next.isNull() ) {
                m_findNode = m_findNode.parentNode();
                if ( !m_findNode.isNull() ) next = m_findNode.previousSibling();
            }
        }

        m_findNode = next;
        if ( m_findNode.isNull() ) {
            m_findNode = NULL;
            return FALSE;
        }
    }

    return FALSE;
}






/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* popup menu */


/* private */
void KitaHTMLPart::showPopupMenu( const KURL& kurl )
{
    enum {
        ID_Home_Link,
        ID_Temp_Link,
        ID_Koko_Link,
        ID_End_Link,

        ID_COPY_Link,
        ID_Copy_Str,
        ID_Abone_Word,
        ID_Extract,
        ID_Search_Google,
        ID_Open_Browser,

        ID_Show_Imgview,
        ID_Mosaic,
        ID_Del_Image,

        /*-----------------*/
        ID_Back_Link /* stay bottom */
    };

    QPoint point = QCursor::pos();
    QString url = kurl.prettyURL();
    bool showppm = FALSE;
    const int ID_Goto_Mark = ID_Back_Link + m_anchorStack.size();

    QString str;

    /* If selected Text is composed of only digits, then show res popup. */
    if ( !m_pushctrl && showSelectedDigitPopup() ) return ;

    /*-----------------------------------*/
    /* create menu items                 */

    KPopupMenu* popupMenu = new KPopupMenu( view() );
    KPopupMenu* backSubMenu = NULL;
    KPopupMenu* markSubMenu = NULL;

    popupMenu->clear();


    /*-------------*/
    /* image menu  */
    if ( KitaConfig::useImagePopup() ) {

        if ( Kita::ImgManager::isImgFile( kurl, FALSE )
                && !Kita::ImgManager::isLoadingNow( kurl )
           ) {

            if ( Kita::ImgManager::code( kurl ) == 200 && Kita::ImgManager::mosaic( kurl ) )
                popupMenu->insertItem( i18n( "Cancel mosaic" ), ID_Mosaic );

            popupMenu->insertItem( i18n( "Open with Image viewer" ), ID_Show_Imgview );

            if ( Kita::ImgManager::code( kurl ) == 200 )
                popupMenu->insertItem( i18n( "Delete" ) , ID_Del_Image );

            popupMenu->insertSeparator();
        }
    }

    /*------*/
    /* jump */
    if ( m_domtree &&
            ( m_mode == HTMLPART_MODE_MAINPART || m_mode == HTMLPART_MODE_NAVI ) ) {

        showppm = TRUE;

        /* back */
        if ( !m_anchorStack.empty() ) {
            backSubMenu = new KPopupMenu( view() );
            backSubMenu->clear();

            int i = m_anchorStack.size();
            QStringList::iterator it;
            for ( it = m_anchorStack.begin(); it != m_anchorStack.end(); it++, i-- ) {
                str = ( *it ) + "   " + Kita::DatManager::getPlainBody( m_datURL, ( *it ).toInt() ).left( 10 );
                backSubMenu->insertItem( str, ID_Back_Link + ( i - 1 ), 0 );
            }

            popupMenu->insertItem( i18n( "Back" ), backSubMenu );
            popupMenu->insertSeparator();
        }

        /* mark */
        int readNum = Kita::DatManager::getReadNum( m_datURL );
        for ( int i = 1; i <= readNum ; i++ ) {
            if ( Kita::DatManager::isMarked( m_datURL, i ) ) {
                if ( !markSubMenu ) {
                    markSubMenu = new KPopupMenu( view() );
                    markSubMenu->clear();
                    popupMenu->insertItem( i18n( "Mark" ), markSubMenu );
                    popupMenu->insertSeparator();
                }

                str = QString().setNum( i ) + "   " + Kita::DatManager::getPlainBody( m_datURL, i ).left( 10 );
                markSubMenu->insertItem( str, ID_Goto_Mark + i );
            }
        }

        /* home */
        popupMenu->insertItem( i18n( "Start" ), ID_Home_Link );


        /* template */
        if ( m_mode == HTMLPART_MODE_MAINPART ) {

            if ( m_domtree->isMae100Shown() ) popupMenu->insertItem( i18n( "template" ), ID_Temp_Link );

            int kokoyon = Kita::DatManager::getViewPos( m_datURL );
            if ( kokoyon ) {
                str = i18n( "Kokomade Yonda (%1)" ).arg( kokoyon );
                popupMenu->insertItem( str, ID_Koko_Link );
            }
        }


        /* end */
        popupMenu->insertItem( i18n( "End" ), ID_End_Link );
    }


    /*--------------*/
    /* copy & abone */
    if ( hasSelection() ) {
        if ( showppm ) popupMenu->insertSeparator();
        showppm = TRUE;

        popupMenu->insertItem( "Copy", ID_Copy_Str );

        // truncated
        QString selectedStr = selectedText();
        if ( selectedStr.length() > 20 ) {
            selectedStr.truncate( 20 );
            selectedStr.append( "..." );
        }

        popupMenu->insertItem( i18n( "Extract by '%1'" ).arg( selectedStr ), ID_Extract );
        popupMenu->insertItem( i18n( "Add '%1' to abone list" ).arg( selectedStr ), ID_Abone_Word );
        popupMenu->insertItem( i18n( "Search '%1' by Google" ).arg( selectedStr ), ID_Search_Google );
    }


    /*-----------*/
    /* copy link */
    if ( url != QString::null ) {
        if ( showppm ) popupMenu->insertSeparator();
        showppm = TRUE;

        popupMenu->insertItem( i18n( "Open with Web Browser" ), ID_Open_Browser );
        popupMenu->insertSeparator();
        popupMenu->insertItem( i18n( "Copy Link Location" ), ID_COPY_Link );
    }


    /*-----------------------------------*/

    /* show menu */
    if ( showppm ) {

        QClipboard * clipboard = QApplication::clipboard();

        int ret = popupMenu->exec( point );
        switch ( ret ) {

        case ID_Mosaic:
            Kita::ImgManager::setMosaic( kurl, FALSE );
            emit redrawImage( kurl );
            showPopup( kurl, QString::null );
            break;

        case ID_Del_Image:
            Kita::ImgManager::deleteCache( kurl, view() );
            break;

        case ID_Show_Imgview:
            emit openURLRequestExt( url, KParts::URLArgs(), QString::null, 1 );
            break;

        case ID_COPY_Link:
            clipboard->setText( url , QClipboard::Clipboard );
            clipboard->setText( url , QClipboard::Selection );
            break;

        case ID_Open_Browser:
            emit openURLRequestExt( url, KParts::URLArgs(), "text/html" );
            break;

        case ID_Home_Link: gotoAnchor( "header", FALSE ); break;

        case ID_Temp_Link: gotoAnchor( "mae100", FALSE ); break;

        case ID_Koko_Link: slotGotoKokoyon(); break;

        case ID_End_Link: slotClickGotoFooter(); break;

        case ID_Copy_Str:
            clipboard->setText( selectedText(), QClipboard::Clipboard );
            break;

        case ID_Abone_Word:

            if ( QMessageBox::information( view(), "Kita",
                                           i18n( "Do you want to add '%1' to abone list ?" ).arg( selectedText() ),
                                           QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                    == QMessageBox::Ok ) {

                KitaConfig::addAboneWord( selectedText() );
                emit redrawHTMLPart( m_datURL, FALSE );
            }

            break;

        case ID_Extract:
            emit showKitaNaviByWord( m_datURL, selectedText() );
            break;

        case ID_Search_Google: {
                QString google_url = QString( "http://www.google.com/search?ie=UTF-8&q=%1" ).arg(
                                         KURL::encode_string( selectedText(), 106 ) ); // TODO: don't use magic number.
                // 106 == UTF-8
                emit openURLRequestExt( google_url );
            }
            break;

        default:

            /* mark */
            if ( ret >= ID_Goto_Mark ) {
                gotoAnchor( QString().setNum( ret - ID_Goto_Mark ), FALSE );
            }

            /* back */
            else if ( ret >= ID_Back_Link ) {
                for ( int i = 0; i < ret - ID_Back_Link; i++ ) m_anchorStack.pop_back();
                slotGobackAnchor();
            }

            break;
        }

    }

    if ( popupMenu ) delete popupMenu;
    if ( backSubMenu ) delete backSubMenu;
    if ( markSubMenu ) delete markSubMenu;
}

/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* user event */


/* protected */ /* virtual */
void KitaHTMLPart::customEvent( QCustomEvent * e )
{
    if ( e->type() == EVENT_GotoAnchor ) {
        KHTMLPart::gotoAnchor( static_cast< GotoAnchorEvent* >( e ) ->getAnc() );
        return ;
    }

    KHTMLPart::customEvent( e );
}


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/* mouse event */


/* protected */
void KitaHTMLPart::khtmlMousePressEvent( khtml::MousePressEvent* e )
{
    emit mousePressed(); /* to KitaThreadView to focus this view. */

    KURL kurl;
    if ( e->url().string() != QString::null )
        kurl = KURL( Kita::BoardManager::boardURL( m_datURL ), e->url().string() );

    m_pushctrl = m_pushmidbt = m_pushrightbt = FALSE;
    if ( e->qmouseEvent() ->button() & Qt::RightButton ) m_pushrightbt = TRUE;
    if ( e->qmouseEvent() ->state() & Qt::ControlButton ) m_pushctrl = TRUE;
    if ( e->qmouseEvent() ->button() & Qt::MidButton ) m_pushmidbt = TRUE;

    if ( e->url() != NULL ) {

        if ( e->url().string().at( 0 ) == '#' ) { /* anchor */
            kurl = m_datURL;
            kurl.setRef( e->url().string().mid( 1 ) ) ;
        }

        clickAnchor( kurl );
        m_pushctrl = m_pushmidbt = m_pushrightbt = FALSE;
        return ;
    }

    /* popup menu */
    if ( m_pushrightbt ) {
        showPopupMenu( kurl );
        m_pushctrl = m_pushmidbt = m_pushrightbt = FALSE;
        return ;
    }

    KHTMLPart::khtmlMousePressEvent( e );
}





/*-------------------------------------------------------*/
/*-------------------------------------------------------*/
/* click */



/* private slot */
void KitaHTMLPart::slotOpenURLRequest( const KURL& urlin, const KParts::URLArgs& )
{
    clickAnchor( urlin );
}


/*------------------------------------------------------*/
/* This function is called when user clicked res anchor */ /* private */
void KitaHTMLPart::clickAnchor( const KURL& urlin )
{
    QString refstr;
    KURL datURL = Kita::getDatURL( urlin , refstr );

    /*--------------------*/
    /* Ctrl + right click */
    if ( m_pushctrl && m_pushrightbt ) {
        showPopupMenu( urlin );
        return ;
    }

    /*--------------------------------*/
    /* If this is not anchor, then    */
    /* emit openURLRequest and return */

    if ( datURL.host() != m_datURL.host() || datURL.path() != m_datURL.path() ) {

        /* right click */
        if ( m_pushrightbt ) {

            /* show image menu */
            if ( KitaConfig::useImagePopup()
                    && Kita::ImgManager::isImgFile( urlin, FALSE ) ) showPopupMenu( urlin );

            /* start multi-popup mode or show popup menu */
            else if ( !startMultiPopup() ) showPopupMenu( urlin );

            return ;
        }
        /* right click */


        /* bmp file */
        if ( Kita::ImgManager::isBMP( urlin ) ) {
            QMessageBox::warning( view(),
                                  "Kita",
                                  i18n( "This is a BMP file. It might contain a Troy." ) );
            return ;
        }
        /* bmp file */


        /* image */
        if ( Kita::ImgManager::isImgFile( urlin, FALSE ) ) {

            int active = 1;

            if ( m_pushctrl || m_pushmidbt ) {

                /* stop loading */
                if ( Kita::ImgManager::isLoadingNow( urlin ) ) {

                    if ( QMessageBox::information( view(),
                                                   "Kita",
                                                   i18n( "Do you want to stop loading ?" ),
                                                   QMessageBox::Ok,
                                                   QMessageBox::Cancel | QMessageBox::Default )
                            == QMessageBox::Ok ) Kita::ImgManager::stop( urlin );

                    return ;
                }

                if ( Kita::ImgManager::code( urlin ) != 200 ) showPopup( urlin, "<DIV>loading image...</DIV>" );
                active = 0;
            }

            QString id = getCurrentIDofNode();
            QString tmpurl = m_datURL.prettyURL();
            if ( id != QString::null ) tmpurl += "#" + id;

            emit openURLRequestExt( urlin, KParts::URLArgs(), QString::null, active, tmpurl );
            return ;
        }
        /* image */

        emit openURLRequestExt( urlin );
        return ;
    }

    if ( refstr == QString::null ) return ;

    /*---------------------------*/
    /* show popupmenu for #write */

    if ( refstr.left( 5 ) == "write" ) {
        showWritePopupMenu( refstr );
        return ;
    }

    /*----------------------------*/
    /* extract responses by ID    */

    if ( refstr.left( 5 ) == "idpop" ) {
        showIDPopup( refstr );
        return ;
    }

    /*---------------------------*/
    /* show popupmenu for #bepop */

    if ( refstr.left( 5 ) == "bepop" ) {
        showBePopupMenu( refstr );
        return ;
    }

    /*-------------------------*/
    /* start multi-popup mdde  */
    if ( m_pushrightbt && startMultiPopup() ) return ;


    /*----------------------------*/
    /* next 100 ,before 100 ,etc. */
    if ( showNext100Etc( refstr ) ) return ;


    /*-------------------------------*/
    /* open Kita Navi or goto anchor */

    int refNum, refNum2;

    int i = refstr.find( "-" );
    if ( i != -1 ) {
        refNum = refstr.left( i ).toInt();
        refNum2 = refstr.mid( i + 1 ).toInt();
        if ( refNum2 < refNum ) refNum2 = refNum;
    } else refNum = refNum2 = refstr.toInt();

    if ( !refNum ) return ;

    if ( m_pushctrl || m_pushmidbt ) emit showKitaNavi( m_datURL, refNum, refNum2 );
    else if ( ( m_mode == HTMLPART_MODE_KHTML || m_mode == HTMLPART_MODE_POPUP )
              || ( m_mode == HTMLPART_MODE_NAVI && !m_domtree->isResShown( refNum ) ) ) {
        emit openURLRequestExt( urlin );
    } else gotoAnchor( QString().setNum( refNum ), TRUE );
}



/*---------------------------------------------------------*/
/* popup menu that is opened when user clicked res number. */
/* This funtcion is called in only clickAnchor().          */ /* private */
void KitaHTMLPart::showWritePopupMenu( const QString& refstr )
{
    enum{
        WRITEMENU_RES,
        WRITEMENU_QUOTE,
        WRITEMENU_SHOWBROWSER,
        WRITEMENU_SHOWNAVI,
        WRITEMENU_RESTREE,
        WRITEMENU_REVERSERESTREE,
        WRITEMENU_EXTRACTNAME,
        WRITEMENU_COPY,
        WRITEMENU_COPYURL,
        WRITEMENU_COPYTHREADNAME,
        WRITEMENU_SETKOKOYON,
        WRITEMENU_MARK,
        WRITEMENU_ABONENAME
    };

    QClipboard * clipboard = QApplication::clipboard();
    QString str, resstr;
    int resNum = refstr.mid( 5 ).toInt();
    QString namestr = Kita::DatManager::getPlainName( m_datURL, resNum );

    /* show res tree */
    if ( m_pushrightbt ) {
        int num;
        QString htmlstr = Kita::DatManager::getTreeByRes( m_datURL, resNum, num );
        if ( !num ) return ;
        QString tmpstr = QString( "<DIV>No.%1 : [%2]<BR>" ).arg( resNum ).arg( num );
        tmpstr += htmlstr + "<BR><BR></DIV>";
        showPopup( m_datURL, tmpstr );
        startMultiPopup();
        return ;
    }

    /* open kitanavi */
    else if ( m_pushctrl | m_pushmidbt ) {
        emit showKitaNaviResTree( m_datURL, resNum );
        return ;
    }

    /*---------------------*/
    /* create popup menu */
    QString plainStr;

    KPopupMenu *popupMenu = new KPopupMenu( view() );
    popupMenu->clear();

    /* write */
    if ( m_mode == HTMLPART_MODE_MAINPART || m_mode == HTMLPART_MODE_NAVI ) {

        popupMenu->insertItem( i18n( "write response" ), WRITEMENU_RES );
        popupMenu->insertItem( i18n( "quote this" ), WRITEMENU_QUOTE );
        popupMenu->insertSeparator();

    }

    /* mark */
    popupMenu->insertItem( i18n( "Mark" ), WRITEMENU_MARK );
    popupMenu->setItemChecked( WRITEMENU_MARK, Kita::DatManager::isMarked( m_datURL, resNum ) );
    popupMenu->insertSeparator();

    /* open */
    popupMenu->insertItem( i18n( "Open with KitaNavi" ), WRITEMENU_SHOWNAVI );
    popupMenu->insertItem( i18n( "Open with Web Browser" ), WRITEMENU_SHOWBROWSER );
    popupMenu->insertSeparator();

    /* util */
    popupMenu->insertItem( i18n( "res tree" ), WRITEMENU_RESTREE );
    popupMenu->insertItem( i18n( "reverse res tree" ), WRITEMENU_REVERSERESTREE );
    popupMenu->insertItem( i18n( "extract by name" ), WRITEMENU_EXTRACTNAME );
    popupMenu->insertSeparator();

    /* copy */
    popupMenu->insertItem( i18n( "copy URL" ), WRITEMENU_COPYURL );
    popupMenu->insertItem( i18n( "Copy title and URL" ), WRITEMENU_COPYTHREADNAME );
    popupMenu->insertItem( i18n( "copy" ), WRITEMENU_COPY );

    /* kokkoma de yonda */
    if ( m_domtree && m_mode == HTMLPART_MODE_MAINPART ) {
        popupMenu->insertSeparator();
        popupMenu->insertItem( i18n( "set Kokomade Yonda" ), WRITEMENU_SETKOKOYON );
    }

    /* abone */
    popupMenu->insertSeparator();
    popupMenu->insertItem( i18n( "add name to abone list" ), WRITEMENU_ABONENAME );


    /*--------------------------------------*/
    /* show popup menu */

    int ret = popupMenu->exec( QCursor::pos() );
    delete popupMenu;
    switch ( ret ) {

    case WRITEMENU_RES:
        resstr = ">>" + QString().setNum( resNum ) + "\n";
        emit openWriteDialog( resstr );
        break;

    case WRITEMENU_QUOTE:
        resstr = ">>" + QString().setNum( resNum ) + "\n"
                 + "> " + Kita::DatManager::getPlainTitle( m_datURL, resNum ) + "\n"
                 + "> " + Kita::DatManager::getPlainBody( m_datURL, resNum ).replace( "\n", "\n> " ) + "\n";
        emit openWriteDialog( resstr );
        break;

    case WRITEMENU_MARK:
        Kita::DatManager::setMark( m_datURL, resNum, ! Kita::DatManager::isMarked( m_datURL, resNum ) );
        break;

    case WRITEMENU_COPY:
    case WRITEMENU_COPYURL:
    case WRITEMENU_COPYTHREADNAME:
        str = QString::null;

        /* title */
        if ( ret == WRITEMENU_COPYTHREADNAME || ret == WRITEMENU_COPY ) {
            str = Kita::DatManager::threadName( m_datURL );
        }

        /* url */
        if ( str != QString::null ) str += "\n";
        str += Kita::DatManager::threadURL( m_datURL ) + "/" + QString().setNum( resNum ) + "\n";

        /* body */
        if ( ret == WRITEMENU_COPY ) {
            str += "\n"
                   + Kita::DatManager::getPlainTitle( m_datURL, resNum ) + "\n"
                   + Kita::DatManager::getPlainBody( m_datURL, resNum ) + "\n";
        }

        /* copy */
        clipboard->setText( str , QClipboard::Clipboard );
        clipboard->setText( str , QClipboard::Selection );
        break;


    case WRITEMENU_SETKOKOYON:
        Kita::DatManager::setViewPos( m_datURL, resNum );
        emit updateSubjectTab( m_datURL );
        m_updatedKokoyon = TRUE;
        updateScreen( TRUE, TRUE );
        gotoAnchor( QString().setNum( resNum ), FALSE );
        break;

    case WRITEMENU_SHOWBROWSER:
        str = Kita::DatManager::threadURL( m_datURL ) + "/" + QString().setNum( resNum );

        emit openURLRequestExt( str, KParts::URLArgs(), "text/html" );
        break;


    case WRITEMENU_SHOWNAVI:
        emit showKitaNavi( m_datURL, resNum, resNum );
        break;


    case WRITEMENU_RESTREE:
        emit showKitaNaviResTree( m_datURL, resNum );
        break;

    case WRITEMENU_REVERSERESTREE:
        emit showKitaNaviRevResTree( m_datURL, resNum );
        break;

    case WRITEMENU_EXTRACTNAME:
        emit showKitaNaviByName( m_datURL, namestr );
        break;

    case WRITEMENU_ABONENAME:
        if ( QMessageBox::information( view(), "Kita",
                                       i18n( "Do you want to add '%1' to abone list ?" ).arg( namestr ),
                                       QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                == QMessageBox::Ok ) {

            KitaConfig::addAboneName( namestr );
            emit redrawHTMLPart( m_datURL, FALSE );
        }
        break;

    default:
        break;
    }

}



/*--------------------------------------------------*/
/* popup that is opened when user clicked ID        */
/* This funtcion is called in only clickAnchor().   */ /* private */
void KitaHTMLPart::showIDPopup( const QString& refstr )
{
    QString strid = refstr.mid( 5 )
                    .replace( "%2B", "+" )
                    .replace( "%2F", "/" );

    /* popup */
    if ( m_pushrightbt ) {
        int num;
        QString htmlstr
        = Kita::DatManager::getHtmlByID( m_datURL, strid, num );
        if ( num <= 1 ) return ;
        QString tmpstr = QString( "<DIV>ID:%1:[%2]<BR>" ).arg( strid ).arg( num );
        tmpstr += htmlstr + "<BR><BR></DIV>";
        showPopup( m_datURL, tmpstr );
        startMultiPopup();
    }

    /* open kitanavi when user pushed Ctrl+Left or Mid button. */
    else if ( m_pushctrl | m_pushmidbt ) emit showKitaNaviByID( m_datURL, strid );

    else {

        enum{
            IDMENU_EXTRACT,
            IDMENU_ABONE
        };

        KPopupMenu *popupMenu = new KPopupMenu( view() );
        popupMenu->clear();
        if ( Kita::DatManager::getNumByID( m_datURL, strid ) > 1 )
            popupMenu->insertItem( i18n( "extract by ID" ), IDMENU_EXTRACT );
        popupMenu->insertItem( i18n( "add id to abone list" ), IDMENU_ABONE );
        int ret = popupMenu->exec( QCursor::pos() );
        delete popupMenu;
        switch ( ret ) {

        case IDMENU_EXTRACT:
            emit showKitaNaviByID( m_datURL, strid );
            break;

        case IDMENU_ABONE:
            /* add ID to abone list */
            if ( QMessageBox::information( view(), "Kita",
                                           i18n( "Do you want to add '%1' to abone list ?" ).arg( strid ),
                                           QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                    == QMessageBox::Ok ) {

                KitaConfig::addAboneID( strid );
                emit redrawHTMLPart( m_datURL, FALSE );
            }
            break;

        default:
            break;
        }
    }
}



/*---------------------------------------------------------*/
/* popup menu that is opened when user clicked be anchor.  */
/* This funtcion is called in only clickAnchor().          */ /* private */
void KitaHTMLPart::showBePopupMenu( const QString& refstr )
{
    QString strURL = "http://be.2ch.net/test/p.php?i=" + refstr.mid( 5 )
                    + "&u=d:" + Kita::DatManager::threadURL( m_datURL ) + "/l50";

    if ( m_pushrightbt ) {
        /*---------------------*/
        /* create popup menu */
        KPopupMenu *popupMenu = new KPopupMenu( view() );
        popupMenu->clear();

        enum{
            BEMENU_COPYURL,
            BEMENU_SHOWBROWSER
        };
    
        QClipboard * clipboard = QApplication::clipboard();
    
        popupMenu->insertItem( i18n( "copy URL" ), BEMENU_COPYURL );
        popupMenu->insertItem( i18n( "Open with Web Browser" ), BEMENU_SHOWBROWSER );
    
        /*--------------------------------------*/
        /* show popup menu */
    
        int ret = popupMenu->exec( QCursor::pos() );
        delete popupMenu;
        switch ( ret ) {
    
        case BEMENU_COPYURL:
            /* copy */
            clipboard->setText( strURL, QClipboard::Clipboard );
            clipboard->setText( strURL, QClipboard::Selection );
            break;
    
        case BEMENU_SHOWBROWSER:
            emit openURLRequestExt( strURL, KParts::URLArgs(), "text/html" );
            break;
    
        default:
            break;
        }
    } else {
        emit openURLRequestExt( strURL, KParts::URLArgs(), "text/html" );
    }
}



/*-------------------------------------------------------*/
/* funtcion when user clicked next 100 ,before 100 ,etc. */
/* This funtcion is called in only clickAnchor().        */ /* private */
bool KitaHTMLPart::showNext100Etc( const QString& refstr )
{
    if ( m_mode != HTMLPART_MODE_MAINPART ) return FALSE;

    if ( refstr.left( 7 ) == "tugi100" ) {
        slotClickTugi100();
        return TRUE;

    } else if ( refstr.left( 6 ) == "nokori" ) {
        slotClickNokori();
        return TRUE;

    } else if ( refstr.left( 7 ) == "tosaigo" ) {
        slotClickGotoFooter();
        return TRUE;

    } else if ( refstr.left( 6 ) == "mae100" ) {
        slotClickMae100();
        return TRUE;

    } else if ( refstr.left( 8 ) == "maezenbu" ) {
        slotClickMaeZenbu();
        return TRUE;

    } else if ( refstr.left( 6 ) == "tmp100" ) {
        slotClickTmpNext100();
        return TRUE;

    } else if ( refstr.left( 5 ) == "zenbu" ) {
        slotClickShowAll();
        return TRUE;
    }

    return FALSE;
}



/*-------------------------------------------------------*/
/*-------------------------------------------------------*/
/* popup */


/* public */
bool KitaHTMLPart::isPopupVisible()
{
    if ( !m_popup ) return FALSE;
    return m_popup->isVisible();
}


/* public slot */
void KitaHTMLPart::slotDeletePopup()
{
    if ( m_popup ) delete m_popup;
    m_popup = NULL;
    m_multiPopup = FALSE;
}


/* for convenience */ /* public slot */
void KitaHTMLPart::slotShowResPopup( QPoint point, int refNum, int refNum2 )
{
    QString innerHTML = Kita::DatManager::getHtml( m_datURL, refNum, refNum2 );
    if ( innerHTML == QString::null ) return ;

    showPopupCore( m_datURL, innerHTML, point );
}


/* for convenience */ /* private */
void KitaHTMLPart::showPopup( const KURL& url, const QString& innerHTML )
{
    showPopupCore( url, innerHTML, QCursor::pos() );
}


/* show popup window   */  /* private */
void KitaHTMLPart::showPopupCore( const KURL& url, const QString& innerHTML, QPoint point )
{
    slotDeletePopup();
    m_multiPopup = FALSE;

    m_popup = new Kita::ResPopup( view() , url );

    connect( m_popup, SIGNAL( hideChildPopup() ), SLOT( slotHideChildPopup() ) );

    m_popup->setText( innerHTML );
    m_popup->adjustSize();
    m_popup->adjustPos( point );
    m_popup->show();
}


/*------------------------*/
/* start multi-popup mode */ /* private */
bool KitaHTMLPart::startMultiPopup()
{

    if ( m_popup && m_popup->isVisible() ) {
        m_multiPopup = TRUE;
        m_popup->moveMouseAbove();
    } else m_multiPopup = FALSE;

    return m_multiPopup;
}


/* Is it multi-popup mode now ? */ /* private */
bool KitaHTMLPart::isMultiPopupMode()
{
    if ( !m_popup ) m_multiPopup = FALSE;
    else if ( m_popup->isHidden() ) m_multiPopup = FALSE;

    return m_multiPopup;
}


/* private */
void KitaHTMLPart::hidePopup()
{
    if ( m_popup ) m_popup->hide();
    m_multiPopup = FALSE;
}



/* return TRUE if this view is under mouse. */ /* private */
bool KitaHTMLPart::isUnderMouse( int mrgwd, int mrght )
{
    QPoint pos = QCursor::pos();
    int cx = pos.x(), cy = pos.y();

    QPoint viewpos = view() ->mapToGlobal( QPoint( 0, 0 ) );
    int px = viewpos.x(), py = viewpos.y();
    int wd = view() ->visibleWidth(), ht = view() ->visibleHeight();

    if ( ( px < cx && cx < px + wd + mrgwd )
            && ( py < cy && cy < py + ht + mrght ) ) {
        return TRUE;
    }

    return FALSE;
}


/* private slot */
void KitaHTMLPart::slotLeave()
{
    if ( isMultiPopupMode() ) return ;
    if ( view() ->isHorizontalSliderPressed() ) return ;
    if ( view() ->isVerticalSliderPressed () ) return ;

    hidePopup();

    /* emit signal to have parent hide this if this is popup . */
    if ( m_mode == HTMLPART_MODE_POPUP && !isUnderMouse( 0, 0 ) ) emit hideChildPopup();
}


/* private slot */
void KitaHTMLPart::slotVSliderReleased()
{

    QScrollBar * bar = view() ->verticalScrollBar();
    QRect rt = bar->sliderRect();
    int mrg = rt.right() - rt.left();

    hidePopup();

    /* emit signal to have parent hide this if this is popup . */
    if ( m_mode == HTMLPART_MODE_POPUP && !isUnderMouse( mrg, 0 ) ) emit hideChildPopup();
}


/* private slot */
void KitaHTMLPart::slotHSliderReleased()
{

    QScrollBar * bar = view() ->horizontalScrollBar();
    QRect rt = bar->sliderRect();
    int mrg = rt.bottom() - rt.top();

    hidePopup();

    /* emit signal to have parent hide this if this is popup . */
    if ( m_mode == HTMLPART_MODE_POPUP && !isUnderMouse( 0, mrg ) ) emit hideChildPopup();
}



/* private slot */
void KitaHTMLPart::slotHideChildPopup()
{
    hidePopup();

    /* emit signal to have parent hide this if this is popup . */
    if ( m_mode == HTMLPART_MODE_POPUP && !isUnderMouse( 0, 0 ) ) emit hideChildPopup();
}



/*------------------------------------------*/
/* called back when kita is active .
   see also an explanation in slotOnURL.    */ /* private slot */
void KitaHTMLPart::slotKitaIsActive()
{
    m_kitaIsActive = TRUE;
}



/*---------------------------------------------------*/
/* This slot is called when mouse moves onto the URL */ /* private slot */
void KitaHTMLPart::slotOnURL( const QString& url )
{
    /* config */

    const int maxpopup = 10;  /* max number of responses shown in the popup window */

    /*----------------------------*/

    if ( isMultiPopupMode() ) return ;

    slotDeletePopup();

    if ( url.isEmpty() ) return ;
    if ( url.left( 7 ) == "mailto:" ) return ;

    /* Is Kita active now ?

       emit SIGNAL( isKitaActive() ) to KitaMainWindow, KitaNavi, etc. ,
       and if one of them is active, then slotKitaIsActive() is called
       back, and m_kitaIsActive is set to TRUE.   */
    m_kitaIsActive = FALSE;
    emit isKitaActive();
    if ( !m_kitaIsActive ) return ;

    /* get reference */
    QString refstr;
    KURL datURL = m_datURL;
    if ( url.at( 0 ) == '#' ) refstr = url.mid( 1 );
    else datURL = Kita::getDatURL( KURL( m_datURL, url ) , refstr );

    /*------------------------*/
    /* id popup               */

    if ( url.left( 6 ) == "#idpop" ) {
        int num = Kita::DatManager::getNumByID( m_datURL, url.mid( 6 ) );
        QString tmpstr;
        if ( num >= 2 ) tmpstr = QString( "<DIV>ID:%1:[%2]</DIV>" ).arg( url.mid( 6 ) ).arg( num );
        else tmpstr = "<DIV>" + i18n( "None" ) + "</DIV>";
        showPopup( m_datURL, tmpstr );
        return ;
    }


    /*------------------------*/
    /* show reffered num      */

    if ( refstr.left( 5 ) == "write" ) {
        int no = refstr.mid( 5 ).toInt();
        int num = 0;
        Kita::DatManager::getTreeByRes( m_datURL, no, num );
        QString tmpstr;
        if ( num ) tmpstr = QString( "<DIV>No.%1 : [%2]</DIV>" ).arg( no ).arg( num );
        else tmpstr = "<DIV>" + i18n( "None" ) + "</DIV>";
        showPopup( m_datURL, tmpstr );
        return ;
    }


    /*------------------------*/
    /* abone                  */

    if ( url.left( 6 ) == "#abone" ) {
        int no = url.mid( 6 ).toInt();
        QString tmpstr = Kita::DatManager::getHtml( m_datURL, no, no, FALSE );
        showPopup( m_datURL, tmpstr );
        return ;
    }

    /*------------------------*/
    /* image popup            */
    if ( KitaConfig::useImagePopup() ) {

        if ( Kita::ImgManager::isImgFile( url, FALSE ) ) {

            /* now loading */
            if ( Kita::ImgManager::isLoadingNow( url ) ) {
                showPopup( url, "<DIV>loading image...</DIV>" );
                return ;
            }

            /* show error code */
            int code = Kita::ImgManager::code( url );
            if ( code == -1 ) return ;
            if ( code != 200 ) {
                QString tmpstr = QString( "<DIV>error %1 </DIV>" ).arg( code );
                showPopup( url, tmpstr );
                return ;
            }

            /* show image */
            showPopup( url, QString::null );
            return ;
        }
    }


    /*-------------------------*/
    /* popup for anchor        */

    QString innerHTML = QString::null;
    int refNum;
    int refNum2;

    int i = refstr.find( "-" );
    if ( i != -1 ) { /* >>refNum-refNum2 */

        refNum = refstr.left( i ).toInt();
        refNum2 = refstr.mid( i + 1 ).toInt();

        if ( refNum ) {
            if ( refNum2 < refNum ) refNum2 = refNum;
            if ( refNum2 - refNum > maxpopup - 1 ) refNum2 = refNum + maxpopup - 1;
        }

    } else { /* >>refNum */
        refNum = refstr.toInt();
        refNum2 = refNum;
    }

    /* another thread ? */
    if ( datURL.host() != m_datURL.host() || datURL.path() != m_datURL.path() ) {

        /* get board name */
        QString boardName = Kita::BoardManager::boardName( datURL );
        if ( boardName != QString::null ) innerHTML += "[" + boardName + "] ";

        /* If idx file of datURL is not read, thread name cannot be obtained.
           so, create DatInfo if cache exists, and read idx file in DatInfo::DatInfo(). */
        Kita::DatManager::getDatInfoPointer( datURL );

        /* get thread Name */
        QString subName = Kita::DatManager::threadName( datURL );
        if ( subName != QString::null ) innerHTML += subName + "<br><br>";

        if ( !refNum ) refNum = refNum2 = 1;
    }

    /* get HTML and show it */
    if ( !refNum ) return ;
    innerHTML += Kita::DatManager::getHtml( datURL, refNum, refNum2 );

    if ( innerHTML != QString::null ) showPopup( datURL, innerHTML );
}



/* If selected Text is composed of only digits,
   then show res popup.                          */ /* private */
bool KitaHTMLPart::showSelectedDigitPopup()
{

    if ( !hasSelection() ) return FALSE;

    QString linkstr;
    int refNum;
    QString selectText = selectedText();
    const QChar *chpt = selectText.unicode();
    unsigned int length = selectText.length();

    if ( ( refNum = Kita::stringToPositiveNum( chpt, length ) ) != -1 ) {
        QString innerHTML = Kita::DatManager::getHtml( m_datURL, refNum, refNum );
        if ( innerHTML != QString::null ) {
            showPopup( m_datURL, innerHTML );
            startMultiPopup();
            return TRUE;
        }
    }

    return FALSE;
}




/*--------------------------------------------------------------*/
/*--------------------------------------------------------------*/
/* class ResPopup */


namespace Kita
{
    ResPopup::ResPopup( KHTMLView* view, const KURL& url )
            : QFrame( view, "res_popup",
                      WStyle_Customize
                      | WStyle_NoBorder
                      | WStyle_Tool
                      | WType_TopLevel
                      | WX11BypassWM
                    )
    {
        m_url = url;
        m_htmlPart = NULL;
        m_imgview = NULL;

        /* image extension */
        if ( KitaConfig::useImagePopup() ) {

            if ( Kita::ImgManager::isImgFile( url, FALSE ) ) {

                connect( Kita::ImgManager::getInstance(),
                         SIGNAL( receiveImgData( const KURL&, unsigned int, unsigned int ) ),
                         SLOT( slotReceiveImgData( const KURL&, unsigned int, unsigned int ) ) );
                connect( Kita::ImgManager::getInstance(), SIGNAL( finishImgLoad( const KURL& ) ),
                         SLOT( slotFinishImgLoad( const KURL& ) ) );

                /* show image */
                if ( !Kita::ImgManager::isLoadingNow( url ) && Kita::ImgManager::code( url ) == 200 ) {
                    showImage( url );
                    return ;
                }
            }
        }

        m_htmlPart = new KitaHTMLPart( this );
        m_htmlPart->setup( HTMLPART_MODE_POPUP , url );
        connect( m_htmlPart, SIGNAL( hideChildPopup() ), SIGNAL( hideChildPopup() ) );
    }


    ResPopup::~ResPopup()
    {
        if ( m_htmlPart ) delete m_htmlPart;
        if ( m_imgview ) delete m_imgview;
    }



    /* public */
    void ResPopup::setText( const QString& str )
    {
        const int maxwd = 1600;
        const int maxht = 1200;

        QString style = QString( "body.pop {"
                                 " font-size: %1pt; "
                                 " font-family: %2; "
                                 " color: %3; "
                                 " background-color: %4; "
                                 " border-width: 0;"
                                 "}" )
                        .arg( KitaConfig::popupFont().pointSize() )
                        .arg( KitaConfig::popupFont().family() )
                        .arg( KitaConfig::popupColor().name() )
                        .arg( KitaConfig::popupBackgroundColor().name() );

        QString text = "<html><head><style>";
        text += KitaConfig::defaultStyleSheetText();
        text += style;
        if ( KitaConfig::useStyleSheet() ) {
            text += KitaConfig::styleSheetText();
        }
        text += "</style></head><body class=\"pop\">";
        text += str;
        text += "</body></html>";

        if ( m_htmlPart ) {
            m_htmlPart->view() ->resize( maxwd, maxht );
            m_htmlPart->setJScriptEnabled( false );
            m_htmlPart->setJavaEnabled( false );
            m_htmlPart->begin( "file:/dummy.htm" );
            m_htmlPart->write( text );
            m_htmlPart->end();
            m_htmlPart->view() ->setVScrollBarMode( QScrollView::AlwaysOff );
        }
    }


    /* public */
    void ResPopup::adjustSize()
    {
        if ( !m_htmlPart ) return ;

        int width = 0, xx = 0, leftmrg = 0;
        int maxwidth = 0, maxheight = 0;
        DOM::Node curnode = m_htmlPart->htmlDocument().body().firstChild();

        for ( ;; ) {

            QRect qr = curnode.getRect();
            int tmpwd = qr.right() - qr.left();

            /*----------------------------------*/

            if ( curnode.nodeType() == DOM::Node::TEXT_NODE ) {
                if ( xx == 0 ) xx = qr.left();
                width += tmpwd;
            }

            /*----------------------------------*/

            else if ( curnode.nodeName().string() == "div" ) {
                if ( leftmrg == 0 ) leftmrg = qr.left();
                width = 0;
                xx = 0;
            }

            /*----------------------------------*/

            else if ( curnode.nodeName().string() == "br" ) {
                width = 0;
                xx = 0;
            }


            /*----------------------------------*/

            if ( leftmrg + xx + width > maxwidth ) maxwidth = leftmrg + xx + width;
            if ( qr.bottom() > maxheight ) maxheight = qr.bottom();

            /* move to the next node */
            DOM::Node next = curnode.firstChild();

            if ( next.isNull() ) next = curnode.nextSibling();

            while ( !curnode.isNull() && next.isNull() ) {
                curnode = curnode.parentNode();
                if ( !curnode.isNull() ) next = curnode.nextSibling();
            }

            curnode = next;

            if ( curnode.isNull() ) break;
        }

        const int mrg = 32;

        int wd = maxwidth + mrg;
        int ht = maxheight + mrg;

        m_htmlPart->view() ->resize( wd, ht );
        QFrame::adjustSize();
    }


    /* public */
    void ResPopup::adjustPos( QPoint pos )
    {
        enum{
            POS_LeftUp,
            POS_RightUp,
            POS_LeftDown,
            POS_RightDown
        };

        /* config */

        const int mrg = 16;

        /*----------------------------*/

        if ( !m_htmlPart && !m_imgview ) return ;

        QRect qr = QApplication::desktop() ->rect();
        int sw = qr.width(), sh = qr.height();
        int wd = width(), ht = height();
        int x = pos.x(), y = pos.y();
        int idx;

        if ( ( x + mrg ) + wd < sw
                && ( y - mrg ) - ht >= 0 ) idx = POS_RightUp;

        else if ( ( x - mrg ) - wd >= 0
                  && y - ( ht + mrg ) >= 0 ) idx = POS_LeftUp;

        else if ( ( x + mrg ) + wd < sw
                  && ( y + mrg ) + ht < sh ) idx = POS_RightDown;

        else if ( ( x - mrg ) - wd >= 0
                  && ( y + mrg ) + ht < sh ) idx = POS_LeftDown;

        else {
            int area[ 4 ];
            area[ 0 ] = ( sw - x ) * y;
            area[ 1 ] = x * y;
            area[ 2 ] = ( sw - x ) * ( sh - y );
            area[ 3 ] = x * ( sh - y );

            idx = 0;
            for ( int i = 1; i < 4; ++i ) if ( area[ i ] > area[ idx ] ) idx = i;
        }

        switch ( idx ) {

        case POS_RightUp:
            x = x + mrg;
            y = ( y - mrg ) - ht;
            break;

        case POS_LeftUp:
            x = ( x - mrg ) - wd;
            y = ( y - mrg ) - ht;
            break;

        case POS_RightDown:
            x = x + mrg;
            y = y + mrg;
            break;

        case POS_LeftDown:
            x = ( x - mrg ) - wd;
            y = y + mrg;
            break;
        }

        if ( x < 0 ) {

            x = ht % 16;
        }
        if ( x + wd >= sw ) {

            x = sw - wd - ( ht % 16 );

            if ( x < 0 ) {
                if ( m_htmlPart ) m_htmlPart->view() ->setVScrollBarMode( QScrollView::AlwaysOn );
                x = 0;
                wd = sw;
            }
        }

        if ( y < 0 ) {
            if ( x <= pos.x() && pos.x() < x + wd ) {
                if ( m_htmlPart ) m_htmlPart->view() ->setVScrollBarMode( QScrollView::AlwaysOn );
                ht += y;
            }
            y = 0;
        }
        if ( y + ht >= sh ) {

            if ( x <= pos.x() && pos.x() < x + wd ) {
                if ( m_htmlPart ) m_htmlPart->view() ->setVScrollBarMode( QScrollView::AlwaysOn );
                ht = sh - y;
            } else {
                y = sh - ht;

                if ( y < 0 ) {
                    if ( m_htmlPart ) m_htmlPart->view() ->setVScrollBarMode( QScrollView::AlwaysOn );
                    y = 0;
                    ht = sh;
                }
            }
        }

        pos.setX( x );
        pos.setY( y );
        move( pos );

        if ( m_htmlPart ) m_htmlPart->view() ->resize( wd, ht );
        resize( wd , ht );
    }


    /* move mouse pointer above the popup frame */  /* public */
    void ResPopup::moveMouseAbove()
    {
        /* config */

        const int mrg = 10;

        /*-------------------------------*/

        QPoint pos = QCursor::pos();
        int cx = pos.x(), cy = pos.y();
        int px = x();
        int py = y();
        int wd = width();
        int ht = height();

        if ( cx <= px ) cx = px + mrg;
        else if ( cx >= px + wd ) cx = px + wd - mrg;

        if ( cy <= py ) cy = py + mrg;
        else if ( cy >= py + ht ) cy = py + ht - mrg;

        QCursor::setPos( cx, cy );
    }


    /*----------------------------------------------------------*/
    /* image extension */

    /* private */
    void ResPopup::showImage( const KURL& url )
    {
        /* size of image popup */
        const int wd = 320;
        const int ht = 240;

        /*---------------------------*/

        if ( m_imgview ) return ;
        if ( m_htmlPart ) m_htmlPart->hide();

        if ( Kita::ImgManager::code( url ) == 200 ) {

            QString path = Kita::ImgManager::getPath( url );

            m_imgview = new KitaImgView( url, KURL(), NULL, this, "Viewer", 0L );
            m_imgview->resize( wd, ht );
            resize( wd, ht );
            m_imgview->show();
            m_imgview->slotCustomSize( wd, ht );
        }
    }


    /* private slot */
    void ResPopup::slotReceiveImgData( const KURL& url, unsigned int size, unsigned int total )
    {
        if ( url != m_url ) return ;

        QString tmpstr = QString( "<DIV>loading image...<br>%1k/%2k</DIV>" )
                         .arg( size / 1024 ).arg( total / 1024 );
        setText( tmpstr );
    }

    /* private slot */
    void ResPopup::slotFinishImgLoad( const KURL& url )
    {
        if ( url != m_url ) return ;
        if ( m_htmlPart == NULL ) return ;

        int code = Kita::ImgManager::code( url );

        if ( code == 200 ) {
            showImage( url );
        } else {
            QString tmpstr = QString( "<DIV>error %1</DIV>" ).arg( code );
            setText( tmpstr );
        }
    }

}

