/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

// include files for Qt
#include <qtoolbutton.h>
#include <qtextbrowser.h>
#include <qtextcodec.h>
#include <qregexp.h>
#include <qfile.h>
#include <qclipboard.h>

// kdelibs/kio
#include <kio/slaveconfig.h>
#include <kio/netaccess.h>

// include files for KDE
#include <kfilterdev.h>
#include <klistview.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <krun.h>

#include "kitasubjectview.h"
#include "favoritelistview.h"
#include "kita.h"
#include "kitacacheinfo.h"
#include "part/kita2ch.h"
#include "libkita/thread.h"

enum SubjectViewRows {
  Row_Mark,
  Row_ID,
  Row_Icon,
  Row_Subject,
  Row_ResNum,
  Row_Read,
  Row_Unread,
  Row_DatName,
  Row_DatURL
};

KitaSubjectView::KitaSubjectView(QWidget *parent, const char *name)
  : KitaSubjectViewBase(parent, name)
{
  {
    SearchButton->setPixmap( SmallIcon("find") );
    HideButton->setPixmap( SmallIcon("filter") );
  }

  subjectList->addColumn("");
  subjectList->addColumn(i18n("No."));
  subjectList->addColumn("");
  subjectList->addColumn(i18n("Title"));
  subjectList->addColumn(i18n("ResNum"));
  subjectList->addColumn(i18n("ReadNum"));
  subjectList->addColumn(i18n("Unread"));
  subjectList->addColumn(i18n("Dat"));

  subjectList->setColumnWidth(Row_Subject, 400);
  subjectList->setColumnWidthMode(Row_Subject, QListView::Manual);

  connect( SearchButton, SIGNAL( clicked() ), SLOT( slotSearchButton() ) );
  connect( SearchCombo, SIGNAL( activated(int) ), SLOT( slotSearchButton() ) );
  connect( HideButton, SIGNAL( toggled(bool) ), SLOT( slotHideButton(bool) ) );
  connect( subjectList, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
                        SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
  connect( subjectList, SIGNAL( mouseButtonClicked( int, QListViewItem*, const QPoint&, int ) ),
                        SLOT( loadThread( int, QListViewItem* ) ) );
}

KitaSubjectView::~KitaSubjectView()
{
}

void KitaSubjectView::reloadSubject()
{
}

void KitaSubjectView::loadThread( int, QListViewItem* item )
{
  if( ! item ) return;

  QString datName = item->text(Row_DatName);
  KURL datURL = m_board.url();
  datURL.addPath("/dat/" + datName);

  Kita::Thread* thread = new Kita::Thread( m_board, datURL,
                                           item->text( Row_Subject ),
                                           item->text( Row_Read ).toInt() );

  emit signalShowThread( *thread );
}

void KitaSubjectView::loadBoard(const Kita::Board& board)
{
  // reset member variables.
  {
    m_hitList.clear();
    m_nextHitIndex = 0;
    m_prevquery = "";
  }

  m_board = board;

  KitaCacheInfo* cache = KitaCacheInfo::getInstance();
  QPtrList<Kita::Thread> threadList = m_board.getThreadList();

  // clear list
  subjectList->clear();

  for(unsigned i=0; i<threadList.count(); i++) {
    Kita::Thread* thread = threadList.at(i);

    int readNum = cache->readNum( thread->datURL() );
    int resNum = thread->resNum();

    KListViewItem* item = new KListViewItem( subjectList );
    item->setText( Row_ID, QString("%1").arg(i+1, 4) );
    item->setText( Row_Subject, thread->name() );
    item->setText( Row_ResNum, QString("%1").arg( resNum, 4 ) );
    item->setText( Row_Read, (readNum > 0) ? QString("%1").arg( readNum, 4 ) : QString("") );
    item->setText( Row_Unread, (readNum > 0 && resNum != readNum) ? QString("%1").arg( resNum - readNum, 4 ) : QString("") );
    item->setText( Row_DatName, thread->datURL().fileName() );
    item->setText( Row_DatURL, thread->datURL().url() );

    if ( readNum > 0 && resNum != readNum ) {
      // ̤ɤ
      item->setPixmap( Row_Mark, SmallIcon( "unread" ) );
    } else if ( readNum > 0 ) {
      // 
      item->setPixmap( Row_Mark, SmallIcon( "read" ) );
    }
  }

  if( HideButton->isOn() ) {
    HideButton->toggle();
  }
  emit loadBoardCompleted(m_board.url());
}

void KitaSubjectView::slotSearchButton()
{
  insertSearchCombo();
  QStringList list = parseSearchQuery( SearchCombo->currentText() );
  searchNext( list );
}

void KitaSubjectView::insertSearchCombo()
{
  int count;
  bool found = false;

  for( count = 0; count < SearchCombo->count(); ++count ) {
    if ( SearchCombo->text( count ) ==  SearchCombo->currentText() ) {
      found = true;
      break;
    }
  }
  if ( ! found ) SearchCombo->insertItem( SearchCombo->currentText() );
}

QStringList KitaSubjectView::parseSearchQuery(const QString &input)
{
  QStringList tmp = QStringList::split( ' ', input );
  QStringList ret_list;
  QRegExp truncSpace("\\s*$");
  QStringList::iterator it = tmp.begin();
  for( ; it != tmp.end(); ++it )
    ret_list += (*it).replace( truncSpace, "" );
  return ret_list;
}

void KitaSubjectView::searchNext(const QStringList &query)
{
  if ( query.isEmpty() ) return;

  if ( query != m_prevquery ) {
    searchAll( query );
    slotHideButton( HideButton->isOn() );
    m_nextHitIndex = 0; //A next jump-search target reset to '0'.
    return;
  }

  if(m_nextHitIndex >= m_hitList.size()) return;

  KListViewItem* item = m_hitList[m_nextHitIndex];
  subjectList->ensureItemVisible(item);
  subjectList->setSelected(item, true);
  m_nextHitIndex++;
  if(m_nextHitIndex >= m_hitList.size()) m_nextHitIndex = 0;
}

void KitaSubjectView::searchAll(const QStringList &query)
{
  m_hitList.clear();
  m_prevquery = query;

  QListViewItemIterator listIt( subjectList );
  while( listIt.current() != 0 ) {
    KListViewItem* item = static_cast<KListViewItem*>(listIt.current());
    item->setPixmap(Row_Icon, 0);

    QStringList::const_iterator queryIt = query.begin();
    for( ; queryIt != query.end(); ++queryIt ) {
      if( item->text( Row_Subject ).contains( *queryIt, false ) ) {
        item->setPixmap(Row_Icon, SmallIcon("find") );
        m_hitList.append(item);
        break;
      }
    }
    ++listIt;
  }
}

void KitaSubjectView::slotHideButton(bool on)
{
  if(m_hitList.empty()) return;

  QListViewItemIterator listIt( subjectList );
  while( listIt.current() != 0 ) {
    KListViewItem* item = static_cast<KListViewItem*>(listIt.current());
    if( on && ! item->pixmap(Row_Icon)) {
      item->setVisible(false);
    } else {
      item->setVisible(true);
    }
    ++listIt;
  }
}

void KitaSubjectView::setFont( const QFont& font )
{
  subjectList->setFont(font);
}

void KitaSubjectView::updateThread(const Kita::Thread& updated_thread)
{
  for(QListViewItem* item = subjectList->firstChild(); item; item = item->nextSibling() ) {
    if( item->text(Row_DatURL) == updated_thread.datURL().url() ) {
      int resNum = item->text( Row_ResNum ).toInt();
      int readNum = updated_thread.resNum();
      item->setText( Row_Read, QString("%1").arg( readNum, 4 ) );
      item->setText( Row_Unread, readNum > 0 && resNum != readNum ? QString("%1").arg( resNum - readNum, 4 ) : QString(""));
    }
  }
}

void KitaSubjectView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
  if( item == 0 ) {
    return;
  }

  KPopupMenu popup( 0 );
  popup.insertItem( i18n("Open with Web Browser"), 0 );
  popup.insertItem( i18n("Open with new tab"), 1 );
  popup.insertItem( i18n("Copy title and URL"), 2 );

  QString datName = item->text(Row_DatName);
  KURL datURL = m_board.url();
  datURL.addPath("/dat/" + datName);

  Kita::Thread thread( m_board, datURL );
  thread.setName( item->text( Row_Subject ) );
  thread.setResNum( item->text( Row_Read ).toInt() );

  QClipboard* clipboard = QApplication::clipboard();

  switch( popup.exec( point ) ) {
  case 0:
    KRun::runURL( thread.url(), "text/html" );
    break;
  case 1:
    emit signalShowThreadWithNewTab( thread );
    break;
  case 2:
    clipboard->setText( thread.name() + "\n" + thread.url().url() );
    break;
  default:
    break;
  }
}
