/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@users.sourceforge.jp                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <kio/netaccess.h>

#include <qfile.h>
#include <qtextcodec.h>
#include <qstringlist.h>
#include <qregexp.h>
#include <klistview.h>
#include <kpopupmenu.h>
#include <klocale.h>
#include <krun.h>
#include <kdebug.h>
#include <qvaluelist.h>

#include "libkita/qcp932codec.h"
#include "part/kita2ch.h"
#include "kitaboardview.h"
#include "kita.h"
#include "libkita/board.h"
#include "libkita/category.h"


KitaBoardView::KitaBoardView( QWidget *parent, const char *name )
    : KitaBoardViewBase(parent, name)
{
  m_boardList->setSorting( -1 );
  m_boardList->addColumn( "name" );
  connect( m_boardList, SIGNAL( executed( QListViewItem* ) ), SLOT( loadBoard( QListViewItem* ) ) );
  connect( m_boardList, SIGNAL( returnPressed( QListViewItem* ) ), SLOT( loadBoard( QListViewItem* ) ) );
  connect( m_boardList, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
                        SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
}

KitaBoardView::~KitaBoardView()
{}

QString KitaBoardView::getCategory( const QString& line ) const
{
  QRegExp regexp( "<BR><BR><B>(.*)</B><BR>", false );
  if(regexp.search( line ) != -1) {
    return regexp.cap(1);
  } else {
    return QString::null;
  }
}

Kita::Board* KitaBoardView::getBoard( const QString& line ) const
{
  QRegExp regexp( "<A HREF=(.*)>(.*)</A>", false );
  if( regexp.search(line) != -1 ) {
    QString board_url = regexp.cap(1);
    QString board_title = regexp.cap(2);
    Kita::Board::setName( board_url, board_title );
    return Kita::Board::getByURL( board_url );
  } else {
    return Kita::Board::getByURL( QString::null );
  }
}

bool KitaBoardView::isBoardUrl(const KURL& url) const
{
  QRegExp url_2ch("http://.*\\.2ch\\.net/.*");
  QRegExp url_bbspink("http://.*\\.bbspink\\.com/.*");
  QRegExp url_www_2ch("http://www\\.2ch\\.net/.*");

  if ( url.isEmpty() ) return false;

  QString str = url.url();

  if ( url_2ch.search(str) == -1 && url_bbspink.search(str) == -1 ) return false;
  if ( url_www_2ch.search(str) != -1 ) return false;

  return true;
}

void KitaBoardView::loadBoardList()
{
  QString tmpFile;
  QString url = KitaConfig::boardListURL();
  if( ! KIO::NetAccess::download(url, tmpFile) )
  {
    return;
  }

  QFile file(tmpFile);
  if( ! file.open(IO_ReadOnly) )
  {
    return;
  }

  QTextStream stream(&file);
  QCp932Codec cp932Codec;
  stream.setCodec(&cp932Codec);
  QPtrList<Kita::Category> categoryList = getCategoryList(stream.read());

  // clear list
  m_boardList->clear();

  KListViewItem* previous = 0;
  for( Kita::Category* category = categoryList.first(); category; category = categoryList.next() )
  {
    QPtrList<Kita::Board> board_list = category->getBoardList();

    if( board_list.isEmpty() ) {
      continue;
    }

    KListViewItem* categoryItem = new KListViewItem( m_boardList, previous, category->name() );

    KListViewItem* previousBoard = 0;
    QPtrListIterator<Kita::Board> it( board_list );
    for(; *it; ++it)
    {
      Kita::Board* board = *it;
      previousBoard = new KListViewItem( categoryItem, previousBoard, board->name(), board->url() );
    }
    previous = categoryItem;
  }
}

QPtrList<Kita::Category> KitaBoardView::getCategoryList( const QString& html ) const
{
  QPtrList<Kita::Category> result;

  QStringList lines = QStringList::split("\n", html);
  QStringList::iterator it;

  Kita::Category* current_category = 0;
  for(it = lines.begin(); it != lines.end(); ++it) {
    QString category_name = getCategory(*it);
    if( category_name != QString::null ) {
      current_category = new Kita::Category( category_name );
      result.append( current_category );
    } else {
      Kita::Board* board = getBoard(*it);

      if( isBoardUrl( board->url() ) ) {
        if( current_category ) {
          current_category->append( board );
        }
      }
    }
  }
  return result;
}

void KitaBoardView::loadBoard( QListViewItem* item )
{
  QString boardName = item->text(0);
  QString url = item->text(1);

  if( url.isEmpty() ) {
    return;
  }

  Kita::Board* board = Kita::Board::getByURL( url );
  emit clicked( board );
}

void KitaBoardView::setFont( const QFont& font )
{
  m_boardList->setFont(font);
}

void KitaBoardView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
  if( item == 0 ) {
    return;
  }

  KPopupMenu popup( 0 );
  popup.insertItem( i18n("Open with Web Browser"), 0 );
  popup.insertItem( i18n("Open with New Tab"), 1 );

  KURL url = item->text(1);

  switch( popup.exec( point ) ) {
  case 0:
    KRun::runURL( url, "text/html" );
    break;
  case 1:
    emit openBoardRequested( Kita::Board::getByURL( url.url() ), true );
    break;
  default:
    break;
  }
}
