/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitasubjectview.h"

// include files for Qt
#include <qtoolbutton.h>
#include <qtextbrowser.h>
#include <qtextcodec.h>
#include <qregexp.h>
#include <qfile.h>
#include <qclipboard.h>
#include <qapplication.h>
#include <qdatetime.h>

// kdelibs/kio
#include <kio/slaveconfig.h>
#include <kio/netaccess.h>

// include files for KDE
#include <kwin.h>
#include <kfilterdev.h>
#include <klistview.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kdebug.h>

#include "libkita/threadinfo.h"
#include "libkita/thread.h"
#include "libkita/kita_misc.h"
#include "part/kita2ch.h"
#include "kitaconfig.h"

KitaSubjectView::KitaSubjectView( QWidget* parent, const char* name )
        : Kita::ThreadListView( parent, name )
        , m_board( 0 )
{
    connect( subjectList, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
             SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
    connect( subjectList, SIGNAL( mouseButtonClicked( int, QListViewItem*, const QPoint&, int ) ),
             SLOT( slotMouseButtonClicked( int, QListViewItem* ) ) );
    connect( subjectList, SIGNAL( returnPressed( QListViewItem* ) ),
             SLOT( loadThread( QListViewItem* ) ) );
    connect( ReloadButton, SIGNAL( clicked() ),
             SLOT( slotReloadButton() ) );
}

KitaSubjectView::~KitaSubjectView()
{}

void KitaSubjectView::reloadSubject()
{}

const QString KitaSubjectView::boardURL()
{
    if ( m_board ) {
        return m_board->url();
    }
    return "";
}

const QString KitaSubjectView::boardName()
{
    if ( m_board ) {
        return m_board->name();
    }
    return "";
}

void KitaSubjectView::loadThread( QListViewItem* item )
{
    if ( ! item ) return ;

    KURL datURL = item->text( Col_DatURL );

    Kita::Thread* thread = Kita::Thread::getByURL( datURL.url() );

    if ( KitaConfig::alwaysUseTab() ) {
        emit showThreadRequested( thread, true );
    } else {
        emit showThreadRequested( thread, false );
    }
}

void KitaSubjectView::loadBoard( const Kita::Board* board )
{
    setActiveWindow();
    topLevelWidget() ->raise();
    QDateTime currentdat = QDateTime::currentDateTime();

    // reset member variables.
    // FIXME: FavoriteListView::update()˥ԡ
    {
        m_hitList.clear();
        m_nextHitIndex = 0;
        m_prevquery = "";
    }

    m_board = board;

    KitaThreadInfo* cache = KitaThreadInfo::getInstance();
    QPtrList<Kita::Thread> threadList = m_board->getThreadList();

    // clear list
    subjectList->clear();

    for ( unsigned i = 0; i < threadList.count(); i++ ) {
        Kita::Thread* thread = threadList.at( i );

        QDateTime since;
        since.setTime_t( KURL( thread->datURL() ).fileName().section( '.', 0, 0 ).toInt() );

        int readNum = cache->readNum( thread->datURL() );
        int resNum = thread->resNum();

        Kita::ThreadListViewItem* item = new Kita::ThreadListViewItem( subjectList );
        item->setText( Col_ID, QString( "%1" ).arg( i + 1, 4 ) );
        item->setText( Col_Subject, Kita::unescape( thread->name() ) );
        item->setText( Col_ResNum, QString( "%1" ).arg( resNum, 4 ) );
        item->setText( Col_Read, ( readNum > 0 ) ? QString( "%1" ).arg( readNum, 4 ) : QString( "" ) );
        item->setText( Col_Unread, ( readNum > 0 && resNum != readNum ) ? QString( "%1" ).arg( resNum - readNum, 4 ) : QString( "" ) );
        item->setText( Col_Board, thread->boardName() );
        item->setText( Col_Since, since.toString( "yy/MM/dd hh:mm" ) );
        item->setText( Col_DatURL, thread->datURL() );

        // FIXME: updateThread˥ԡƤ
        // TODO: 'N'Υ
        /*
        ͥ
        0: ɤ1000ĶƤ
        1: ̤1ʾФäƤ
        2: ̤1Τ
        3: ɤ1000ʲΤ
        4: ̤ɤ
        */
        if ( readNum > 0 && resNum != readNum ) {
            // ̤ɤ
            item->setPixmap( Col_Mark, SmallIcon( "unread" ) );
            item->setText( Col_MarkOrder, "4" );
        } else if ( readNum > 0 ) {
            // 
            item->setPixmap( Col_Mark, SmallIcon( "read" ) );
            if( readNum > 1000 ) {
                item->setText( Col_MarkOrder, "0" );
            } else {
                item->setText( Col_MarkOrder, "3" );
            }
        } else if ( since.secsTo( currentdat ) < 3600*24 ){
            item->setPixmap( Col_Mark, SmallIcon( "newthread" ) );
            item->setText( Col_MarkOrder, "2" );
        } else {
            item->setText( Col_MarkOrder, "1" );
        }
    }

    if ( HideButton->isOn() ) {
        HideButton->toggle();
    }
    emit loadBoardCompleted( m_board->url() );

    subjectList->setSorting( Col_ID );

    subjectList->setFocus();
}

void KitaSubjectView::setFont( const QFont& font )
{
    subjectList->setFont( font );
}

void KitaSubjectView::updateThread( const Kita::Thread* updated_thread )
{
    for ( QListViewItem * item = subjectList->firstChild(); item; item = item->nextSibling() ) {
        if ( item->text( Col_DatURL ) == updated_thread->datURL() ) {
            int resNum = updated_thread->resNum();
            int readNum = KitaThreadInfo::readNum( updated_thread->datURL() );
            item->setText( Col_ResNum, QString( "%1" ).arg( resNum, 4 ) );
            item->setText( Col_Read, QString( "%1" ).arg( readNum, 4 ) );
            item->setText( Col_Unread, readNum > 0 && resNum != readNum ? QString( "%1" ).arg( resNum - readNum, 4 ) : QString( "" ) );

            if ( readNum > 0 && resNum != readNum ) {
                // ̤ɤ
                item->setPixmap( Col_Mark, SmallIcon( "unread" ) );
                item->setText( Col_MarkOrder, "2" );
            } else if ( readNum > 0 ) {
                // 
                item->setPixmap( Col_Mark, SmallIcon( "read" ) );
                item->setText( Col_MarkOrder, "1" );
            } else {
                item->setText( Col_MarkOrder, "0" );
            }
        }
    }
}

void KitaSubjectView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
    if ( item == 0 ) {
        return ;
    }

    KPopupMenu popup( 0 );
    popup.insertItem( i18n( "Open with Web Browser" ), 0 );
    popup.insertItem( i18n( "Open with new tab" ), 1 );
    popup.insertItem( i18n( "Copy title and URL" ), 2 );

    Kita::Thread* thread = Kita::Thread::getByURL( item->text( Col_DatURL ) );

    QClipboard* clipboard = QApplication::clipboard();

    switch ( popup.exec( point ) ) {
    case 0:
        KRun::runURL( thread->url(), "text/html" );
        break;
    case 1:
        emit showThreadRequested( thread, true );
        break;
    case 2:
        clipboard->setText( thread->name() + "\n" + thread->url() );
        break;
    default:
        break;
    }
}

void KitaSubjectView::slotReloadButton()
{
    if ( m_board ) {
        loadBoard( m_board );
    }
}

void KitaSubjectView::slotMouseButtonClicked( int button, QListViewItem* item )
{
    if ( ! item ) return ;

    KURL datURL = item->text( Col_DatURL );

    Kita::Thread* thread = Kita::Thread::getByURL( datURL.url() );

    switch ( button ) {
    case MidButton:
        emit showThreadRequested( thread, true );
        break;
    case LeftButton:
        if ( KitaConfig::alwaysUseTab() ) {
            emit showThreadRequested( thread, true );
        } else {
            emit showThreadRequested( thread, false );
        }
        break;
    }
}
