/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "comment.h"

#include <qstringlist.h>
#include <qregexp.h>
#include <klocale.h>

using namespace Kita;

CommentList::CommentList( const QString& str )
{
    QStringList lines = QStringList::split( "\n", str );
    for ( QStringList::iterator it = lines.begin(); it != lines.end(); ++it ) {
        QString line = ( *it );
        if ( line.isEmpty() ) {
            continue;
        }
        Kita::Comment comment( line );
        m_list << comment;
    }
}

CommentList::~CommentList()
{}

Comment::Comment() : m_isValid( false )
{}

Comment::Comment( const QString& line ) : m_isValid( false )
{
    QStringList list = QStringList::split( "<>", line, true );

    if ( list.size() == 5 ) {
        m_isValid = true;
    } else if ( list.size() < 5 ) {
        return ;
    }
    m_name = list[ 0 ];
    m_address = list[ 1 ];
    parseDateId( list[ 2 ] );
    m_body = list[ 3 ];
    m_subject = list[ 4 ];
}

void Comment::parseDateId( const QString& str )
{
    QRegExp regexp( "(\\d\\d)/(\\d\\d)/(\\d\\d) (\\d\\d):(\\d\\d)( ID:(.*))?" );

    if ( regexp.search( str ) == -1 ) {
        m_dateId = str;
        return ;
    }

    int year = regexp.cap( 1 ).toInt();
    if ( year >= 70 ) {
        year += 1900;
    } else {
        year += 2000;
    }

    QDateTime dateTime( QDate( year, regexp.cap( 2 ).toInt(), regexp.cap( 3 ).toInt() ),
                        QTime( regexp.cap( 4 ).toInt(), regexp.cap( 5 ).toInt() ) );

    m_dateTime = dateTime;
    m_id = regexp.cap( 7 );
}

QString Comment::linkedAddress( bool showMailAddress ) const
{
    if ( m_address != "" && showMailAddress ) {
        return QString( "<b><a href=\"mailto:" ) + m_address + "\" title=\"" + m_address + "\">" + m_name + "</a></b> <span style='color: blue'>[" + m_address + "]</span>";
    } else if ( m_address != "" ) {
        return QString( "<b><a href=\"mailto:" ) + m_address + "\" title=\"" + m_address + "\">" + m_name + "</a></b>";
    } else {
        return QString( "<b>%1</b>" ).arg( m_name );
    }
}

QString Comment::linkedBody() const
{
    QString ret;
    QString str = m_body;
    // see RFC 1738.
    QRegExp url_rx( "(h?ttp://([-.0-9a-zA-Z]+(:[0-9]+)?(/[;:@&=$-_.+!*'(),%~/?#0-9a-zA-Z]*)?))" );

    int i;
    while ( ( i = str.find( url_rx ) ) != -1 ) {
        if ( i == 0 ) {
            // starts with URL.
            ret += QString( "<a href=\"http://" ) + url_rx.cap( 2 ) + "\">" + url_rx.cap( 1 ) + "</a>";
            str.remove( 0, url_rx.cap( 1 ).length() );
        } else {
            ret += str.left( i );
            str.remove( 0, i );
        }
    }
    ret += str;

    if ( m_isValid == false ) {
        ret = QString::fromLocal8Bit( "<font color='red'>ϲƤޤ</font><br />" ) + ret;
    }
    return ret;
}

QString Comment::getDateId() const
{
    if ( ! m_dateId.isEmpty() ) {
        return m_dateId;
    } else {
        QString dateTime = m_dateTime.toString( i18n( "yyyy/MM/dd hh:mm" ) );
        if ( ! m_id.isEmpty() ) {
            dateTime.append( QString( " ID:%1" ).arg( m_id ) );
        }
        return dateTime;
    }
}

QString Comment::toHtml( int num, bool showMailAddress ) const
{
    QString result;
    result += QString( "<dl><dt><span id=\"%1\"/>%2 " ).arg( num ).arg( num );
    result += linkedAddress( showMailAddress );
    result += " " + getDateId() + "</dt><dd>" + linkedBody() + "<br/><br/></dd></dl>";

    return result;
}

const QString& Comment::getName() const
{
    return m_name;
}

const QString& Comment::getAddress() const
{
    return m_address;
}

const QString& Comment::getBody() const
{
    return m_body;
}

const QString& Comment::getSubject() const
{
    return m_subject;
}
