/**************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto , (c)2004 by 421                 *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
**************************************************************************/

/* This class keeps information of the DOM tree, creates the node,
   and do parsing, rendering, etc. */

#include "kitaconfig.h"
#include "kitadomtree.h"
#include "kitathreadview.h"

/* UTF-16 */
#define KITADOM_BRACKET  0xFF1E /* > */
#define KITADOM_0 0xFF10 /* 0 */
#define KITADOM_9 0xFF19 /* 9 */
#define KITADOM_EQ 0xFF1D /* = */
#define KITADOM_COMMA 0xFF0C /* , */

/* UTF-8 */
#include "kita-utf8.h"


enum{
    DOM_RES_NOTVALID,
    DOM_RES_VALID,
    DOM_RES_ABONED
};


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/


/*-------------------------------------------------------------------*/
/* public functions                                                  */
/*-------------------------------------------------------------------*/

KitaDomTree::KitaDomTree( KitaHTMLPart* threadPart )
{
    QTextCodec * codec = QTextCodec::codecForName( "utf8" );
    m_colonstr = codec->toUnicode( KITAUTF8_COLON );
    m_clnamestr = codec->toUnicode( KITAUTF8_NAME );

    m_semap = new QSemaphore( 1 );
    m_krt = new KitaRenderThread( this );

    m_threadPart = threadPart;
    if ( m_threadPart != NULL ) {
        m_hdoc = m_threadPart->htmlDocument(); /* == NULL */
    }

    resetAllVar();
}


KitaDomTree::~KitaDomTree()
{

    if ( m_krt->running() ) {
        m_krt->terminate();
        m_krt->wait();
    }
    delete m_semap;
    delete m_krt;
}



/*--------------------------*/
/* setup & create HTML doc. */
/* call this before parsing */
/*--------------------------*/
void KitaDomTree::parseResInit( Kita::Thread* thread,
                                const QString& style,
                                int templateNum )
{
    m_thread = thread;
    if ( m_thread ) {
        m_url = m_thread->datURL();
    } else {
        m_url = "";
    }

    m_showAA = FALSE;

    m_showaddr = KitaConfig::showMailAddress();
    m_imgsize = 50; /* initial width of inline image */

    m_templateNum = templateNum;

    m_namelist = KitaConfig::aboneNameList();
    m_idlist = KitaConfig::aboneIDList();
    m_wordlist = KitaConfig::aboneWordList();
    m_resAbone = FALSE;

    /* create HTML Document */
    createHTMLDocument( style );
    m_hdoc = m_threadPart->htmlDocument();

    /* create the nodes of footer, header, etc. */
    createKokoyon();
    m_footer = createFooter( "footer" );
    m_header = createFooter( "header" );
    createTugi100();
    createMae100();

    resetAllVar();
}



/*------------------------------------*/
/* copy thread data to buffer         */
/*------------------------------------*/
void KitaDomTree::setDat( const QString& line, int num )
{
    if ( num < 1 || num >= KITADOM_MAXRES ) return ;

    m_linedat[ num ] = line;
    m_linedatset[ num ] = TRUE;

    /* get subject */
    if ( num == 1 ) {
        QStringList list = QStringList::split( "<>", m_linedat[ num ], true );
        if ( list[ 4 ] != NULL ) {
            m_subject = list[ 4 ];
        }
    }

    /* check if dat is broken */
    if ( line.contains( "<>" ) != 4 ) {
        m_broken = TRUE;
    }

    if ( num > m_maxNum ) {
        m_maxNum = num;
    }
}



/*--------------------------------------------*/
/* DOM parsing function                       */
/*                                            */
/* to show a res, call appendRes later.       */
/*--------------------------------------------*/
void KitaDomTree::parseRes(
    int num,      /* the number of buffer */
    int resnum /* the number of response , usually, num = resnum */
)
{
    if ( m_hdoc == NULL ) return ;
    if ( num < 1 || num > m_maxNum ) return ;
    if ( !m_linedatset[ num ] ) return ;
    if ( m_resparsed[ num ] ) return ; /* already parsed */

    ( *m_semap ) ++;

    DOM::Element rootnode, dlnode;
    DOM::Element bodynode, titlenode;
    QString str_name = "";
    QString str_address = "", str_body = "";
    QString str_id = "", str_dateId = "";
    QDateTime dt_dateTime;

    /* split 2ch dat */
    QStringList list = QStringList::split( "<>", m_linedat[ num ], true );

    if ( list.size() == 5 ) { /* valid data */

        str_name = list[ 0 ];
        str_address = list[ 1 ];
        parseDateId( list[ 2 ], str_id, str_dateId, dt_dateTime );
        str_body = list[ 3 ];

        /* create both title & body node */
        titlenode = createTitleNode( resnum,
                                     m_showaddr,
                                     str_name,
                                     str_address,
                                     str_id,
                                     str_dateId,
                                     dt_dateTime );

        bodynode = createBodyTextNode( str_body );
    } else { /* data is broken */

        titlenode = m_hdoc.createElement( "DT" );
        titlenode.appendChild( m_hdoc.createTextNode(
                                   QString().setNum( resnum ) + " " + m_clnamestr ) );

        bodynode = m_hdoc.createElement( "DD" );
        QString str = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_KOWARE );
        appendColoredTextNode( bodynode, str, "color: red" );
        bodynode.appendChild( m_hdoc.createElement( "BR" ) );
        bodynode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    /*-------------*/
    /* create node */
    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", "res" );
        rootnode.setAttribute( "id", QString().setNum( resnum ) );
        rootnode.setAttribute( "kita_rname", str_name );
        rootnode.setAttribute( "kita_rid", str_id );

        dlnode = rootnode.appendChild( m_hdoc.createElement( "DL" ) );
        {
            dlnode.appendChild( titlenode );
            dlnode.appendChild( bodynode );
        }
    }

    m_res[ num ] = rootnode;
    m_resparsed[ num ] = TRUE;

    ( *m_semap ) --;
}



/*---------------------------------*/
/* append  a response to document  */
/*---------------------------------*/
bool KitaDomTree::appendRes( int num, bool binsert )
{
    int res = checkIsNumValid( num );
    if ( res == DOM_RES_NOTVALID ) return FALSE;

    if ( res == DOM_RES_VALID ) {

        if ( !binsert ) {
            m_hdoc.body().appendChild( m_res[ num ] );
        } else {
            DOM::Node first = m_hdoc.body().firstChild();
            if ( first == NULL ) {
                m_hdoc.body().appendChild( m_res[ num ] );
            } else {/* !! very slow !! */
                m_hdoc.body().insertBefore( m_res[ num ], first );
            }
        }

        m_resshown[ num ] = TRUE;
    }

    if ( num < m_topNum ) m_topNum = num;
    if ( num > m_bottomNum ) m_bottomNum = num;
    if ( m_topNum <= m_templateNum ) m_topNum = 1;

    return TRUE;
}


/* remove shown res */
bool KitaDomTree::removeRes( int num )
{
    if ( checkIsNumValid( num ) != DOM_RES_VALID ) return FALSE;

    if ( m_resshown[ num ] ) m_hdoc.body().removeChild( m_res[ num ] );
    m_resshown[ num ] = FALSE;

    return TRUE;
}

/*---------------------------------*/
/* parse all res in the backgound  */
/*---------------------------------*/
void KitaDomTree::parseAllRes()
{
    if ( m_hdoc == NULL ) return ;

    if ( m_krt->running() ) m_krt->wait();
    m_krt->setMaxNum( m_maxNum );
    m_krt->start();
}


void KitaDomTree::TerminateParseThread()
{
    if ( m_krt->running() ) {
        m_krt->terminate();
        m_krt->wait();
    }
}

void KitaDomTree::StopParseThread()
{
    if ( m_krt->running() ) {
        m_krt->stopRender();
        m_krt->wait();
    }
}


/*-----------------------------------------------*/
/* public utilities                              */

/* get string data of res */
QString KitaDomTree::getDat( int num )
{
    if ( num < 1 || num > m_maxNum ) {
        return QString::null;
    }

    if ( m_linedatset[ num ] ) {
        return m_linedat[ num ];
    } else {
        return QString::null;
    }
}


/*------------------------------------*/
/* get HTML documents for popup, etc. */
/*------------------------------------*/
QString KitaDomTree::getHtml( int startnum, int endnum )
{
    QString str = QString::null;

    for ( int i = startnum; i <= endnum; i++ ) {
        if ( checkIsNumValid( i ) == DOM_RES_VALID ) {
            str += getHtmlCore( i );
        }
    }
    return str;
}

/* return html documents & number of resposes which have ID=strid */
QString KitaDomTree::getHtmlByID( QString strid, int &num )
{

    QString outstr = QString::null;
    num = 0;

    for ( int i = 1; i <= m_maxNum; i++ ) {
        if ( checkIsNumValid( i ) == DOM_RES_VALID ) {
            if ( CheckID( strid, i ) ) {
                outstr += getHtmlCore( i );
                num++;
            }
        }
    }

    return outstr;
}


/* return only number of resposes which have ID=strid */
int KitaDomTree::getNumByID( QString strid )
{
    int num = 0;

    for ( int i = 1; i <= m_maxNum; i++ ) {
        if ( CheckID( strid, i ) ) {
            if ( CheckID( strid, i ) ) num++;
        }
    }

    return num;
}



/*--------------------------------------------------*/
/* get strings of response for quotaion, copy, etc. */
/*--------------------------------------------------*/
QString KitaDomTree::getResStr( int num,
                                QString qtstr /* quotation mark such as "> " */
                              )
{
    QString str, strtmp;
    QString str_id;

    if ( checkIsNumValid( num ) != DOM_RES_VALID ) return QString::null;

    DOM::Node node = m_res[ num ];
    DOM::Node titlenode = node.firstChild().firstChild();
    DOM::Node bodynode = node.firstChild().lastChild();

    strtmp = qtstr + extractText( bodynode, "dd", "\n" + qtstr );

    /* remove >\n>\n */
    strtmp = strtmp.left( strtmp.length() - ( qtstr.length() + 1 ) * 2 );

    str = qtstr + extractText( titlenode, "dt", QString::null ) + "\n";
    str += strtmp + "\n";

    return str;
}



/* check if ID == strid */
bool KitaDomTree::CheckID( QString strid, int num )
{
    if ( checkIsNumValid( num ) != DOM_RES_VALID ) return FALSE;

    QString str_rid
    = static_cast<DOM::Element>( m_res[ num ] ).getAttribute( "kita_rid" ).string();

    if ( strid == str_rid ) return TRUE;

    return FALSE;
}


/* check if this res includes some keywords */
bool KitaDomTree::CheckWord( QStringList stlist, int num,
                             bool checkOR /* AND or OR search */
                           )
{
    if ( stlist.size() == 0 ) return FALSE;
    if ( checkIsNumValid( num ) != DOM_RES_VALID ) return FALSE;

    DOM::Node node = m_res[ num ];
    QString str_text = extractText( node, "div", QString::null );

    for ( QStringList::iterator it = stlist.begin(); it != stlist.end(); ++it ) {

        if ( checkOR ) { /* OR */
            if ( str_text.find( ( *it ), 0, FALSE ) != -1 ) {
                return TRUE;
            }
        } else { /* AND */
            if ( str_text.find( ( *it ), 0, FALSE ) == -1 ) return FALSE;
        }
    }

    if ( checkOR ) return FALSE;

    return TRUE;
}


/* check if this res has an anchor for target */
bool KitaDomTree::CheckRes( int target, int num )
{

    if ( checkIsNumValid( num ) != DOM_RES_VALID ) return FALSE;
    if ( checkIsNumValid( target ) != DOM_RES_VALID ) return FALSE;

    DOM::Node node = m_res[ num ];
    DOM::Node bodytext = node.firstChild().lastChild();

    DOM::NodeList nodelist = bodytext.childNodes();
    for ( unsigned int i = 0; i < nodelist.length(); ++i ) {

        DOM::HTMLElement anode = nodelist.item( i );
        if ( anode.nodeName().string() == "a" ) {
            int refNum, refNum2;
            QString url = anode.getAttribute( "href" ).string();
            KitaThreadView::getRefNumber( url, m_thread, refNum, refNum2 );
            if ( target >= refNum && target <= refNum2 ) return TRUE;
        }
    }

    return FALSE;
}

/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* public functions for header Node, footer Node, kokomadeyonda Node, etc... */


/*----------------------------------------*/
/* append templates at the head of thread */
/*----------------------------------------*/
void KitaDomTree::appendTemplate()
{
    if ( m_hdoc == NULL ) return ;

    for ( int i = 1; i <= QMIN( m_templateNum, m_maxNum ); i++ ) {
        if ( checkIsNumValid( i ) != DOM_RES_VALID ) continue;
        m_hdoc.body().appendChild( m_res[ i ] );
        m_resshown[ i ] = TRUE;
    }
}


/*-------------------------*/
/* append footer & header  */
/*-------------------------*/
void KitaDomTree::appendFooter( int resnum )
{
    if ( m_hdoc == NULL ) return ;

    DOM::Element tmpelm, tmpelm2, tmpelm3, tmpelm4;
    DOM::Element targetelm = m_footer;

    for ( int i = 0; i < 2; i++, targetelm = m_header ) {

        /* < a href="#tosaigo"> & <br> */
        tmpelm2 = targetelm.removeChild( targetelm.lastChild() );
        if ( i == 1 ) {
            tmpelm3 = targetelm.removeChild( targetelm.lastChild() );
            tmpelm4 = targetelm.removeChild( targetelm.lastChild() );
        }

        DOM::Node node = targetelm.firstChild(); /* <a href="#" */
        node = node.nextSibling(); /* " " */
        node = node.nextSibling(); /* <a href="#" */
        node = node.nextSibling(); /* " " */
        node = node.nextSibling();

        for ( int target = 1; target < resnum ; target += 100 ) {
            QString str = QString( "#%1" ).arg( target );
            if ( node == NULL ) {
                tmpelm = targetelm.appendChild( m_hdoc.createElement( "A" ) );
                tmpelm.setAttribute( "href", str );
                tmpelm.appendChild( m_hdoc.createTextNode( QString().setNum( target ) + "-" ) );

                node = targetelm.appendChild( m_hdoc.createTextNode( " " ) );
                node = node.nextSibling();
            } else {
                node = node.nextSibling();
                node = node.nextSibling();
            }
        }

        if ( i == 1 ) {
            targetelm.appendChild( tmpelm4 );
            targetelm.appendChild( tmpelm3 );
        }
        targetelm.appendChild( tmpelm2 );
    }

    m_hdoc.body().appendChild( m_footer );
    m_hdoc.body().insertBefore(
        m_header,
        m_hdoc.body().firstChild() );
}



/*------------------------------------------*/
/* append kokomadeyonda & tugi 100 & mae 100*/
/*------------------------------------------*/
void KitaDomTree::appendKokoyon()
{
    if ( m_hdoc == NULL ) return ;

    /* call setKokoyonNum to set m_kokoyonNum */
    if ( m_kokoyonNum <= 0 ) return ;

    int i = m_kokoyonNum + 1;
    while ( !m_resshown[ i ] && i <= m_maxNum ) i++;

    if ( i <= m_maxNum ) {
        m_hdoc.body().insertBefore( m_kokoyon, m_res[ i ] );
        return ;
    }

    m_hdoc.body().appendChild( m_kokoyon );
}


void KitaDomTree::appendTugi100()
{
    if ( m_hdoc == NULL ) return ;

    if ( m_bottomNum == m_maxNum ) {
        removeTugi100();
        return ;
    }

    m_hdoc.body().appendChild( m_tugi100 );
    m_appendtugi100 = TRUE;
}

void KitaDomTree::removeTugi100()
{
    if ( m_hdoc == NULL ) return ;

    if ( m_appendtugi100 ) m_hdoc.body().removeChild( m_tugi100 );
    m_appendtugi100 = FALSE;
}


void KitaDomTree::appendMae100()
{
    if ( m_hdoc == NULL ) return ;

    if ( m_topNum <= m_templateNum ) {
        removeMae100();
        return ;
    }

    int i = m_topNum;
    while ( !m_resshown[ i ] && i <= m_maxNum ) i++;

    if ( i <= m_maxNum ) {
        m_hdoc.body().insertBefore( m_mae100, m_res[ i ] );
    } else {
        m_hdoc.body().appendChild( m_mae100 );
    }
    m_appendmae100 = TRUE;
}

void KitaDomTree::removeMae100()
{
    if ( m_hdoc == NULL ) return ;

    if ( m_appendmae100 ) m_hdoc.body().removeChild( m_mae100 );
    m_appendmae100 = FALSE;
}



/*------------------------------*/
/* create & append comment node */
/*------------------------------*/
void KitaDomTree::createCommentNode( QString comment, QString idstr,
                                     int beforeBR,
                                     int afterBR,
                                     bool binsert )
{
    if ( m_hdoc == NULL ) return ;

    DOM::Element rootnode;

    rootnode = m_hdoc.createElement( "DIV" );
    {
        for ( int i = 0; i < beforeBR; i++ )
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.setAttribute( "kita_type", "comment" );
        rootnode.setAttribute( "id", idstr );
        rootnode.appendChild( m_hdoc.createTextNode( comment ) );
        for ( int i = 0; i < afterBR; i++ )
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    if ( binsert ) {
        if ( m_hdoc.body().firstChild() != NULL ) {
            m_hdoc.body().insertBefore( rootnode, m_hdoc.body().firstChild() );
            return ;
        }
    }

    m_hdoc.body().appendChild( rootnode );
}


/* create belt node */
void KitaDomTree::createBeltNode( QString idstr )
{
    DOM::Element rootnode;

    QString style = "background-color: #CCCCCC; text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "style", style );
        rootnode.setAttribute( "id", idstr );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    m_hdoc.body().appendChild( rootnode );
}


/*--------------------------------------------------------*/

/*-----------------------------------*/
/*  private functions                */
/*-----------------------------------*/



/*----------------------*/
/* create HTML Document */
/*----------------------*/
void KitaDomTree::createHTMLDocument( const QString& style )
{
    QString text;

    /* I use JavaScript for resizing images */
    text = "<html><head>";
    text += "<script language=\"JavaScript\">";
    text += "function resizeImage(id,w){document.getElementById(id).width  = w;} ";
    text += "</script>";
    text += "</head>";
    text += "<body " + style + ">";
    text += "</body></html>";

    m_threadPart->setJScriptEnabled( true );
    m_threadPart->setJavaEnabled( false );

    /* Use dummy URL here, and it's protocol should be "file:".
       If protocol is "http:", local image files are not shown
       (for security reasons ?).
     */
    m_threadPart->begin( "file:/dummy.htm" );
    m_threadPart->write( text );
    m_threadPart->end();
}


/*-----------*/
/* reset all */
/*-----------*/
void KitaDomTree::resetAllVar()
{

    for ( int i = 0; i <= KITADOM_MAXRES; i++ ) {
        m_resparsed[ i ] = FALSE;
        m_resshown[ i ] = FALSE;
        m_linedatset[ i ] = FALSE;
        m_checkAbone[ i ] = FALSE;
        m_abone[ i ] = FALSE;
    }

    m_topNum = KITADOM_MAXRES + 2;
    m_bottomNum = -1;
    m_maxNum = 0;

    m_imgnum = 0;
    m_kokoyonNum = 0;
    m_appendtugi100 = FALSE;
    m_appendmae100 = FALSE;
    m_broken = FALSE;
}




/*----------------------------------------*/
/* create the node of name, date and id.  */
/*                                        */
/* called from  KitaDomTree::parseRes    */
/*----------------------------------------*/
DOM::Element KitaDomTree::createTitleNode(
    int num, bool showMailAddress,
    QString &str_name,
    QString &str_address,
    QString &str_id,
    QString &str_dateId,
    QDateTime &dt_dateTime )
{
    DOM::Element titlenode = m_hdoc.createElement( "DT" );
    {
        /* number */
        DOM::Element numberLink = titlenode.appendChild( m_hdoc.createElement( "A" ) );
        {
            /* set anchor id = number */
            QString str = QString( "%1" ).arg( num );
            numberLink.setAttribute( "href", "#write" + str );
            numberLink.appendChild( m_hdoc.createTextNode( str ) );
        }

        /* name */
        titlenode.appendChild( m_hdoc.createTextNode( " " + m_clnamestr ) );
        DOM::Element namenode = titlenode.appendChild( m_hdoc.createElement( "B" ) );
        {
            /* remove </b> and <b> inside name */
            str_name.remove( "<b>" );
            str_name.remove( "</b>" );

            /* show mail address */
            if ( showMailAddress ) {

                /* when name is composed of only numbers, create link */
                unsigned int i;
                const QChar *chpt = str_name.unicode();
                char strlink[ KITADOM_RESDIGIT + 1 ];
                for ( i = 0;i < QMIN( str_name.length(), KITADOM_RESDIGIT ) ;i++ ) {
                    unsigned short c = chpt[ i ].unicode();
                    if ( ( c < KITADOM_0 || c > KITADOM_9 ) && ( c < '0' || c > '9' ) ) {
                        i = 0;
                        break;
                    }
                    
                    if ( c >= KITADOM_0 ) {
                        strlink[ i ] = '0' + chpt[ i ].unicode() - KITADOM_0;
                    } else {
                        strlink[ i ] = c;
                    }
                }
                strlink[ i ] = '\0';

                if ( i ) {
                    DOM::Element nameLink = namenode.appendChild( m_hdoc.createElement( "A" ) );
                    {
                        nameLink.setAttribute( "href", "#" + QString( strlink ) );
                        nameLink.appendChild( m_hdoc.createTextNode( str_name ) );
                    }
                } else appendColoredTextNode( titlenode, str_name, "color: green" );
            }
            /* not show mail address */
            else if ( str_address != "" ) {

                DOM::Element mailtoLink = namenode.appendChild( m_hdoc.createElement( "A" ) );
                {
                    QString str = QString( "mailto:" ) + str_address;
                    mailtoLink.setAttribute( "href", str );
                    mailtoLink.setAttribute( "title", str_address );
                    mailtoLink.appendChild( m_hdoc.createTextNode( str_name ) );
                }
            }
            /* show name only */
            else {
                appendColoredTextNode( titlenode, str_name, "color: green" );
            }
        }

        /* mail address */
        if ( str_address != "" && showMailAddress ) {
            QString str = QString( " [" ) + str_address + "]";
            titlenode.appendChild( m_hdoc.createTextNode( str ) );
        }

        /* date & id */
        if ( ! str_dateId.isEmpty() ) {
            QString str = m_colonstr + str_dateId;
            titlenode.appendChild( m_hdoc.createTextNode( str ) );
        } else {
            QString str = " " + m_colonstr + dt_dateTime.toString( i18n( "yyyy/MM/dd hh:mm" ) ) + " ";
            titlenode.appendChild( m_hdoc.createTextNode( str ) );

            /* id */
            if ( ! str_id.isEmpty() ) {
                if ( str_id == "???" ) {
                    str = "ID:" + str_id;
                } else {
                    DOM::Element idLink = titlenode.appendChild( m_hdoc.createElement( "A" ) );
                    {
                        idLink.setAttribute( "href", "#idpop" + str_id );
                        idLink.appendChild( m_hdoc.createTextNode( "ID" ) );
                    }
                    str = ":" + str_id;
                }

                titlenode.appendChild( m_hdoc.createTextNode( str ) );
            }
        }
    }

    return titlenode;
}



/*-------------------------------------*/
/* create the node of body text.       */
/*                                     */
/* called from  KitaDomTree::parseRes */
/*-------------------------------------*/
DOM::Element KitaDomTree::createBodyTextNode(
    QString &str_body )
{
    int i;

    /* remove space after <> & copy */
    //    QString str = str_body;
    QString str = str_body.mid( 1 );

    // see RFC 1738.
    QRegExp url_rx( "(h?ttp://([-.0-9a-zA-Z]+(:[0-9]+)?(/[;:@&=$-_.+!*'(),%~/?#0-9a-zA-Z]*)?))" );

    DOM::Element bodynode = m_hdoc.createElement( "DD" );
    {
        /* put the span node at the head of each line */
        if ( m_showAA ) {
            appendColoredTextNode( bodynode, "", "color: white" );
        }

        while ( ( i = str.find( url_rx ) ) != -1 ) {

            if ( i == 0 ) {
                // starts with URL.
                DOM::Element linkElement = bodynode.appendChild( m_hdoc.createElement( "A" ) );
                {
                    linkElement.setAttribute( "href", "http://" + url_rx.cap( 2 ) );
                    linkElement.appendChild( m_hdoc.createTextNode( url_rx.cap( 1 ) ) );
                }

                str.remove( 0, url_rx.cap( 1 ).length() );
            } else {

                QString leftstr = str.left( i );
                parseHTMLdat( bodynode, leftstr );

                str.remove( 0, i );
            }
        }

        parseHTMLdat( bodynode, str );

        bodynode.appendChild( m_hdoc.createElement( "BR" ) );
        bodynode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    return bodynode;
}





/*-----------------------------------*/
/* parsing function of HTML document */
/*-----------------------------------*/
void KitaDomTree::parseHTMLdat(
    DOM::Element &bodynode, QString &str )
{

    struct LocalFunc {
        static int isEqual( const QChar *chpt, const QString& str, const int ret )
        {
            int i = 0;
            while ( str.at( i ) != '\0' ) {
                if ( *chpt != str.at( i ) ) return 0;
                chpt++;i++;
            }
            return ret;
        }
    };

    unsigned int i, i2, index, index2, tail, length = str.length();
    QString urlstr, textstr;
    const QChar *chpt = str.unicode();
    QString replacestr;
    QTextCodec *codec = QTextCodec::codecForName( "utf-8" );

    DOM::Element tmpelm;

    m_linestr = "";

    for ( i = index = 0 ; i < length ; i++ ) {

        switch ( chpt[ i ].unicode() ) {

        case '<':

            /* <br> */
            if ( LocalFunc::isEqual( chpt + i + 1, "br>", 4 ) ) {

                /* remove space before <br> */
                //		    m_linestr += str.mid(index,i-index);
                m_linestr += str.mid( index, ( i - index > 0 ? i - index - 1 : 0 ) );

                bodynode.appendChild( m_hdoc.createTextNode( m_linestr ) );
                bodynode.appendChild( m_hdoc.createElement( "BR" ) );

                /* put the span node after br node */
                if ( m_showAA ) {
                    appendColoredTextNode( bodynode, "", "color: white" );
                }

                m_linestr = "";
                index = i + 4;

                /* remove space after <br> */
                index++;

                i = index - 1;
            }

            /* <a href> */
            /* I don't consider the existence of space ' ' here... */
            else if ( LocalFunc::isEqual( chpt + i + 1, "a href", 7 ) ) {

                /* obtain URL */
                index2 = i + 9; tail = index2;
                while ( chpt[ tail ] != '"' && tail < length ) tail++;
                if ( tail >= length ) break;
                urlstr = str.mid( index2, tail - index2 );

                while ( chpt[ tail ] != '>' && tail < length ) tail++;
                if ( tail >= length ) break;

                /* obtain text */
                index2 = tail + 1; tail = index2;
                while ( chpt[ tail ] != '<' && tail < length ) tail++;
                if ( tail >= length ) break;
                textstr = str.mid( index2, tail - index2 );

                while ( chpt[ tail ] != '>' && tail < length ) tail++;
                if ( tail >= length ) break;

                /* create A node */
                m_linestr += str.mid( index, i - index );
                bodynode.appendChild( m_hdoc.createTextNode( m_linestr ) );
                m_linestr = "";

                tmpelm = bodynode.appendChild( m_hdoc.createElement( "A" ) );
                {
                    tmpelm.setAttribute( "href", urlstr );
                    tmpelm.setAttribute( "target", "_blank" );
                    parseHTMLdat( tmpelm, textstr );
                }

                index = tail + 1;
                i = index;
                while ( createResAnchor( bodynode, str, chpt, i, index ) );
            }

            break;

            /*----------------------------------*/
        case '&':

            i2 = 0;

            if ( LocalFunc::isEqual( chpt + i + 1, "gt;", 4 ) )
                while ( createResAnchor( bodynode, str, chpt, i, index ) );

            else if ( ( i2 = LocalFunc::isEqual( chpt + i + 1, "lt;", 4 ) ) )
                replacestr = "<";

            else if ( ( i2 = LocalFunc::isEqual( chpt + i + 1, "nbsp;", 6 ) ) )
                replacestr = " ";

            else if ( ( i2 = LocalFunc::isEqual( chpt + i + 1, "amp;", 5 ) ) )
                replacestr = "&";

            else if ( ( i2 = LocalFunc::isEqual( chpt + i + 1, "quot;", 6 ) ) )
                replacestr = "\"";

            else if ( ( i2 = LocalFunc::isEqual( chpt + i + 1, "hearts;", 8 ) ) )
                replacestr = codec->toUnicode( KITAUTF8_HEART );

            else if ( ( i2 = LocalFunc::isEqual( chpt + i + 1, "diams;", 7 ) ) )
                replacestr = codec->toUnicode( KITAUTF8_DIA );

            else if ( ( i2 = LocalFunc::isEqual( chpt + i + 1, "clubs;", 7 ) ) )
                replacestr = codec->toUnicode( KITAUTF8_CLUB );

            else if ( ( i2 = LocalFunc::isEqual( chpt + i + 1, "spades;", 8 ) ) )
                replacestr = codec->toUnicode( KITAUTF8_SPADE );

            if ( i2 ) {
                m_linestr += str.mid( index, i - index ) + replacestr;
                index = i + i2;
                i = index - 1;
            }

            break;

            /*----------------------------------------*/

            /* unicode '>'  */
        case KITADOM_BRACKET:

            while ( createResAnchor( bodynode, str, chpt, i, index ) );
            break;
        }
    }

    m_linestr += str.mid( index, i - index );
    bodynode.appendChild( m_hdoc.createTextNode( m_linestr ) );
    m_linestr = "";
}




/*---------------------------*/
/*  append colored text node */
/*---------------------------*/
void KitaDomTree::appendColoredTextNode(
    DOM::Element &root,
    QString str, QString style )
{
    DOM::Element tmpelm;

    tmpelm = root.appendChild( m_hdoc.createElement( "SPAN" ) );
    {
        tmpelm.setAttribute( "style", style );
        tmpelm.appendChild( m_hdoc.createTextNode( str ) );
    }
}



/*---------------------------------*/
/* create  anchor (>>number)  node */
/*---------------------------------*/
bool KitaDomTree::createResAnchor(
    DOM::Element &bodynode, QString &str, const QChar *chpt,
    unsigned int &i, unsigned int &index )
{

    struct LocalFunc {
        static bool isHYPHEN( unsigned short c )
        {

            /* UTF-16 */
            if ( c == '-'
                    || ( c >= 0x2010 && c <= 0x2015 )
                    || ( c == 0x2212 )
                    || ( c == 0xFF0D )      /* UTF8: 0xEFBC8D */
               ) {
                return TRUE;
            }

            return FALSE;
        }
    };

    unsigned int newi;
    unsigned int strlng = str.length();
    int i2, i3;
    QString qstrlink = "";
    char strlink[ KITADOM_RESDIGIT + 1 ];
    DOM::Element tmpelm;
    bool bCreateNode = FALSE;
    bool bBodyisA = ( bodynode.nodeName().string() == "a" );

    newi = i;

    /* check '>' twice */
    for ( i2 = 0;i2 < 2;i2++ ) {
        if ( chpt[ newi ].unicode() == KITADOM_BRACKET ) {
            qstrlink += chpt[ newi ];
            newi++;
        } else if ( chpt[ newi ] == '&' && chpt[ newi + 1 ] == 'g'  /* &gt; */
                    && chpt[ newi + 2 ] == 't' && chpt[ newi + 3 ] == ';' ) {
            qstrlink += ">";
            bCreateNode = TRUE;
            newi += 4;
        }
    }

    /* check ',' */
    if ( i == newi ) {
        if ( chpt[ newi ] == ',' || chpt[ newi ].unicode() == KITADOM_COMMA ) {
            qstrlink += ",";
            newi ++;
        }
    }

    /* check '=' */
    if ( i == newi ) {
        if ( chpt[ newi ] == '=' || chpt[ newi ].unicode() == KITADOM_EQ ) {
            qstrlink += "=";
            newi ++;
        }
    }

    if ( i == newi ) {
        index = newi;
        i--;
        return FALSE;
    }

    /* check numbers */
    bool hyphen = FALSE;
    for ( i2 = 0 , i3 = 0; i2 < KITADOM_RESDIGIT + 1 && newi + i3 < strlng ;i2++, i3++ ) {

        unsigned short c = chpt[ newi + i3 ].unicode();

        if ( ( c < KITADOM_0 || c > KITADOM_9 )
                && ( c < '0' || c > '9' )
                && ( !LocalFunc::isHYPHEN( c )
                     || ( i2 == 0 && LocalFunc::isHYPHEN( c ) )
                     || ( hyphen && LocalFunc::isHYPHEN( c ) ) )
           ) break;

        qstrlink += chpt[ newi + i3 ];
        if ( LocalFunc::isHYPHEN( c ) ) {
            strlink[ i3 ] = '-';
            hyphen = TRUE;
            i2 = -1;
        } else if ( c >= KITADOM_0 )
            strlink[ i3 ] = '0' + chpt[ newi + i3 ].unicode() - KITADOM_0;
        else strlink[ i3 ] = c;
    }

    strlink[ i3 ] = '\0';

    if ( i3 == 0
            || bBodyisA ) { /* If body element is A, return here */

        if ( bCreateNode ) { /* replace &gt; with '>' without link  */
            m_linestr += str.mid( index, i - index ) + qstrlink;
            index = newi + i3;
            i = index - 1;
        }

        return FALSE;
    }

    /* create 'A' element */
    m_linestr += str.mid( index, i - index );
    bodynode.appendChild( m_hdoc.createTextNode( m_linestr ) );
    m_linestr = "";

    tmpelm = bodynode.appendChild( m_hdoc.createElement( "A" ) );
    {
        tmpelm.setAttribute( "href", "#" + QString( strlink ) );
        tmpelm.appendChild( m_hdoc.createTextNode( qstrlink ) );
    }

    index = newi + i3;
    i = index; /* call this function again, so, don't set i = index-1 here ! */

    return TRUE;
}



/*!!! copied from comment.cpp !!!*/
void KitaDomTree::parseDateId( const QString& str ,
                               QString &str_id,
                               QString &str_dateId, QDateTime &dt_dateTime )

{
    QRegExp regexp( "(\\d\\d)/(\\d\\d)/(\\d\\d) (\\d\\d):(\\d\\d)( ID:(.*))?" );

    if ( regexp.search( str ) == -1 ) {
        str_dateId = str;
        return ;
    }

    int year = regexp.cap( 1 ).toInt();
    if ( year >= 70 ) {
        year += 1900;
    } else {
        year += 2000;
    }

    QDateTime
    dateTime( QDate( year, regexp.cap( 2 ).toInt(), regexp.cap( 3 ).toInt() ),
              QTime( regexp.cap( 4 ).toInt(), regexp.cap( 5 ).toInt() ) );
    dt_dateTime = dateTime;
    str_id = regexp.cap( 7 );
}


/*------------------------------------------------------------*/
/*------------------------------------------------------------*/
/*------------------------------------------------------------*/


/*--------------------------------*/
/* private utils                  */
/*--------------------------------*/


/*---------------------------------*/
/* check if this response is valid */

/* call this before appending
   or getting a res.               */ 
/*---------------------------------*/
int KitaDomTree::checkIsNumValid( int num )
{
    if ( m_hdoc == NULL ) return DOM_RES_NOTVALID;
    if ( num < 1 || num > m_maxNum ) return DOM_RES_NOTVALID;
    if ( m_krt->running() ) m_krt->wait();
    if ( !m_resparsed[ num ] ) parseRes( num, num ); /* do paring here */
    if ( !m_resparsed[ num ] ) return DOM_RES_NOTVALID;
    if ( checkAbone( num ) ) return DOM_RES_ABONED;

    return DOM_RES_VALID;
}



/*----------------------------------------------------------------------*/
/* extract all texts from the body text node for searchig , Abone & etc */
/*----------------------------------------------------------------------*/
QString KitaDomTree::extractText( DOM::Node rootnode,
                                  const char* termstr,      /* name of terminating node, e.g. "div","dt","dd"... */
                                  const QString br_str /* strings to replace instead of <br> */
                                )
{
    QString str_ret = "";
    DOM::Node node = rootnode;
    DOM::Node nextnode;

    while ( node != NULL ) {

        if ( node.nodeType() == DOM::Node::TEXT_NODE
                || node.nodeType() == DOM::Node::CDATA_SECTION_NODE
           ) {
            DOM::DOMString nodeText = node.nodeValue();
            str_ret += nodeText.string();
        }
        /* replace <br> with str_ret */
        else if ( br_str != NULL && node.nodeName().string() == "br" ) {
            str_ret += br_str;
        }

        nextnode = node.firstChild();
        if ( nextnode == NULL ) nextnode = node.nextSibling();
        while ( node != NULL && nextnode == NULL ) {
            node = node.parentNode();
            if ( node != NULL ) {

                if ( node.nodeName().string() == termstr ) return str_ret;

                nextnode = node.nextSibling();
            }
        }

        node = nextnode;
    }

    return str_ret;
}



/*------------------------*/
/* get html document      */
/*------------------------*/
QString KitaDomTree::getHtmlCore( int num )      /* the number of response */
{
    QString retstr = QString::null;

    DOM::Node node = m_res[ num ];

    /* get strings as HTML document */
    retstr = "<div>"
             + static_cast<DOM::HTMLElement>( node ).innerHTML().string()
             + "</div>";

    return retstr;
}




/*--------------*/
/* check Abone  */
/*--------------*/

/* !!!  checkAbone should be always called from checkIsNumValid !!! */
/* don't call from other functions. */

bool KitaDomTree::checkAbone( int num )
{
    struct LocalFunc {
        static int StrlistFind( QStringList &strlist, QString str )
        {

            int ret = -1;

            for ( QStringList::iterator it = strlist.begin();
                    it != strlist.end(); ++it ) {
                ret = str.find( ( *it ) );
                if ( ret != -1 ) return ret;
            }
            return ret;
        }
    };

    if ( num == 1 ) return FALSE; /* Don't abone the first res. */
    if ( m_checkAbone[ num ] ) return m_abone[ num ]; /* already checkd */

    m_checkAbone[ num ] = TRUE;
    m_abone[ num ] = TRUE;

    DOM::Node node = m_res[ num ];
    DOM::Node bodytext = node.firstChild().lastChild();
    QString str_id = QString().setNum( num );
    QString str_rname, str_rid, str_text;

    /* id abone */
    str_rid = static_cast<DOM::Element>
              ( node ).getAttribute( "kita_rid" ).string();
    if ( LocalFunc::StrlistFind( m_idlist, str_rid ) != -1 ) {
        return TRUE;
    }

    /* name abone */
    str_rname = static_cast<DOM::Element>
                ( node ).getAttribute( "kita_rname" ).string();
    if ( LocalFunc::StrlistFind( m_namelist, str_rname ) != -1 ) {
        return TRUE;
    }

    /* word abone */
    str_text = extractText( bodytext, "dd", QString::null );
    if ( LocalFunc::StrlistFind( m_wordlist, str_text ) != -1 ) {
        return TRUE;
    }

    /* abone res to respond aboned res */
    if ( m_resAbone ) {

        DOM::NodeList nodelist = bodytext.childNodes();
        for ( int i = 0; i < ( int ) nodelist.length(); ++i ) {

            DOM::HTMLElement anode = nodelist.item( i );
            if ( anode.nodeName().string() == "a" ) {

                int refNum, refNum2;
                QString url = anode.getAttribute( "href" ).string();
                KitaThreadView::getRefNumber( url, m_thread, refNum, refNum2 );

                /* I don't want to enter loop... */
                if ( refNum >= num ) refNum = num - 1;
                if ( refNum2 >= num ) refNum = num - 1;
                if ( refNum2 < refNum ) refNum2 = refNum;

                if ( refNum ) {
                    for ( i = refNum; i <= refNum2; i++ ) {
                        if ( i != num ) {
                            if ( checkIsNumValid( i ) == DOM_RES_ABONED ) {
                                return TRUE;
                            }
                        }
                    }
                }
            }
        }
    }

    m_abone[ num ] = FALSE;

    return FALSE;
}


/*------------------------------------------------------------*/
/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/*---------------------------------------------*/
/* private functions for footer, header , etc. */
/*---------------------------------------------*/


/*-------------------------------*/
/* create footer & header node   */
/*-------------------------------*/
DOM::Element KitaDomTree::createFooter( QString kitatype )
{
    struct LocalFunc {
        static void appendNode( DOM::HTMLDocument hdoc,
                                DOM::Element rootnode, QString attr, QString str )
        {
            DOM::Element tmpelm;
            tmpelm = rootnode.appendChild( hdoc.createElement( "A" ) );
            {
                tmpelm.setAttribute( "href", attr );
                tmpelm.appendChild( hdoc.createTextNode( str ) );
            }
        }
    };

    QString str;
    DOM::Element rootnode, tmpelm;

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", kitatype );
        rootnode.setAttribute( "id", kitatype );
        str = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_KOKOYON );
        LocalFunc::appendNode( m_hdoc, rootnode, "#kokomade_yonda", str );
        rootnode.appendChild( m_hdoc.createTextNode( " " ) );

        str = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_ZENBU );
        LocalFunc::appendNode( m_hdoc, rootnode, "#zenbu", str );
        rootnode.appendChild( m_hdoc.createTextNode( " " ) );

        str = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_SAIGO );
        LocalFunc::appendNode( m_hdoc, rootnode, "#tosaigo", str );

        if ( kitatype == "header" ) {
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
            rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        }
    }

    return rootnode;
}




/*---------------------------*/
/* create kokomadeyonda node */
/*---------------------------*/
void KitaDomTree::createKokoyon()
{
    QString str, style;
    DOM::Element rootnode;

    str = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_KOKOYON2 );
    style = "background-color: #CCCCCC; text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", "kokoyon" );
        rootnode.setAttribute( "id", "kokomade_yonda" );
        rootnode.setAttribute( "style", style );
        rootnode.appendChild( m_hdoc.createTextNode( str ) );
    }

    m_kokoyon = rootnode;
}



/*-----------------------*/
/* create tugi 100 node  */
/*-----------------------*/
void KitaDomTree::createTugi100()
{
    struct LocalFunc {
        static void appendNode( DOM::HTMLDocument hdoc,
                                DOM::Element rootnode, QString attr, QString str )
        {
            DOM::Element tmpelm;
            tmpelm = rootnode.appendChild( hdoc.createElement( "A" ) );
            {
                tmpelm.setAttribute( "href", attr );
                tmpelm.appendChild( hdoc.createTextNode( str ) );
            }
        }
    };

    QString str, style;
    DOM::Element rootnode;

    style = "text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", "tugi100" );
        rootnode.setAttribute( "id", "tugi100" );
        rootnode.setAttribute( "style", style );

        str = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_TUGI100 );
        LocalFunc::appendNode( m_hdoc, rootnode, "#tugi100", str );

        rootnode.appendChild( m_hdoc.createTextNode( "  " ) );

        str = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_ZENBUNOKORI );
        LocalFunc::appendNode( m_hdoc, rootnode, "#nokori", str );

        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    m_tugi100 = rootnode;
}



/*------------------------------------------*/
/* create mae 100 (template)  node          */
/* copied from createTugi100...             */
/*------------------------------------------*/
void KitaDomTree::createMae100()
{
    struct LocalFunc {
        static void appendNode( DOM::HTMLDocument hdoc,
                                DOM::Element rootnode, QString attr, QString str )
        {
            DOM::Element tmpelm;
            tmpelm = rootnode.appendChild( hdoc.createElement( "A" ) );
            {
                tmpelm.setAttribute( "href", attr );
                tmpelm.appendChild( hdoc.createTextNode( str ) );
            }
        }
    };

    QString style;
    DOM::Element rootnode;

    style = "text-align: center";

    rootnode = m_hdoc.createElement( "DIV" );
    {
        rootnode.setAttribute( "kita_type", "mae100" );
        rootnode.setAttribute( "id", "mae100" );
        rootnode.setAttribute( "style", style );

        /* 100 */
        QString tugi100 = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_TUGI100 );
        LocalFunc::appendNode( m_hdoc, rootnode, "#tmp100", tugi100 );

        /* */
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );

        /* */
        style = "background-color: #CCCCCC; text-align: center";
        DOM::Element divElement = rootnode.appendChild( m_hdoc.createElement( "DIV" ) );
        {
            divElement.setAttribute( "style", style );
            QString templateStr = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_TEMPLATE );
            divElement.appendChild( m_hdoc.createTextNode( templateStr ) );
        }

        /* */
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );

        /* 100 */
        QString mae100 = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_MAE100 );
        LocalFunc::appendNode( m_hdoc, rootnode, "#mae100", mae100 );

        /* */
        rootnode.appendChild( m_hdoc.createTextNode( "  " ) );

        /*  */
        QString zenbumae = QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_ZENBUMAE );
        LocalFunc::appendNode( m_hdoc, rootnode, "#maezenbu", zenbumae );

        /* */
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
        rootnode.appendChild( m_hdoc.createElement( "BR" ) );
    }

    m_mae100 = rootnode;
}

/*----------------------------------------------------------------*/
/* for Search */

/*
You need to call findTextInit() before calling findText().
For example, call findTextInit() in KitaThreadView::update_finish().
 
Then call findText() in KitaThreadView::slotSearchButton()
instead of calling searchNext().
*/


/*-----------*/
/* init find */
/*-----------*/  /* public */
void KitaDomTree::findTextInit()
{
    m_initFindText = TRUE;
    m_findNode = NULL;
    m_findPos = -1;
}



/*-------------*/
/* search text */
/*-------------*/ /* public */
bool KitaDomTree::findText( const QString &str, bool reverse )
{
    if ( m_hdoc == NULL ) return FALSE;

    static int m_line = 0;

    QRect qr;

    int cy = 0, ch = 0;
    KHTMLView *view = m_threadPart->view();

    /* init */
    if ( m_findNode.isNull() || m_initFindText ) {

        m_findNode = m_hdoc.body();
        m_line = 0;

        /* move to the last child node */
        if ( reverse ) {
            while ( !m_findNode.lastChild().isNull() ) {
                m_findNode = m_findNode.lastChild();
            }
        }

        m_findPos = -1;
    }

    if ( m_initFindText ) {
        cy = view->contentsY();
        ch = view->visibleHeight();
    }

    while ( 1 ) {

        if ( m_findNode.nodeType() == DOM::Node::TEXT_NODE
                || m_findNode.nodeType() == DOM::Node::CDATA_SECTION_NODE
           ) {

            /* move to the initial node */
            if ( m_initFindText ) {
                QRect rt = m_findNode.getRect();
                if ( !reverse && rt.top() > cy ) {
                    m_initFindText = 0;
                } else if ( rt.bottom() < cy + ch ) {
                    m_initFindText = 0;
                }
            } else {
                /* find the word in the current node */
                DOM::DOMString nodeText = m_findNode.nodeValue();
                QString str2 = nodeText.string();
                int lastpos = str2.length();
                if ( reverse && m_findPos != -1 ) {
                    lastpos = m_findPos;
                }
                QConstString cstr( str2.unicode(), lastpos );

                if ( reverse ) {
                    int foo;
                    foo = m_findPos = -1;
                    while ( ( foo = cstr.string().find( str, m_findPos + 1, FALSE ) ) != -1 )
                        m_findPos = foo;
                } else m_findPos = cstr.string().find( str, m_findPos + 1, FALSE );

                /* scroll & select */
                if ( m_findPos != -1 ) {
                    int matchLen = str.length();

                    qr = m_findNode.getRect();
                    view->setContentsPos( qr.left() - 50
                                          , m_find_y + m_line - 100 );
                    DOM::Range rg( m_findNode, m_findPos,
                                   m_findNode, m_findPos + matchLen );
                    m_threadPart->setSelection( rg );
                    return TRUE;
                }
            }
        } else if (
            m_findNode.nodeName().string() == "dt"
            || m_findNode.nodeName().string() == "dd" ) {

            qr = m_findNode.getRect();
            if ( reverse ) {
                m_find_y = qr.bottom();
            } else {
                m_find_y = qr.top();
            }
            m_line = 0;
        } else if ( m_findNode.nodeName().string() == "br" ) {
            DOM::Node tmpnode = m_findNode.previousSibling();
            if ( tmpnode != NULL ) {
                qr = tmpnode.getRect();
                if ( reverse ) {
                    m_line -= qr.bottom() - qr.top();
                } else {
                    m_line += qr.bottom() - qr.top();
                }
            }
        }

        m_findPos = -1;
        DOM::Node next;

        /* move to the next node */
        if ( !reverse ) {

            next = m_findNode.firstChild();
            if ( next.isNull() ) {
                next = m_findNode.nextSibling();
            }
            while ( !m_findNode.isNull() && next.isNull() ) {
                m_findNode = m_findNode.parentNode();
                if ( !m_findNode.isNull() ) {
                    next = m_findNode.nextSibling();
                }
            }
        } else { /* revearse */

            next = m_findNode.lastChild();
            if ( next.isNull() ) {
                next = m_findNode.previousSibling();
            }
            while ( !m_findNode.isNull() && next.isNull() ) {
                m_findNode = m_findNode.parentNode();
                if ( !m_findNode.isNull() ) {
                    next = m_findNode.previousSibling();
                }
            }
        }

        m_findNode = next;
        if ( m_findNode.isNull() ) {
            m_findNode = NULL;
            return FALSE;
        }
    }

    return FALSE;
}
