/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitahtmlpart.h"

#include "kitathreadview.h"
#include "kitahtmlview.h"

#include <kpopupmenu.h>
#include <klocale.h>
#include <khtmlview.h>
#include <dom/html_element.h>

#include <qprogressdialog.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qclipboard.h>

#include "libkita/thread.h"

enum {
    ID_Copy_Link
};

KitaHTMLPart::KitaHTMLPart( QWidget* parent, const char* name )
        : KHTMLPart( new KitaHTMLView( this, parent, name ) )
        , m_parent( parent )
{
    connect( this, SIGNAL( popupMenu( const QString&, const QPoint& ) ),
             SLOT( slotPopupMenu( const QString&, const QPoint& ) ) );

    m_popupMenu = new KPopupMenu( view() );
}


KitaHTMLPart::~KitaHTMLPart()
{}

void KitaHTMLPart::reset()
{
    m_prevQuery = "";
    m_nextHit = -1;
    m_hitCount = 0;
    m_contents = "";
    m_pointStack.clear();
}

void KitaHTMLPart::highLightenNodes( DOM::NodeList nodes, const QStringList& query )
{
    for ( unsigned int count = 0; count < nodes.length(); ++count ) {
        DOM::HTMLElement dd = nodes.item( count );
        QStringList lt_gt_split = lt_gt_spliter( dd.innerHTML().string() );

        QProgressDialog progress( m_parent );
        progress.setLabelText( "Searching..." );
        progress.setTotalSteps( lt_gt_split.size() );
        progress.show();
        int step = 0;
        int divide = lt_gt_split.size() / ( 100 / 5 );
        int next = divide;

        //loop order not optimized. Need optim. that also needs any codes & members?
        QStringList::iterator it_msg = lt_gt_split.begin();
        for ( ; it_msg != lt_gt_split.end(); ++it_msg ) {
            QStringList::const_iterator it = query.begin();
            for ( ; it != query.end(); ++it ) {
                int pos = 1;
                QString message = *it_msg;
                if ( message.at( 0 ) == '<' ) continue;
                while ( ( pos = ( *it_msg ).find( *it, pos, false ) ) != -1 ) {
                    QString replace = QString( "<font class=\"highlight\" id=\"highlighten%2\">" ).arg( m_hitCount );
                    //	  replace += *it;
                    replace += message.mid( pos, ( *it ).length() );
                    replace += "</font>";
                    message.replace( pos, ( *it ).length(), replace );
                    pos += replace.length();
                    lt_gt_split.insert( it_msg, message );
                    it_msg = lt_gt_split.remove( it_msg );
                    --it_msg;
                    ++m_hitCount;
                }
            }
            if ( ++step > next ) {
                progress.setProgress( step );
                next += divide;
            }
        }

        dd.setInnerHTML( lt_gt_split.join( " " ) );

    }
}

QStringList KitaHTMLPart::lt_gt_spliter( const QString& input )
{
    QStringList lt_gt_splitted;
    QRegExp lt_gt( "[<>]" );
    int start = 0;
    int end = 0;
    while ( ( end = input.find( lt_gt, end + 1 ) ) != -1 ) {
        if ( input.at( end ) == '<' ) {
            lt_gt_splitted += input.mid( start, end - start );
            start = end;
        } else
            if ( input.at( end ) == '>' ) {
                lt_gt_splitted += input.mid( start, end + 1 - start );
                start = end + 1;
            } else QMessageBox::warning( 0, "", "KitaThreadView::lt_gt_spliter()\n !!Bug!! Not Match.\n Will you contact with authors?" );
    }
    lt_gt_splitted += input.mid( start );
    return lt_gt_splitted;
}

void KitaHTMLPart::displayContents( const QString& input, const KURL& url )
{
    m_contents = input;
    m_url = url;

    begin( url );
    write( input );
    end();
    if ( ! url.isEmpty() ) emit setLocationBarURL( url.url() );

    view() ->setFocus();
}

void KitaHTMLPart::gotoURL( const Kita::Thread* thread, int prevResNum )
{
    if ( KURL( thread->datURL() ).hasRef() ) {
        if ( ! gotoAnchor( KURL( thread->datURL() ).encodedHtmlRef() ) )
            gotoAnchor( KURL( thread->datURL() ).htmlRef() );
    } else {
        gotoAnchor( QString().setNum( prevResNum ) );
    }
    clearPosition();
    pushCurrentPosition();
}

bool KitaHTMLPart::gotoAnchor( const QString& name )
{
    pushCurrentPosition();
    return KHTMLPart::gotoAnchor( name );
}

int KitaHTMLPart::nextHit()
{
    m_nextHit++;
    if ( m_nextHit >= m_hitCount ) m_nextHit = 0;

    return m_nextHit;
}

void KitaHTMLPart::highLighten( bool yes, const QStringList& query )
{
    if ( ! yes ) {
        setUserStyleSheet( QString( "" ) );
        return ;
    }

    // Use style sheet to highlighten.
    QString highlightCSS( ".highlight { color: %1; background: %2; } " );
    highlightCSS = highlightCSS.arg( "yellow" ).arg( "black" );

    if ( prevQuery() != query && !query.isEmpty() ) {
        m_prevQuery = query;
        displayContents( contents(), m_url );
        DOM::NodeList nodes;
        resetHit();
        m_hitCount = 0;
        nodes = document().getElementsByTagName( "body" );
        highLightenNodes( nodes, query );
    }
    setUserStyleSheet( highlightCSS );
}

/*-------------------------------------------------*/
/* shobon extension                                */
/*-------------------------------------------------*/

#include "kitadomtree.h"
#include <khtml_events.h>

#include "libkita/access.h"

enum {
    ID_COPY_Link,
    ID_Home_Link,
    ID_Temp_Link,
    ID_Koko_Link,
    ID_End_Link,
    ID_SAVEC_Link,
    ID_Back_Link,
    ID_Copy_Str,
    ID_Ext_Str,
    ID_Del_IMGCache,
    ID_COPY_IMGCachepath,
    ID_DEL_ThreadLog
};

/*-------------------------------------------------*/

/*------------------*/
/* public functions */
/*------------------*/

/* setup */
void KitaHTMLPart::setupEx( KitaThreadView* threadView, KitaDomTree* domtree,
                            Kita::Thread* thread )
{
    m_threadView = threadView;
    m_domtree = domtree;
    m_thread = thread;
    m_anchorStack.clear();
}


void KitaHTMLPart::pushCurrentPosition()
{
    QString anchor, ancundermouse;
    DOM::Node node;

    /* find the current node */
    node = nodeUnderMouse();
    while ( node != NULL && node.nodeName().string() != "div" )
        node = node.parentNode();
    if ( node == NULL ) return ;
    anchor = static_cast<DOM::Element>( node ).getAttribute( "id" ).string();

    m_anchorStack += anchor;
}


/* jump to kokomade yonda */
void KitaHTMLPart::gotoKokoyon()
{

    int num = m_domtree->getKokoyonNum() - 1;

    while ( num > 0 ) {
        if ( m_domtree->isResShown( num ) ) break;
        num--;
    }

    KHTMLPart::gotoAnchor( QString().setNum( num ) );
}



/*-------------------------------------------------------------------*/

/* public slots */

bool KitaHTMLPart::gobackAnchor()
{
    QString anchor;

    if ( m_anchorStack.empty() ) return FALSE;

    anchor = m_anchorStack.last();
    m_anchorStack.pop_back();

    return KHTMLPart::gotoAnchor( anchor );
}



/*--------------------------------------------------*/

/* private slot */

/* show popup menu */
void KitaHTMLPart::slotPopupMenu( const QString& url, const QPoint& point )
{
    QString str;
    DOM::Node node = nodeUnderMouse();

    m_popupMenu->clear();
    
    if ( !m_anchorStack.empty() ) {
        str = "Back (" + m_anchorStack.last() + ")";
        m_popupMenu->insertItem( str, ID_Back_Link );
        m_popupMenu->insertSeparator();
    }
    
    m_popupMenu->insertItem( "Home", ID_Home_Link );
    
    if ( m_domtree->isMae100Shown() ) {
        m_popupMenu->insertItem( "template", ID_Temp_Link );
    }
    
    if ( m_domtree->getKokoyonNum() ) {
        str = "Kokomade Yonda (" + QString().setNum( m_domtree->getKokoyonNum() ) + ")";
        m_popupMenu->insertItem( str, ID_Koko_Link );
    }
    
    m_popupMenu->insertItem( "End", ID_End_Link );
    
    if ( hasSelection() ) {
        m_popupMenu->insertSeparator();
        m_popupMenu->insertItem( "Copy", ID_Copy_Str );
    }
    
    if ( url != NULL ) {
        m_popupMenu->insertSeparator();
        m_popupMenu->insertItem( i18n( "Copy Link Location" ), ID_COPY_Link );

    }
    
    if ( m_threadView->getViewMode() == VIEWMODE_PARENT ) {
        m_popupMenu->insertSeparator();
        m_popupMenu->insertItem( i18n( "Clear Log and Reload" ), ID_DEL_ThreadLog );
    }

    QClipboard* clipboard = QApplication::clipboard();

    switch ( m_popupMenu->exec( point ) ) {
    case ID_COPY_Link:
        clipboard->setText( url , QClipboard::Clipboard );
        clipboard->setText( url , QClipboard::Selection );
        break;
    case ID_Home_Link: gotoAnchor( "header" ); break;
    case ID_Temp_Link: gotoAnchor( "mae100" ); break;
    case ID_Koko_Link: gotoKokoyon();break;
    case ID_End_Link: gotoAnchor( "footer" ); break;
    case ID_Back_Link: gobackAnchor(); break;

    case ID_Copy_Str:
        clipboard->setText( selectedText(), QClipboard::Clipboard );
        break;

    case ID_DEL_ThreadLog:
        if ( Kita::Access::deleteLog( m_thread, m_parent ) )
            m_threadView->showThread( m_thread );
        break;

    default:
        break;
    }
}


/*----------------------------------------------------*/

/* protected functions */

void KitaHTMLPart::khtmlMousePressEvent( khtml::MousePressEvent* e )
{
    m_pushctrl = FALSE;
    m_pushmidbt = FALSE;

    /* check if ctrl key is pushed */
    if ( e->url() != NULL ) {

        if ( e->qmouseEvent() ->state() & Qt::ControlButton ) {
            m_pushctrl = TRUE;
        }
        if ( e->qmouseEvent() ->button() & Qt::MidButton ) {
            m_pushmidbt = TRUE;
        }

        if ( m_pushctrl || m_pushmidbt ) {
            KURL kurl;
            if ( e->url().string().at( 0 ) == '#' ) {
                kurl = KURL( m_thread->datURL(), e->url().string() );
            } else {
                kurl = KURL( m_thread->boardURL(), e->url().string() );
            }

            if ( m_pushctrl ) {
                emit browserExtension() ->openURLRequest( kurl );
            }

            return ;
        }
    }
    KHTMLPart::khtmlMousePressEvent( e );
}

void KitaHTMLPart::khtmlMouseMoveEvent( khtml::MouseMoveEvent* e )
{
    KHTMLPart::khtmlMouseMoveEvent( e );
}

void KitaHTMLPart::khtmlMouseReleaseEvent( khtml::MouseReleaseEvent *e )
{
    if ( e->qmouseEvent() ->button() & Qt::RightButton ) {

        /* popup here & return */
        slotPopupMenu( e->url().string(), e->qmouseEvent() ->globalPos() );
        return ;
    }
    KHTMLPart::khtmlMouseReleaseEvent( e );
}
