/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*   linux 󥫥פ餰ο                                        *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitathreadview.h"

#include <kpopupmenu.h>
#include <krun.h>
#include <kurl.h>
#include <khtmlview.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kaction.h>
#include <kmessagebox.h>
#include <kdebug.h>

#include <kio/netaccess.h>

#include <dom/html_inline.h>
#include <dom/html_base.h>
#include <dom/html_list.h>

#include <qclipboard.h>
#include <qlayout.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qprogressdialog.h>
#include <qcursor.h>
#include <qtextbrowser.h>
#include <qmessagebox.h>
#include <qdatetime.h>
#include <qeucjpcodec.h>

#include "kitahtmlpart.h"
#include "kitawritedialog.h"
#include "kita2ch.h"
#include "kitaconfig.h"
#include "kita-utf8.h"

#include "libkita/access.h"
#include "libkita/threadinfo.h"
#include "libkita/comment.h"
#include "libkita/qcp932codec.h"
#include "libkita/favoritethreads.h"

#define MAX_LABEL_LENGTH 60

static const char* cookie_title = "񤭹߳ǧ";
static const char* cookie_message =
    "Ƴǧ\n"
    "Ƥ줿Ƥϥԡ¸ѡž礬ޤ\n"
    "Ƥ˴ؤȯǤƼԤ˵ޤ\n"
    "\n"
    "Ǥ餦Ȥƽ񤭹ߤޤ\n";

static const char* kokomade_yonda =
    "<p id='kokomade_yonda' style=\"background-color: #CCCCCC; text-align: center\">"
    "ޤɤ"
    "</p>";

KitaThreadView::KitaThreadView( QWidget* parent, const char* name )
        : KitaThreadViewBase( parent, name )
        , m_access( 0 )
        , m_popup( 0 )
{
    m_thread = new Kita::NullThread();
    m_threadPart = new KitaHTMLPart( threadFrame );
    QHBoxLayout* aLayout = new QHBoxLayout( threadFrame );
    aLayout->addWidget( m_threadPart->view() );

    {
        SearchButton->setPixmap( SmallIcon( "find" ) );
        HighLightButton->setPixmap( SmallIcon( "idea" ) );
        ReloadButton->setPixmap( SmallIcon( "reload" ) );
        GobackAnchorButton->setPixmap( SmallIcon( "2leftarrow" ) );
        BookmarkButton->setPixmap( SmallIcon( "bookmark_add" ) );
    }

    setAcceptDrops( true ); // DND Drop eneble: 2nd stage. - enable on "KitaThreadView" widget and disable on the others(child widgets of "KitaThreadView").
    threadFrame->setAcceptDrops( false ); // don't treat Drop event on child.
    m_threadPart->view() ->setAcceptDrops( false ); // don't treat Drop event on child.

    m_threadPart->enableMetaRefresh( false ); //disable <meta refresh="...">


    connect( writeButton, SIGNAL( clicked() ),
             SLOT( slotWriteButtonClicked() ) );
    connect( m_threadPart, SIGNAL( nodeActivated( const DOM::Node& ) ),
             SLOT( slotDOMNodeActivated( const DOM::Node& ) ) );
    connect( m_threadPart, SIGNAL( onURL( const QString& ) ),
             SLOT( slotOnURL( const QString& ) ) );
    connect( m_threadPart, SIGNAL( setLocationBarURL( const QString& ) ),
             SIGNAL( setLocationBarURL( const QString& ) ) );
    connect( BookmarkButton, SIGNAL( toggled( bool ) ),
             SLOT( slotBookmarkButtonClicked( bool ) ) );
    connect( SearchButton, SIGNAL( clicked() ),
             SLOT( slotSearchButton() ) );
    connect( SearchCombo, SIGNAL( activated( int ) ),
             SLOT( slotSearchButton() ) );
    connect( HighLightButton, SIGNAL( toggled( bool ) ),
             SLOT( slotHighLightenButton( bool ) ) );
    connect( GobackAnchorButton, SIGNAL( clicked() ),
             m_threadPart, SLOT( gobackAnchor() ) );
    connect( ReloadButton, SIGNAL( clicked() ),
             SLOT( slotReloadButton() ) );

    KParts::BrowserExtension* ext = m_threadPart->browserExtension();
    connect( ext, SIGNAL( openURLRequest( const KURL&, const KParts::URLArgs& ) ),
             SLOT( slotOpenURLRequest( const KURL&, const KParts::URLArgs& ) ) );
    connect( ext, SIGNAL( createNewWindow ( const KURL&, const KParts::URLArgs& ) ),
             SIGNAL( createNewWindow ( const KURL&, const KParts::URLArgs& ) ) );
    connect( ext, SIGNAL( setLocationBarURL( const QString& ) ),
             SIGNAL( setLocationBarURL( const QString& ) ) );
    connect( ext, SIGNAL( enableAction( const char*, bool ) ),
             SIGNAL( enableAction( const char*, bool ) ) );

    connect( ext, SIGNAL( popupMenu( KXMLGUIClient*, const QPoint&, const KURL&, const QString&, mode_t ) ),
             this, SLOT( slotPopupMenu( KXMLGUIClient*, const QPoint&, const KURL&, const QString&, mode_t ) ) );

    /* init  */
    /* re-set them later in KitaThreadView::setupEx again */
    m_domtree = new KitaDomTree( m_threadPart );
    m_viewparent = this;
    m_domtreeparent = m_domtree;
    m_viewmode = VIEWMODE_PARENT;
    m_rescode = 200;

    m_threadPart->setupEx( this, m_domtree, NULL );
    m_threadPart->begin();
    m_threadPart->write( " " );
    m_threadPart->end();
    m_threadPart->show();

}

KitaThreadView::~KitaThreadView()
{

    if ( m_popup ) {
        delete m_popup;
        m_popup = NULL;
    }

    /* If this widget is parent, delete kitaNavi first */
    if ( m_viewmode == VIEWMODE_PARENT ) {
        killJob(); /* don't forget killing the job first... */
        if ( m_access ) {
            delete m_access;
            m_access = NULL;
        }
    }

    /* don't delete m_threadPart before deleting m_domtree */
    if ( m_domtree ) {
        delete m_domtree;
        m_domtree = NULL;
    }
    if ( m_threadPart ) {
        delete m_threadPart;
        m_threadPart = NULL;
    }
}

const QString KitaThreadView::threadName() const
{
    return m_thread->name();
}

const KURL KitaThreadView::threadURL() const
{
    return m_thread->url();
}

void KitaThreadView::slotDOMNodeActivated( const DOM::Node& node )
{
    { //process Anchor tags. Anchor tags not proccessed here cause 'emit KParts::BrowserExtention::openURLRequest()'
        DOM::HTMLAnchorElement anchor = node;

        if ( ! anchor.href().isEmpty() ) {
            kdDebug() << "AnchorNodeActivated::" << endl;
        } // end: anchor.href().isEmpty()
    } // end of Anchor tags.
}

void KitaThreadView::setSubjectLabel( const QString& boardName, const QString& threadName )
{
    QString disp;
    if ( boardName.isEmpty() ) {
        disp = threadName;
    } else {
        disp = QString( "[%1] %2" ).arg( boardName ).arg( threadName );
    }

    disp.truncate( MAX_LABEL_LENGTH );
    subjectLabel->setText( disp );
}

void KitaThreadView::updateButton()
{
    writeButton->setEnabled( true );
    BookmarkButton->setEnabled( true );
    ReloadButton->setEnabled( true );

    if ( HighLightButton->isOn() ) {
        HighLightButton->toggle();
    }
    if ( FavoriteThreads::getInstance() ->contains( m_thread->datURL() ) ) {
        BookmarkButton->setOn( true );
    } else {
        BookmarkButton->setOn( false );
    }
}

const QString KitaThreadView::footer() const
{
    QString text;
    int target;
    for ( target = 1; target < m_thread->resNum(); target += 100 ) {
        text += QString( " <a href=\"#%1\">%3</a> " ).arg( target ).arg( target );
    }
    text += " <a href=\"#kokomade_yonda\">New</a> ";
    text += "<br/>";

    return text;
}

void KitaThreadView::slotShowErrorDialog( const QString& input, const KURL& )
{
    QTextCodec * eucjpCodec = QTextCodec::codecForName( "euc-jp" );

    kdDebug() << "'" << input << "'" << endl;
    Kita::WriteResult writeResult( input );

    kdDebug() << "code = " << writeResult.code() << endl;
    switch ( writeResult.code() ) {
    case Kita::K2ch_Unknown:
        // probably OK.
        emit writeSucceeded();
        slotReloadButton();
        break;
    case Kita::K2ch_True:
    case Kita::K2ch_False:
    case Kita::K2ch_Error:
    case Kita::K2ch_Check:
        KMessageBox::error( 0, writeResult.message(), writeResult.title() );
        break;
    case Kita::K2ch_Cookie:
        if ( KMessageBox::questionYesNo( 0, eucjpCodec->toUnicode( cookie_message ),
                                         eucjpCodec->toUnicode( cookie_title ) )
                == KMessageBox::Yes ) {
            KitaWriteDialog * dialog = openDialog( m_postInfo );
            dialog->postMessage();
        } else {
            KitaWriteDialog* dialog = openDialog( m_postInfo );
            dialog->setupEx( m_thread, m_access, m_viewparent, m_domtreeparent,
                             m_domtreeparent->getMaxResNumber() + 1 );
            dialog->show();
        }
        break;
    default:
        break;
    }
}

KitaWriteDialog* KitaThreadView::openDialog( const Kita::PostInfo& info )
{
    KitaWriteDialog * new_dialog = KitaWriteDialog::open( info, m_thread );
    connect( new_dialog, SIGNAL( postStarted( KIO::Job*, const Kita::PostInfo& ) ),
             this, SLOT( slotPostStarted( KIO::Job*, const Kita::PostInfo& ) ) );
    connect( new_dialog, SIGNAL( postResponse( const QString&, const KURL& ) ),
             this, SLOT( slotShowErrorDialog( const QString&, const KURL& ) ) );
    connect( new_dialog, SIGNAL( postResponse( const QString&, const KURL& ) ),
             this, SIGNAL( postResponse( const QString&, const KURL& ) ) );
    return new_dialog;
}

/*--------------------*/
/* write response     */
/*--------------------*/  /* private slots */
void KitaThreadView::slotWriteButtonClicked()
{
    slotWriteButtonClickedCore( "" );
}

void KitaThreadView::slotWriteButtonClickedCore( QString resstr )
{
    //  QSjisCodec cp932Codec;
    if ( ! m_access ) {
        kdError() << __func__ << ": m_access is null" << endl;
        return ;
    }
    KURL bbscgiURL = KURL( m_thread->boardURL(), "../test/bbs.cgi" );
    bbscgiURL.setProtocol( "http" );

    Kita::PostInfo info;
    info.host = bbscgiURL.host();
    info.bbs = m_thread->boardID();
    info.key = m_thread->datID();
    info.time = QString( "%1" ).arg( m_access->serverTime() );
    info.body = resstr;
    info.mail = m_mailaddr;

    KitaWriteDialog* dialog = openDialog( info );
    dialog->setupEx( m_thread, m_access, m_viewparent, m_domtreeparent,
                     m_domtreeparent->getMaxResNumber() + 1 );
    dialog->show(); // work asynchronus.
}

void KitaThreadView::slotHighLightenButton( bool yes )
{
    insertSearchCombo();
    QStringList list = parseSearchQuery( SearchCombo->currentText() );
    m_threadPart->highLighten( yes, list );
}

void KitaThreadView::insertSearchCombo()
{
    for ( int count = 0; count < SearchCombo->count(); ++count ) {
        if ( SearchCombo->text( count ) == SearchCombo->currentText() ) {
            // found
            return ;
        }
    }
    SearchCombo->insertItem( SearchCombo->currentText() );
}

QStringList KitaThreadView::parseSearchQuery( const QString& input ) const
{
    QStringList tmp = QStringList::split( ' ', input );
    QStringList ret_list;
    QRegExp truncSpace( "\\s*$" );
    QStringList::iterator it = tmp.begin();
    for ( ; it != tmp.end(); ++it )
        ret_list += ( *it ).replace( truncSpace, "" );
    return ret_list;
}

void KitaThreadView::searchNext( const QStringList& query )
{
    if ( query.isEmpty() )
        return ;
    if ( ! HighLightButton->isOn() ) {
        HighLightButton->toggle();
        m_threadPart->resetHit(); //A next jump-search target reset to '0'.
        // Process works asynchronusly. So Firstly, we don't do jump-search as a simple solution.
        return ;
    }
    if ( query != m_threadPart->prevQuery() ) {
        m_threadPart->highLighten( true, query );
        m_threadPart->resetHit(); //A next jump-search target reset to '0'.
        return ;
    }

    m_threadPart->gotoAnchor( QString( "highlighten%1" ).arg( m_threadPart->nextHit() ) );
}

void KitaThreadView::hidePopup()
{
    if ( m_popup ) {
        m_popup->hide();
    }
}

void KitaThreadView::slotPopupMenu( KXMLGUIClient* client, const QPoint& global, const KURL& url, const QString& mimeType, mode_t mode )
{
    KActionCollection * collection = client->actionCollection();
    KAction* action;
    action = new KAction( i18n( "goback anchor" ), SmallIcon( "idea" ), KShortcut(), m_threadPart, SLOT( gobackAnchor() ), collection, "goback_anchor" );
    emit popupMenu( client, global, url, mimeType, mode );
}

void KitaThreadView::setFont( const QFont& font )
{
    //  m_threadPart->setStandardFont(font.family());
    subjectLabel->setFont( font );

    DOM::CSSStyleDeclaration style = m_threadPart->htmlDocument().body().style();
    style.setProperty( "font-family", font.family(), "" );
    style.setProperty( "font-size", QString( "%1pt" ).arg( font.pointSize() ), "" );
}

void KitaThreadView::slotPostStarted( KIO::Job*, const Kita::PostInfo& info )
{
    m_postInfo = info;
}

void KitaThreadView::slotBookmarkButtonClicked( bool on )
{
    emit bookmarked( m_thread, on );
}

void KitaThreadView::killJob()
{
    if ( m_viewparent != this ) {
        return ;
    }

    if ( m_access ) {
        m_access->killJob();
    }
}

void KitaThreadView::focusSearchCombo()
{
    SearchCombo->setFocus();
}

namespace Kita
{

    PopupTextBrowser::PopupTextBrowser( QFrame* frame )
            : QTextBrowser( frame )
    {
        zoomOut( static_cast<int>( pointSize() * 0.3 ) );
        setWordWrap( QTextEdit::NoWrap );
        setResizePolicy( QScrollView::AutoOne );
        setFont( KitaConfig::threadFont() );
        setPaletteForegroundColor( KitaConfig::popupColor() );
        setPaletteBackgroundColor( KitaConfig::popupBackgroundColor() );
    }

    PopupTextBrowser::~ PopupTextBrowser()
    {}

    ResPopup::ResPopup( KHTMLView* view )
            : QFrame( view, "res_popup", WStyle_Customize | WStyle_NoBorder | WStyle_Tool | WType_TopLevel | WX11BypassWM )
    {
        m_browser = new Kita::PopupTextBrowser( this );
    }

    ResPopup::~ ResPopup()
    {}

    void ResPopup::setText( QString& str )
    {
        m_browser->setText( str );
    }

    void ResPopup::adjustSize()
    {
        m_browser->resize( m_browser->contentsWidth() + 10, m_browser->contentsHeight() );
        QFrame::adjustSize();
    }

};


/*-------------------------------------------------*/
/* shobon extension                                */
/*-------------------------------------------------*/

void KitaThreadView::setupEx( const Kita::Thread* thread,
                              Kita::Access* access,
                              KitaThreadView* viewparent,
                              KitaDomTree* domtreeparent,
                              int viewmode )
{

    /* config. */

    /* show m_preShowNum responses before 'kokomade yonda',
       and m_afterShowNum responses after 'kokomade yonda' */
    int tmplateNum = 20;
    m_preShowNum = 50;
    m_afterShowNum = 100;

    /* max size of responses in the popup window */
    m_maxpopup = 10;

    /* style */
    m_style = QString( "style=\"font-size: %1pt; "
                       "font-family: %2; "
                       "color: %3; "
                       "background-color: %4; border-width: 0; \"" )
              .arg( KitaConfig::threadFont().pointSize() )
              .arg( KitaConfig::threadFont().family() )
              .arg( KitaConfig::threadColor().name() )
              .arg( KitaConfig::threadBackgroundColor().name() );

    /* default name */
    m_mailaddr = "sage";

    /* online mode */
    m_online = TRUE;

    m_thread = const_cast<Kita::Thread *>( thread );

    /* setup DOM */
    /* m_domtree is created in KitaThreadView::KitaThreadView */
    m_domtree->TerminateParseThread();
    m_domtree->parseResInit( m_thread,
                             m_style,
                             tmplateNum );

    /* setup HTMLPart */
    /* m_threadPart is created in KitaThreadView::KitaThreadView */
    m_threadPart->setupEx( this, m_domtree, m_thread );
    m_threadPart->reset();

    /* set parent thread view of this view. If this widget is parent,
       then, m_viewparent == this  */
    m_viewparent = viewparent;

    /* If this widget is on the KitaNavi, m_domtreeparent is used
       for quotation, etc. Otherwise (i.e. this widget is NOT on the KitaNavi),
       m_domtreeparent == m_domtree. */
    m_domtreeparent = domtreeparent;


    /* mode. Mode is defined in kitathreadview.h */
    m_viewmode = viewmode;

    /* dat loader */
    m_access = access;

    m_revsearch = FALSE;
}


/*--------------------------------------------------------*/
/* Show thread                                            */
/* This function is called from KitaThreadTabWidget class */
/*--------------------------------------------------------*/
void KitaThreadView::showThread( const Kita::Thread* thread )
{

    /* If this widget is not parent, then do nothing. */
    if ( m_viewmode != VIEWMODE_PARENT ) return ;

    if ( topLevelWidget() ->isMinimized() ) topLevelWidget() ->showNormal();
    topLevelWidget() ->raise();
    setActiveWindow();

    /*------------------------------------------------*/
    /* setup */
    if ( m_access != NULL ) {
        killJob();
        delete m_access;
        m_access = NULL;
    }
    setupEx( thread ,
             m_access,        /* = NULL */
             this,
             m_domtree,
             VIEWMODE_PARENT /* parent mode */
           );

    /* end setup */
    /*------------------------------------------------*/

    /* read cache */
    update_readCache();

    /* update cahce & do rendering */
    update_rendering( m_online );

    topLevelWidget() ->raise();
    setActiveWindow();
}



/*---------*/
/* reload  */
/*---------*/ /* public slot */
void KitaThreadView::slotReloadButton()
{
    /* If this widget is not parent, then do nothing. */
    if ( m_viewmode != VIEWMODE_PARENT ) return ;

    update_rendering( m_online );
}




/*---------------------------------------------------*/
/* This function is called when user clicks the link */
/*---------------------------------------------------*/  /* public slot */
void KitaThreadView::slotOpenURLRequest( const KURL& urlin, const KParts::URLArgs& args )
{
    KURL url = urlin;

    /* convert to the correct URL */
    /* see also KitaDomTree::createHTMLDocument*/
    if ( urlin.protocol() == "file" ) {


        if ( urlin.prettyURL().mid( 5, 14 ) == "/test/read.cgi" )
            url = KURL( m_thread->datURL(), ".." + urlin.prettyURL().mid( 5 ) );
        else
            url = KURL( m_thread->datURL(), urlin.prettyURL().mid( 15 ) );

    }

    KURL datURL = filterReadCGI( url );

    /*----------------*/
    /* jump to anchor */
    /*----------------*/
    if ( datURL.host() == KURL( m_thread->datURL() ).host()
            && datURL.path() == KURL( m_thread->datURL() ).path() ) {

        if ( m_viewmode == VIEWMODE_PREVIEW ) return ;

        if ( datURL.hasRef() ) {

            int top = m_domtree->getTopResNumber();
            int bottom = m_domtree->getBottomResNumber();
            int totalNum = m_domtreeparent->getMaxResNumber();

            /* write */
            if ( datURL.ref().left( 5 ) == "write" ) {

                QClipboard * clipboard = QApplication::clipboard();
                QString str, resstr;
                int resNum = datURL.ref().mid( 5 ).toInt();


                KPopupMenu *popupMenu = new KPopupMenu( m_threadPart->view() );
                popupMenu->clear();
                popupMenu->insertItem( "write response", 0 );
                popupMenu->insertItem( "quote this", 1 );
                popupMenu->insertItem( "copy", 3 );
                if ( m_viewmode == VIEWMODE_PARENT ) {
                    popupMenu->insertSeparator();
                    popupMenu->insertItem( "set Kokomade Yonda", 2 );
                }

                switch ( popupMenu->exec( QCursor::pos() ) ) {
                case 0:
                    resstr = ">>" + QString().setNum( resNum ) + "\n";
                    break;

                case 1:
                    resstr = ">>" + QString().setNum( resNum ) + "\n";
                    resstr += m_domtreeparent->getResStr( resNum, "> " );
                    break;

                case 2:
                    delete popupMenu;
                    m_domtree->setKokoyonNum( resNum );
                    KitaThreadInfo::setReadNum( m_thread->datURL(), resNum );
                    update_finish();
                    gotoAnchor( datURL.ref().mid( 5 ) );
                    return ;

                case 3:
                    delete popupMenu;
                    str = threadName()
                          + "\n" + threadURL().url() + QString().setNum( resNum )
                          + "\n\n" + m_domtreeparent->getResStr( resNum, "" );
                    clipboard->setText( str , QClipboard::Clipboard );
                    clipboard->setText( str , QClipboard::Selection );
                    return ;

                default:
                    delete popupMenu;
                    return ;
                }

                delete popupMenu;

                slotWriteButtonClickedCore( resstr );

                return ;
            }

            /*---------------------------------*/

            else if ( datURL.ref().left( 7 ) == "tugi100" ) { /* show next 100 responses */
                if ( totalNum != bottom ) {
                    appendRes( bottom + 1, bottom + 100 );
                }
                gotoAnchor( QString().setNum( bottom ) );
            } else if ( datURL.ref().left( 6 ) == "nokori" ) {
                if ( totalNum != bottom ) {
                    appendRes( bottom + 1, totalNum );
                }
                gotoAnchor( QString().setNum( bottom ) );
            } else if ( datURL.ref().left( 7 ) == "tosaigo" ) {
                if ( totalNum != bottom ) {
                    appendRes( bottom + 1, totalNum );
                }
                gotoAnchor( "footer" );
            } else if ( datURL.ref().left( 6 ) == "mae100" ) {
                if ( top != 1 ) {
                    m_domtree->appendTemplate();
                    appendRes( top - 100, bottom );
                    gotoAnchor( QString().setNum( top ) );
                }
            } else if ( datURL.ref().left( 8 ) == "maezenbu" ) {
                if ( top != 1 ) {
                    appendRes( 1, bottom );
                    gotoAnchor( QString().setNum( top ) );
                }
            } else if ( datURL.ref().left( 6 ) == "tmp100" ) {
                int tmpnum = m_domtree->getTemplateNumber();

                if ( tmpnum < top ) {
                    m_domtree->setTemplateNumber( tmpnum + 100 );
                    m_domtree->appendTemplate();
                    appendRes( top, bottom );
                    gotoAnchor( QString().setNum( tmpnum ) );
                }
            } else if ( datURL.ref().left( 5 ) == "zenbu" ) {
                if ( top != 1 || bottom != totalNum ) {
                    appendRes( 1, totalNum );
                    gotoAnchor( "header" );
                }
            }

            /*---------------------------------------*/

            else { /* go to anchor */

                QString refstr = datURL.ref();
                int refNum, refNum2;

                int i = refstr.find( "-" );
                if ( i != -1 ) {
                    refNum = refstr.left( i ).toInt();
                    refNum2 = QMIN( refstr.mid( i + 1 ).toInt(), totalNum );
                    if ( refNum2 < refNum ) {
                        refNum2 = refNum;
                    }
                    refstr = datURL.ref().left( i );
                } else {
                    refNum = refstr.toInt();
                    refNum2 = refNum;
                }

                if ( refNum != 0 ) {

                    if ( refNum > totalNum ) return ;

                    /* data is not set... */
                    if ( !m_domtree->isResDataSet( refNum2 ) ) return ;
                }

                gotoAnchor( refstr );
            }
        }
        return ;
    }

    /*------------------*/
    /* open web browser */
    /*------------------*/

    /* This widget is not parent */
    /* So, call slotOpenURLRequest of the parent view again. */
    if ( m_viewmode != VIEWMODE_PARENT ) {
        m_viewparent->slotOpenURLRequest( url, args );
        return ;
    }

    emit openURLRequest( datURL, args );
}



/*-----------------------------------------*/
/* show the information at the statusbar.  */
/*-----------------------------------------*/
void KitaThreadView::showStatusBar( QString info )
{
    if ( m_viewmode != VIEWMODE_PARENT ) return ;

    int prevNum = m_domtree->getKokoyonNum();
    int totalNum = m_domtree->getMaxResNumber();
    bool broken = m_domtree->isBroken();

    QString errstr = QString::null;
    if ( m_rescode != 200 && m_rescode != 206 )
        errstr = QString( "Error %1" ).arg( m_rescode );
    if ( broken ) info += " This thread is broken.";

    if ( totalNum ) {
        /* show status bar */
        QString infostr = m_thread->name() +
                          QString( " [Total: %1 New: %2]" ).arg( totalNum ).arg( totalNum - prevNum )
                          + info + " " + errstr;

        emit signalChangeStatusbar( infostr );
        emit showThreadCompleted( m_thread->url() );
        topLevelWidget() ->setCaption( m_thread->name() );
    } else { /* DOM tree is not initialized */
        emit signalChangeStatusbar( errstr );
        topLevelWidget() ->setCaption( "" );
    }
}

KURL KitaThreadView::filterReadCGI( const KURL& url )
{
    return ( filterReadCGICore( url, m_thread ) );
}

/* TODO: need to write test case. */
KURL KitaThreadView::filterReadCGICore( const KURL& url, Kita::Thread* thread )
{
    KURL newURL = url;

    if ( url.path().contains( "/test/read.cgi" ) ) {
        newURL.setProtocol( KURL( thread->datURL() ).protocol() );
        QString tmp = url.path().section( "/test/read.cgi", 1 );

        QString newPath = QString( "/%1/dat/%2.dat" )
                          .arg( tmp.section( '/', 1, 1 ) )
                          .arg( tmp.section( '/', 2, 2 ) );
        newURL.setPath( newPath );

        QString refBase = tmp.section( '/', 3 );
        if ( ! refBase.isEmpty() ) {

            QString newRef;
            if ( refBase.at( 0 ) == '-' ) newRef = "1" + refBase;
            else newRef = refBase;

            newURL.setRef( newRef );

        }
    }
    kdDebug() << "newURL: " << newURL.url() << endl;
    return newURL;
}

/*----------------------------------*/
/* get reference numbers            */
/*                                  */
/* ex) if url = ..../12345.dat#5-10 */
/* then refNum = 5, refNum2 = 10    */
/*----------------------------------*/  /* !!! public & static !!! */
void KitaThreadView::getRefNumber( const KURL& url, Kita::Thread* thread,        /* input */
                                   int& refNum, int& refNum2 /* output */
                                 )
{

    refNum = 0;
    refNum2 = 0;
    QString urlstr = url.prettyURL();
    QString refstr;
    int i;

    /* get ref */
    if ( urlstr.at( 0 ) == '#' ) {
        refstr = urlstr.mid( 1 );
    } else {
        KURL datURL = filterReadCGICore(
                          KURL( thread->datURL(), urlstr ), thread );

        if ( datURL.host() == KURL( thread->datURL() ).host()
                && datURL.path() == KURL( thread->datURL() ).path() )
            refstr = datURL.ref();
        else return ;
    }

    /* get numbers */
    i = refstr.find( "-" );
    if ( i != -1 ) {
        refNum = refstr.left( i ).toInt();
        refNum2 = refstr.mid( i + 1 ).toInt();
        if ( refNum2 < refNum ) refNum2 = refNum;
    } else {
        refNum = refstr.toInt();
        refNum2 = refNum;
    }
}




/*----------------------*/
/* move to anchor       */
/*----------------------*/
void KitaThreadView::gotoAnchor( QString anc )
{

    int top = m_domtree->getTopResNumber();
    int bottom = m_domtree->getBottomResNumber();
    int maxres = m_domtree->getMaxResNumber();
    int res = anc.toInt();

    if ( res == 1 ) {
        anc = "header";
    } else if ( res > 1 ) {

        res = QMIN( maxres, res );
        if ( !m_domtree->isResShown( res ) ) {

            if ( res > bottom ) appendRes( bottom + 1, res + 99 );
            else if ( res < top ) {
                m_domtree->appendTemplate();
                appendRes( res, bottom );
            }
        }

        /* aboned */
        if ( !m_domtree->isResShown( res ) ) {
            while ( res > 0 ) {
                if ( m_domtree->isResShown( res ) ) break;
                res--;
            }
        }

        anc = QString().setNum( res );
    }

    m_threadPart->gotoAnchor( anc );
}

/*------------*/
/* read cache */
/*------------*/
void KitaThreadView::update_readCache()
{

    /* create dat loader */
    if ( m_access == NULL ) {
        m_access = new Kita::Access( m_thread );
        connect( m_access, SIGNAL( redirection( const QString& ) ),
                 SIGNAL( setLocationBarURL( const QString& ) ) );
    }

    /* get log from cahce & copy to dom */
    QString logline = m_access->getcache();
    if ( logline.length() ) {
        update_copydata( logline, 1 );
        m_domtree->parseAllRes();
    }
}

/*-----------------------*/
/* copy data to dom tree */
/*-----------------------*/
void KitaThreadView::update_copydata( const QString& linedata,
                                      int basenum )
{
    int num = basenum;
    QStringList lines = QStringList::split( "\n", linedata );

    m_domtree->StopParseThread();
    for ( QStringList::iterator it = lines.begin(); it != lines.end(); ++it ) {
        m_domtree->setDat( ( *it ) , num );
        num++;
    }
}

/*-------------------------*/
/* update & rendering      */
/*-------------------------*/
void KitaThreadView::update_rendering( bool breload )
{
    if ( topLevelWidget() ->isMinimized() ) {
        topLevelWidget() ->showNormal();
    }
    topLevelWidget() ->raise();
    setActiveWindow();

    /* show clock cursor */
    QCursor qc; qc.setShape( Qt::WaitCursor );
    QApplication::setOverrideCursor( qc );

    showStatusBar( QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_NOWRENEW ) );

    /* get new data from server & copy */
    int maxres = m_domtree->getMaxResNumber();
    m_rescode = 200;
    if ( breload && m_access != NULL ) {
        QString updateline = m_access->getupdate();

        m_rescode = m_access->responseCode();
        if ( m_rescode == 200 || m_rescode == 206 ) {
            if ( updateline.length() > 1 ) update_copydata( updateline, maxres + 1 );
        }
    }
    m_domtree->StopParseThread();

    maxres = m_domtree->getMaxResNumber();
    int prevReadNum = KitaThreadInfo::readNum( m_thread->datURL() );
    int bottom = m_domtree->getBottomResNumber();

    if ( maxres == 0 ) {
        showStatusBar( "" );
        QApplication::restoreOverrideCursor();
        return ;
    }

    /* This dat file is destroyed */
    if ( maxres < prevReadNum ) {
        prevReadNum = maxres;
        bottom = -1;
    }

    /* This thread is not shown */
    if ( bottom < 0 ) {
        bottom = QMAX( 0, prevReadNum - m_preShowNum );
        m_domtree->appendTemplate();
    }

    /* bottom <= prevReadNum <= maxres */

    /* show resposes from bottom to prevReadNum */
    if ( bottom != prevReadNum ) {
        for ( int i = bottom + 1 ; i <= prevReadNum ; i++ )
            m_domtree->appendRes( i, FALSE );
    }

    /* bottom = prevReadNum <= maxres */

    m_domtree->setKokoyonNum( prevReadNum );

    /* show updated responses  */
    if ( prevReadNum < maxres ) {
        for ( int i = prevReadNum + 1 ; i <= QMIN( prevReadNum + m_afterShowNum, maxres ); i++ )
            m_domtree->appendRes( i, FALSE );
    }

    /* prevReadNum <= bottom <= maxres */

    m_thread->setResNum( maxres );
    KitaThreadInfo::setReadNum( m_thread->datURL(), maxres );
    update_finish();

    m_threadPart->gotoKokoyon();
    m_domtree->parseAllRes();

    QApplication::restoreOverrideCursor();
}

/*-------------------*/
/* finish rendering  */
/*-------------------*/
void KitaThreadView::update_finish()
{
    int totalNum = m_domtree->getMaxResNumber();

    /* set header, footer , etc. */
    m_domtree->appendMae100();
    m_domtree->appendTugi100();
    m_domtree->appendKokoyon();
    m_domtree->appendFooter( totalNum );

    /* update display */
    m_threadPart->htmlDocument().applyChanges();
    m_threadPart->view() ->layout();
    m_threadPart->view() ->setVScrollBarMode( QScrollView::AlwaysOn );

    /* uptate informations */
    showStatusBar( "" );
    Kita::Thread::setName( m_thread->datURL(), m_domtree->getSubject() );
    emit thread( m_thread );
    setSubjectLabel( m_thread->boardName(), m_thread->name() );
    m_domtree->findTextInit();
    updateButton();
    emit showThreadCompleted( m_thread->url() );
    m_threadPart->view() ->setFocus();
}

/*-------------------*/
/* append  responses */
/*-------------------*/
void KitaThreadView::appendRes( int startnum, int endnum )
{

    showStatusBar( QTextCodec::codecForName( "utf8" ) ->toUnicode( KITAUTF8_NOWEXPAND ) );

    startnum = QMAX( 1, startnum );
    endnum = QMIN( m_domtree->getMaxResNumber(), endnum );

    QCursor qc; qc.setShape( Qt::WaitCursor );
    QApplication::setOverrideCursor( qc );

    for ( int i = startnum ; i <= endnum; i++ ) m_domtree->appendRes( i, FALSE );

    update_finish();
    QApplication::restoreOverrideCursor();
}

/*---------------------*/
/* open popup window   */
/*---------------------*/
void KitaThreadView::showPopup( QString innerHTML, QString imgfile )
{

    if ( m_popup ) {
        delete m_popup;
        m_popup = NULL;
    }

    /* text popup */
    if ( imgfile == NULL ) {
        m_popup = new Kita::ResPopup( m_threadPart->view() );
        m_popup->setText( innerHTML );
        m_popup->adjustSize();
    }

    connect( m_threadPart->view(), SIGNAL( leave() ), SLOT( hidePopup() ) );

    QRect qr = QApplication::desktop() ->rect();
    int sw = qr.width(), sh = qr.height();
    int pw = m_popup->width(), ph = m_popup->height();
    const int mrg = 10;

    QPoint pos = QCursor::pos();
    int x = pos.x(), y = pos.y();

    if ( x + mrg + pw < sw ) {
        x = x + mrg;
        y = QMAX( 0, y - ( ph + mrg ) );
    } else if ( y - ( mrg + ph ) >= 0 ) {
        x = QMAX( 0, sw - pw );
        y = y - ( ph + mrg );
    } else if ( y + ( mrg + ph ) <= sh ) {
        x = QMAX( 0, sw - pw );
        y = y + mrg;
    } else if ( x - ( mrg + pw ) >= 0 ) {
        x = x - ( mrg + pw );
        y = QMAX( 0, y - ( ph + mrg ) );
    } else {
        x = x + mrg;
        y = QMAX( 0, y - ( ph + mrg ) );
    }

    pos.setX( x );
    pos.setY( y );

    m_popup->move( pos );
    m_popup->show();

}

/*------------------------------------------------------*/
/* This function is called when the mouse is on the URL */
/*------------------------------------------------------*/ /* private slots */
void KitaThreadView::slotOnURL( const QString& url )
{
    if ( m_domtreeparent == NULL ) return ; /* This widget is on the ImagekitaNavi.*/

    if ( !isActiveWindow() ) return ;

    if ( !url.isEmpty() ) {

        if ( m_popup ) {
            delete m_popup;
            m_popup = NULL;
        }

        int refNum = 0;
        int refNum2 = 0;
        int totalNum = m_domtreeparent->getMaxResNumber();
        QString refstr;
        QString innerHTML;
        QString imgfile = QString::null;

        if ( url.left( 6 ) == "#idpop" ) {	/* show id popup */
            int num = m_domtreeparent->getNumByID( url.mid( 6 ) );
            if ( num <= 1 ) return ;
            innerHTML = QString( "<DIV>ID:%1:[%2]<BR><BR></DIV>" ).arg( url.mid( 6 ) ).arg( num );

            refNum = 1;  /* dummy , meaningless */
        } else { /* show popup */

            if ( url.at( 0 ) == '#' ) refstr = url.mid( 1 );
            else if ( url.left( 7 ) == "mailto:" ) {
                DOM::Node node = m_threadPart->nodeUnderMouse().firstChild();
                QString nodeValue = node.nodeValue().string();
                refstr = nodeValue;
            } else {

                KURL datURL = filterReadCGI( KURL( m_thread->datURL(), url ) );

                if ( datURL.host() == KURL( m_thread->datURL() ).host()
                        && datURL.path() == KURL( m_thread->datURL() ).path() )
                    refstr = datURL.ref();
                else return ; /* TODO: show the name of the another thread */

            }

            int i = refstr.find( "-" );
            if ( i != -1 ) {

                refNum = refstr.left( i ).toInt();
                refNum2 = QMIN( refstr.mid( i + 1 ).toInt(), totalNum );
                if ( refNum > totalNum ) {
                    return ;
                }
                if ( refNum2 < refNum ) {
                    refNum2 = refNum;
                }

                if ( refNum ) {
                    if ( refNum2 - refNum > m_maxpopup - 1 ) {
                        refNum2 = refNum + m_maxpopup - 1;
                    }
                    innerHTML = m_domtreeparent->getHtml( refNum, refNum2 );
                }
            } else {
                refNum = refstr.toInt();
                refNum2 = refNum;
                if ( refNum > totalNum ) {
                    return ;
                }
                if ( refNum ) {
                    innerHTML = m_domtreeparent->getHtml( refNum, refNum2 );
                }
            }
        }

        if ( refNum && innerHTML != NULL ) showPopup( innerHTML, imgfile );

    } else if ( m_popup != NULL ) m_popup->hide();

}

/*------------------------*/
/* search function        */
/*------------------------*/  /* private slots */
void KitaThreadView::slotSearchButton()
{
    if ( m_access == NULL ) return ;
    if ( m_viewmode == VIEWMODE_PREVIEW ) return ;

    QString str = SearchCombo->currentText();
    if ( str.at( 0 ) == ':' ) {
        QString anc = str.mid( 1 );
        gotoAnchor( anc );
        SearchCombo->setFocus();
        return ;
    }

    /* if this is parent, then show all responses, and search */
    if ( m_viewmode == VIEWMODE_PARENT ) {
        int top = m_domtree->getTopResNumber();
        int bottom = m_domtree->getBottomResNumber();
        int totalNum = m_domtree->getMaxResNumber();
        if ( top != 1 || bottom != totalNum ) {
            appendRes( 1, totalNum );
        }
    }

    insertSearchCombo();
    QStringList list = parseSearchQuery( SearchCombo->currentText() );
    m_domtree->findText( SearchCombo->currentText(), m_revsearch );
    SearchCombo->setFocus();
}
// vim:sw=2:



/*--------------------------------------------------------*/
/* These public functions are used for KitaNavi & preview */
/*--------------------------------------------------------*/

/*---------------------*/
/* set data & parse it */
/*---------------------*/ /* public */
void KitaThreadView::domAppendData( QString& str, int bufnum , int num )
{
    m_domtree->setDat( str , bufnum );
    m_domtree->parseRes( bufnum, num );
    m_domtree->appendRes( bufnum, FALSE );
}


/* public */
void KitaThreadView::domDeleteRes( int bufnum )
{
    m_domtree->removeRes( bufnum );
}


/* public */
void KitaThreadView::domAppendBelt( QString idstr )
{
    m_domtree->createBeltNode( idstr );
}

/* public */
void KitaThreadView::domAppendHTML( QString& url, QString& html )
{

    m_threadPart->begin( url );
    m_threadPart->write( html );
    m_threadPart->end();
    m_threadPart->show();

}

/* public */
void KitaThreadView::domApplyChange( QString lbstr,
                                     QString hstr,
                                     QString fstr,
                                     QString anchor,
                                     QString cmbstr )
{

    /* label, header & footer */
    if ( hstr != NULL ) m_domtree->createCommentNode( hstr, "header", 0, 2, TRUE );
    if ( fstr != NULL ) m_domtree->createCommentNode( fstr, "footer", 0, 0, FALSE );
    if ( lbstr != NULL ) subjectLabel->setText( lbstr );

    /* update display */
    m_threadPart->htmlDocument().applyChanges();
    m_threadPart->view() ->layout();
    m_threadPart->view() ->setVScrollBarMode( QScrollView::AlwaysOn );
    m_threadPart->view() ->setFocus();

    GobackAnchorButton->setEnabled( FALSE );
    writeButton->setEnabled( FALSE );
    BookmarkButton->setEnabled( FALSE );
    ReloadButton->setEnabled( FALSE );
    if ( m_viewmode == VIEWMODE_PREVIEW ) {
        SearchCombo->setEnabled( FALSE );
        SearchButton->setEnabled( FALSE );
        HighLightButton->setEnabled( FALSE );
    }

    if ( cmbstr != NULL ) SearchCombo->insertItem( cmbstr );
    if ( anchor != NULL ) gotoAnchor( anchor );
}
