/*
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version. This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details. You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.l2jserver.gameserver.script.faenor;
//package net.sf.l2j.gameserver.script.faenor;

import java.sql.PreparedStatement;
import java.util.Date;
import java.util.logging.Logger;

import javax.script.ScriptContext;

import javolution.text.TypeFormat;

import org.w3c.dom.Node;

import com.l2jserver.L2DatabaseFactory;
import com.l2jserver.gameserver.datatables.NpcTable;
import com.l2jserver.gameserver.datatables.SpawnTable;
import com.l2jserver.gameserver.model.L2Spawn;
import com.l2jserver.gameserver.script.DateRange;
import com.l2jserver.gameserver.script.IntList;
import com.l2jserver.gameserver.script.Parser;
import com.l2jserver.gameserver.script.ParserFactory;
import com.l2jserver.gameserver.script.ScriptEngine;
import com.l2jserver.gameserver.templates.chars.L2NpcTemplate;


/**
 * @author Luis Arias
 * http://my-svn.assembla.com/svn/100NT/duoTM/java/net/sf/l2j/gameserver/script/faenor/FaenorEventParser.java
 */
public class FaenorEventParserEx extends FaenorParser
{
	static class FaenorEventParserFactory extends ParserFactory
	{

		@Override
		public Parser create()
		{
			return (new FaenorEventParser());
		}
	}
	static Logger _log = Logger.getLogger(FaenorEventParser.class.getName());

	private DateRange _eventDates = null;

	static
	{
		ScriptEngine.parserFactories.put(getParserName("Event"), new FaenorEventParserFactory());
	}

	private void doCleanAllPlayers(final Node cleanNode)
	{
		try
		{
			for (Node node = cleanNode.getFirstChild(); node != null; node = node.getNextSibling())
				if (isNodeName(node, "ItemsRange"))
				{
					_log.warning("Event cleaning : ItemsRange.");
					final int minItemID = TypeFormat.parseInt(attribute(node, "MinItemID"));
					final int maxItemID = TypeFormat.parseInt(attribute(node, "MaxItemID"));
					sql_DeleteItemsToItemIdRange(minItemID, maxItemID);
				}
				else if (isNodeName(node, "Item"))
				{
					_log.warning("Event cleaning : Item.");
					final int ItemID = TypeFormat.parseInt(attribute(node, "ItemID"));
					sql_DeleteItemsToItemId(ItemID);
				}
		}
		catch (final Exception e)
		{
			_log.warning("Error in event parser.");
			e.printStackTrace();
		}
	}

	private void parceSpawnNpc(final Node spawnNode)
	{
		try
		{
			final int npcid = TypeFormat.parseInt(attribute(spawnNode, "NpcId"));
			final int coordinatesX = TypeFormat.parseInt(attribute(spawnNode, "SpawnX"));
			final int coordinatesY = TypeFormat.parseInt(attribute(spawnNode, "SpawnY"));
			final int coordinatesZ = TypeFormat.parseInt(attribute(spawnNode, "SpawnZ"));
			final int head = TypeFormat.parseInt(attribute(spawnNode, "Heading"));
			final int respawn_delay = TypeFormat.parseInt(attribute(spawnNode, "RespawnDelay"));

			L2NpcTemplate template1;
			template1 = NpcTable.getInstance().getTemplate(npcid);
			if (template1 == null)
				return;
			final L2Spawn spawn = new L2Spawn(template1);
		// final L2Spawn spawn = L2Spawn.valueOf(template1);	//-
			spawn.setLocx(coordinatesX);
			spawn.setLocy(coordinatesY);
			spawn.setLocz(coordinatesZ);
			spawn.setAmount(1);
			spawn.setHeading(head);
			spawn.setRespawnDelay(respawn_delay);
			spawn.init();
			SpawnTable.getInstance().addNewSpawn(spawn, false);
			if (respawn_delay <= 0)
				spawn.stopRespawn();
		}
		catch (final Exception e)
		{
			_log.warning("Error in event parser.");
			e.printStackTrace();
		}
	}

	private void parseEventClean(final Node CommandNode)
	{
		try
		{
			for (Node node = CommandNode.getFirstChild(); node != null; node = node.getNextSibling())
				if (isNodeName(node, "Take")
						&& attribute(node, "From").equalsIgnoreCase("AllPlayers"))
					doCleanAllPlayers(node);
		}
		catch (final Exception e)
		{
			_log.warning("Error in event parser.");
			e.printStackTrace();
		}
	}

	private void parseEventDropList(final Node dropList)
	{
		if (DEBUG)
			_log.fine("Parsing Droplist.");

		DateRange _activeDate = null;
		_activeDate = DateRange.parse(attribute(dropList, "Active"), DATE_FORMAT);
		final Date currentDate = new Date();

		if (!_activeDate.isWithinRange(currentDate))
		{
			_log.warning("FaenorEventParser: Drops has disabled! Out of date!");
			return;
		}

		for (Node node = dropList.getFirstChild(); node != null; node = node.getNextSibling())
			if (isNodeName(node, "AllDrop"))
				parseForAllEventDrop(node);
			else if (isNodeName(node, "NpcsDrop"))
				parseForNpcsEventDrop(node);
	}

	private void parseEventMessage(final Node sysMsg)
	{
		if (DEBUG)
			_log.fine("Parsing Event Message.");

		DateRange _activeDate = null;
		_activeDate = DateRange.parse(attribute(sysMsg, "Active"), DATE_FORMAT);
		final Date currentDate = new Date();

		if (!_activeDate.isWithinRange(currentDate))
			return;

		try
		{
			final String type = attribute(sysMsg, "Type");
			final String[] message = attribute(sysMsg, "Msg").split("\n");

			if (type.equalsIgnoreCase("OnJoin"))
				_bridge.onPlayerLogin(message, _eventDates);
		}
		catch (final Exception e)
		{
			_log.warning("Error in event parser.");
			e.printStackTrace();
		}
	}

	private void parseForAllEventDrop(final Node drop)
	{
		if (DEBUG)
			_log.fine("Parsing Drop.");

		try
		{
			final int[] items = IntList.parse(attribute(drop, "Items"));
			final int[] count = IntList.parse(attribute(drop, "Count"));
			final double chance = getPercent(attribute(drop, "Chance"));

			_bridge.addEventDrop(items, count, chance, _eventDates/*, -1*/);
		}
		catch (final Exception e)
		{
			System.err.println("ERROR(parseEventDrop):" + e.getMessage());
		}
	}

	private void parseForNpcsEventDrop(final Node drop)
	{
		throw new UnsupportedOperationException();	//TODO:
	//	if (DEBUG)
	//		_log.fine("Parsing for npcs Drop.");
	//
	//	try
	//	{
	//		final int[] items = IntList.parse(attribute(drop, "Items"));
	//		final int[] count = IntList.parse(attribute(drop, "Count"));
	//		final double chance = getPercent(attribute(drop, "Chance"));
	//		final int[] npcs = IntList.parse(attribute(drop, "Npcs"));
    //
	//		for (final int npc : npcs)
	//		{
	//			if (npc <= 0)
	//				continue;
    //
	//			_bridge.addEventDrop(items, count, chance, _eventDates, npc);
	//		}
	//	}
	//	catch (final Exception e)
	//	{
	//		System.err.println("ERROR(parseEventDrop):" + e.getMessage());
	//	}
	}

	@Override
	public void parseScript(final Node eventNode, final ScriptContext context)
	{
		final String ID = attribute(eventNode, "ID");

		if (DEBUG)
			_log.fine("Parsing Event \"" + ID + "\"");

		_eventDates = DateRange.parse(attribute(eventNode, "Active"), DATE_FORMAT);

		final Date currentDate = new Date();
		if (!_eventDates.isWithinRange(currentDate))
		{
			_log.warning("Event ID: (" + ID + ") has passed... Ignored.");
			for (Node node = eventNode.getFirstChild(); node != null; node = node.getNextSibling())
				if (isNodeName(node, "EventEnd"))
					parseEventClean(node);
			return;
		}

		_log.warning("Event ID: (" + ID + ") ... Started.");
		for (Node node = eventNode.getFirstChild(); node != null; node = node.getNextSibling())
			if (isNodeName(node, "DropList"))
				parseEventDropList(node);
			else if (isNodeName(node, "Message"))
				parseEventMessage(node);
			else if (isNodeName(node, "SpawnList"))
				parseSpawnList(node);
	}

	private void parseSpawnList(final Node spawnNode)
	{
		try
		{
			for (Node node = spawnNode.getFirstChild(); node != null; node = node.getNextSibling())
				if (isNodeName(node, "Spawn"))
					parceSpawnNpc(node);
		}
		catch (final Exception e)
		{
			_log.warning("Error in event parser.");
			e.printStackTrace();
		}
	}

	private void sql_DeleteItemsToItemId(final int _itemId)
	{
		java.sql.Connection con = null;
		try
		{
			con = L2DatabaseFactory.getInstance().getConnection();
			final PreparedStatement statement = con.prepareStatement("DELETE FROM items WHERE item_id=?");
			statement.setInt(1, _itemId);
			statement.executeUpdate();
			statement.close();
		}
		catch (final Exception e)
		{
			_log.warning("Could not delete item " + _itemId + " in DB: "
					+ e.getMessage());
		}
		finally
		{
			L2DatabaseFactory.close(con);
		}
	}

	private void sql_DeleteItemsToItemIdRange(final int _itemIdMin,
			final int _itemIdMax)
	{
		java.sql.Connection con = null;
		try
		{
			con = L2DatabaseFactory.getInstance().getConnection();
			final PreparedStatement statement = con.prepareStatement("DELETE FROM items WHERE item_id>=? AND item_id<=?");
			statement.setInt(1, _itemIdMin);
			statement.setInt(2, _itemIdMax);
			statement.executeUpdate();
			statement.close();
		}
		catch (final Exception e)
		{
			_log.warning("Could not delete items " + _itemIdMin + " <> "
					+ _itemIdMax + " in DB: " + e.getMessage());
		}
		finally
		{
			L2DatabaseFactory.close(con);
		}
	}
}
