/*
 * Copyright (C) 2004-2014 L2J DataPack
 * 
 * This file is part of L2J DataPack.
 * 
 * L2J DataPack is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * L2J DataPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package ai.npc.ForgeOfTheGods;

import java.util.ArrayList;

import jp.sf.l2j.arrayMaps.SortedIntObjectArrayMap;
import jp.sf.l2j.troja.FastIntObjectMap;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import com.l2jserver.gameserver.GeoData;
import com.l2jserver.gameserver.ThreadPoolManager;
import com.l2jserver.gameserver.datatables.NpcData;
import com.l2jserver.gameserver.engines.DocumentParser;
import com.l2jserver.gameserver.model.L2Spawn;
import com.l2jserver.gameserver.model.L2Territory;
import com.l2jserver.gameserver.model.actor.L2Npc;
import com.l2jserver.util.Rnd;

/**
 * Tar Beetle zone spawn
 * @author malyelfik
 */
public class TarBeetleSpawn extends DocumentParser
{
	static final int REMAINING_BULLETS = 5;
	
	private static final SortedIntObjectArrayMap<SpawnZone> allZones = new SortedIntObjectArrayMap<>();
	static final ArrayList<SpawnZone> lowerZones = new ArrayList<>();
	static final ArrayList<SpawnZone> upperZones = new ArrayList<>();
	
	static final FastIntObjectMap<L2Npc> lowerNpcs = new FastIntObjectMap<L2Npc>().shared();
	static final FastIntObjectMap<L2Npc> upperNpcs = new FastIntObjectMap<L2Npc>().shared();
	
	TarBeetleSpawn()
	{
		load();
	}
	
	@Override
	public void load()
	{
		allZones.clear();
		lowerZones.clear();
		upperZones.clear();
		lowerNpcs.clear();
		upperNpcs.clear();
		parseDatapackFile("data/spawnZones/tar_beetle.xml");
		_log.info(TarBeetleSpawn.class.getSimpleName() + ": Loaded " + allZones.size() + " spawn zones.");
	}
	
	@Override
	protected void parseDocument()
	{
		final Node n = getCurrentDocument().getFirstChild();
		for (Node d = n.getFirstChild(); d != null; d = d.getNextSibling())
		{
			if (d.getNodeName().equals("spawnZones"))
			{
				for (Node r = d.getFirstChild(); r != null; r = r.getNextSibling())
				{
					if (r.getNodeName().equals("zone"))
					{
						NamedNodeMap attrs = r.getAttributes();
						int id = parseInt(attrs, "id");
						int minZ = parseInt(attrs, "minZ");
						int maxZ = parseInt(attrs, "maxZ");
						String type = parseString(attrs, "type");
						
						int[] bZones = null;
						String bZonesStr = parseString(attrs, "bZones", "");
						if (!bZonesStr.isEmpty())
						{
							String[] str = bZonesStr.split(";");
							bZones = new int[str.length];
							for (int i = 0; i < str.length; i++)
							{
								bZones[i] = Integer.parseInt(str[i]);
							}
						}
						
						SpawnZone zone = new SpawnZone(id, bZones);
						for (Node c = r.getFirstChild(); c != null; c = c.getNextSibling())
						{
							if (c.getNodeName().equals("point"))
							{
								attrs = c.getAttributes();
								int x = parseInt(attrs, "x");
								int y = parseInt(attrs, "y");
								zone.add(x, y, minZ, maxZ, 0);
							}
						}
						allZones.put(id, zone);
						
						if (type.equals("lower"))
						{
							lowerZones.add(zone);
						}
						else if (type.equals("upper"))
						{
							upperZones.add(zone);
						}
					}
				}
			}
		}
	}
	
	void removeBeetle(L2Npc npc)
	{
		npc.deleteMe();
		if (lowerNpcs.remove(npc.getObjectId()) == null)
			upperNpcs.remove(npc.getObjectId());
	}
	
	L2Npc spawn(ArrayList<SpawnZone> zones)
	{
		try
		{
			int[] loc = zones.get(Rnd.get(zones.size())).getRandomPoint();
			
			final L2Spawn spawn = new L2Spawn(NpcData.getInstance().getTemplate(18804));
			spawn.setHeading(Rnd.get(65536));
			spawn.setX(loc[0]);
			spawn.setY(loc[1]);
			spawn.setZ(GeoData.getInstance().getSpawnHeight(loc[0], loc[1], loc[2], loc[3]));
			spawn.setIsNoRndWalk(true);
			
			final L2Npc npc = spawn.doSpawn();
			npc.setIsImmobilized(true);
			npc.setIsInvul(true);
			npc.disableCoreAI(true);
			npc.setScriptValue(REMAINING_BULLETS);
			
			return npc;
		}
		catch (Exception e)
		{
			_log.warning(TarBeetleSpawn.class.getSimpleName() + ": Could not spawn npc! Error: " + e.getMessage());
			return null;
		}
	}
	
	void startTasks()
	{
		ThreadPoolManager.getInstance().scheduleGeneralAtFixedRate(new SpawnTask(), 1000, 60000);
		ThreadPoolManager.getInstance().scheduleGeneralAtFixedRate(new NumShotTask(), 300000, 300000);
	}
	
	SpawnZone getSpawnZoneById(int id)
	{
		return allZones.get(id);
	}
	
	private class SpawnZone extends L2Territory
	{
		private final int[] _bZones;
		
		public SpawnZone(int terr, int[] bZones)
		{
			super(terr);
			_bZones = bZones;
		}
		
		@Override
		public int[] getRandomPoint()
		{
			int[] loc;
			do loc = super.getRandomPoint(); while (isInsideBannedZone(loc));
			return loc;
		}
		
		private boolean isInsideBannedZone(int[] loc)
		{
			if (_bZones != null)
			{
				for (int i : _bZones)
				{
					if (getSpawnZoneById(i).isInside(loc[0], loc[1]))
					{
						return true;
					}
				}
			}
			return false;
		}
	}
	
	class SpawnTask implements Runnable
	{
		@Override
		public void run()
		{
			while (lowerNpcs.size() < 4)
			{
				L2Npc npc = spawn(lowerZones);
				lowerNpcs.put(npc.getObjectId(), npc);
			}
			
			while (upperNpcs.size() < 12)
			{
				L2Npc npc = spawn(upperZones);
				upperNpcs.put(npc.getObjectId(), npc);
			}
		}
	}
	
	class NumShotTask implements Runnable
	{
		@Override
		public void run()
		{
			recovery(lowerNpcs);
			recovery(upperNpcs);
		}
		private void recovery(FastIntObjectMap<L2Npc> npcs)
		{
			for (L2Npc npc : npcs.values())
			{
				int val = npc.getScriptValue() + 1;
				if (val > REMAINING_BULLETS)
				{
					removeBeetle(npc);
				}
				else
				{
					npc.setScriptValue(val);
				}
			}
		}
	}
}