/*
 * Copyright (C) 2004-2014 L2J DataPack
 * 
 * This file is part of L2J DataPack.
 * 
 * L2J DataPack is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * L2J DataPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package handlers.admincommandhandlers;

import java.io.File;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import javax.script.ScriptException;

import com.l2jserver.gameserver.handler.IAdminCommandHandler;
import com.l2jserver.gameserver.instancemanager.QuestManager;
import com.l2jserver.gameserver.model.actor.L2Character;
import com.l2jserver.gameserver.model.actor.instance.L2PcInstance;
import com.l2jserver.gameserver.model.events.EventType;
import com.l2jserver.gameserver.model.events.ListenerRegisterType;
import com.l2jserver.gameserver.model.events.listeners.AbstractEventListener;
import com.l2jserver.gameserver.model.quest.Quest;
import com.l2jserver.gameserver.model.quest.QuestTimer;
import com.l2jserver.gameserver.network.serverpackets.NpcHtmlMessage;
import com.l2jserver.gameserver.scripting.L2ScriptEngineManager;
import com.l2jserver.gameserver.util.Util;

public class AdminQuest implements IAdminCommandHandler
{
	private static final String[] ADMIN_COMMANDS =
	{
		"admin_quest_reload",
		"admin_script_load",
		"admin_script_unload",
		"admin_show_quests",
		"admin_quest_info"
	};
	
	@Override
	public boolean useAdminCommand(String command, L2PcInstance activeChar)
	{
		if (activeChar == null)
		{
			return false;
		}
		
		// syntax will either be:
		// //quest_reload <id>
		// //quest_reload <questName>
		// The questName MUST start with a non-numeric character for this to work,
		// regardless which of the two formats is used.
		// Example: //quest_reload orc_occupation_change_1
		// Example: //quest_reload chests
		// Example: //quest_reload SagasSuperclass
		// Example: //quest_reload 12
		if (command.startsWith("admin_quest_reload"))
		{
			String[] parts = command.split(" ");
			if (parts.length < 2)
			{
				activeChar.sendMessage("Usage: //quest_reload <questFolder>.<questSubFolders...>.questName> or //quest_reload <id>");
			}
			else
			{
				// try the first param as id
				try
				{
					int questId = Integer.parseInt(parts[1]);
					if (QuestManager.getInstance().reload(questId))
					{
						activeChar.sendMessage("Quest Reloaded Successfully.");
					}
					else
					{
						activeChar.sendMessage("Quest Reloaded Failed");
					}
				}
				catch (NumberFormatException e)
				{
					if (QuestManager.getInstance().reload(parts[1]))
					{
						activeChar.sendMessage("Quest Reloaded Successfully.");
					}
					else
					{
						activeChar.sendMessage("Quest Reloaded Failed");
					}
				}
			}
		}
		// script load should NOT be used in place of reload. If a script is already loaded
		// successfully, quest_reload ought to be used. The script_load command should only
		// be used for scripts that failed to load altogether (eg. due to errors) or that
		// did not at all exist during server boot. Using script_load to re-load a previously
		// loaded script may cause unpredictable script flow, minor loss of data, and more.
		// This provides a way to load new scripts without having to reboot the server.
		else if (command.startsWith("admin_script_load"))
		{
			String[] parts = command.split(" ");
			if (parts.length < 2)
			{
				// activeChar.sendMessage("Example: //script_load <questFolder>/<questSubFolders...>/<filename>.<ext> ");
				activeChar.sendMessage("Example: //script_load quests/SagasSuperclass/__init__.py");
			}
			else
			{
				File file = new File(L2ScriptEngineManager.SCRIPT_FOLDER, parts[1]);
				// Trying to reload by script name.
				if (!file.exists())
				{
					Quest quest = QuestManager.getInstance().getQuest(parts[1]);
					if (quest != null)
					{
						file = new File(L2ScriptEngineManager.SCRIPT_FOLDER, quest.getClass().getName().replaceAll("\\.", "/") + ".java");
					}
				}
				
				// Reloading by full path
				if (file.isFile())
				{
					try
					{
						L2ScriptEngineManager.getInstance().executeScript(file);
						
						// This part should be called only when the script is successfuly loaded.
						activeChar.sendMessage("Script Successfully Loaded.");
					}
					catch (ScriptException e)
					{
						activeChar.sendMessage("Failed loading: " + parts[1]);
						L2ScriptEngineManager.getInstance().reportScriptFileError(file, e);
					}
					catch (Exception e)
					{
						activeChar.sendMessage("Failed loading: " + parts[1]);
					}
				}
				else
				{
					activeChar.sendMessage("File Not Found: " + parts[1]);
				}
			}
			
		}
		else if (command.startsWith("admin_script_unload"))
		{
			String[] parts = command.split(" ");
			if (parts.length < 2)
			{
				activeChar.sendMessage("Example: //script_unload questName/questId");
			}
			else
			{
				Quest q = Util.isDigit(parts[1]) ? QuestManager.getInstance().getQuest(Integer.parseInt(parts[1])) : QuestManager.getInstance().getQuest(parts[1]);
				
				if (q != null)
				{
					if (q.unload())
					{
						activeChar.sendMessage("Script Successfully Unloaded [" + q.getName() + "/" + q.getId() + "]");
					}
					else
					{
						activeChar.sendMessage("Failed unloading [" + q.getName() + "/" + q.getId() + "].");
					}
				}
				else
				{
					activeChar.sendMessage("The quest [" + parts[1] + "] was not found!.");
				}
			}
		}
		else if (command.startsWith("admin_show_quests"))
		{
			if (activeChar.getTarget() == null)
			{
				activeChar.sendMessage("Get a target first.");
			}
			else if (!activeChar.getTarget().isCharacter())
			{
				activeChar.sendMessage("Invalid Target.");
			}
			else
			{
				final L2Character character = (L2Character) activeChar.getTarget();
				final StringBuilder sb = new StringBuilder(256);
				final Set<String> questNames = new TreeSet<>();
				for (EventType type : EventType.values())
				{
					for (AbstractEventListener listener : character.getListeners(type))
					{
						if (listener.getOwner() instanceof Quest)
						{
							final Quest quest = (Quest) listener.getOwner();
							if (questNames.contains(quest.getName()))
							{
								continue;
							}
							sb.append("<tr><td><font color=LEVEL><a action=\"bypass -h admin_quest_info ").append(quest.getName()).append("\">").append(quest.getName()).append("</a></font></td></tr>");
							questNames.add(quest.getName());
						}
					}
				}
				
				final NpcHtmlMessage msg = new NpcHtmlMessage(0, 1);
				msg.setFile(activeChar.getHtmlPrefix(), "data/html/admin/npc-quests.htm");
				msg.replace("%quests%", sb.toString());
				msg.replace("%objid%", character.getObjectId());
				msg.replace("%questName%", "");
				activeChar.sendPacket(msg);
			}
		}
		else if (command.startsWith("admin_quest_info "))
		{
			final String questName = command.substring("admin_quest_info ".length());
			final Quest quest = QuestManager.getInstance().getQuest(questName);
			String events = "", npcs = "", items = "", timers = "";
			int counter, size;
			if (quest == null)
			{
				activeChar.sendMessage("Couldn't find quest or script with name " + questName + " !");
				return false;
			}
			
			// Events:
			final Set<String> listenerTypes = new TreeSet<>();
			for (AbstractEventListener t : quest.getListeners())
				listenerTypes.add(t.getType().name());
			counter = 0; size = listenerTypes.size();
			for (String eventType : listenerTypes)
			{
				if (++counter > 10 && counter < size)
				{
					events += ", ...";
					break;
				}
				events += ", " + eventType;
			}
			if (!events.isEmpty())
			{
				events = "(" + size + ") " + events.substring(2);
			}
			
			// NPCs:
			final Set<Integer> npcIds = new TreeSet<>(quest.getRegisteredIds(ListenerRegisterType.NPC));
			counter = 0; size = npcIds.size();
			for (int npcId : npcIds)
			{
				if (++counter > 24 && counter < size)
				{
					npcs += ", ...";
					break;
				}
				npcs += ", " + npcId;
			}
			if (!npcs.isEmpty())
			{
				npcs = "(" + size + ") " + npcs.substring(2);
			}
			
			// Items:
			if (quest.getRegisteredItemIds() != null)
			{
				counter = 0; size = quest.getRegisteredItemIds().length;
				for (int itemId : quest.getRegisteredItemIds())
				{
					if (++counter > 24 && counter < size)
					{
						items += ", ...";
						break;
					}
					items += ", " + itemId;
				}
				items = "(" + size + ") " + items.substring(2);
			}
			
			// Timers:
			counter = 0; size = quest.getQuestTimers().size();
			for (List<QuestTimer> list : quest.getQuestTimers().values())
			{
				for (QuestTimer timer : list)
				{
					if (++counter >= 10 && counter < size)
					{
						timers += "<tr><td>...</td></tr>";
						break;
					}
					final String font1, font2, font3;
					if (timer.getIsActive()) {
						font1 = "<font color=00FFFF>";
						font2 = "<font color=FFFF80>";
						font3 = "<font color=00FF00>";
					} else {
						font1 = "<font color=AAAAAA>";
						font2 = "<font color=888888>";
						font3 = "<font color=666666>";
					}
					timers += "<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=270>"
						+ font1 + "\"" + timer.getName() + "\":</font>"
					//	+ font2 + " Active:</font>"  + font3 + timer.getIsActive() + "</font>"
						+ font2 + " Remain:</font>"  + font3 + timer.getRemainDelay() + "</font>"
						+ font2 + " Npc:</font>"  + font3 + timer.getNpc() + "</font>"
						+ font2 + " Player:</font>"  + font3 + timer.getPlayer() + "</font>"
						+ font2 + " Repeat:</font>"  + font3 + timer.getIsRepeating() + "</font>"
						+ "</td></tr></table></td></tr>";
				}
			}
			
			final StringBuilder sb = new StringBuilder(256);
			sb.append("<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=270><font color=LEVEL>ID:</font> <font color=00FF00>").append(quest.getId()).append("</font></td></tr></table></td></tr>"
				+ "<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=270><font color=LEVEL>Name:</font> <font color=00FF00>").append(quest.getName()).append("</font></td></tr></table></td></tr>"
				+ "<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=270><font color=LEVEL>Descr:</font> <font color=00FF00>").append(quest.getDescr()).append("</font></td></tr></table></td></tr>"
				+ "<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=270><font color=LEVEL>Path:</font> <font color=00FF00>").append(quest.getClass().getName().substring(0, quest.getClass().getName().lastIndexOf('.')).replaceAll("\\.", "/")).append("</font></td></tr></table></td></tr>");
			if (!events.isEmpty())
			{
				sb.append("<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=270><font color=LEVEL>Events:</font> <font color=00FF00>").append(events).append("</font></td></tr></table></td></tr>");
			}
			if (!npcs.isEmpty())
			{
				sb.append("<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=270><font color=LEVEL>NPCs:</font> <font color=00FF00>").append(npcs).append("</font></td></tr></table></td></tr>");
			}
			if (!items.isEmpty())
			{
				sb.append("<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=270><font color=LEVEL>Items:</font> <font color=00FF00>").append(items).append("</font></td></tr></table></td></tr>");
			}
			if (!timers.isEmpty())
			{
				sb.append("<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=270><font color=LEVEL>Timers:</font> <font color=00FF00></font></td></tr></table></td></tr>");
				sb.append(timers);
			}
			
			final NpcHtmlMessage msg = new NpcHtmlMessage(0, 1);
			msg.setFile(activeChar.getHtmlPrefix(), "data/html/admin/npc-quests.htm");
			msg.replace("%quests%", sb.toString());
			msg.replace("%questName%", "<table><tr><td width=50 align=left><a action=\"bypass -h admin_script_load " + quest.getName() + "\">Reload</a></td> <td width=150  align=center><a action=\"bypass -h admin_quest_info " + quest.getName() + "\">" + quest.getName() + "</a></td> <td width=50 align=right><a action=\"bypass -h admin_script_unload " + quest.getName() + "\">Unload</a></tr></td></table>");
			activeChar.sendPacket(msg);
		}
		return true;
	}
	
	@Override
	public String[] getAdminCommandList()
	{
		return ADMIN_COMMANDS;
	}
}
