// This file is part of Notepad++ project
// Copyright (C)2003 Don HO <don.h@free.fr>
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// Note that the GPL places important restrictions on "derived works", yet
// it does not provide a detailed definition of that term.  To avoid
// misunderstandings, we consider an application to constitute a
// "derivative work" for the purpose of this license if it does any of the
// following:
// 1. Integrates source code from Notepad++.
// 2. Integrates/includes/aggregates Notepad++ into a proprietary executable
//    installer, such as those produced by InstallShield.
// 3. Links to a library or executes a program that does any of the above.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.



#include <stdexcept>
#include "ListView.h"
#include "Parameters.h"
#include "localization.h"

void ListView::init(HINSTANCE hInst, HWND parent)
{
	Window::init(hInst, parent);
    INITCOMMONCONTROLSEX icex;

    // Ensure that the common control DLL is loaded.
    icex.dwSize = sizeof(INITCOMMONCONTROLSEX);
    icex.dwICC  = ICC_LISTVIEW_CLASSES;
    InitCommonControlsEx(&icex);

    // Create the list-view window in report view with label editing enabled.
	int listViewStyles = LVS_REPORT | LVS_NOSORTHEADER\
						| LVS_SINGLESEL | LVS_AUTOARRANGE\
						| LVS_SHAREIMAGELISTS | LVS_SHOWSELALWAYS;

	_hSelf = ::CreateWindow(WC_LISTVIEW,
                                TEXT(""),
                                WS_CHILD | listViewStyles,
                                0,
                                0,
                                0,
                                0,
                                _hParent,
                                (HMENU) NULL,
                                hInst,
                                NULL);
	if (!_hSelf)
	{
		throw std::runtime_error("ListView::init : CreateWindowEx() function return null");
	}

	::SetWindowLongPtr(_hSelf, GWLP_USERDATA, (LONG_PTR)this);
	_defaultProc = reinterpret_cast<WNDPROC>(::SetWindowLongPtr(_hSelf, GWLP_WNDPROC, (LONG_PTR)staticProc));

	DWORD exStyle = ListView_GetExtendedListViewStyle(_hSelf);
	exStyle |= LVS_EX_FULLROWSELECT | LVS_EX_BORDERSELECT ;
	ListView_SetExtendedListViewStyle(_hSelf, exStyle);

	LVCOLUMN lvColumn;
	lvColumn.mask = LVCF_TEXT|LVCF_WIDTH;

	NativeLangSpeaker *pNativeSpeaker = (NppParameters::getInstance())->getNativeLangSpeaker();
	generic_string valStr = pNativeSpeaker->getAttrNameStr(TEXT("Value"), "AsciiInsertion", "ColumnVal");
	generic_string octStr = pNativeSpeaker->getAttrNameStr(TEXT("Oct"), "AsciiInsertion", "ColumnOct");	//+[JOJO]
	generic_string hexStr = pNativeSpeaker->getAttrNameStr(TEXT("Hex"), "AsciiInsertion", "ColumnHex");
	generic_string charStr = pNativeSpeaker->getAttrNameStr(TEXT("Character"), "AsciiInsertion", "ColumnChar");

	lvColumn.cx = NppParameters::getInstance()->_dpiManager.scaleX(38);
	lvColumn.pszText = (TCHAR *)valStr.c_str();
	ListView_InsertColumn(_hSelf, 0, &lvColumn);

	lvColumn.cx = NppParameters::getInstance()->_dpiManager.scaleY(34);	//+[JOJO]
	lvColumn.pszText = (TCHAR *)octStr.c_str();	//+[JOJO]
	ListView_InsertColumn(_hSelf, 1, &lvColumn);	//+[JOJO]

	lvColumn.cx = NppParameters::getInstance()->_dpiManager.scaleY(38);
	lvColumn.pszText = (TCHAR *)hexStr.c_str();
	ListView_InsertColumn(_hSelf, 2, &lvColumn);

	lvColumn.cx = NppParameters::getInstance()->_dpiManager.scaleY(48);
	lvColumn.pszText = (TCHAR *)charStr.c_str();
	ListView_InsertColumn(_hSelf, 3, &lvColumn);

	lvColumn.cx = NppParameters::getInstance()->_dpiManager.scaleY(52);	//+[JOJO]
	lvColumn.pszText = (TCHAR *)L"";
	ListView_InsertColumn(_hSelf, 4, &lvColumn);
}

#ifdef MOD_CHAR_PANEL_ENCODING
void ListView::changeCodepage(int codepage)
{
	//if (_codepage == codepage)
	//	return;

	_codepage = codepage;
	for (int i = 128; i < 256; ++i)
	{
		generic_string s = getAscii((unsigned char)i);
		ListView_SetItemText(_hSelf, i, 3, (LPTSTR)s.c_str());
	}
}
#else
void ListView::resetValues(int codepage)
{
	if (codepage == -1)
		codepage = 0;

	if (_codepage == codepage)
		return;

	ListView_DeleteAllItems(_hSelf);
	setValues(codepage);
}
#endif

generic_string ListView::getAscii(unsigned char value)
{
	switch (value)
	{
		case 0:
			return TEXT("NULL");
		case 1:
			return TEXT("SOH");
		case 2:
			return TEXT("STX");
		case 3:
			return TEXT("ETX");
		case 4:
			return TEXT("EOT");
		case 5:
			return TEXT("ENQ");
		case 6:
			return TEXT("ACK");
		case 7:
			return TEXT("BEL");
		case 8:
			return TEXT("BS");
		case 9:
			return TEXT("TAB");
		case 10:
			return TEXT("LF");
		case 11:
			return TEXT("VT");
		case 12:
			return TEXT("FF");
		case 13:
			return TEXT("CR");
		case 14:
			return TEXT("SO");
		case 15:
			return TEXT("SI");
		case 16:
			return TEXT("DLE");
		case 17:
			return TEXT("DC1");
		case 18:
			return TEXT("DC2");
		case 19:
			return TEXT("DC3");
		case 20:
			return TEXT("DC4");
		case 21:
			return TEXT("NAK");
		case 22:
			return TEXT("SYN");
		case 23:
			return TEXT("ETB");
		case 24:
			return TEXT("CAN");
		case 25:
			return TEXT("EM");
		case 26:
			return TEXT("SUB");
		case 27:
			return TEXT("ESC");
		case 28:
			return TEXT("FS");
		case 29:
			return TEXT("GS");
		case 30:
			return TEXT("RS");
		case 31:
			return TEXT("US");
		case 32:
			return TEXT("Space");
		case 127:
			return TEXT("DEL");
		default:
		{
			TCHAR charStr[10];
			const char ascii[] = { (char)value, '\0' };
#ifdef FIX_CHAR_PANEL_CODEPAGE
			if (MultiByteToWideChar(_codepage, MB_ERR_INVALID_CHARS, ascii, -1, charStr, _countof(charStr)) > 0)
			{
				const static char *repsC1[] = {
					"PAD", "HOP", "BPH", "NBH", "IND", "NEL", "SSA", "ESA",
					"HTS", "HTJ", "VTS", "PLD", "PLU", "RI", "SS2", "SS3",
					"DCS", "PU1", "PU2", "STS", "CCH", "MW", "SPA", "EPA",
					"SOS", "SGCI", "SCI", "CSI", "ST", "OSC", "PM", "APC",
					"NBSP",
				};
				WCHAR u = charStr[0];
				if (0x0080 <= u && u <= 0x00A0)
					return WcharMbcsConvertor::getInstance()->char2wchar(repsC1[u - 0x0080], 65001);
				else
					return charStr;
			}
			else
				return L"";
#else
			MultiByteToWideChar(_codepage, 0, ascii, -1, charStr, sizeof(charStr));
			return charStr;
#endif
		}

	}
}

void ListView::setValues(int codepage)
{
	_codepage = codepage;

	for (int i = 0 ; i < 256 ; ++i)
	{
		LVITEM item;
		item.mask = LVIF_TEXT;
		TCHAR dec[8];
		TCHAR oct[8];	//+[JOJO]
		TCHAR hex[8];
		TCHAR ctr[8];
		generic_sprintf(dec, TEXT("%3d"), i);
		generic_sprintf(oct, TEXT("%03o"), i);	//+[JOJO]
		generic_sprintf(hex, TEXT("%02X"), i);

		WCHAR *p = ctr; *p = L'\0';
		if (i < 0x20) {
			WCHAR *q;
			switch (i) {
			case 0x07: q = L"\\a"; break;
			case 0x08: q = L"\\b"; break;
			case 0x09: q = L"\\t"; break;
			case 0x0A: q = L"\\n"; break;
			case 0x0B: q = L"\\v"; break;
			case 0x0C: q = L"\\f"; break;
			case 0x0D: q = L"\\r"; break;
			case 0x1B: q = L"\\e"; break;
			case 0x20: q = L"\\s"; break;
			default:   q = L"";
			}
			p += swprintf(p, L"^%c %s", 0x40 + i, q);
		}

		item.pszText = dec;
		item.iItem = i;
		item.iSubItem = 0;
		ListView_InsertItem(_hSelf, &item);

		ListView_SetItemText(_hSelf, i, 1, (LPTSTR)oct);	//+[JOJO]
		ListView_SetItemText(_hSelf, i, 2, (LPTSTR)hex);

		generic_string s = getAscii((unsigned char)i);
		ListView_SetItemText(_hSelf, i, 3, (LPTSTR)s.c_str());

		ListView_SetItemText(_hSelf, i, 4, (LPTSTR)ctr);	//+[JOJO]
	}
}


void ListView::destroy()
{
	::DestroyWindow(_hSelf);
	_hSelf = NULL;
}


LRESULT ListView::runProc(HWND hwnd, UINT Message, WPARAM wParam, LPARAM lParam)
{
	return ::CallWindowProc(_defaultProc, hwnd, Message, wParam, lParam);
}

