/*
  liboftp: this is an FTP library to simplify the work to a Developer
  who want to work with FTP servers (RFC 959).

  Copyright (c) 2009 hirohito higashi. All rights reserved.
  This file is distributed under BSD license.
*/


/***** Feature test switches ************************************************/
/***** System headers *******************************************************/
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>


/***** Local headers ********************************************************/
#include "liboftp.h"
#include "sub.h"


/***** Constat values *******************************************************/
#define	TRANSFER_SEGMENT_SIZE	4096		/* 一度のrecv()で転送するバイト数 */


/***** Macros ***************************************************************/
/***** Typedefs *************************************************************/
/***** Function prototypes **************************************************/
/***** Local variables ******************************************************/
/***** Global variables *****************************************************/
/***** Signal catching functions ********************************************/
/***** Local functions ******************************************************/


/****************************************************************************/
/*! バッファからファイル送信
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	buf	バッファへのポインタ
 *@param	bufsiz	バッファサイズ
 *@param	fname	サーバ上のファイル名
 *@param	cmd	送信するFTPコマンド
 *@retval	int	エラーコード
 *@note
 */
static int ftp_put_buffer_main( LIBOFTP *ftp, const char *buf, int bufsiz, const char *fname, const char *cmd )
{
    int data_socket;
    const char *p = buf;
    int n, res;

    if( ftp->socket < 0 ) return LIBOFTP_ERROR;

    /*
     * 送信準備
     */
    if( ftp->flag_passive ) {
	data_socket = ftp_getready_pasv( ftp, cmd, fname );
    } else {
	data_socket = ftp_getready_active( ftp, cmd, fname );
    }
    if( data_socket < 0 ) {
	return data_socket;
    }

    /*
     * タイムアウトが意図通りに働くように、分割してsendする。
     */
    while( 1 ) {
	int len = bufsiz;
	if( len > TRANSFER_SEGMENT_SIZE ) {
	    len = TRANSFER_SEGMENT_SIZE;
	}
	n = sendn( data_socket, p, len, 0 );
	DEBUGPRINT1( "SEND: n=%d\n", n );
	if( n < 0 ) {
	    DEBUGPRINT1( "recv error. %s\n", strerror(errno) );
	    copy_strerror();
	    close( data_socket );
	    return LIBOFTP_ERROR_OS;
	}

	p += n;
	bufsiz -= n;
	if( bufsiz <= 0 ) {
	    break;
	}
    }
    close( data_socket );

    /*
     * receive response.
     */
    res = ftp_receive_response( ftp, ftp->error_message, sizeof(ftp->error_message)-1 );
    if( res != 226 ) {						/* 226: Closing data connection. */
	DEBUGPRINT1( "got illegal response %d\n", res );
	return res < 0? res: LIBOFTP_ERROR_PROTOCOL;
    }

    return LIBOFTP_NOERROR;
}



/***** Global functions *****************************************************/

/****************************************************************************/
/*! バッファからファイル送信
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	buf	バッファへのポインタ
 *@param	bufsiz	バッファサイズ
 *@param	fname	サーバ上のファイル名
 *@retval	int	エラーコード
 *@note
 */
int ftp_put_buffer( LIBOFTP *ftp, const char *buf, int bufsiz, const char *fname )
{
    return ftp_put_buffer_main( ftp, buf, bufsiz, fname, "STOR" );
}



/****************************************************************************/
/*! バッファからファイル送信 アペンドモード
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	buf	バッファへのポインタ
 *@param	bufsiz	バッファサイズ
 *@param	fname	サーバ上のファイル名
 *@retval	int	エラーコード
 *@note
 */
int ftp_append_buffer( LIBOFTP *ftp, const char *buf, int bufsiz, const char *fname )
{
    return ftp_put_buffer_main( ftp, buf, bufsiz, fname, "APPE" );
}
