﻿using System;
using System.ComponentModel;
using System.Linq;
using System.Windows;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using Microsoft.Kinect;

namespace BeginningKinect.Chapter9.SeatedSkeletion
{
    /// <summary>
    /// MainWindow.xaml の相互作用ロジック
    /// </summary>  
    public partial class MainWindow : Window
    {
        #region メンバー変数
        private KinectSensor KinectDevice;
        private WriteableBitmap ColorImageBitmap;
        private Int32Rect ColorImageBitmapRect;
        private int ColorImageStride;
        private byte[] ColorImagePixelData;
        private BackgroundWorker Worker;
        #endregion メンバー変数


        #region コンストラクタ
        public MainWindow()
        {
            InitializeComponent();

            this.Worker = new BackgroundWorker();
            this.Worker.DoWork += Worker_DoWork;
            this.Worker.RunWorkerAsync();

            this.Unloaded += (s, e) => { this.Worker.CancelAsync(); };
        }
        #endregion コンストラクタ


        private void DiscoverKinectSensor()
        {
            if (this.KinectDevice != null && this.KinectDevice.Status != KinectStatus.Connected)
            {
                this.KinectDevice = null;
            }

            if (this.KinectDevice == null)
            {
                this.KinectDevice = KinectSensor.KinectSensors
                                           .FirstOrDefault(x => x.Status == KinectStatus.Connected);

                if (this.KinectDevice != null)
                {
                    this.KinectDevice.ColorStream.Enable(ColorImageFormat.RgbResolution640x480Fps30);
                    this.KinectDevice.DepthStream.Enable(DepthImageFormat.Resolution640x480Fps30);
                    this.KinectDevice.SkeletonStream.Enable();
                    this.KinectDevice.Start();

                    ColorImageStream colorStream = this.KinectDevice.ColorStream;

                    this.ColorImageStride = colorStream.FrameWidth * colorStream.FrameBytesPerPixel;
                    this.ColorImagePixelData = new byte[colorStream.FramePixelDataLength];
                    this.Rgb_Image.Dispatcher.BeginInvoke(new Action(() =>
                    {
                        this.ColorImageBitmap = new WriteableBitmap(colorStream.FrameWidth,
                                                                    colorStream.FrameHeight,
                                                                    96,
                                                                    96,
                                                                    PixelFormats.Bgr32,
                                                                    null);
                        this.ColorImageBitmapRect = new Int32Rect(0,
                                                                  0,
                                                                  colorStream.FrameWidth,
                                                                  colorStream.FrameHeight);
                        this.Rgb_Image.Source = this.ColorImageBitmap;
                    }));
                }
            }
        }


        private void PollColorImageStream()
        {
            if (this.KinectDevice == null)
            {
                //TODO:使用可能なセンサーが存在しないことを通知する
            }
            else
            {
                try
                {
                    using (ColorImageFrame frame = this.KinectDevice.ColorStream.OpenNextFrame(100))
                    {
                        if (frame != null)
                        {
                            frame.CopyPixelDataTo(this.ColorImagePixelData);

                            this.Rgb_Image.Dispatcher.BeginInvoke(new Action(() =>
                            {
                                this.ColorImageBitmap.WritePixels(this.ColorImageBitmapRect,
                                                                  this.ColorImagePixelData,
                                                                  this.ColorImageStride,
                                                                  0);
                            }));
                        }
                    }
                    using (SkeletonFrame frame = this.KinectDevice.SkeletonStream.OpenNextFrame(100))
                    {
                        if (frame != null)
                        {
                            DrawSkeleton(frame);
                        }
                    }
                }
                catch
                {
                }
            }
        }

        private void DrawSkeleton(SkeletonFrame frame)
        {
            Skeleton[] skeletons = new Skeleton[frame.SkeletonArrayLength];
            frame.CopySkeletonDataTo(skeletons);
            this.Skeleton_Canvas.Dispatcher.BeginInvoke(new Action(() =>
            {
                this.Skeleton_Canvas.Children.Clear();
                foreach (Skeleton _skeleton in skeletons)
                {
                    if (_skeleton.TrackingState == SkeletonTrackingState.Tracked)
                    {
                        foreach (Joint _joint in _skeleton.Joints)
                        {
                            if (_joint.TrackingState == JointTrackingState.Tracked)
                            {
                                ColorImagePoint position =
                                        KinectDevice.MapSkeletonPointToColor(_joint.Position,
                                                                             KinectDevice.ColorStream.Format);
                                this.Skeleton_Canvas.Children.Add(
                                    new System.Windows.Shapes.Ellipse()
                                    {
                                        Margin = new Thickness(position.X - 5,
                                                               position.Y - 5,
                                                               0,
                                                               0),
                                        Fill = new SolidColorBrush(Colors.Red),
                                        Width = 10,
                                        Height = 10
                                    });
                            }
                        }
                    }
                }
            }));
        }

        private void Worker_DoWork(object sender, DoWorkEventArgs e)
        {
            BackgroundWorker worker = sender as BackgroundWorker;

            if (worker != null)
            {
                while (!worker.CancellationPending)
                {
                    DiscoverKinectSensor();
                    PollColorImageStream();
                }
            }
        }

        private void Mode_ComboBox_SelectionChanged(object sender,
                                                    System.Windows.Controls.SelectionChangedEventArgs e)
        {
            if (this.KinectDevice != null && this.KinectDevice.SkeletonStream != null)
            {
                string tagValue = (string)((FrameworkElement)this.Mode_ComboBox.SelectedItem).Tag;
                switch (tagValue)
                {
                    case "1":
                        this.KinectDevice.SkeletonStream.TrackingMode = SkeletonTrackingMode.Default;
                        break;
                    case "2":
                        this.KinectDevice.SkeletonStream.TrackingMode = SkeletonTrackingMode.Seated;
                        break;
                }
            }
        }
    }
}
