#!/usr/bin/python
#coding: utf-8

# hb_cibgen : Heartbeat cib.xml generator
#
# Copyright (C) 2008 NIPPON TELEGRAPH AND TELEPHONE CORPORATION
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import sys
import os
import commands
import popen2
import codecs
import csv
from optparse import OptionParser, make_option
from xml.dom.minidom import *

'''
	DEFINE設定部
'''
#テンプレートファイルが置かれるディレクトリ
TEMPLATE_DIR = '/usr/share/heartbeat/hb-cibgen/'
#crm_verifyコマンド名
CRM_VERIFY_CMD = 'crm_verify'
#crm_verifyファイル指定オプション
CRM_VERIFY_FILE_OPT = '-x'
#crm_verify文字列指定オプション
CRM_VERIFY_STR_OPT = '-X'

#実行時のプログラム名
PROG_NAME = 'hb_cibgen'
#hb_cibgenのバージョン情報
PROG_VERSION = '1.0'

#入力ファイルの文字コード
INPUT_STRING_CODE = 'japanese.shift-jis'
#出力ファイルの文字コード
OUTPUT_STRING_CODE = 'utf-8'
#テンプレートXMLファイルの文字コード
TEMPLATE_STRING_CODE = 'utf-8'

#メッセージレベル指定用文字
VERIFY_V = 'V'

#メインヘッダ列位置情報
M_HEADER_POS = 1

#ファイルパスの区切り文字を設定
DIR_SEPARATER = os.sep

#サブヘッダ用の区切り文字
SEPARATE_CHAR = ':'

#IDを区切るときに使用する文字
ID_SEPARATER = '-'
#IDとシーケンスを区切るときに使用する文字
IDSEQ_SEPARATER = ':'

#CSVファイルのコメント用文字(文字列も可)
COMMENT_CHAR = '#'

#instance_attribute用、id特定用文字列
ID_TARGET_STR = 'ID='

#pingd & diskd使用時のキーワード(CSV上では大文字でも小文字でも可）
USE_CHECK_LIST = ['y', 'yes']

#メッセージレベルの定義
ERRORLEVEL  = 0
WARNLEVEL   = 1
NOTICELEVEL = 2
INFOLEVEL   = 3
DEBUGLEVEL  = 4
DEBUG1LEVEL = 5

LOGLEVEL_DIC = {
	0: 'ERROR ',
	1: 'WARN  ',
	2: 'NOTICE',
	3: 'INFO  ',
	4: 'DEBUG ',
	5: 'DEBUG1'
}

#リターンコード
GOK_VOK   = 0
GERROR    = 1
GWARN_VOK = 2
GOK_VNG   = 3
GWARN_VNG = 4
VERROR    = 5

#Resourcesモード必須項目名
RESOURCE_ITEM = 'resourceitem'
RESOURCE_ID = 'id'
#Operationsモード必須項目名
PRIMITIVE_ID = 'primitiveid'
#Constraints_RscLocationモード必須項目名
CRL_RESOURCE_ID = 'resourceid'

#テンプレートXML名辞書
CIB_TEMPLATE_DIC = {
	'Base_cib': 'base.xml',
	'Primitive': 'primitive.xml',
	'Group': 'group.xml',
	'Clone': 'clone.xml',
	'MasterSlave': 'master_slave.xml',
	'RLnode': 'node.xml',
	'RLpingd': 'pingd.xml',
	'RLdiskd': 'diskd.xml'
}

#CSV上のモード識別文字列(内部的に小文字化)をプログラム内で対応付けた文字列の辞書
MODE_DIC = {
	'cluster_property': 'ClusterPropertySet_Attributes',
	'resource': 'Resources',
	'instance_attribute': 'InstanceAttributes',
	'operation': 'Operations',
	'rsc_location': 'Constraints_RscLocation'
}

#CSV上のリソース種別識別文字列(内部的に小文字化)をプログラム内で対応付けた文字列の辞書
SUBMODE_DIC = {
	'primitive': 'Primitive',
	'group': 'Group',
	'clone': 'Clone',
	'masterslave': 'MasterSlave'
}

#プログラム内で対応付けた文字列と、テンプレートXML内の識別用文字列の辞書
TARGET_NODE_DIC = {
	'InstanceAttributes': 'INSTANCE_ATTRIBUTE',
	'Operations': 'OPERATION',
	'ClusterPropertySet_Attributes': 'CLUSTER_PROPERTY',
	'Resources': 'RESOURCE',
	'Constraints_RscLocation': 'RSC_LOCATION'
}

#ID挿入先指定用文字列
REQUIRED_ID = 'REQUIRED_ID'
REQUIRED_ID_SEQ = 'REQUIRED_ID_SEQ'
#リソース配置制約 - attribute属性値挿入先指定用文字列
EXPRESSION_ATTRIBUTE = 'EXPRESSION_ATTRIBUTE'
#リソース配置制約 - value属性値挿入先指定用文字列
EXPRESSION_VALUE = 'EXPRESSION_VALUE'

try:
	#stdoutのエンコーディングをutf-8にする
	sys.stdout = codecs.getwriter(OUTPUT_STRING_CODE)(sys.stdout)
	#stderrのエンコーディングをutf-8にする
	sys.stderr = codecs.getwriter(OUTPUT_STRING_CODE)(sys.stderr)
except Exception:
	sys.stderr.write('failed to encode stdout and stderr.\n')
	sys.exit(1)

'''
	cib.xmlのDOM情報を操作するクラス
'''
class CreateCIBXML:

	'''
		CreateCIBXMLクラスの初期化処理
		引数
			logMsgObj：CsvReadクラスで作成した、LogMsgクラスオブジェクト
		戻り値
			成功時は何も返さない
			初期化処理に失敗(exit(1))

	'''
	def __init__(self, logMsgObj):

		#LogMsgクラスのインスタンス作成(CsvReadクラスで作ったインスタンスをコピー）
		self.logMsg = logMsgObj

		#文字を連結するために空文字で初期化
		buff = ''

		#Baseテンプレートのファイルパスを作成
		cib_template = TEMPLATE_DIR + CIB_TEMPLATE_DIC.get('Base_cib')

		#Baseテンプレートのオープン処理
		fd = self.openFile(cib_template)

		try:
			#ファイルから1行ずつ読み込む
			for line in fd:
				#テンプレートXMLの文字コードをunicode化
				line = unicode(line, TEMPLATE_STRING_CODE)
				#テンプレートファイルの1行の前後の空白・タブ・改行を取り、1つの文字列にする
				buff = buff + del_blank(line)
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルの読み込みに失敗しました。[%s]' % cib_template)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=cib_template)
			self.logMsg.print_quit_message()
			sys.exit(1)

		try:
			#ファイルのクローズ
			fd.close()
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルのクローズに失敗しました。[%s]' % cib_template)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=cib_template)
			self.logMsg.print_quit_message()
			sys.exit(1)

		try:
			#文字列からDOMオブジェクトを作成する
			self.cibxml = parseString(buff)
		except Exception, msg:
			#テンプレートファイルが空などの時
			self.logMsg.print_log(ERRORLEVEL, u'ファイルの解析に失敗しました。[%s]' % cib_template)
			#エラー情報をDEBUGレベルのメッセージで出力する
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=cib_template)
			self.logMsg.print_quit_message()
			sys.exit(1)

		#テンプレートから読み込み直後の変更の加えられてない、cib.xmlを保存(最後に変更が加えられたかの判定に使う
		try:
			self.original_ciblines = self.cibxml.documentElement.toxml(encoding=OUTPUT_STRING_CODE)
		except Exception, msg:
			self.logMsg.print_inner_error(u'XMLの文字列化に失敗')
			#エラー情報をDEBUGレベルのメッセージで出力する
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=cib_template)
			self.logMsg.print_quit_message()
			sys.exit(1)


	'''
		ファイルを開いてファイルオブジェクトを返すメソッド
		引数
			file：開きたいファイル名
		戻り値
			成功：ファイルディスクリプタ
			失敗：プログラム終了(exit(1))
	'''
	def openFile(self, file):

		try:
			fd = open(file, 'r')
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルのオープンに失敗しました。[%s]' % file)
			#エラー情報をDEBUGレベルのメッセージで出力する
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=file)
			self.logMsg.print_quit_message()
			sys.exit(1)

		return fd

	'''
		cib.xmlテンプレートからDOMオブジェクトを作成するメソッド
		引数
			templateFile：作成したいcib.xmlテンプレートファイル名
		戻り値
			成功：DOMのルートエレメント
			失敗：プログラム終了(exit(1))
	'''
	def create_template_node(self, templateFile):

		buff = ''

		fd = self.openFile(templateFile)

		try:
			for line in fd:
				#テンプレートXMLの文字コードをunicode化
				line = unicode(line, TEMPLATE_STRING_CODE)
				buff = buff + del_blank(line)
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルの読み込みに失敗しました。[%s]' % templateFile)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=templateFile)
			self.logMsg.print_quit_message()
			sys.exit(1)

		try:
			fd.close()
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルのクローズに失敗しました。[%s]' % templateFile)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=templateFile)
			self.logMsg.print_quit_message()
			sys.exit(1)

		try:
			nodeDom = parseString(buff)
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルの解析に失敗しました。[%s]' % templateFile)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=templateFile)
			self.logMsg.print_quit_message()
			sys.exit(1)

		return nodeDom.documentElement

	'''
		指定した名前のノードオブジェクトを作成するメソッド
		引数
			name：作成したいノードのタグ名
		戻り値
			成功：作成したノードオブジェクト
			失敗：プログラム終了(exit(1))
	'''
	def create_node(self, name):

		try:
			node = self.cibxml.createElement(name)
		except Exception, msg:
			self.logMsg.print_inner_error(u'エレメントの生成に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

		return node

	'''
		指定したノードオブジェクトに属性と、属性値を設定するメソッド
		引数
			node：対象ノードオブジェクト
			attr：設定する属性名
			value：設定する属性値
		戻り値
			属性の設定に失敗(exit(1))
			成功時は何も返さない
	'''
	def set_attribute(self, node, attr, value):

		try:
			node.setAttribute(attr, value)
		except Exception, msg:
			self.logMsg.print_inner_error(u'エレメントの属性の設定に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

	'''
		指定したノードオブジェクトにattrListに登録された属性名と、CSVファイルの列情報を
		使って、属性を設定するメソッド
		引数
			node：対象ノードオブジェクト
			attrList：サブヘッダの情報を保持したリスト
			csvList：CSVファイルの1行分のカンマ区切りのリスト
		戻り値
			True：空のデータが1つもなく設定終了
			False：空のデータが1つでもあった。
	'''
	def set_dic_attribute(self, node, attrList, csvList):

		flg = True

		for attrSet in attrList:
			key, value = tuple(attrSet)
			#値があればそれは有効なものとみなし属性に設定する。
			if csvList[value] != '':
				self.set_attribute(node, key, csvList[value])
			else:
				flg = False

		return flg

	'''
		指定したノードオブジェクトに子ノードを追加するメソッド
		引数
			parentNode：対象親ノードオブジェクト
			appendNode：追加する子ノードオブジェクト
		戻り値
			成功：何も返さない(Noneが返る)
			失敗：プログラム終了(exit(1))
	'''
	def append_child_node(self, parentNode, appendNode):

		try:
			parentNode.appendChild(appendNode)
		except Exception, msg:
			self.logMsg.print_inner_error(u'子要素の追加に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

	'''
		指定したノードオブジェクトから、指定した属性名の値を取得して返すメソッド
		引数
			currentNode：対象ノードオブジェクト
			attr：対象属性名
		戻り値
			成功：属性値(属性名が存在しない場合は空文字列)
			失敗：プログラム終了(exit(1))
	'''
	def get_attribute(self, currentNode, attr):
		try:
			#指定属性がなかった場合空文字が返される
			attrVal = currentNode.getAttribute(attr)
		except Exception, msg:
			self.logMsg.print_inner_error(u'エレメントの属性の取得に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

		return attrVal

	'''
		指定したノードオブジェクトから、指定した階層分の親ノードオブジェクトを取得して返すメソッド
		引数
			currentNode：対象ノードオブジェクト
			cnt：指定階層
		戻り値
			成功：指定階層分上のノードオブジェクト
			失敗：プログラム終了(exit(1))
	'''
	def get_parentNode(self, currentNode, cnt):

		for i in range(cnt):
			try:
				currentNode = currentNode.parentNode
			except Exception, msg:
				self.logMsg.print_inner_error(u'エレメントの親ノードの取得に失敗')
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
				self.logMsg.print_quit_message()
				sys.exit(1)

		return currentNode

	'''
		指定したノードオブジェクトの親ノードのid属性値を取得して返すメソッド
		直近の親がid属性を持たない場合、さらに親のノードがid属性を持っていないか見る
		引数
			node：基準となるノードオブジェクト
		戻り値
			成功：親ノードのid属性値
			失敗：空文字列または、プログラム終了(exit(1))
	'''
	def get_parentNodeId(self, node):

		parentId = None
		try:
			parentNode = node.parentNode
		except Exception, msg:
			self.logMsg.print_inner_error(u'エレメントの親ノードの取得に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

		while True:
			try:
				parentId = parentNode.getAttribute('id')
			except Exception, msg:
				self.logMsg.print_inner_error(u'エレメントの属性の取得に失敗')
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
				self.logMsg.print_quit_message()
				sys.exit(1)

			if parentId != '':
				return parentId
			else:
				try:
					parentNode = parentNode.parentNode
				except Exception, msg:
					self.logMsg.print_inner_error(u'エレメントの親ノードの取得に失敗')
					self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
					self.logMsg.print_quit_message()
					sys.exit(1)

			#一番親までノードをたどったがidを持つノードが見つからなかった。
			if parentNode.nodeType == node.DOCUMENT_NODE:
				return ''

	'''
		DOM全体から指定したタグ名のノードを探してリスト化して返すメソッド
		引数
			currentNode：基準となるノードオブジェクト
			searchNode_name：探すノードのタグ名
		戻り値
			成功：特定したノードリスト、該当ノードがない場合、空のリスト
			失敗：プログラム終了(exit(1))
	'''
	def searchNodeList(self, currentNode, searchNode_name):

		try:
			if currentNode != None:
				#ノードトップの指定があった場合、そのノード以下から該当するものを探し出す
				matchNodeList = currentNode.getElementsByTagName(searchNode_name)
			else:
				#ノードトップの指定がない場合、DOM全体から該当するものを探し出す
				matchNodeList = self.cibxml.getElementsByTagName(searchNode_name)
		except Exception, msg:
			self.logMsg.print_inner_error(u'指定タグのエレメントの取得に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

		return matchNodeList

	'''
		DOM全体から指定したノードを削除するメソッド
		引数
			delNode：削除対象のノードオブジェクト
		戻り値
			成功：何も返さない（Noneは返っている)
			失敗：プログラム終了(exit(1))
	'''
	def find_node_del(self, delNode):

		try:
			matchNodeList = self.cibxml.getElementsByTagName(delNode)
		except Exception, msg:
			self.logMsg.print_inner_error(u'指定タグのエレメントの取得に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

		for matchdelNode in matchNodeList:
			parentNode = matchdelNode.parentNode
			try:
				delNode = parentNode.removeChild(matchdelNode)
				delNode.unlink()
			except Exception, msg:
				self.logMsg.print_inner_error(u'子エレメントの削除に失敗')
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
				self.logMsg.print_quit_message()
				sys.exit(1)

	'''
		指定した属性値を持つノードをリスト化して返すメソッド
		引数
			nodeTop：基準となるノードオブジェクト
			findAttr：探す属性名
			findValue：探す属性値
			rc：特定したノードオブジェクトを保持するリスト
		戻り値
			成功：特定したノードオブジェクトのリスト
			失敗：空のリスト
	'''
	def find_haveAttr_nodelist(self, nodeTop, findAttr, findValue, rc):
		seq = 0
		if nodeTop == None:
			root = self.cibxml.documentElement
		else:
			root = nodeTop
		#指定されたノードの全ての下位ノードを取得
		for node in root.childNodes:
			seq = seq + 1
			try:
				if node.nodeType == Node.ELEMENT_NODE and node.getAttribute(findAttr) == findValue:
					#指定された属性値を持つノードをリストに保存
					rc.append((node,seq))
			except Exception, msg:
				self.logMsg.print_inner_error(u'エレメントの属性の取得に失敗')
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
				self.logMsg.print_quit_message()
				sys.exit(1)
			#ノードリストを全部見るまで、再帰処理
			self.find_haveAttr_nodelist(node, findAttr, findValue, rc)

		#ノードが特定できない場合は空のリストを返す(要エラーハンドリング）
		return rc

	#デバッグ用
	def debug_print(self):

		print self.cibxml.documentElement.toprettyxml(indent='  ', newl='\n')

	'''
		生成したcib.xmlがテンプレートから読み込んだ状態かどうか判断するメソッド
		引数
			input_csv:入力に使用したCSVファイル名
		戻り値
			True：テンプレートの状態から変化があった
			False：テンプレートのまま
	'''
	def check_cibxml(self, input_csv):

		#生成したcib.xmlがテンプレートから読み込んだ状態と一緒だった。
		try:
			if self.original_ciblines == self.cibxml.documentElement.toxml(encoding=OUTPUT_STRING_CODE):
				self.logMsg.print_log(ERRORLEVEL, u'CSVファイルが不正です。[%s]' % input_csv)
				self.logMsg.print_log(DEBUGLEVEL, u'CSVファイルに有効なデータが指定されていません。')
				return False
		except Exception, msg:
			self.logMsg.print_inner_error(u'XMLの文字列化に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

		return True

	'''
		DOMの内容をバッファに格納し、返すメソッド
		引数なし
		戻り値
			成功：DOMの内容が格納されたバッファ
			失敗：プログラム終了(exit(1))
	'''
	def outputXml(self):

		try:
			buff = self.cibxml.documentElement.toprettyxml(indent='  ', newl='\n')
		except Exception, msg:
			self.logMsg.print_inner_error(u'XMLの文字列化に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

		return buff

class CsvRead:

	'''
		CsvReadクラスの初期化処理
		__init__は何も返さない(Noneを返す)
	'''
	def __init__(self):

		#処理中のCSVファイルの行番号を保存
		self.lineNo = 0
		#モード識別用文字列保存
		self.modeStr = ''
		#cib.xml生成中に警告が発生した場合に使用するフラグ
		self.warnFlg = False
		#rsc_location id用シーケンス
		self.rsclocSeq = 0
		#idを一意にするために使用するシーケンス番号
		self.seq = 0
		#cib.xml生成中にエラーが発生した場合に使用するフラグ
		self.errFlg = False
		#crm_verifyに渡すVを保存する変数
		self.verifyLevel = ''
		#デフォルトのメッセージレベル
		self.default_logLevel = ERRORLEVEL

		ret = self.optionParser()

		#オプションの解析処理でエラーが起きたらプログラムをエラー終了
		if ret == False:
			sys.exit(1)

		#ユーザ指定のメッセージレベルにデフォルトのメッセージレベル値を加算
		self.logLevel = self.logLevel + self.default_logLevel

		#LogMsgクラスのインスタンス作成
		self.logMsg = LogMsg(self.logLevel)

		#実行コマンドラインの生成
		args = ''
		try:
			for arg in sys.argv:
				arg = unicode(arg, 'utf-8')
				args = args + u' ' + arg
			args = args + u' '
		except Exception, msg:
			self.logMsg.print_inner_error(u'コマンドライン文字列のunicode化に失敗')
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.logMsg.print_quit_message()
			sys.exit(1)

		#実行コマンドラインの表示
		self.logMsg.print_log(INFOLEVEL, u'実行コマンドライン [%s]' % args)

		#CIBインスタンスの作成
		self.createCib = CreateCIBXML(self.logMsg)

	'''
		プログラムに指定されたオプションを解析し、変数に格納する。
		引数なし
		戻り値
			成功：True
			失敗(不正なオプションの指定)：False
	'''
	def optionParser(self):

		flg = True

		#usageの出力形式定義
		usage='%s [options] CSVFILE' % PROG_NAME

		#オプションの定義
		option_list = [
			make_option('-o', dest='output_file',
				help=u'output generated cib.xml to the named file (default: stdout)'),
			make_option('-O', dest='output_dtd_check',
				help=u'output crm_verify\'s log to the named file (default: stderr)'),
			make_option('-l', dest='log_level', default='',
				help=u'turn on hb_cibgen\'s debug info. additional instances increase verbosity'),
			make_option('-L', dest='verify_level', default='',
				help=u'turn on crm_verify\'s debug info. additional instances increase verbosity'),
			make_option('-Q', dest='exec_verify', default=True, action='store_false',
				help=u'without DTD check with crm_verify'),
			make_option('-v', '--version',  action='callback', callback=print_version,
				help=u'output version information and exit')
		]

		#パーサオブジェクトの作成
		parser = OptionParser(option_list=option_list, usage=usage)

		#コマンドラインオプションのパーシング結果を取得
		#	options :定義したオプション
		#	args	:引数
		try:
			(options, args) = parser.parse_args()
		except SystemExit, retcode:
			if str(retcode) != str(0):
				#定義されていない引数が指定された場合、メッセージを出力する
				sys.stderr.write(u'未定義のオプションが指定されています。\n')
				parser.print_help(sys.stderr)
				return False
			#-vと-hが指定された場合は、optionparserがバージョン情報、またはヘルプを表示して正常終了する
			#その時exit()はSystemExitを送出するため、それをハンドリングしてプログラムを正常終了する
			sys.exit(0)
		except Exception:
			sys.stderr.write(u'オプションの解析に失敗しました。\n')
			sys.exit(1)

		if len(options.verify_level) == options.verify_level.count(VERIFY_V):
			if len(options.verify_level) > 0:
				#ユーザ指定のVと-を結合し、crm_verifyのオプションを作成する
				self.verifyLevel = '-' + options.verify_level
			else:
				#ユーザから何も指定がなかったら、空文字列を入れる
				self.verifyLevel = ''
		else:
			sys.stderr.write(u'-Lオプションの指定に誤りがあります。\n')
			parser.print_help(sys.stderr)
			flg = False

		if len(options.log_level) == options.log_level.count(VERIFY_V):
			#cibgenのログレベル指定
			self.logLevel = options.log_level.count(VERIFY_V)
		else:
			sys.stderr.write(u'-lオプションの指定に誤りがあります。\n')
			parser.print_help(sys.stderr)
			flg = False

		if len(args) == 0:
			#引数が0ならファイルの指定がなかったと判断
			sys.stderr.write(u'CSVファイルが指定されていません。\n')
			parser.print_help(sys.stderr)
			flg = False
		elif len(args) > 1:
			#引数が2つ以上ならファイルが複数されていたと判断
			sys.stderr.write(u'CSVファイルが複数指定されています。\n')
			parser.print_help(sys.stderr)
			flg = False
		else:
			#上記以上ならファイルが1つ指定されていたと判断
			try:
				self.input_csv = unicode(args[0], 'utf-8')
			except Exception, msg:
				sys.stderr.write(u'CSVファイル名のunicode化に失敗しました。\n')
				sys.exit(1)

		if options.output_file != None:
			#xmlファイルのアウトプット先を指定。指定がなければ標準出力
			try:
				self.output_file = unicode(options.output_file, 'utf-8')
			except Exception, msg:
				sys.stderr.write(u'cib.xml出力ファイル名のunicode化に失敗しました。\n')
				sys.exit(1)
		else:
			self.output_file = sys.stdout

		'''
		crm_verifyを実行するかの指定
			True:実行する
			False:実行しない
		'''
		self.exec_verify = options.exec_verify

		if options.output_dtd_check != None:
			try:
				self.output_dtd_check = unicode(options.output_dtd_check, 'utf-8')
			except Exception, msg:
				sys.stderr.write(u'crm_verifyのログファイル名のunicode化に失敗しました。\n')
				sys.exit(1)
		else:
			#crm_verifyの結果の出力先を指定、指定がなければ標準エラー出力
			self.output_dtd_check = sys.stderr

		if options.output_dtd_check != None and options.output_file != None:
			#出力ファイルが2つとも同じファイルの場合エラー
			if os.path.abspath(options.output_file) == os.path.abspath(options.output_dtd_check):
				sys.stderr.write(u'cib.xmlとcrm_verifyのログの出力先に同じファイルが指定されています。\n')
				parser.print_help(sys.stderr)
				flg = False

		'''
		flg
			True：オプション解析成功
			False：オプション解析失敗
		'''
		return flg


	'''
		CSVファイルの1行の文字コードをsjisからunicodeに変換
		さらに各要素の前後の全角半角スペース＆タブ文字を削除する
		引数
			csvList：CSVファイル1行分のリスト
		戻り値
			成功：変換済みのcsvList
			失敗：プログラム終了(exit(1))
	'''
	def conversion_csv(self, csvList):

		#各列の要素をsjisからunicodeに変換して再格納
		for i, csvRow in enumerate(csvList):
			#要素が空なら変換処理を行わない
			if csvRow == '':
				continue

			try:
				#元ファイルの形式(sjis)をunicodeに変換
				buff = unicode(csvRow, INPUT_STRING_CODE)
			except Exception, msg:
				self.logMsg.print_inner_error(u'unicodeへの変換に失敗')
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, self.input_csv, self.lineNo)
				self.logMsg.print_quit_message()
				sys.exit(1)

			#前後の空白を取り去る処理
			buff = del_blank(buff)
			#処理後の各要素の保存
			csvList[i] = buff

		return csvList

	'''
		ID=の文字列を探して、その右にあるリソースidを取得して返すメソッド
		引数
			csvList：CSVファイル1行分のリスト
		戻り値
			成功：リソースID(文字列)
			失敗：None
	'''
	def get_rscid(self, csvList):

		i = 0
		#ID=の個数を保存する変数
		idTargetList = []
		idTargetPos = None
		rscIdList = []
		rscId = None

		#CSVファイルのメインヘッダ列より右側の列が存在しない場合
		if self.check_list_length(csvList, M_HEADER_POS + 1) == False:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL,
				u'"ID="が存在しません。', self.input_csv, self.lineNo)
			return None

		#M_HEADER_POS以降のデータから取得
		for csvRow in csvList:
			#メインヘッダ列まではスキップ
			if i < M_HEADER_POS + 1:
				i = i + 1
				continue

			if ID_TARGET_STR == csvRow.upper():
				#ID=となっている列情報を保存
				idTargetList.append(i)

			i = i + 1

		#ID=の数が0だった(ID=が指定されなかった)
		if len(idTargetList) == 0:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL,
				u'"ID="が存在しません。', self.input_csv, self.lineNo)
			return None
		#ID=の数が2個以上だった。エラー終了
		elif len(idTargetList) > 1:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL,
				u'"ID="が複数存在します。', self.input_csv, self.lineNo)
			return None

		idTargetPos = idTargetList[0]

		#id特定用文字以外の有効なデータで1番最初に見つかったものを使用する。
		for csvRow in csvList[idTargetPos + 1:]:
			if csvRow != '':
				rscIdList.append(csvRow)

		#ID=の指定はあったが、有効なデータが存在しなかった。
		if len(rscIdList) == 0:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'リソースIDが指定されていません。', self.input_csv, self.lineNo)
			return None
		#有効なデータが複数存在した。警告を出し、処理は継続
		elif len(rscIdList) > 1:
			self.logMsg.print_log(WARNLEVEL, u'リソースIDが複数指定されています。', self.input_csv, self.lineNo)
			self.warnFlg = True

		#先に特定したものを使用する
		rscId = rscIdList[0]

		return rscId


	'''
		指定されたノード中にREQUIRED_IDが存在する場合、そのIDを設定する
		引数
			node：探し始める基準となるノード
		戻り値
			対象が存在しなかった。-> 1
			正常に終了した -> 0
	'''
	def set_reqid(self, node):

		#node内からREQUIRED_IDを持つnodeをリストにして得る
		matchList = self.createCib.find_haveAttr_nodelist(node, 'id', REQUIRED_ID, NodeList())

		if len(matchList) == 0:
			#埋めるべきところが何もなかったので、何もしない
			return 1

		for matchNode in matchList:
			mnode, seq = tuple(matchNode)
			#親のidを取得する。（直近の親がid属性を持たない場合さらに親のidを取得する
			parentId = self.createCib.get_parentNodeId(mnode)

			if parentId == '':
				self.seq = self.seq + 1
				self.logMsg.print_log(DEBUG1LEVEL, u'親ノードのidが見つからないため、「タグ名::シーケンス番号」をidに設定します。',
					file=self.input_csv, lineNo=self.lineNo)
				self.createCib.set_attribute(mnode, 'id', mnode.tagName + IDSEQ_SEPARATER * 2 + str(self.seq))
			else:
				self.createCib.set_attribute(mnode, 'id', parentId + ID_SEPARATER + mnode.tagName)

		return 0

	'''
		指定されたノード中にREQUIRED_ID_SEQが存在する場合、そのIDを設定する
		引数
			node：探し始める基準となるノード
		戻り値
			対象が存在しなかった。-> 1
			正常に終了した -> 0

	'''
	def set_reqid_seq(self, node):

		seq = 0
		prevNode = None
		dic = {}
		list = []

		#node内からREQUIRED_ID_SEQを持つnodeをリストにして得る
		matchList = self.createCib.find_haveAttr_nodelist(node, 'id', REQUIRED_ID_SEQ, NodeList())

		if len(matchList) == 0:
			#埋めるべきところがなかったので、何もしない
			return 1

		for matchNode in matchList:
			mnode, seq = tuple(matchNode)
			#親のidを取得する。（直近の親がid属性を持たない場合さらに親のidを取得する
			parentId = self.createCib.get_parentNodeId(mnode)

			if parentId == '':
				self.seq = self.seq + 1
				self.logMsg.print_log(DEBUG1LEVEL, u'親ノードのidが見つからないため、「タグ名::シーケンス番号」をidに設定します。',
					file=self.input_csv, lineNo=self.lineNo)
				self.createCib.set_attribute(mnode, 'id', mnode.tagName + IDSEQ_SEPARATER * 2 + str(self.seq))
			else:
				self.createCib.set_attribute(mnode, 'id', parentId + ID_SEPARATER + mnode.tagName + IDSEQ_SEPARATER + str(seq))

		return 0

	'''
		ノード中の、指定された属性に属性値を設定する
		引数
			node     ：属性値を設定する対象ノード
			attrName ：属性値を設定する対象属性（名）
			attrValue：属性値を設定する対象属性（値）
			value    ：設定する属性値
		戻り値
			設定対象の属性が存在しなかった -> 1
			属性値を設定した（正常終了） -> 0
	'''
	def set_attribute_value(self, node, attrName, attrValue, value):
		#node内から、設定対象属性を持つnodeをリストにして得る
		matchList = self.createCib.find_haveAttr_nodelist(
						node, attrName, attrValue, NodeList())

		if len(matchList) == 0:
			return 1

		for matchNode in matchList:
			mnode, seq = tuple(matchNode)
			self.createCib.set_attribute(mnode, attrName, value)
		return 0

	'''
		列数が指定された値より大きいかをチェックするメソッド
		引数
			csvList：CSVファイル1行分のリスト
			num：比較したい列番号
		戻り値
			成功：True -> リストの長さが足りている
			失敗：False -> リストの長さが足りていない
	'''
	def check_list_length(self, csvList, num):
		if len(csvList) < num + 1:
			return False

		return True

	'''
		属性値の列に値が入っているかを判定するメソッド
		引数
			csvList：CSVファイル1行分のリスト
			attrList：サブヘッダ(必須項目以外)の位置情報を保持したリスト
		戻り値
			1つでも値が入ってる：True
			1つも入っていない：False
	'''
	def have_effective_val(self, csvList, attrList):
		for attrSet in attrList:
			key, pos = tuple(attrSet)
			#サブヘッダで指定した列に値が入っているか？
			if csvList[pos] != '':
				#値があるなら有効行とみなす
				return True

		return False

	'''
		必須属性列に値が入っているかを判定するメソッド
		引数
			csvList：CSVファイル1行分のリスト
			posList：必須属性列の位置情報リスト
		戻り値
			1つでも値が入ってる：True
			1つも入っていない：False
	'''
	def have_effective_required_val(self, csvList, posList):
		for pos in posList:
			#サブヘッダで指定した列に値が入っているか？
			if csvList[pos] != '':
				#値があるなら有効行とみなす
				return True

		return False

	'''
		hb_cibgen内部で起きた警告フラグと、crm_verifyの実行結果を利用し、hb_cibgenのexitコードを生成する。
		exitコード１はcibgen内のエラー発生箇所でexit(1)しているため、このメソッド内には不要
		引数
			cibgen：hb_cibgen実行中に警告が発生した場合に使用するフラグ(1になっていたら警告が発生している)
			verify：crm_verifyの実行結果(戻り値)
		戻り値
			生成したexitコード
	'''
	def generate_return_code(self, cibgen, verify):
		#crm_verifyが正常に終了しなかった
		if verify == 2:
			return VERROR #5

		#cibgenはcib.xml生成処理で警告が発生していた場合1になる
		if cibgen == 1 and verify == 1:
			return GWARN_VNG #4
		if cibgen == 1 and verify == 0:
			return GWARN_VOK #2
		#cibgenで警告が発生していないが、crm_verifyでerrorまたはwarningが発生した
		if cibgen == 0 and verify == 1:
			return GOK_VNG #3
		#両方エラーも警告も発生しなかった
		if cibgen == 0 and verify == 0:
			return GOK_VOK #0

		#上記で全ての組み合わせを網羅しているので、else処理は行わない

	'''
		メインヘッダの文字列をモード文字列に変換して返すメソッド
		引数
			csvList：CSVファイル1行分のリスト
		戻り値
			モード文字列
			辞書から特定できない場合は空文字列
	'''
	def get_mode(self, csvList):

		#CSVファイル内のメインヘッダ文字列が、メインヘッダ辞書内にkeyとして存在していれば、モード特定
		mode = MODE_DIC.get(csvList[M_HEADER_POS].lower())

		if mode == None:
			return ''
		else:
			return mode

	'''
		メインヘッダのモードを特定するメソッド
		引数
			csvList：CSVファイル1行分のリスト
		戻り値
			0:モードの特定に成功
			1:モードの特定に失敗
	'''
	def analyze_main_header(self, csvList):

		#モードを特定する処理
		self.modeStr = self.get_mode(csvList)

		#メインヘッダが不正ならエラー終了
		if self.modeStr == '':
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'未定義のメインヘッダ"%s"が指定されています。' % csvList[M_HEADER_POS],
				file=self.input_csv, lineNo=self.lineNo)
			return 1

		#モードの切り替えを知らせる
		self.logMsg.print_log(DEBUG1LEVEL, u'内部処理モードに「%s」がセットされました。' % self.modeStr,
			file=self.input_csv, lineNo=self.lineNo)

		return 0

	'''
		Resourceモード時の初期化処理メソッド
		引数なし
		戻り値
			0:初期化に成功
			1:2度目以降のResourceモードなのでCSVのフォーマットエラー
	'''
	def init_resource_mode(self):
		#2回目のResourceモードがきたら、エラー
		if self.resource_mode_flg == True:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'Resourcesの設定を2度以上行うことはできません。',
				file=self.input_csv, lineNo=self.lineNo)
			return 1

		#前回追加したリソースの階層を保存する変数（初回は階層0のみ許容するため初期値を-1に）
		self.prevDepth = -1
		#前回追加したリソースを保存するための変数
		self.prevNode = None
		#1度でもResourceモードのヘッダがきたらフラグを立てる
		self.resource_mode_flg = True

		return 0

	'''
		instance_attributesモード時の初期化処理メソッド(rscidからリソースの特定を行う
		引数
			rscId:リソースID
		戻り値
			findedIdNodeList[0]:リソースの特定に成功
			None:リソースの特定に失敗
	'''
	def init_instattr_mode(self, rscId):
		#対象idを持つノードをリスト化して得る
		findedIdNodeList = self.createCib.find_haveAttr_nodelist(None, 'id', rscId, NodeList())

		cnt = 0
		for i, findedIdNode in enumerate(findedIdNodeList):
			node, seq = tuple(findedIdNode)

			#この処理で最初に見つかった「<INSTANCE_ATTRIBUTE id=prmId/>」のノードを特定
			if node.tagName == TARGET_NODE_DIC.get(self.modeStr):
				cnt = cnt + 1
				if cnt == 1:
					pos = i

		#該当するリソースが存在しなかった。
		if cnt == 0:
			self.logMsg.print_log(WARNLEVEL, u'リソース("%s")にパラメータを設定できませんでした。' % rscId,
				file=self.input_csv, lineNo=self.lineNo)
			self.warnFlg = True
			self.logMsg.print_log(DEBUGLEVEL, u'対象リソースが特定できませんでした。',
				file=self.input_csv, lineNo=self.lineNo)
			#モードを初期化する。
			self.modeStr = 'skip'
			#次の行へ
			return None
		#該当するリソースが1つより多く存在した場合、警告を出して処理は継続
		elif cnt > 1:
			self.logMsg.print_log(WARNLEVEL, u'パラメータ設定対象のリソース("%s")が複数見つかりました。' % rscId,
				file=self.input_csv, lineNo=self.lineNo)
			self.warnFlg = True

		return findedIdNodeList[pos]

	'''
		サブヘッダを解析するメソッド
		引数
			csvList：CSVファイル1行分のリスト
			reqAttrList:必須属性列の情報を入れるリスト
			attrList:必須属性以外の列の情報を入れるリスト
		戻り値なし
	'''
	def analyze_sub_header(self, csvList, reqAttrList, attrList, allAttrList):

		rscItemList = []
		idPosList = []

		#行内の全ての列を参照し、列情報を必須リスト非必須のリストに分類する
		#各リソースの必須属性は以下の通り(各サブヘッダは大文字小文字混在可)
		#	Resources	:ResourceItem, id
		#	Operations	:PrimitiveId
		#	Constraints_RscLocation	:ResourceId
		for i, csvRow in enumerate(csvList):
			dplflg = False

			if i < M_HEADER_POS + 1:
				#メインヘッダ列以前は見ない
				continue

			#列にデータがあるなら、その列を属性列とみなす
			if csvRow != '':
				allAttrList.append(i)
				if self.modeStr == 'Resources' and csvRow.lower() == RESOURCE_ITEM:
					rscItemList.append(i)
				elif self.modeStr == 'Resources' and csvRow.lower() == RESOURCE_ID:
					idPosList.append(i)
				elif self.modeStr == 'Operations' and csvRow.lower() == PRIMITIVE_ID \
					or self.modeStr == 'Constraints_RscLocation' and csvRow.lower() == CRL_RESOURCE_ID:
					reqAttrList.append(i)
				#上記、準必須列以外は属性列として、値と、列番号をリストに保存
				else:
					if self.check_attrlist(attrList, csvRow) == False:
						#サブヘッダに同じ属性が設定されていたら、2列目以降は
						#警告を出し、処理を行わない
						self.logMsg.print_log(WARNLEVEL, u'"%s"列が複数指定されています。' % csvRow.lower(),
							self.input_csv, self.lineNo)
						self.warnFlg = True
						continue

					if self.modeStr == 'Constraints_RscLocation':
						if ((csvRow.lower().startswith('score' + SEPARATE_CHAR)) or \
							(csvRow.lower().startswith('pingd' + SEPARATE_CHAR)) or \
							(csvRow.lower().startswith('diskd' + SEPARATE_CHAR))):
							attrList.append((csvRow, i))
							continue

					#必須以外のアトリビュートは全て小文字にして、列番号と共にリストに保存
					attrList.append((csvRow.lower(), i))
			else:
				#列のデータが空だったら、処理をしない（飛ばす）
				pass

		if self.modeStr == 'Resources':
			#Resoucesの場合のみ、必須属性が複数あるため、それらを1つのリストに結合する
			reqAttrList.append((rscItemList, idPosList))

	'''
		サブヘッダの属性が重複してないかチェックするメソッド
		引数
			attrList：サブヘッダの情報を保持したリスト(属性名, 列番号)
			str:チェックしたい属性名
		戻り値
			True:重複していない
			False:重複していた
	'''
	def check_attrlist(self, attrList, str):
		#サブヘッダの属性が重複してないかチェックする処理
		for attr in attrList:
			attr, pos = tuple(attr)
			if attr.lower() == str.lower():
				return False
		return True

	'''
		必須属性列が1列以上存在するかチェックするメソッド
		引数
			list:必須属性列の位置情報を保持したリスト
			str:属性列名
		戻り値
			True:必須属性列が1列以上ある
			False:必須属性列がない
	'''
	def check_req_attrlist(self, list, str):
		#必須列が存在しない場合
		if len(list) == 0:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'%s列が存在しません。' % str,
				file=self.input_csv, lineNo=self.lineNo)
			return False

		return True

	'''
		必須属性列が1列だけかチェックするメソッド
		引数
			list:必須属性列の位置情報を保持したリスト
			str:属性列名
		戻り値
			True:必須属性列が1列だけある
			False:必須属性列がない、または必須属性列が2列以上ある
	'''
	def check_req_attrlist_solitary(self, list, str):
		#必須列が存在しない場合
		if len(list) == 0:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'%s列が存在しません。' % str,
				file=self.input_csv, lineNo=self.lineNo)
			return False
		#必須列が複数列ある場合
		elif len(list) > 1:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'%s列が複数存在します。' % str,
				file=self.input_csv, lineNo=self.lineNo)
			return False

		return True

	'''
		OPERATION設定のサブヘッダのフォーマットをチェックするメソッド
		引数
			attrList：サブヘッダの情報を保持したリスト(属性名, 列番号)
		戻り値
			True:フォーマットエラーなし
			False:フォーマットにエラーを発見
	'''
	def check_op_subheader_format(self, attrList):
		#アトリビュートリストから、属性名と、列情報のペアを得る
		for attrSet in attrList:
			opKey, colPos = tuple(attrSet)
			if SEPARATE_CHAR in opKey:
				#':'が2つ以上ある
				if opKey.count(SEPARATE_CHAR) > 1:
					self.logMsg.print_format_error(self.input_csv, self.lineNo)
					self.logMsg.print_log(DEBUGLEVEL, u'サブヘッダに区切り文字[%s]が2つ以上存在します。[%s]' % (SEPARATE_CHAR, opKey),
						file=self.input_csv, lineNo=self.lineNo)
					return False

				#サブヘッダの値を':'でname属性値と、属性に分割する
				[opName, attr] = opKey.split(SEPARATE_CHAR)
				#空文字列が含まれていた
				if opName == '' or attr == '':
					self.logMsg.print_format_error(self.input_csv, self.lineNo)
					self.logMsg.print_log(DEBUGLEVEL, u'サブヘッダに空文字列が含まれています。[%s]' % opKey,
						file=self.input_csv, lineNo=self.lineNo)
					return False
			else:
				#指定された文字列が識別できるものではなかった
				self.logMsg.print_format_error(self.input_csv, self.lineNo)
				self.logMsg.print_log(DEBUGLEVEL, u'未定義のサブヘッダ"%s"が指定されました。' % opKey,
					file=self.input_csv, lineNo=self.lineNo)
				return False
		return True

	'''
		RSC_LOCATION設定のサブヘッダのフォーマットをチェックするメソッド
		引数
			attrList：サブヘッダの情報を保持したリスト(属性名, 列番号)
		戻り値
			True:フォーマットエラーなし
			False:フォーマットにエラーを発見
	'''
	def check_rscloc_subheader_format(self, attrList):
		#アトリビュートリストから、属性名と、列情報のペアを得る
		for attrSet in attrList:
			rlKey, colPos = tuple(attrSet)

			#':'が属性名内にあるか？
			if SEPARATE_CHAR in rlKey:

				if ((rlKey.lower().startswith('score' + SEPARATE_CHAR)) or \
					(rlKey.lower().startswith('diskd' + SEPARATE_CHAR))):
					#':'が2つ以上ある
					if rlKey.count(SEPARATE_CHAR) > 1:
						self.logMsg.print_format_error(self.input_csv, self.lineNo)
						self.logMsg.print_log(DEBUGLEVEL,
							u'サブヘッダに区切り文字[%s]が2つ以上存在します。[%s]' \
							% (SEPARATE_CHAR,rlKey), file=self.input_csv, lineNo=self.lineNo)
						return False

					[keyStr, value] = rlKey.split(SEPARATE_CHAR)

					#文字列が空だった
					if value == '':
						self.logMsg.print_format_error(self.input_csv, self.lineNo)
						self.logMsg.print_log(DEBUGLEVEL,
							u'サブヘッダに空文字列が含まれています。[%s]' \
							% rlKey, file=self.input_csv, lineNo=self.lineNo)
						return False

				elif (rlKey.lower().startswith('pingd' + SEPARATE_CHAR)):
					#':'が2つない
					if rlKey.count(SEPARATE_CHAR) < 2:
						self.logMsg.print_format_error(self.input_csv, self.lineNo)
						self.logMsg.print_log(DEBUGLEVEL,
							u'サブヘッダに区切り文字[%s]が2つ存在しません。[%s]' \
							% (SEPARATE_CHAR,rlKey), file=self.input_csv, lineNo=self.lineNo)
						return False

					#':'が3つ以上ある
					elif rlKey.count(SEPARATE_CHAR) > 2:
						self.logMsg.print_format_error(self.input_csv, self.lineNo)
						self.logMsg.print_log(DEBUGLEVEL,
							u'サブヘッダに区切り文字[%s]が3つ以上存在します。[%s]' \
							% (SEPARATE_CHAR,rlKey), file=self.input_csv, lineNo=self.lineNo)
						return False

					[keyStr, value1, value2] = rlKey.split(SEPARATE_CHAR)

					#文字列が空だった
					if value1 == '' or value2 == '':
						self.logMsg.print_format_error(self.input_csv, self.lineNo)
						self.logMsg.print_log(DEBUGLEVEL,
							u'サブヘッダに空文字列が含まれています。[%s]' \
							% rlKey, file=self.input_csv, lineNo=self.lineNo)
						return False

				else:
					#score、pingd、diskd以外の文字列が指定された
					self.logMsg.print_format_error(self.input_csv, self.lineNo)
					self.logMsg.print_log(DEBUGLEVEL,
						u'未定義のサブヘッダ"%s"が指定されました。' \
						% rlKey, file=self.input_csv, lineNo=self.lineNo)
					return False

			else:
				#指定された文字列が識別できるものではなかった
				self.logMsg.print_format_error(self.input_csv, self.lineNo)
				self.logMsg.print_log(DEBUGLEVEL,
					u'未定義のサブヘッダ"%s"が指定されました。' \
					% rlKey, file=self.input_csv, lineNo=self.lineNo)
				return False

		return True

	'''
		cluster_propertyモード時、サブヘッダで指定した列(属性列)に実データが入っているかチェック
		引数
			csvList：CSVファイル1行分のリスト
			attrList：サブヘッダの情報を保持したリスト(属性名, 列番号)
		戻り値
			True:attrListに登録してある属性列に1列でも実データが入っている
			False:属性列に実データが入っていない
	'''
	def check_cluster_propertyset(self, csvList, attrList):
		#実データがあるかどうかの判定
		isEffective =  self.have_effective_val(csvList, attrList)
		#属性列に実データが1つもなかった。
		if isEffective == False:
			self.logMsg.print_log(DEBUG1LEVEL, u'属性値が設定されていません。',
				file=self.input_csv, lineNo=self.lineNo)
			#実データがないので、次の行へ
			return False

		return True

	'''
		resourcesモード時、サブヘッダで指定した列(属性列)に実データが入っているかチェック
		引数
			csvList：CSVファイル1行分のリスト
			attrList：サブヘッダの情報を保持したリスト(属性名, 列番号)
			idPos:id属性列の列番号
			rscPosList:ResourceItem列の列番号を格納したリスト
		戻り値
			True:attrListに登録してある属性列に1列でも実データが入っている
			False:idPos列、ResourceItem列が空
	'''
	def check_resources(self, csvList, attrList, idPos, rscPosList):
		#id列に実データがあるかどうかの判定
		isEffective =  self.have_effective_required_val(csvList, [idPos])
		#必須属性列に実データが1つもなかった。
		if isEffective == False:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'%s列に値が設定されていません。' % RESOURCE_ID,
				file=self.input_csv, lineNo=self.lineNo)
			self.errFlg = True
			#エラーが発生しても、1モード分の処理は続けるため、次の行の処理へ移る
			return False

		#resourceItem列に実データがあるかどうかの判定
		isEffective =  self.have_effective_required_val(csvList, rscPosList)
		#必須属性列に実データが1つもなかった。
		if isEffective == False:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'%s列に値が設定されていません。' % RESOURCE_ITEM,
				file=self.input_csv, lineNo=self.lineNo)
			self.errFlg = True
			#エラーが発生しても、1モード分の処理は続けるため、次の行の処理へ移る
			return False

		#必須以外の属性列に実データがあるかどうかの判定
		isEffective =  self.have_effective_val(csvList, attrList)
		#属性列に実データが1つもなかった。(groupやcloneリソースの場合は正常なので、エラーではない)
		if isEffective == False:
			self.logMsg.print_log(DEBUG1LEVEL, u'属性値が設定されていません。',
				file=self.input_csv, lineNo=self.lineNo)

		return True

	'''
		operationsモード時、サブヘッダで指定した列(属性列)に実データが入っているかチェック
		引数
			csvList：CSVファイル1行分のリスト
			attrList：サブヘッダの情報を保持したリスト(属性名, 列番号)
			prmIdPos:対象リソースid属性列の列番号
		戻り値
			True:attrListに登録してある属性列に1列でも実データが入っている
			False:prmIdPos列が空、attrListで保持した属性列に実データが1つもない
	'''
	def check_operations(self, csvList, attrList, prmIdPos):
		#id列に実データがあるかどうかの判定
		isEffective =  self.have_effective_required_val(csvList, [prmIdPos])
		#必須属性列に実データが1つもなかった。
		if isEffective == False:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'%s列に値が設定されていません。' % PRIMITIVE_ID,
				file=self.input_csv, lineNo=self.lineNo)
			self.errFlg = True
			#エラーが発生しても、1モード分の処理は続けるため、次の行の処理へ移る
			return False

		#必須以外の属性列に実データがあるかどうかの判定
		isEffective =  self.have_effective_val(csvList, attrList)
		#属性列に実データが1つもなかった。
		if isEffective == False:
			self.logMsg.print_log(DEBUG1LEVEL, u'属性値が設定されていません。',
				file=self.input_csv, lineNo=self.lineNo)
			#実データがないので、次の行へ
			return False

		return True

	'''
		instance_attributesモード時、サブヘッダで指定した列(属性列)に実データが入っているかチェック
		引数
			csvList：CSVファイル1行分のリスト
			attrList：サブヘッダの情報を保持したリスト(属性名, 列番号)
		戻り値
			True:attrListに登録してある属性列に1列でも実データが入っている
			False:attrListで保持した属性列に実データが1つもない
	'''
	def check_inst_attributes(self, csvList, attrList):
		#属性列に実データがあるかどうかの判定
		isEffective =  self.have_effective_val(csvList, attrList)
		#属性列に実データが1つもなかった。
		if isEffective == False:
			self.logMsg.print_log(DEBUG1LEVEL, u'属性値が設定されていません。',
				file=self.input_csv, lineNo=self.lineNo)
			#実データがないので、次の行へ
			return False

		return True

	'''
		constraints_rsc_locationモード時、サブヘッダで指定した列(属性列)に実データが入っているかチェック
		引数
			csvList：CSVファイル1行分のリスト
			rscIdPos:対象リソースid属性列の列番号
		戻り値
			True:rscIdPos列に実データが入っている
			False:rscIdPos列に実データが入ってない
	'''
	def check_const_rsc_location(self, csvList, rscIdPos):
		#resourceId列に実データがあるかどうかの判定
		isEffective =  self.have_effective_required_val(csvList, [rscIdPos])
		#必須属性列に実データが1つもなかった。
		if isEffective == False:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'%s列に値が設定されていません。' % CRL_RESOURCE_ID,
				file=self.input_csv, lineNo=self.lineNo)
			self.errFlg = True
			#エラーが発生しても、1モード分の処理は続けるため、次の行の処理へ移る
			return False

		return True

	'''
		cib.xml生成処理のメインメソッド
		引数なし
		戻り値
			0:正常にcib.xml生成処理が終わり、crm_verifyでも以上がない
			1:cib.xml生成処理中にエラー発生
			2:cib.xml生成中に警告発生
			3:crm_verifyでエラーか警告が発生
			4:cib.xml生成中に警告＆crm_verifyでエラーか警告が発生
			5:crm_verifyが実行できなかった時や、crm_verifyの実行に失敗した時
	'''
	def generate_cib(self):

		#cib.xml生成処理の開始を知らせる
		self.logMsg.print_log(DEBUG1LEVEL, u'cib.xml生成処理を開始します。', file=self.input_csv)

		#変数の初期化
		csvList = []
		attrList = []
		cibgen_ret = 0
		verify_ret = 0
		self.resource_mode_flg = False
		#nameアトリビュートが存在しない場合に使用するIDとして使用するシーケンス番号
		#nameアトリビュートは現在DTD上必須ではあるが、安全策のため
		self.idSeq = 0
		self.instattr_rscList = []
		self.operation_rscList = []
		self.rscloc_rscList = []

		try:
			fd = open(self.input_csv, 'r')
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルのオープンに失敗しました。[%s]' % self.input_csv)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=self.input_csv)
			self.logMsg.print_quit_message()
			sys.exit(1)

		#CSVファイルのOPEN & CSVオブジェクトの作成
		try:
			csvReader = csv.reader(fd)
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルの読み込みに失敗しました。[%s]' % self.input_csv)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=self.input_csv)
			self.logMsg.print_quit_message()
			#closeに失敗しても終了するだけなので、エラーチェックは行わない
			fd.close()
			sys.exit(1)

		#CSVファイルから1行読み出す処理をファイルの最後まで行う。
		while 1:
			try:
				csvList = csvReader.next()
			except StopIteration:
				#StopIterationが送出された場合、csvReaderオブジェクトから取り出す要素が
				#もうないことを表す(CSVファイルを終端まで読み込んだ)
				break
			except Exception, msg:
				self.logMsg.print_log(ERRORLEVEL, u'ファイルの読み込みに失敗しました。[%s]' % self.input_csv)
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=self.input_csv)
				self.logMsg.print_quit_message()
				#closeに失敗しても終了するだけなので、エラーチェックは行わない
				fd.close()
				sys.exit(1)


			#各モード処理の戻り値初期化
			mode_ret = 0

			#行番号をループごとに + 1
			self.lineNo = self.lineNo + 1

			#listの各要素を文字コード変換し、前後の空白を削除する。
			csvList = self.conversion_csv(csvList)

			#カンマなしの改行のみの行に対する処理
			if len(csvList) == 0:
				self.logMsg.print_log(DEBUG1LEVEL, u'改行のみの行です。',
					file=self.input_csv, lineNo=self.lineNo)
				continue

			#要素の先頭が'#'なら何もしない
			if (csvList[0].startswith(COMMENT_CHAR)):
				self.logMsg.print_log(DEBUG1LEVEL, u'コメント行です。',
					file=self.input_csv, lineNo=self.lineNo)
				continue
			elif csvList[0] != '':
				self.logMsg.print_log(DEBUG1LEVEL, u'列Aに#以外から始まる文字列があります。',
					file=self.input_csv, lineNo=self.lineNo)
				#空文字を代入
				csvList[0] = ''

			#行が空行(改行・空白文字のみ)
			if (csvList.count('') == (len(csvList) - csvList.count('\n'))):
				self.logMsg.print_log(DEBUG1LEVEL, u'行内に有効なデータがありません。',
					file=self.input_csv, lineNo=self.lineNo)
				continue

			'''

				メインヘッダ列解析処理(mode切り替え処理)部分

			'''
			if csvList[M_HEADER_POS] != '':

				#前回処理したモードでエラーが出ていた場合、終了する。
				if self.errFlg == True:
					self.logMsg.print_quit_message()
					#closeに失敗しても終了するだけなので、エラーチェックは行わない
					fd.close()
					return 1

				#メインヘッダ行解析処理の開始を知らせる
				self.logMsg.print_log(DEBUG1LEVEL, u'メインヘッダ解析処理を開始します。',
					file=self.input_csv, lineNo=self.lineNo)

				#新しいモードがセットされたら、アトリビュートリストをNoneクリア
				attrList = None

				#メインヘッダがどのモードなのか特定する
				ret = self.analyze_main_header(csvList)
				if ret == 1:
					self.logMsg.print_quit_message()
					#メッセージは呼び出し先で出力しているので、ここでは出力しない
					#closeに失敗しても終了するだけなので、エラーチェックは行わない
					fd.close()
					return 1

				if self.modeStr == 'Resources':
					#モードがResources時の初期化処理
					ret = self.init_resource_mode()
					if ret == 1:
						self.logMsg.print_quit_message()
						#メッセージは呼び出し先で出力しているので、ここでは出力しない
						#closeに失敗しても終了するだけなので、エラーチェックは行わない
						fd.close()
						return 1

				elif self.modeStr == 'InstanceAttributes':
					#rscIDのノードを特定
					rscId = self.get_rscid(csvList)
					if rscId in self.instattr_rscList:
						self.logMsg.print_log(WARNLEVEL, u'リソース("%s")のパラメータ設定部が複数あります。' % rscId,
							file=self.input_csv, lineNo=self.lineNo)
						self.warnFlg = True

					self.instattr_rscList.append(rscId)

					#対象リソースidの取得に失敗した。
					if rscId == None:
						self.logMsg.print_quit_message()
						#closeに失敗しても終了するだけなので、エラーチェックは行わない
						fd.close()
						return 1

					#対象リソースIDを持つノードが複数ある場合(1つでも)、一番初めに見つかったノードとシーケンスのペアを返す
					rsc_tuple = self.init_instattr_mode(rscId)
					if rsc_tuple != None:
						#tupleをリソースノードとシーケンスに分割
						findedIdNode, seq = tuple(rsc_tuple)

				#モードの切り替え処理を終えて、次の行へ
				continue

			#skipモード中は次のメインヘッダ行が来るまで、何もしない
			if self.modeStr == 'skip':
				self.logMsg.print_log(DEBUG1LEVEL, u'次のメインヘッダ行までスキップします。',
					file=self.input_csv, lineNo=self.lineNo)
				continue

			'''

				サブヘッダ行解析処理部分

			'''
			if attrList == None and self.modeStr != '':

				#サブヘッダ解析処理のの開始を知らせる
				self.logMsg.print_log(DEBUG1LEVEL, u'サブヘッダ解析処理を開始します。', file=self.input_csv, lineNo=self.lineNo)

				#attrListの初期化
				attrList = []
				#必須アトリビュートを保存するリスト
				reqAttrList = []
				#サブヘッダの属性列の位置情報を全て保持するリスト
				allAttrList = []

				#サブヘッダ解析処理
				self.analyze_sub_header(csvList, reqAttrList, attrList, allAttrList)

				#各モードの必須項目に関する処理
				if self.modeStr == 'Resources':
					rscPosList, idPosList = tuple(reqAttrList[0])
					#Resourceモード時、ResourceItem列の存在チェック
					if self.check_req_attrlist(rscPosList, RESOURCE_ITEM) == False:
						self.logMsg.print_quit_message()
						#メッセージは呼び出し先で出力しているので、ここでは出力しない
						#closeに失敗しても終了するだけなので、エラーチェックは行わない
						fd.close()
						return 1

					#Resourceモード時、id列の存在チェック
					if self.check_req_attrlist_solitary(idPosList, RESOURCE_ID) == False:
						self.logMsg.print_quit_message()
						#メッセージは呼び出し先で出力しているので、ここでは出力しない
						#closeに失敗しても終了するだけなので、エラーチェックは行わない
						fd.close()
						return 1

					#対象列が1つの場合、その列を使用する
					idPos = idPosList[0]

				elif self.modeStr == 'Operations':
					#Operationモード時、PrimitiveId列の存在チェック
					if self.check_req_attrlist_solitary(reqAttrList, PRIMITIVE_ID) == False:
						self.logMsg.print_quit_message()
						#メッセージは呼び出し先で出力しているので、ここでは出力しない
						#closeに失敗しても終了するだけなので、エラーチェックは行わない
						fd.close()
						return 1

					#Operationモード時、サブヘッダ必須以外の属性列のフォーマットチェック
					if self.check_op_subheader_format(attrList) == False:
						self.logMsg.print_quit_message()
						#メッセージは呼び出し先で出力しているので、ここでは出力しない
						#closeに失敗しても終了するだけなので、エラーチェックは行わない
						fd.close()
						return 1

					#対象列が複数ある場合(1つでも)、一番初めに見つかった列を使用する
					prmIdPos = reqAttrList[0]

				elif self.modeStr == 'Constraints_RscLocation':
					#Constraints_RscLocationモード時、ResourceId列の存在チェック
					if self.check_req_attrlist_solitary(reqAttrList, CRL_RESOURCE_ID) == False:
						self.logMsg.print_quit_message()
						#メッセージは呼び出し先で出力しているので、ここでは出力しない
						#closeに失敗しても終了するだけなので、エラーチェックは行わない
						fd.close()
						return 1

					#Constraints_RscLocationモード時、サブヘッダ必須以外の属性列のフォーマットチェック
					if self.check_rscloc_subheader_format(attrList) == False:
						self.logMsg.print_quit_message()
						#メッセージは呼び出し先で出力しているので、ここでは出力しない
						#closeに失敗しても終了するだけなので、エラーチェックは行わない
						fd.close()
						return 1

					#対象列が複数ある場合(1つでも)、一番初めに見つかった列を使用する
					rscIdPos = reqAttrList[0]

				#サブヘッダ解析処理を終えて、次の行へ
				continue

			'''

				実データ解析処理部分

			'''
			if self.modeStr != '' and attrList != None:
				#実データ列数が、サブヘッダに指定された属性列数より少ない
				isListCount = self.check_list_length(csvList, max(allAttrList))
				if isListCount == False:
					self.logMsg.print_log(DEBUG1LEVEL, u'サブヘッダと実データの列数を合わせるため、実データに空の列を追加しました。',
						file=self.input_csv, lineNo=self.lineNo)

					#max(allAttrList)は位置情報なので、リストの数と合わせるため、+1している
					num = (max(allAttrList) + 1) - len(csvList)

					for i in range(num):
						#足りない分の列を空文字で埋める
						csvList.append('')

				#一時フラグ
				flg = False
				#サブヘッダ列(属性値の列)に値が入っているかチェック
				for i in allAttrList:
					if csvList[i] != '':
						flg = True
				if flg == False:
					continue

			if self.modeStr == '':
				#モードがセットされる前に実データ行が来たときの処理
				self.logMsg.print_format_error(self.input_csv, self.lineNo)
				self.logMsg.print_log(DEBUGLEVEL, u'メインヘッダ行の前に実データ行があります。',
					file=self.input_csv, lineNo=self.lineNo)
				self.logMsg.print_quit_message()
				return 1

			if self.modeStr == 'ClusterPropertySet_Attributes':
				self.logMsg.print_log(DEBUG1LEVEL, u'クラスタ・プロパティ部のcib.xml生成処理を行います。',
					file=self.input_csv, lineNo=self.lineNo)

				#ClusterPropertySetの属性列にデータが入っているかチェック
				check_ret = self.check_cluster_propertyset(csvList, attrList)
				if check_ret == False:
					continue

				#ClusterPropertySetのcib.xml生成処理を行う。
				mode_ret = self.create_cluster_property(csvList, attrList)

			elif self.modeStr == 'Resources':

				self.logMsg.print_log(DEBUG1LEVEL, u'リソース構成部のcib.xml生成処理を行います。',
					file=self.input_csv, lineNo=self.lineNo)

				#Resourcesの属性列にデータが入っているかチェック
				check_ret = self.check_resources(csvList, attrList, idPos, rscPosList)
				if check_ret == False:
					continue

				#Resourcesのcib.xml生成処理を行う。
				mode_ret = self.create_resource(csvList, attrList, rscPosList, idPos)

			elif self.modeStr == 'Operations':

				self.logMsg.print_log(DEBUG1LEVEL, u'リソース動作部のcib.xml生成処理を行います。',
					file=self.input_csv, lineNo=self.lineNo)

				#Operationsの属性列にデータが入っているかチェック
				check_ret = self.check_operations(csvList, attrList, prmIdPos)
				if check_ret == False:
					continue

				#Operationsのcib.xml生成処理を行う。
				mode_ret = self.create_operation(csvList, attrList, prmIdPos)

			elif self.modeStr == 'InstanceAttributes':

				self.logMsg.print_log(DEBUG1LEVEL, u'リソース・パラメータ部のcib.xml生成処理を行います。',
					file=self.input_csv, lineNo=self.lineNo)

				#InstanceAttributesの属性列にデータが入っているかチェック
				check_ret = self.check_inst_attributes(csvList, attrList)
				if check_ret == False:
					continue

				#InstanceAttributesのcib.xml生成処理を行う。
				mode_ret = self.create_instance_attribute(findedIdNode, csvList, attrList)

			elif self.modeStr == 'Constraints_RscLocation':

				self.logMsg.print_log(DEBUG1LEVEL, u'リソース配置制約部のcib.xml生成処理を行います。',
					file=self.input_csv, lineNo=self.lineNo)

				#Constraints_RscLocationの属性列にデータが入っているかチェック
				check_ret = self.check_const_rsc_location(csvList, rscIdPos)
				if check_ret == False:
					continue

				#Constraints_RscLocationのcib.xml生成処理を行う。
				mode_ret = self.create_resource_location(csvList, attrList, rscIdPos)

			else:
				#メインヘッダ解析時にエラーチェック済みのため、ここではチェック不要
				pass

			#各モードでエラーなどが発生したら知らせる
			if mode_ret == 1:
				self.logMsg.print_quit_message()
				#closeに失敗しても終了するだけなので、エラーチェックは行わない
				fd.close()
				return 1
			elif mode_ret == 2:
				self.logMsg.print_log(DEBUG1LEVEL, u'%sモードのcib.xml生成処理中に問題が発生しました。' % self.modeStr,
					file=self.input_csv)
				continue

		#ここまででCSVファイル読み込み終了

		#一番最後に処理したモードでエラーが出ていた場合、終了する。
		if self.errFlg == True:
			self.logMsg.print_quit_message()
			#closeに失敗しても終了するだけなので、エラーチェックは行わない
			fd.close()
			return 1

		try:
			fd.close()
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルのクローズに失敗しました。[%s]' % self.input_csv)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg, file=self.input_csv)
			self.logMsg.print_quit_message()
			#closeに失敗後に処理が続くため、エラー終了
			sys.exit(1)

		#CSV変換処理中に警告が出ていた場合の処理
		if self.warnFlg == True:

			cibgen_ret = 1

		#生成したcib.xmlがテンプレートから読み込んだ状態から変化があるかチェック
		ret = self.createCib.check_cibxml(self.input_csv)
		if ret == False:
			self.logMsg.print_quit_message()
			return 1

		#生成したcib.xmlを出力する処理
		cibBuff = self.call_output()

		#-Qオプションが指定されなかったら
		if self.exec_verify == True:
			verify_ret = self.check_crm_verify(cibBuff)

		if self.errFlg == True:
			self.logMsg.print_quit_message()
		elif self.warnFlg == True and self.logLevel < WARNLEVEL:
			#WARNLEVELのログを出力しないようにしていた場合の処理
			sys.stderr.write(u'cib.xml生成処理で警告が発生しています。\n警告のメッセージを出力するには、"-l V"オプションを指定してください。\n')

		#最終的なプログラムの戻り値を決定する処理
		return self.generate_return_code(cibgen_ret, verify_ret)

	'''
		クラスタプロパティを設定するcib.xml作成処理
		引数
			csvList：入力ファイルの１行分の文字データ
			attrList：クラスタプロパティのアトリビュートリスト
		戻り値
			0：cib.xml作成処理成功
			1：モードをやりきらずに終了すべきエラーが発生（テンプレート内から挿入位置の特定に失敗など
	'''
	def create_cluster_property(self, csvList, attrList):

		#挿入先ノードの特定
		markNodeList = self.createCib.searchNodeList(None, TARGET_NODE_DIC.get(self.modeStr))

		#挿入先が見つからない場合
		if len(markNodeList) == 0:
			self.logMsg.print_log(ERRORLEVEL, u'XMLファイルが不正です。[%s]' % CIB_TEMPLATE_DIC.get('Base_cib'))
			self.logMsg.print_log(DEBUGLEVEL, u'"<%s/>"が存在しません。' % TARGET_NODE_DIC.get(self.modeStr))
			return 1
		elif len(markNodeList) > 1:
			self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"<%s/>"が複数存在します。[%s]' % (TARGET_NODE_DIC.get(self.modeStr), CIB_TEMPLATE_DIC.get('Base_cib')))
			self.warnFlg = True

		markNode = markNodeList[0]

		#ノードオブジェクトの作成、失敗時にはエラー終了するため、戻り値チェックなし
		node = self.createCib.create_node('nvpair')

		self.createCib.set_dic_attribute(node, attrList, csvList)

		parentId = self.createCib.get_parentNodeId(markNode)

		if parentId == '':
			self.logMsg.print_log(DEBUG1LEVEL, u'親ノードのidが存在しません。',
				file=self.input_csv, lineNo=self.lineNo)

		#自ノードのname属性値を取得
		attrVal = self.createCib.get_attribute(node, 'name')
		if attrVal == '':
			self.logMsg.print_log(DEBUG1LEVEL, u'自ノードにname属性が存在しません。',
				file=self.input_csv, lineNo=self.lineNo)

		#自ノードのタグ名+全体シーケンス
		if attrVal == '' or parentId == '':
			#自ノードにname属性がなかった、また親ノードにid属性がなかった場合、シーケンス番号を振る
			self.idSeq = self.idSeq + 1
			self.createCib.set_attribute(node, 'id', node.tagName + IDSEQ_SEPARATER * 2 + str(self.idSeq))
		else:
			#自ノードのname属性でidを作成する
			self.createCib.set_attribute(node, 'id', parentId + ID_SEPARATER + attrVal)

		self.createCib.append_child_node(markNode.parentNode, node)

		return 0

	'''
		リソース構成設定用cib.xml作成処理
		引数
			csvList：入力ファイルの１行分の文字データ
			attrList：クラスタプロパティのアトリビュートリスト
			rscPosList:ResourceItem列の列情報を保持したリスト
		戻り値
			0：cib.xml作成処理成功
			1：モードをやりきらずに終了すべきエラーが発生（テンプレート内から挿入位置の特定に失敗など
			2：エラーが発生したが1モード分の処理はやりきる（CSVファイルのフォーマットが異常など
	'''
	def create_resource(self, csvList, attrList, rscPosList, idPos):

		rscModeList = []
		rscMode = None
		#currentNodeの階層情報
		depth = 0
		#currentNodeが前回追加nodeと親子関係の時だけ
		#フラグをTrueにする
		wFlg = False

		#現在何のリソースを追加しようとしているのか特定する
		for i, rscPos in enumerate(rscPosList):
			#指定列に値が入っていたら
			if csvList[rscPos] != '':
				#何のリソースを追加するのかリソースモードを特定
				rscMode = SUBMODE_DIC.get(csvList[rscPos].lower())
				depth = i
				if rscMode == None:
					self.logMsg.print_format_error(self.input_csv, self.lineNo)
					self.logMsg.print_log(DEBUGLEVEL, u'未定義のリソース種別"%s"が指定されています。' % csvList[rscPos],
						file=self.input_csv, lineNo=self.lineNo)
					self.errFlg = True
					return 2
				else:
					rscSet = (rscMode, depth)
					rscModeList.append(rscSet)

		if len(rscModeList) == 0:
			#リソース構成可能範囲に1つもリソースが設定されていない
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'リソース種別が指定されていません。',
				file=self.input_csv, lineNo=self.lineNo)
			self.errFlg = True
			return 2
		elif len(rscModeList) > 1:
			#リソース構成可能範囲に2つ以上のリソースが設定されていた
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'リソース種別が複数設定されています。',
				file=self.input_csv, lineNo=self.lineNo)
			self.errFlg = True
			return 2
		else:
			(rscMode, depth) = tuple(rscModeList[0])

		#現在追加する階層が前回追加した階層+1より深かった
		if depth > self.prevDepth + 1:
			self.logMsg.print_format_error(self.input_csv, self.lineNo)
			self.logMsg.print_log(DEBUGLEVEL, u'リソース構成の設定に誤りがあります。',
				file=self.input_csv, lineNo=self.lineNo)
			self.errFlg = True
			return 2

		if depth > self.prevDepth:
			#現在の階層のほうが深い＝前回追加したノードの下に追加する
			markNodeList = self.createCib.searchNodeList(self.prevNode, TARGET_NODE_DIC.get(self.modeStr))
			wFlg = True
		elif depth == self.prevDepth:
			#現在の階層と前回追加した階層が同じ
			#	＝前回追加したノードと同じ階層に追加する
			parentNode = self.createCib.get_parentNode(self.prevNode, 1)
			markNodeList = self.createCib.searchNodeList(parentNode, TARGET_NODE_DIC.get(self.modeStr))
		elif depth < self.prevDepth:
			#現在の階層のほうが浅い
			#	＝前回追加したノードの階層との差分を求め、
			#	その差分から自ノードが追加されるべき親ノードを捜す
			parentNode = self.createCib.get_parentNode(self.prevNode, self.prevDepth - depth + 1)
			markNodeList = self.createCib.searchNodeList(parentNode, TARGET_NODE_DIC.get(self.modeStr))
		#挿入先が見つからない場合
		if len(markNodeList) == 0:
			self.logMsg.print_log(ERRORLEVEL, u'次のいずれかのXMLファイルが不正です。[%s/%s/%s/%s/%s]' \
				% (CIB_TEMPLATE_DIC.get('Base_cib'), CIB_TEMPLATE_DIC.get('Primitive'), CIB_TEMPLATE_DIC.get('Group'),
					CIB_TEMPLATE_DIC.get('Clone'), CIB_TEMPLATE_DIC.get('MasterSlave')))
			self.logMsg.print_log(DEBUGLEVEL, u'"<%s/>"が存在しません。' % TARGET_NODE_DIC.get(self.modeStr))
			return 1
		#フラグがTrueの状態なら、テンプレート.xml内に識別用ノードが
		#複数存在する状態になっている（通常は1つのはず)
		elif len(markNodeList) > 1 and wFlg == True:
			self.logMsg.print_log(WARNLEVEL, u'次のいずれかのXMLファイル内に"<%s/>"が複数存在します。[%s/%s/%s/%s/%s]' \
				% (TARGET_NODE_DIC.get(self.modeStr),
					CIB_TEMPLATE_DIC.get('Base_cib'), CIB_TEMPLATE_DIC.get('Primitive'),
					CIB_TEMPLATE_DIC.get('Group'), CIB_TEMPLATE_DIC.get('Clone'),
					CIB_TEMPLATE_DIC.get('MasterSlave')))
			self.warnFlg = True

		markNode = markNodeList[0]


		#特定したリソースでテンプレートノード作成(テンプレート使用) 失敗時にはエラー終了するため、戻り値チェックなし
		node = self.createCib.create_template_node(TEMPLATE_DIR + CIB_TEMPLATE_DIC.get(rscMode))

		#id属性をノードに追加
		self.createCib.set_attribute(node, 'id', csvList[idPos])

		self.createCib.set_dic_attribute(node, attrList, csvList)

		ret = self.set_reqid(node)
		#テンプレート上にREQUIRED_IDがひとつもなかった。
		if ret == 1:
			self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"%s"が存在しません。[%s]' \
				% (REQUIRED_ID, CIB_TEMPLATE_DIC.get(rscMode)))
			self.warnFlg = True

		#リソース・パラメータを設定する場所を示す識別ノード（<INSTANCE_ATTRIBUTE/>）に
		#リソースIDを設定する（-> <INSTANCE_ATTRIBUTE id="リソースID"/>）
		list = self.createCib.searchNodeList(node, TARGET_NODE_DIC.get('InstanceAttributes'))

		#テンプレートXMLに「<INSTANCE_ATTRIBUTE/>」を探しidを振る
		if len(list) == 0:
			#フォーマット異常としてエラー終了
			self.logMsg.print_log(ERRORLEVEL,
				u'XMLファイルが不正です。[%s]' % CIB_TEMPLATE_DIC.get(rscMode))
			self.logMsg.print_log(DEBUGLEVEL, u'"<%s/>"が存在しません。[%s]' \
				% (TARGET_NODE_DIC.get('InstanceAttributes'), CIB_TEMPLATE_DIC.get(rscMode)))
			#1を返しプログラム終了
			return 1

		if len(list) > 1:
			#フォーマット異常として警告
			self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"<%s/>"が複数存在します。[%s]' \
				% (TARGET_NODE_DIC.get('InstanceAttributes'), CIB_TEMPLATE_DIC.get(rscMode)))
			self.warnFlg = True

		#1つ以上存在する場合一番最初に特定したノードにidを振る
		self.createCib.set_attribute(list[0], 'id', csvList[idPos])

		if node.tagName == 'primitive':
			#リソース動作を設定する場所を示す識別ノード（<OPERATION/>）に
			#リソースIDを設定する（-> <OPERATION id="リソースID"/>）
			list = self.createCib.searchNodeList(node, TARGET_NODE_DIC.get('Operations'))

			#テンプレートXMLに「<OPERATION/>」を探す
			if len(list) == 0:
				#フォーマット異常としてエラー終了
				self.logMsg.print_log(ERRORLEVEL,
					u'XMLファイルが不正です。[%s]' % CIB_TEMPLATE_DIC.get(rscMode))
				self.logMsg.print_log(DEBUGLEVEL, u'"<%s/>"が存在しません。[%s]' \
					% (TARGET_NODE_DIC.get('Operations'), CIB_TEMPLATE_DIC.get(rscMode)))
				#1を返しプログラム終了
				return 1

			if len(list) > 1:
				#フォーマット異常として警告
				self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"<%s/>"が複数存在します。[%s]' \
					% (TARGET_NODE_DIC.get('Operations'), CIB_TEMPLATE_DIC.get(rscMode)))
				self.warnFlg = True

			#1つ以上存在する場合一番最初に特定したノードにidを振る
			self.createCib.set_attribute(list[0], 'id', csvList[idPos])

		self.createCib.append_child_node(markNode.parentNode, node)

		self.prevNode = node
		self.prevDepth = depth
		self.addMissDepth = None

		return 0

	'''
		リソース動作設定用cib.xml作成処理
		引数
			csvList：入力ファイルの１行分の文字データ
			attrList：クラスタプロパティのアトリビュートリスト
			prmIdPos:PrimitiveId列の列情報
		戻り値
			0：cib.xml作成処理成功
			1：モードをやりきらずに終了すべきエラーが発生（テンプレート内から挿入位置の特定に失敗など
			2：エラーが発生したが1モード分の処理はやりきる（CSVファイルのフォーマットが異常など
	'''
	def create_operation(self, csvList, attrList, prmIdPos):

		prmId = csvList[prmIdPos]
		if prmId in self.operation_rscList:
			self.logMsg.print_log(WARNLEVEL, u'リソース("%s")の動作設定部が複数あります。' % prmId,
				file=self.input_csv, lineNo=self.lineNo)
			self.warnFlg = True

		self.operation_rscList.append(prmId)

		opList = []

		#primitiveIDのノードを特定
		findedIdNodeList = self.createCib.find_haveAttr_nodelist(None, 'id', prmId, NodeList())

		cnt = 0
		for i, findedIdNode in enumerate(findedIdNodeList):
			node, seq = tuple(findedIdNode)

			#この処理で最初に見つかった「<OPERATION id=prmId/>」のノードを特定
			if node.tagName == TARGET_NODE_DIC.get(self.modeStr):
				cnt = cnt + 1
				if cnt == 1:
					pos = i

		#ノードが見つからない場合
		if cnt == 0:
			self.logMsg.print_log(WARNLEVEL, u'リソース("%s")に動作を設定できませんでした。' % prmId,
				file=self.input_csv, lineNo=self.lineNo)
			self.warnFlg = True
			return 2
		elif cnt > 1:
			self.logMsg.print_log(WARNLEVEL, u'動作設定対象のリソース("%s")が複数見つかりました。' % prmId,
				file=self.input_csv, lineNo=self.lineNo)
			self.warnFlg = True

		markNode, seq = tuple(findedIdNodeList[pos])

		#アトリビュートリストから、属性名と、列情報のペアを得る
		for attrSet in attrList:
			opKey, colPos = tuple(attrSet)
			node = None

			#サブヘッダの値を':'でname属性値と、属性に分割する
			[opName, attr] = opKey.split(SEPARATE_CHAR)

			for op in opList:
				#行内で既に同じ"name=opName"に対して処理を行っていた場合、
				#opListにその"<op name=opName/>"のノードがあるはずなのでそのノードを使用する
				if opName == self.createCib.get_attribute(op, 'name'):
					node = op
					break

			if node == None:
				#<op/>を作る、失敗時にはエラー終了するので、戻り値チェックなし
				node = self.createCib.create_node('op')
				#<op name='opName(start etc..)'/>になる
				self.createCib.set_attribute(node, 'name', opName)
				#<op id='prmId-opName(start etc..)' name='opName(start etc..)'/>になる
				self.createCib.set_attribute(node, 'id', prmId + ID_SEPARATER + opName)

			if csvList[colPos] != '':
				#ノードに属性と属性値を追加する。
				self.createCib.set_attribute(node, attr, csvList[colPos])

			#cib.xml上の挿入位置に、opノードを追加する。
			self.createCib.append_child_node(markNode.parentNode, node)
			opList.append(node)

		#1行分(for)の処理を終えたら、次の行へ
		return 0

	'''
		リソースパラメータ設定用cib.xml作成処理
		引数
			findedIdNode:メインヘッダ行のIDで特定したノード
			csvList：入力ファイルの１行分の文字データ
			attrList：クラスタプロパティのアトリビュートリスト
		戻り値
			0：cib.xml作成処理成功
			1：モードをやりきらずに終了すべきエラーが発生（テンプレート内から挿入位置の特定に失敗など
	'''
	def create_instance_attribute(self, findedIdNode, csvList, attrList):

		#ノードオブジェクトの作成、失敗時にはエラー終了するため、戻り値チェックなし
		node = self.createCib.create_node('nvpair')

		#ノードにアトリビュートを設定
		self.createCib.set_dic_attribute(node, attrList, csvList)

		#1行分のノードをcib.xmlに追加
		self.createCib.append_child_node(findedIdNode.parentNode, node)

		#親ノードのidを取得し自ノードのid生成に使用する。
		parentId = self.createCib.get_parentNodeId(node)
		if parentId == '':
			self.logMsg.print_log(DEBUG1LEVEL, u'親ノードのidが存在しません。',
				file=self.input_csv, lineNo=self.lineNo)

		#自ノードのname属性値を取得
		attrVal = self.createCib.get_attribute(node, 'name')
		if attrVal == '':
			self.logMsg.print_log(DEBUG1LEVEL, u'自ノードにname属性が存在しません。',
				file=self.input_csv, lineNo=self.lineNo)

		if attrVal == '' or parentId == '':
			#自ノードにname属性がなかった、また親ノードにid属性がなかった場合、シーケンス番号を振る
			self.idSeq = self.idSeq + 1
			self.createCib.set_attribute(node, 'id', node.tagName + IDSEQ_SEPARATER * 2 + str(self.idSeq))
		else:
			#自ノードのname属性でidを作成する
			self.createCib.set_attribute(node, 'id', parentId + ID_SEPARATER + attrVal)

		return 0

	'''
		リソース配置制約設定用cib.xml作成処理
		引数
			csvList：入力ファイルの１行分の文字データ
			attrList：クラスタプロパティのアトリビュートリスト
			rscIdPos:ResourceId列の列情報
		戻り値
			0：cib.xml作成処理成功
			1：モードをやりきらずに終了すべきエラーが発生（テンプレート内から挿入位置の特定に失敗など
			2：エラーが発生したが1モード分の処理はやりきる（CSVファイルのフォーマットが異常など
	'''
	def create_resource_location(self, csvList, attrList, rscIdPos):

		if csvList[rscIdPos] in self.rscloc_rscList:
			self.logMsg.print_log(WARNLEVEL, u'リソース("%s")の配置制約部が複数あります。' \
				% csvList[rscIdPos], file=self.input_csv, lineNo=self.lineNo)
			self.warnFlg = True

		self.rscloc_rscList.append(csvList[rscIdPos])

		#指定したノード上から挿入位置</RSC_LOCATION>を探す
		markNodeList = self.createCib.searchNodeList(None, TARGET_NODE_DIC.get(self.modeStr))

		#挿入先が見つからない場合
		if len(markNodeList) == 0:
			self.logMsg.print_log(ERRORLEVEL,
				u'XMLファイルが不正です。[%s]' % CIB_TEMPLATE_DIC.get('Base_cib'))
			self.logMsg.print_log(DEBUGLEVEL,
				u'"<%s/>"が存在しません。' % TARGET_NODE_DIC.get(self.modeStr))
			#1を返しプログラム終了
			return 1
		elif len(markNodeList) > 1:
			self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"<%s/>"が複数存在します。[%s]' \
				% (TARGET_NODE_DIC.get(self.modeStr), CIB_TEMPLATE_DIC.get('Base_cib')))
			self.warnFlg = True

		markNode = markNodeList[0]

		#アトリビュートリストから、属性名と、列情報のペアを得る
		for attrSet in attrList:
			rlKey, colPos = tuple(attrSet)

			#列のデータが空なら処理を行わない
			if csvList[colPos] == '':
				continue

			if (rlKey.lower().startswith('score' + SEPARATE_CHAR)):
				[keyStr, score_val] = rlKey.split(SEPARATE_CHAR)

				for nodeName in csvList[colPos].split():
					self.rsclocSeq = self.rsclocSeq + 1

					#RSC_LOCATION用エレメント作成(テンプレート使用) 失敗時には
					#エラー終了するため、戻り値チェックなし
					node = self.createCib.create_template_node(
							TEMPLATE_DIR + CIB_TEMPLATE_DIC.get('RLnode'))

					#ruleノードを取得
					rule_nodeList = self.createCib.searchNodeList(node, 'rule')

					#テンプレートノード上からruleノードが見つからない場合
					if len(rule_nodeList) == 0:
						self.logMsg.print_log(ERRORLEVEL,
							u'XMLファイルが不正です。[%s]' % CIB_TEMPLATE_DIC.get('RLnode'))
						self.logMsg.print_log(DEBUGLEVEL,
							u'"<rule/>"が存在しません。[%s]' % CIB_TEMPLATE_DIC.get('RLnode'))
						return 1
					elif len(rule_nodeList) > 1:
						self.logMsg.print_log(WARNLEVEL,
							u'XMLファイル内に"<rule/>"が複数存在します。[%s]' \
							% CIB_TEMPLATE_DIC.get('RLnode'))
						self.warnFlg = True

					rule_node = rule_nodeList[0]

					#score属性をruleノードに追加
					self.createCib.set_attribute(rule_node, 'score', score_val)

					#rsc属性をrsc_locationノードに追加
					self.createCib.set_attribute(node, 'rsc', csvList[rscIdPos])

					#自ノードのrsc属性値と固定文字列'node'+シーケンス番号で
					#id属性をrsc_locationノードに追加
					self.createCib.set_attribute(node, 'id',
						node.getAttribute('rsc') + ID_SEPARATER + 'node' + str(self.rsclocSeq))

					#RSC_LOCATIONテンプレートから作成したDOM上から
					#id=REQUIRED_ID_SEQを探し、idを埋める
					ret = self.set_reqid_seq(node)

					#テンプレート上にREQUIRED_ID_SEQがひとつもなかった。
					if ret == 1:
						self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"%s"が存在しません。[%s]' \
							% (REQUIRED_ID_SEQ, CIB_TEMPLATE_DIC.get('RLnode')))
						self.warnFlg = True

					#expressionノードを取得
					expr_nodeList = self.createCib.searchNodeList(node, 'expression')

					#テンプレートノード上からexprノードが見つからない場合
					if len(expr_nodeList) == 0:
						self.logMsg.print_log(ERRORLEVEL,
							u'XMLファイルが不正です。[%s]' % CIB_TEMPLATE_DIC.get('RLnode'))
						self.logMsg.print_log(DEBUGLEVEL,
							u'"<expression/>"が存在しません。[%s]' % CIB_TEMPLATE_DIC.get('RLnode'))
						return 1
					elif len(expr_nodeList) > 1:
						self.logMsg.print_log(WARNLEVEL,
							u'XMLファイル内に"<expression/>"が複数存在します。[%s]' \
							% CIB_TEMPLATE_DIC.get('RLnode'))
						self.warnFlg = True

					expr_node = expr_nodeList[0]

					#value属性をexpressionノードに追加
					self.createCib.set_attribute(expr_node, 'value', nodeName)

					#cib.xml上の挿入位置に、rsc_locationノードを追加する。
					self.createCib.append_child_node(markNode.parentNode, node)

			#アトリビュートリストのkeyがpingdでかつ、
			#その列の値が'y'または'yes'(大文字、小文字混在可)ならpingd制約の追加処理
			elif (rlKey.lower().startswith('pingd' + SEPARATE_CHAR)):
				if csvList[colPos].lower() in USE_CHECK_LIST:
					#pingdNode取得失敗時にはプログラムがエラー終了するので、戻り値チェックなし
					pingdNode = self.createCib.create_template_node(
								TEMPLATE_DIR + CIB_TEMPLATE_DIC.get('RLpingd'))

					self.createCib.set_attribute(pingdNode, 'rsc', csvList[rscIdPos])
					self.createCib.append_child_node(markNode.parentNode, pingdNode)

					self.rsclocSeq = self.rsclocSeq + 1

					#固定文字列'pingd'+制約対象のリソースIDでid属性をrsc_locationノードに追加
					self.createCib.set_attribute(pingdNode, 'id', 'pingd' + ID_SEPARATER + \
						csvList[rscIdPos] + IDSEQ_SEPARATER + str(self.rsclocSeq))

					ret = self.set_reqid_seq(pingdNode)

					#テンプレート上にREQUIRED_ID_SEQがひとつもなかった。
					if ret == 1:
						self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"%s"が存在しません。[%s]' \
							% (REQUIRED_ID_SEQ, CIB_TEMPLATE_DIC.get('RLpingd')))
						self.warnFlg = True

					#<expression>内の、attribute属性とvalue属性に値を設定
					[keyStr, attribute, value] = rlKey.split(SEPARATE_CHAR)

					ret = self.set_attribute_value(pingdNode, 'attribute',
							EXPRESSION_ATTRIBUTE, attribute)
					if ret == 1:
						self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"%s"が存在しません。[%s]' \
							% ('EXPRESSION_ATTRIBUTE', CIB_TEMPLATE_DIC.get('RLpingd')))
						self.warnFlg = True

					ret = self.set_attribute_value(pingdNode, 'value', EXPRESSION_VALUE, value)
					if ret == 1:
						self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"%s"が存在しません。[%s]' \
							% ('EXPRESSION_VALUE', CIB_TEMPLATE_DIC.get('RLpingd')))
						self.warnFlg = True
				else:
					self.logMsg.print_log(DEBUG1LEVEL,
						u'"%s"が指定されているため、pingdの設定は行いません。' \
						% csvList[colPos], file=self.input_csv, lineNo=self.lineNo)

			#アトリビュートリストのkeyがdiskdでかつ、
			#その列の値が'y'または'yes'(大文字、小文字混在可)ならdiskd制約の追加処理
			elif (rlKey.lower().startswith('diskd' + SEPARATE_CHAR)):
				if csvList[colPos].lower() in USE_CHECK_LIST:
					#diskdNode取得失敗時にはプログラムがエラー終了するので、戻り値チェックなし
					diskdNode = self.createCib.create_template_node(
								TEMPLATE_DIR + CIB_TEMPLATE_DIC.get('RLdiskd'))

					self.createCib.set_attribute(diskdNode, 'rsc', csvList[rscIdPos])
					self.createCib.append_child_node(markNode.parentNode, diskdNode)

					self.rsclocSeq = self.rsclocSeq + 1

					#固定文字列'diskd'+制約対象のリソースIDでid属性をrsc_locationノードに追加
					self.createCib.set_attribute(diskdNode, 'id', 'diskd' + ID_SEPARATER + \
						csvList[rscIdPos] + IDSEQ_SEPARATER + str(self.rsclocSeq))

					ret = self.set_reqid_seq(diskdNode)

					#テンプレート上にREQUIRED_ID_SEQがひとつもなかった。
					if ret == 1:
						self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"%s"が存在しません。[%s]' \
							% (REQUIRED_ID_SEQ, CIB_TEMPLATE_DIC.get('RLdiskd')))
						self.warnFlg = True

					#<expression>内の、attribute属性に値を設定
					[keyStr, attribute] = rlKey.split(SEPARATE_CHAR)

					ret = self.set_attribute_value(diskdNode, 'attribute',
							EXPRESSION_ATTRIBUTE, attribute)
					if ret == 1:
						self.logMsg.print_log(WARNLEVEL, u'XMLファイル内に"%s"が存在しません。[%s]' \
							% ('EXPRESSION_ATTRIBUTE', CIB_TEMPLATE_DIC.get('RLdiskd')))
						self.warnFlg = True
				else:
					self.logMsg.print_log(DEBUG1LEVEL,
						u'"%s"が指定されているため、diskdの設定は行いません。' \
						% csvList[colPos], file=self.input_csv, lineNo=self.lineNo)

		#0を返し次の行へ
		return 0

	'''
		crm_verifyを使用し、DTDチェックを行うメソッド
		引数
			cibBuff：コンソールに出力したcib.xmlのバッファ（コンソール出力時以外は使われない)
		戻り値
			DTDチェック問題なし：0
			DTDチェックでエラーまたは警告が発生：1
			crm_verifyの実行に失敗：2
	'''
	def check_crm_verify(self, cibBuff):

		self.logMsg.print_log(DEBUG1LEVEL, u'%sコマンドによるDTDチェックを行います。' % CRM_VERIFY_CMD)

		#crm_verifyのフルパスを取得し, 存在チェックと実行権チェックを行う
		(status, exec_cmdpath) = \
			commands.getstatusoutput('which ' + CRM_VERIFY_CMD)
		if (os.WIFEXITED(status) == False or os.WEXITSTATUS(status) != 0) \
			or (os.access(exec_cmdpath, os.F_OK | os.X_OK) == False):
			self.logMsg.print_log(ERRORLEVEL, u'コマンドが実行できません。[%s]' % CRM_VERIFY_CMD)
			self.errFlg = True
			return 2

		#実行コマンドの生成
		if self.output_file != sys.stderr and self.output_file != sys.stdout:
			cmdList = [exec_cmdpath, CRM_VERIFY_FILE_OPT, self.output_file]
		else:
			cmdList = [exec_cmdpath, CRM_VERIFY_STR_OPT, cibBuff]

		#crm_verifyに渡すログレベルが0より大きければ、Vオプションを生成する
		if self.verifyLevel != '':
			cmdList.append(self.verifyLevel)

		#popenに失敗＝shの失敗になりtryしてもハンドリングできない
		try:
			p = popen2.Popen4(cmdList)
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'コマンドの実行に失敗しました。[%s]' % CRM_VERIFY_CMD)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.errFlg = True
			return 2

		#crm_verifyの結果をファイルに出力する場合
		if self.output_dtd_check != sys.stderr and self.output_dtd_check != sys.stdout:
			try:
				verifyLog_fd = codecs.open(self.output_dtd_check, 'w', encoding=OUTPUT_STRING_CODE)
			except Exception, msg:
				self.logMsg.print_log(ERRORLEVEL, u'ファイルのオープンに失敗しました。[%s]' % self.output_dtd_check)
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
				self.errFlg = True
				return 2
		else:
			verifyLog_fd = self.output_dtd_check

		try:
			for line in p.fromchild:
				verifyLog_fd.write(line)
		except Exception, msg:
			self.logMsg.print_log(ERRORLEVEL, u'ファイルの書き込みに失敗しました。[%s]' % self.output_dtd_check)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.errFlg = True
			return 2

		#子プロセスのstdinとstdoutをcloseする
		p.fromchild.close()
		p.tochild.close()
		#子プロセスの終了コードを受け取る
		retstat = p.wait()

		try:
			if self.output_dtd_check != sys.stderr and self.output_dtd_check != sys.stdout:
				verifyLog_fd.close()
		except Exception, msg:
			self.logMsg.print_log(WARNLEVEL, u'ファイルのクローズに失敗しました。[%s]' % self.output_dtd_check)
			self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
			self.warnFig = True
			return 2

		if os.WIFEXITED(retstat) == True:
			#crm_verifyがexitにより終了した場合の処理
			if os.WEXITSTATUS(retstat) != 0:
				#crm_verifyがリターンコード0以外で終了した場合
				return 1
			else:
				#crm_verifyがリターンコード0で終了した場合
				return 0
		else:
			#crm_verifyがkillによって終了した場合
			self.errFlg = True
			return 2

	'''
		DOMの内容をバッファに格納し、指定されたファイルディスクリプタに書き出すメソッド
		cib.xmlの出力先にコンソールが指定されていた場合、このバッファをDTDチェック対象とする
		引数：なし
		戻り値
			cib.xml出力メソッドから返されたバッファを返す
			ファイルの操作失敗時はAPIエラーとしてexit(1)
	'''
	def call_output(self):

		#識別用に使ったテンプレート上のノードをリストに従い削除
		for key, val in TARGET_NODE_DIC.iteritems():
			self.createCib.find_node_del(val)

		buff = self.createCib.outputXml()

		if self.output_file == sys.stdout or self.output_file == sys.stderr:
			#self.output_file = 標準出力
			try:
				self.output_file.write(buff)
			except Exception, msg:
				self.logMsg.print_log(ERRORLEVEL, u'cib.xmlの出力に失敗しました。[標準出力]')
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
				self.logMsg.print_quit_message()
				#APIエラー
				sys.exit(1)

		else:
			#self.output_file = ユーザ指定ファイル
			try:
				outputXml_fd = codecs.open(self.output_file, 'w', encoding=OUTPUT_STRING_CODE)
			except Exception, msg:
				self.logMsg.print_log(ERRORLEVEL, u'ファイルのオープンに失敗しました。[%s]' % self.output_file)
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
				self.logMsg.print_quit_message()
				#APIエラー
				sys.exit(1)

			try:
				outputXml_fd.write(buff)
			except Exception, msg:
				self.logMsg.print_log(ERRORLEVEL, u'cib.xmlの出力に失敗しました。[%s]' % self.output_file)
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
				self.logMsg.print_quit_message()
				#outputXml_fd.close() #close()に失敗したときにファイルが残るのを避けるためコメントアウト
				os.remove(self.output_file)
				#APIエラー
				sys.exit(1)

			try:
				outputXml_fd.close()
			except Exception, msg:
				self.logMsg.print_log(ERRORLEVEL, u'ファイルのクローズに失敗しました。[%s]' % self.output_file)
				self.logMsg.print_log(DEBUGLEVEL, u'%s' % msg)
				self.logMsg.print_quit_message()
				#APIエラー
				sys.exit(1)

		#出力したバッファを返す
		return buff

'''
	ログメッセージ用クラス
'''
class LogMsg:

	'''
		初期化メソッド
		引数
			loglevel：基準となるレベル
		戻り値なし
	'''
	def __init__(self, loglevel):

		self.standard_loglevel = loglevel

	'''
		表示したいログレベルが基準レベル以下か判断するメソッド
		引数
			loglevel：判断したいレベル
		戻り値
			True:表示する
			False:表示しない
	'''
	def judge_loglevel(self, loglevel):

		if loglevel <= self.standard_loglevel:
			return True
		else:
			return False

	'''
		内部エラーが発生した場合呼ばれるメソッド
		引数
			factor:内部エラーの原因メッセージ
		戻り値なし
	'''
	def print_inner_error(self, factor):
		self.print_log(ERRORLEVEL, u'内部エラーが発生しました。(%s)' % factor)


	'''
		内部エラーが発生した場合呼ばれるメソッド
		引数
			file:エラーが発生したファイル
			lineNo:エラー発生行
		戻り値なし
	'''
	def print_format_error(self, filePath, lineNum):
		self.print_log(ERRORLEVEL, u'フォーマット・エラーが見つかりました。', file=filePath, lineNo=lineNum)


	'''
		cib.xml生成処理の中止を知らせるメッセージを出力するメソッド
		引数なし
		戻り値なし
	'''
	def print_quit_message(self):
		sys.stderr.write(u'cib.xml生成処理でエラーが発生しました。処理を中止します。\n')

	'''
		ログをstderrに出力するメソッド
		引数
			level：メッセージのレベル
			msg：メッセージ
			file：ファイル(デフォルトはNone)
			lineNo：行番号(デフォルトはNone)
		戻り値なし
	'''
	def print_log(self, level, msg, file=None, lineNo=None):

		judge = self.judge_loglevel(level)

		if judge == False:
			return None

		logLevel = LOGLEVEL_DIC.get(level)
		#行数の指定があった場合
		if lineNo != None:
			lineNo = '(%s)' % lineNo

		#ファイルの指定があった場合、ファイル名だけを取得する
		if file != None:
			file = get_filename(file)

		if file != None and lineNo != None:
			sys.stderr.write('%s: %s%s: %s\n' % (logLevel, file, lineNo, msg))
		elif file != None:
			sys.stderr.write('%s: %s: %s\n' % (logLevel, file, msg))
		else:
			sys.stderr.write('%s: %s\n' % (logLevel, msg))

'''
	前後の空白(全角半角スペース＆タブ)をなくなるまで取り去る
	引数
		string:対象文字列
	戻り値
		処理後の文字列
'''
def del_blank(string):

	#文字列の初期サイズを取得
	length = len(string)

	#半角スペースとタブを消去
	string = string.strip()
	#全角スペースを消去
	string = string.strip(u'　')

	#メソッドに入ったときと文字列長が変化していなければ、前後の空白がなくなったものとする
	if len(string) == length:
		return string
	else:
		string = del_blank(string)

	#前後の空白(改行も)を取り除いた文字列を返す。または空文字列を返す。
	return string

'''
	ファイルパスからファイル名だけを抜き取り、返す
	引数
		filepath：パスつきのファイル名
	戻り値
		filename：パスを取り除いたファイル名
'''
def get_filename(filepath):

	filesep = filepath.split(DIR_SEPARATER)
	filename = filesep[-1]

	return filename

'''
	プログラムのバージョン情報を表示する
	引数
		OptionParserのコールバックを使う場合必ず入れなくてはいけないもの(実際使ってない）
	戻り値
		exit(0)でプログラム終了
'''
def print_version(option, opt, value, parser):
	sys.stdout.write('%s %s\n' % (PROG_NAME, PROG_VERSION))
	sys.exit(0)

'''
	main処理へのエントリーポイント
'''
if __name__ == '__main__':

	#CsvReadクラスインスタンスの作成
	csvRead = CsvRead()

	#cib.xml生成処理の呼び出し
	ret = csvRead.generate_cib()

	#generate_cibからの戻り値でプログラム終了
	sys.exit(ret)

