/**
 * @file  DataManager.cpp
 * @brief ListPaste f[^}l[WNX.
 *
 * @author JIN
 *
 * Copyright (C) 2011- JIN All rights reserved.
 */
#include "StdAfx.h"
#include "DataManager.h"

#include "../GenericLib/MathUtility.h"
#include "../GenericLib/SerializeObject.h"

namespace ListPasteLib {

	namespace {

		/**
		 * text  search Ŏn܂Ă邩ǂ𒲂ׂ.
		 */
		inline bool MatchFirst(const std::wstring& text, const std::wstring& search)
		{
			// HACK: 啶͋ʂȂ
			return _wcsnicmp(text.c_str(), search.c_str(), search.size()) == 0;
		}

		/**
		 * text  search ܂܂Ă邩ǂ𒲂ׂ.
		 */
		inline bool MatchAny(const std::wstring& text, const std::wstring& search)
		{
			// HACK: 啶͋ʂȂ
			return ::StrStrIW(text.c_str(), search.c_str()) != NULL;
		}

	}	// anonymous namespace

	////////////////////////////////////////////////////////////////////////////////

	CDataGroup::CDataGroup()
	{
	}

	CDataGroup::CDataGroup(const std::wstring& name)
		: m_name(name)
	{
	}

	CDataGroup::~CDataGroup()
	{
	}

	void CDataGroup::Swap(CDataGroup& group)
	{
		m_name.swap(group.m_name);
		m_texts.swap(group.m_texts);

		// HACK: Ƒ̕ύXtO𗧂Ă
		SetDirty(true);
		group.SetDirty(true);
	}

	void CDataGroup::SetName(const std::wstring& name)
	{
		m_name = name;

		SetDirty(true);
	}

	const std::wstring& CDataGroup::GetName() const
	{
		return m_name;
	}

	bool CDataGroup::Add(size_t index, const std::wstring& text)
	{
		if (index > m_texts.size()) {
			index = m_texts.size();
		}

		WStrings::iterator it = m_texts.begin();
		std::advance(it, index);
		m_texts.insert(it, text);

		SetDirty(true);
		return true;
	}

	bool CDataGroup::Add(size_t index, const std::vector<std::wstring>& texts)
	{
		if (index > m_texts.size()) {
			index = m_texts.size();
		}

		WStrings::iterator it = m_texts.begin();
		std::advance(it, index);
		m_texts.insert(it, texts.begin(), texts.end());

		SetDirty(true);
		return true;
	}

	bool CDataGroup::Modify(size_t index, const std::wstring& text)
	{
		if (index >= m_texts.size()) {
			return false;
		}

		WStrings::iterator it = m_texts.begin();
		std::advance(it, index);
		
		*it = text;

		SetDirty(true);
		return true;
	}

	bool CDataGroup::Delete(size_t index)
	{
		if (index >= m_texts.size()) {
			return false;
		}

		WStrings::iterator it = m_texts.begin();
		std::advance(it, index);
		m_texts.erase(it);

		SetDirty(true);
		return true;
	}

	const CDataGroup::WStrings& CDataGroup::GetTexts() const
	{
		return m_texts;
	}

	void CDataGroup::Search(
		const std::wstring& search,
		bool matchFirst,
		IndexVec& indices
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		, GenericUtility::CProgressManager* pProgress
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		)
		const
	{
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		if (pProgress) {
			pProgress->StartProgress(m_texts.size());
		}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS

		indices.clear();
		// 擪̂݃}b`
		if (matchFirst) {
			for (WStrings::const_iterator it = m_texts.begin(); it != m_texts.end(); ++it) {
				if (MatchFirst(*it, search)) {
					indices.push_back(std::distance(m_texts.begin(), it));
				}
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
				if (pProgress) {
					pProgress->Increment();
				}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			}
		}
		// ǂł}b`
		else {
			for (WStrings::const_iterator it = m_texts.begin(); it != m_texts.end(); ++it) {
				if (MatchAny(*it, search)) {
					indices.push_back(std::distance(m_texts.begin(), it));
				}
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
				if (pProgress) {
					pProgress->Increment();
				}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			}
		}
	}

	void CDataGroup::Search(
		size_t begin, size_t end,
		const std::wstring& search,
		bool matchFirst,
		IndexVec& indices
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		, GenericUtility::CProgressManager* pProgress
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		)
		const
	{
		end = Math::Min(end, m_texts.size());

		indices.clear();
		// 擪̂݃}b`
		if (matchFirst) {
			for (size_t i = begin; i < end; ++i) {
				if (MatchFirst(m_texts[i], search)) {
					indices.push_back(i);
				}
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
				if (pProgress) {
					pProgress->Increment();
				}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			}
		}
		// ǂł}b`
		else {
			for (size_t i = begin; i < end; ++i) {
				if (MatchAny(m_texts[i], search)) {
					indices.push_back(i);
				}
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
				if (pProgress) {
					pProgress->Increment();
				}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			}
		}
	}

	bool CDataGroup::IsDirty() const
	{
		return CDirtyFlag::IsDirty();
	}

	void CDataGroup::SetDirty(bool dirty) const
	{
		CDirtyFlag::SetDirty(dirty);
	}

	////////////////////////////////////////////////////////////////////////////////

	CDataManager::CDataManager()
	{
		SetupDefaultGroup();
	}

	CDataManager::~CDataManager()
	{
	}

	void CDataManager::SetupDefaultGroup()
	{
		// O[v΁AȂ
		if (!m_groups.empty()) {
			return;
		}

		// HACK: ftHgO[vǉ
		m_groups.push_back(CDataGroup(s_defaultGroupName));

		// HACK: ftHgO[vJgɂ
		m_currentGroup = 0;
	}

	bool CDataManager::Save(const std::wstring& filename) const
	{
		// HACK: ύXtOɂ炸ۑ
		bool ret = GenericUtility::SaveObject(filename.c_str(), *this);
		// HACK: ۑɐAύXtONA
		if (ret) {
			SetDirty(false);
		}
		return ret;
	}

	bool CDataManager::Load(const std::wstring& filename)
	{
		bool ret = GenericUtility::LoadObject(filename.c_str(), *this);

		// HACK: [hƂɗĂꂽύXtOׂăNA
		SetDirty(false);

		// HACK: O[vȂ΁AftHgO[vǉ
		SetupDefaultGroup();
		assert(!m_groups.empty());

		// HACK: ǂݍݓrŎsꍇt@CҏWꂽꍇAJgO[vsɂȂ\
		// HACK: JgO[vmFĂȂ̂ŁANȂ悤ɂŃ`FbNĂ
		if (m_currentGroup >= m_groups.size()) {
			m_currentGroup = m_groups.size() - 1;

			// HACK: iȂ̂ŁAύXtO͗ĂȂ
		}

		return ret;
	}

	bool CDataManager::Add(const std::wstring& name)
	{
		m_groups.push_back(CDataGroup(name));
		// HACK: ǉ̂ɕύXtOĂ̂ŁAł͐ݒ肷KvȂ
		return true;
	}

	bool CDataManager::Delete(size_t index)
	{
		if (index >= m_groups.size()) {
			return false;
		}
		// HACK: ݂̃O[v͍폜łȂ
		if (index == m_currentGroup) {
			return false;
		}
#if 0	// HACK: Ō̈݂͌̃JeS[̂͂Ȃ̂ŁA`FbNKvȂ
		// HACK: Ō̂ЂƂ͍폜łȂ
		if (m_groups.size() <= 1) {
			return false;
		}
#endif

		m_groups.erase(m_groups.begin() + index);
		// JgÔ̂폜ꂽÂw悤ɃJgCfbNX𒲐߂
		if (m_currentGroup > index) {
			--m_currentGroup;
		}
#if 0	// HACK: ݂̃JeS[͍폜łȂ̂ŁA`FbNKvȂ
		// Jg폜ꂽ牽Ȃ (̂̂JgɂȂ) AŌ̗vfꍇ͈Ow悤ɂ
		else if (m_currentGroup >= m_groups.size()) {
			m_currentGroup = m_groups.size() - 1;
		}
#endif

		// HACK: Add()  Move()  CDataGroup ̕ύXtOADelete() ł͖IɕύXtO𗧂ĂKv
		SetDirty(true);
		return true;
	}

	bool CDataManager::Move(size_t index, INT_PTR nMove)
	{
		if (index >= m_groups.size()) {
			return false;
		}
		if (nMove == 0) {
			return false;
		}

		// Vʒu (nMove 𑫂ƕɂȂ\̂ INT_PTR ɂ)
		INT_PTR nNewPos = index + nMove;
		if (nNewPos < 0) {
			nNewPos = 0;
		}
		// (m_groups.size() - 1 ͍Ō)
		else if ((INT_PTR)m_groups.size() <= nNewPos) {
			nNewPos = m_groups.size() - 1;
		}
		// VʒuςȂΉȂ
		if (index == nNewPos) {
			return false;
		}

		// 
		int step = nMove > 0 ? 1 : -1;
		// 炵Ă
		for (size_t i = index; i != nNewPos; i += step) {
			// OƓւ
			m_groups[i].Swap(m_groups[i + step]);
		}

		// JgړAɂ킹
		if (index == m_currentGroup) {
			m_currentGroup = nNewPos;
		}
		// JgÔ̂JgɂAJgOɂ炷
		else if (index < m_currentGroup && m_currentGroup <= (size_t)nNewPos) {
			--m_currentGroup;
		}
		// Jĝ̂JgOɂAJgɂ炷
		else if ((size_t)nNewPos <= m_currentGroup && m_currentGroup < index) {
			++m_currentGroup;
		}

		// HACK: Swap() ŕύXtO̂ŁAł͐ݒ肷KvȂ
		return true;
	}

	size_t CDataManager::Find(const std::wstring& name) const
	{
		for (CDataGroupVec::const_iterator it = m_groups.begin(); it != m_groups.end(); ++it) {
			if (it->GetName() == name) {
				return std::distance(m_groups.begin(), it);
			}
		}
		// Ȃ
		return ~0;
	}

	size_t CDataManager::GetGroupCount() const
	{
		return m_groups.size();
	}

	CDataGroup& CDataManager::GetGroup(size_t index)
	{
		assert(index < m_groups.size());
		return m_groups[index];
	}

	const CDataGroup& CDataManager::GetGroup(size_t index) const
	{
		assert(index < m_groups.size());
		return m_groups[index];
	}

	size_t CDataManager::GetCurrentGroupIndex() const
	{
		return m_currentGroup;
	}

	bool CDataManager::SetCurrentGroup(size_t index, bool setDirtyFlag)
	{
		if (index >= m_groups.size()) {
			return false;
		}

		m_currentGroup = index;

		// HACK: w肳ꂽƂύXtO𗧂Ă
		// HACK: Jg̕ύX̂тɗĂĂAقƂǏɗ悤ɂȂĂ܂
		// HACK: Jg̕ύXȂAƂĂقǑ債ł͂Ȃ͂
		// HACK: łAύXtO𗧂ĂȂꍇ́AAvP[VŕۑĂׂ
		if (setDirtyFlag) {
			SetDirty(true);
		}
		return true;
	}

	CDataGroup& CDataManager::GetCurrentGroup()
	{
		assert(m_currentGroup < m_groups.size());
		return m_groups[m_currentGroup];
	}

	const CDataGroup& CDataManager::GetCurrentGroup() const
	{
		assert(m_currentGroup < m_groups.size());
		return m_groups[m_currentGroup];
	}

	bool CDataManager::IsDirty() const
	{
		// (ꂩύXĂ true)

		// ύXĂAύXĂ
		if (CDirtyFlag::IsDirty()) {
			return true;
		}

		// O[v̂ꂩύXĂAύXĂ
		for (CDataGroupVec::const_iterator it = m_groups.begin(); it != m_groups.end(); ++it) {
			if (it->IsDirty()) {
				return true;
			}
		}

		return false;
	}

	void CDataManager::SetDirty(bool dirty) const
	{
		// ɐݒ肷
		CDirtyFlag::SetDirty(dirty);

		// HACK: ύXtONAƂ́AׂẴO[v̕ύXtONA
		// HACK: (ύXtO || ]̂)
		if (!dirty) {
			for (CDataGroupVec::const_iterator it = m_groups.begin(); it != m_groups.end(); ++it) {
				it->SetDirty(false);
			}
		}
	}

	std::wstring CDataManager::s_defaultGroupName = L"Default";

	void CDataManager::SetDefaultGroupName(const std::wstring& name)
	{
		s_defaultGroupName = name;
	}

}	// namespace ListPasteLib
