/**
 * @file  DataManager.cpp
 * @brief ListPaste f[^}l[WNX.
 *
 * @author JIN
 *
 * Copyright (C) 2011- JIN All rights reserved.
 */
#include "StdAfx.h"
#include "DataManager.h"

#include "../GenericLib/MathUtility.h"
#include "../GenericLib/SerializeObject.h"

/// MessagePack ɂ binary serialization ɑΉ
#define DATAMANAGER_USE_MESSAGE_PACK
#ifdef DATAMANAGER_USE_MESSAGE_PACK
#include <msgpack.hpp>
#endif	// #ifdef DATAMANAGER_USE_MESSAGE_PACK

namespace ListPasteLib {

	namespace {

		/**
		 * text  search Ŏn܂Ă邩ǂ𒲂ׂ.
		 */
		inline bool MatchFirst(const std::wstring& text, const std::wstring& search)
		{
			// HACK: 啶͋ʂȂ
			return _wcsnicmp(text.c_str(), search.c_str(), search.size()) == 0;
		}

		/**
		 * text  search ܂܂Ă邩ǂ𒲂ׂ.
		 */
		inline bool MatchAny(const std::wstring& text, const std::wstring& search)
		{
			// HACK: 啶͋ʂȂ
			return ::StrStrIW(text.c_str(), search.c_str()) != NULL;
		}

	}	// anonymous namespace

	// HACK: f[^ׂ͂ă|C^ł͂Ȃ̂ŎĂ
	// HACK: ̂߁AAdd()  Delete() ̃ptH[}Xɖ肪o\

	////////////////////////////////////////////////////////////////////////////////

	/**
	 * ύXtO.
	 */
	class CDirtyFlag
	{
	public:
		CDirtyFlag()
			// HACK: VK쐬ꂽƂ́AύXtO𗧂ĂĂ
			// HACK: t@Cǂݍō쐬ꂽꍇ́ÂƂɖI off ɂ
			: m_dirty(true)
		{
		}

		/**
		 * ύXĂ邩ǂ擾.
		 */
		bool IsDirty() const
		{
			return m_dirty;
		}
		// HACK:  Save()  SetDirty(false) 邽߂ɁAconst ɂĂ
		/**
		 * ύXtOݒ肷.
		 */
		void SetDirty(bool dirty) const
		{
			m_dirty = dirty;
		}

	private:
		/// ύXtO
		mutable bool m_dirty;
	};

	////////////////////////////////////////////////////////////////////////////////

	// HACK: 蔲 CDirtyFlag  private p ({̓oƂĎׂ)
	/**
	 * f[^O[v.
	 */
	class CDataGroupImpl : public CDataGroup, private CDirtyFlag
	{
		// HACK: CDataGroup  boost::noncopyable ł߂ɁARs[RXgN^ operator= 𖾎Iɒ`Ă
		// HACK: oɕύXƂɒ
	public:
		CDataGroupImpl();
		CDataGroupImpl(const std::wstring& name);
		CDataGroupImpl(const CDataGroupImpl& impl);
		~CDataGroupImpl();

		CDataGroupImpl& operator=(const CDataGroupImpl& impl);

		/**
		 * Ǝւ.
		 */
		void Swap(CDataGroupImpl& group);

		/**
		 * O[vݒ肷.
		 */
		void SetName(const std::wstring& name);
		/**
		 * O[v擾.
		 */
		const std::wstring& GetName() const;

		/**
		 * ǉ.
		 * @param[in] index ǉʒu (>= eLXg: ɒǉ)
		 */
		bool Add(size_t index, const std::wstring& text);
		/**
		 * ǉ.
		 * rɕǉꍇ́Axɒǉ.
		 * @param[in] index ǉʒu (>= eLXg: ɒǉ)
		 */
		bool Add(size_t index, const std::vector<std::wstring>& texts);
		/**
		 * ύX.
		 */
		bool Modify(size_t index, const std::wstring& text);
		/**
		 * 폜.
		 */
		bool Delete(size_t index);

		typedef std::deque<std::wstring> WStrings;
		/**
		 * ׂẴeLXg擾.
		 */
		const WStrings& GetTexts() const;

		typedef std::vector<size_t> IndexVec;
		/**
		 * .
		 * @param[in]  search     
		 * @param[in]  matchFirst ̐擪Ƀ}b`邩ǂ
		 * @param[in]  pProgress  vOX (SetProgressRange() ς݂̂)
		 * @param[out] indices    }b`eLXg̃CfbNX
		 */
		void Search(
			const std::wstring& search,
			bool matchFirst,
			IndexVec& indices
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			, GenericUtility::CProgressManager* pProgress = NULL
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			)
			const;

		/**
		 * w肵CfbNX͈͓.
		 * @param[in] begin     JnCfbNX
		 * @param[in] end       ICfbNX (܂܂Ȃ)
		 * @param[in] search    
		 * @param[in] matchFirst ̐擪Ƀ}b`邩ǂ
		 * @param[in] pProgress vOX (StartProgress() ς݂̂)
		 * @param[out] indices    }b`eLXg̃CfbNX
		 */
		void Search(
			size_t begin, size_t end,
			const std::wstring& search,
			bool matchFirst,
			IndexVec& indices
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			, GenericUtility::CProgressManager* pProgress = NULL
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			)
			const;

		/**
		 * ύXĂ邩ǂ擾.
		 */
		bool IsDirty() const;
		/**
		 * ύXtOݒ肷.
		 */
		void SetDirty(bool dirty) const;

	private:
		/// O[v
		std::wstring m_name;
		/// eLXg
		WStrings m_texts;

	private:
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive& ar, const unsigned int version)
		{
			using boost::serialization::make_nvp;

			ar & make_nvp("Name", m_name);
			ar & make_nvp("Texts", m_texts);
		}

#ifdef DATAMANAGER_USE_MESSAGE_PACK
	public:
		MSGPACK_DEFINE(m_name, m_texts);
#endif	// #ifdef DATAMANAGER_USE_MESSAGE_PACK
	};

	////////////////////////////////////////////////////////////////////////////////

	/**
	 * f[^}l[W.
	 * @note ۑAǂݍ݂ Save/Load łȂ (boost::serialization ɒړnĂ͂ȂȂ; dirty flag ̂)
	 */
	class CDataManagerImpl : public CDataManager, private CDirtyFlag
	{
		// HACK: CDataManager  boost::noncopyable ł߂ɁARs[RXgN^ operator= 𖾎Iɒ`Ă
		// HACK: oɕύXƂɒ
	public:
		CDataManagerImpl();
		CDataManagerImpl(const CDataManagerImpl& impl);
		~CDataManagerImpl();

		CDataManagerImpl& operator=(const CDataManagerImpl& impl);

		/**
		 * O[vȂ΁AftHgO[vǉ.
		 */
		void SetupDefaultGroup();

	public:
		/**
		 * VACYăt@Cɕۑ.
		 * ύXtOɂ炸AۑȂ.
		 * ۑɐAύXtONA.
		 * @note gq ".xml" łȂ΁AoCiŕۑ
		 */
		bool Save(const std::wstring& filename) const;
		/**
		 * t@CǂݍŃfVACY.
		 * ǂݍ񂾌ɕύXtONA邪AftHgꂽ痧Ă.
		 * @note gq ".xml" łȂ΁AoCiƂēǂݍ
		 */
		bool Load(const std::wstring& filename);

		// HACK: VKɃIuWFNg쐬Ȃ̂ŁAClone() ł͂Ȃ Copy() ɂĂ
		/**
		 * ɃRs[.
		 */
		bool Copy(const CDataManagerPtr& manager);

	public:
		/**
		 * f[^O[v𖖔ɒǉ.
		 */
		bool Add(const std::wstring& name);
		/**
		 * f[^O[v폜.
		 */
		bool Delete(size_t index);
		/**
		 * index Ԗڂ̃f[^O[vړ.
		 * [邢͉[𒴂ꍇ́A[邢͉[ɂȂ.
		 * @param[in] nMove ړ (+: , -: )
		 * @return łړǂ.
		 */
		bool Move(size_t index, INT_PTR nMove);
		/**
		 * f[^O[v𖼑OŌ.
		 * @return ŏ̃CfbNX (~0 : Ȃ)
		 */
		size_t Find(const std::wstring& name) const;

		/**
		 * f[^O[v擾.
		 */
		size_t GetGroupCount() const;

		/**
		 * i Ԗڂ̃f[^O[v擾.
		 */
		CDataGroup& GetGroup(size_t index);
		/**
		 * i Ԗڂ̃f[^O[v擾.
		 */
		const CDataGroup& GetGroup(size_t index) const;

		/**
		 * ݂̃f[^O[v擾.
		 */
		size_t GetCurrentGroupIndex() const;
		/**
		 * ݂̃f[^O[vݒ肷.
		 * @param[in] setDirtyFlag ύXtO𗧂Ă
		 */
		bool SetCurrentGroup(size_t index, bool setDirtyFlag);

		/**
		 * ݂̃f[^O[v擾.
		 */
		CDataGroup& GetCurrentGroup();
		/**
		 * ݂̃f[^O[v擾.
		 */
		const CDataGroup& GetCurrentGroup() const;

		typedef std::vector<CDataGroupImpl> CDataGroupVec;
		/**
		 * ׂẴf[^O[v擾.
		 */
		const CDataGroupVec& GetGroups() const;

		/**
		 * ύXĂ邩ǂ擾.
		 */
		bool IsDirty() const;
		/**
		 * ύXtOݒ肷.
		 */
		void SetDirty(bool dirty) const;

		/**
		 * ftHgO[vݒ肷.
		 */
		static void SetDefaultGroupName(const std::wstring& name);

	private:
		/// JgO[ṽCfbNX
		size_t m_currentGroup;
		/// f[^O[v
		CDataGroupVec m_groups;

		/// ftHgO[v
		static std::wstring s_defaultGroupName;

	private:
		friend class boost::serialization::access;
		template<class Archive>
		void serialize(Archive& ar, const unsigned int version)
		{
			using boost::serialization::make_nvp;

			ar & make_nvp("CurrentGroup", m_currentGroup);
			ar & make_nvp("Groups", m_groups);
		}

#ifdef DATAMANAGER_USE_MESSAGE_PACK
	public:
		MSGPACK_DEFINE(m_currentGroup, m_groups);
#endif	// #ifdef DATAMANAGER_USE_MESSAGE_PACK
	};


	////////////////////////////////////////////////////////////////////////////////

	CDataGroupImpl::CDataGroupImpl()
	{
	}

	CDataGroupImpl::CDataGroupImpl(const std::wstring& name)
		: m_name(name)
	{
	}

	CDataGroupImpl::CDataGroupImpl(const CDataGroupImpl& impl)
	{
		*this = impl;
	}

	CDataGroupImpl::~CDataGroupImpl()
	{
	}

	CDataGroupImpl& CDataGroupImpl::operator=(const CDataGroupImpl& impl)
	{
		SetDirty(impl.IsDirty());
		m_name  = impl.m_name;
		m_texts = impl.m_texts;
		return *this;
	}

	void CDataGroupImpl::Swap(CDataGroupImpl& group)
	{
		m_name.swap(group.m_name);
		m_texts.swap(group.m_texts);

		// HACK: Ƒ̕ύXtO𗧂Ă
		SetDirty(true);
		group.SetDirty(true);
	}

	void CDataGroupImpl::SetName(const std::wstring& name)
	{
		m_name = name;

		SetDirty(true);
	}

	const std::wstring& CDataGroupImpl::GetName() const
	{
		return m_name;
	}

	bool CDataGroupImpl::Add(size_t index, const std::wstring& text)
	{
		if (index > m_texts.size()) {
			index = m_texts.size();
		}

		WStrings::iterator it = m_texts.begin();
		std::advance(it, index);
		m_texts.insert(it, text);

		SetDirty(true);
		return true;
	}

	bool CDataGroupImpl::Add(size_t index, const std::vector<std::wstring>& texts)
	{
		if (index > m_texts.size()) {
			index = m_texts.size();
		}

		WStrings::iterator it = m_texts.begin();
		std::advance(it, index);
		m_texts.insert(it, texts.begin(), texts.end());

		SetDirty(true);
		return true;
	}

	bool CDataGroupImpl::Modify(size_t index, const std::wstring& text)
	{
		if (index >= m_texts.size()) {
			return false;
		}

		WStrings::iterator it = m_texts.begin();
		std::advance(it, index);
		
		*it = text;

		SetDirty(true);
		return true;
	}

	bool CDataGroupImpl::Delete(size_t index)
	{
		if (index >= m_texts.size()) {
			return false;
		}

		WStrings::iterator it = m_texts.begin();
		std::advance(it, index);
		m_texts.erase(it);

		SetDirty(true);
		return true;
	}

	const CDataGroup::WStrings& CDataGroupImpl::GetTexts() const
	{
		return m_texts;
	}

	void CDataGroupImpl::Search(
		const std::wstring& search,
		bool matchFirst,
		IndexVec& indices
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		, GenericUtility::CProgressManager* pProgress
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		)
		const
	{
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		if (pProgress) {
			pProgress->StartProgress(m_texts.size());
		}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS

		indices.clear();
		// 擪̂݃}b`
		if (matchFirst) {
			for (WStrings::const_iterator it = m_texts.begin(); it != m_texts.end(); ++it) {
				if (MatchFirst(*it, search)) {
					indices.push_back(std::distance(m_texts.begin(), it));
				}
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
				if (pProgress) {
					pProgress->Increment();
				}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			}
		}
		// ǂł}b`
		else {
			for (WStrings::const_iterator it = m_texts.begin(); it != m_texts.end(); ++it) {
				if (MatchAny(*it, search)) {
					indices.push_back(std::distance(m_texts.begin(), it));
				}
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
				if (pProgress) {
					pProgress->Increment();
				}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			}
		}
	}

	void CDataGroupImpl::Search(
		size_t begin, size_t end,
		const std::wstring& search,
		bool matchFirst,
		IndexVec& indices
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		, GenericUtility::CProgressManager* pProgress
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
		)
		const
	{
		end = Math::Min(end, m_texts.size());

		indices.clear();
		// 擪̂݃}b`
		if (matchFirst) {
			for (size_t i = begin; i < end; ++i) {
				if (MatchFirst(m_texts[i], search)) {
					indices.push_back(i);
				}
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
				if (pProgress) {
					pProgress->Increment();
				}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			}
		}
		// ǂł}b`
		else {
			for (size_t i = begin; i < end; ++i) {
				if (MatchAny(m_texts[i], search)) {
					indices.push_back(i);
				}
#ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
				if (pProgress) {
					pProgress->Increment();
				}
#endif	// #ifdef DATAMANAGER_ENABLE_SEARCH_PROGRESS
			}
		}
	}

	bool CDataGroupImpl::IsDirty() const
	{
		return CDirtyFlag::IsDirty();
	}

	void CDataGroupImpl::SetDirty(bool dirty) const
	{
		CDirtyFlag::SetDirty(dirty);
	}

	////////////////////////////////////////////////////////////////////////////////

#ifdef DATAMANAGER_USE_MESSAGE_PACK
	namespace {

		/**
		 * oCit@Cǂ.
		 * (gq ".xml" łȂ΃oCiƂ݂Ȃ)
		 */
		bool IsBinaryFile(const std::wstring& filename)
		{
			return wcscmp(L".xml", ::PathFindExtensionW(filename.c_str())) != 0;
		}

		/**
		 * t@C|C^.
		 */
		class CAutoFile : boost::noncopyable
		{
		public:
			CAutoFile(FILE* fp) : m_fp(fp)
			{
			}
			~CAutoFile()
			{
				if (m_fp) {
					VERIFY(fclose(m_fp) == 0);
				}
			}

			operator FILE*() const
			{
				return m_fp;
			}

		private:
			FILE* m_fp;
		};

		/**
		 * t@Co̓Xg[.
		 */
		class CFileWriteStream
		{
		public:
			CFileWriteStream(const std::wstring& filename)
				:  m_fp(_wfopen(filename.c_str(), L"wb"))
				, m_error(false)
			{
			}

			/**
			 * Xg[ɏ.
			 */
			void write(const char* buf, unsigned int len)
			{
				if (fwrite(buf, 1, len, m_fp) != len) {
					m_error = true;
				}
			}

			/**
			 * G[ǂ𒲂ׂ.
			 */
			bool error() const
			{
				return m_error;
			}

		private:
			CAutoFile m_fp;
			bool m_error;
		};

		/// ^OTCY
		static const size_t TAG_SIZE = 4;
		/// ^O
		static const char TAG[TAG_SIZE] = { 'L', 'S', 'T', 'P' };
		/// o[W
		static const int VERSION = 1;

		/**
		 * oCit@Cɏ.
		 */
		template <typename T>
		bool SaveBinary(const std::wstring& filename, const T& obj)
		{
			try {
#if 0
				// Xg[

				// VACY
				msgpack::sbuffer sbuf;
				msgpack::pack(sbuf, obj);

				// t@CI[v
				CAutoFile fp = _wfopen(filename.c_str(), L"wb");
				if (!fp) {
					return false;
				}

				// ^O
				if (fwrite(TAG, 1, TAG_SIZE, fp) != TAG_SIZE) {
					return false;
				}
				// o[W
				// HACK: oCiƂď (little endian)
				if (fwrite(&VERSION, sizeof(VERSION), 1, fp) != 1) {
					return false;
				}

				// ct@Cɏ
				// ׂďނƂł琬
				return fwrite(sbuf.data(), 1, sbuf.size(), fp) == sbuf.size();
#else
				// Xg[ (݂͂̕悢)

				// t@Co̓Xg[
				CFileWriteStream stream(filename);
				// VACU
				msgpack::packer<CFileWriteStream> packer(stream);

				// ^O
				stream.write(TAG, TAG_SIZE);
				if (stream.error()) {
					return false;
				}
				// o[W
				// HACK: oCiƂď (little endian)
				stream.write(reinterpret_cast<const char*>(&VERSION), sizeof(VERSION));
				if (stream.error()) {
					return false;
				}

				// VACY
				packer.pack(obj);

				return !stream.error();
#endif
			}
			catch (std::exception&) {
				return false;
			}
		}

		/**
		 * oCit@Cǂݍ.
		 */
		template <typename T>
		bool LoadBinary(const std::wstring& filename, T& obj)
		{
			try {
				// t@CI[v
				CAutoFile fp = _wfopen(filename.c_str(), L"rb");
				if (!fp) {
					return false;
				}

				// ^O
				char tag[TAG_SIZE] = { 0 };
				if (fread(tag, 1, TAG_SIZE, fp) != TAG_SIZE) {
					return false;
				}
				if (memcmp(tag, TAG, TAG_SIZE) != 0) {
					return false;
				}
				// o[W
				int version = 0;
				if (fread(&version, sizeof(version), 1, fp) != 1) {
					return false;
				}
				// HACK: ݂̓o[W 1 ̂ݑΉ
				if (version != 1) {
					return false;
				}

#if 0
				// Xg[

				// fVACYobt@
				msgpack::sbuffer sbuf;

				// ǂݍ݃obt@
				static const size_t BUFSIZE = 0x10000;
				char buf[BUFSIZE];

				while (true) {
					// t@Cǂݍ
					size_t read = fread(buf, 1, BUFSIZE, fp);
					ASSERT(read <= BUFSIZE);

					// fVACYobt@ɒǉ
					if (read > 0) {
						sbuf.write(buf, read);
					}

					// obt@ςɓǂݍ߂Ȃ
					if (read != BUFSIZE) {
						// t@C̏I܂œǂݍ߂Ȃ玸s
						if (!feof(fp)) {
							return false;
						}
						// ǂݍݏI
						break;
					}
				}

				// fVACY
				msgpack::unpacked result;
				msgpack::unpack(&result, sbuf.data(), sbuf.size());

				// fVACŶ擾
				result.get().convert(&obj);
#else
				// Xg[ (ǂݍ݂́A炭Xg[łƕςȂ)

				// fVACU
				msgpack::unpacker unpacker;
				while (true) {
					// obt@mۂ
					unpacker.reserve_buffer();

					// obt@TCY
					size_t capacity = unpacker.buffer_capacity();
					// t@Cǂݍ
					size_t read = fread(unpacker.buffer(), 1, capacity, fp);
					ASSERT(read <= capacity);

					// obt@ɃRs[TCYݒ肷
					unpacker.buffer_consumed(read);

					// obt@ςɓǂݍ߂Ȃ
					if (read != capacity) {
						// t@C̏I܂œǂݍ߂Ȃ玸s
						if (!feof(fp)) {
							return false;
						}
					}

					// fVACY
					// HACK: ܂ׂēǂݍłȂĂAnext() ĂԈӖ͂(?)
					msgpack::unpacked result;
					if (unpacker.next(&result)) {
						// HACK: IuWFNg͈Ȃ̂ŁAfVACYɐꍇ͍Ō܂œǂݍł͂
						if (!feof(fp)) {
							return false;
						}

						// fVACŶ擾
						result.get().convert(&obj);

						return true;
					}
				}
#endif
			}
			catch (std::exception&) {
				return false;
			}
		}

	}	// anonymous namespace
#endif	// #ifdef DATAMANAGER_USE_MESSAGE_PACK

	CDataManagerImpl::CDataManagerImpl()
	{
		SetupDefaultGroup();
	}

	CDataManagerImpl::CDataManagerImpl(const CDataManagerImpl& impl)
	{
		*this = impl;
	}

	CDataManagerImpl::~CDataManagerImpl()
	{
	}

	CDataManagerImpl& CDataManagerImpl::operator=(const CDataManagerImpl& impl)
	{
		SetDirty(impl.IsDirty());
		m_currentGroup = impl.m_currentGroup;
		m_groups       = impl.m_groups;
		return *this;
	}

	void CDataManagerImpl::SetupDefaultGroup()
	{
		// O[v΁AȂ
		if (!m_groups.empty()) {
			return;
		}

		// HACK: ftHgO[vǉ
		m_groups.push_back(CDataGroupImpl(s_defaultGroupName));

		// HACK: ftHgO[vJgɂ
		m_currentGroup = 0;
	}

	bool CDataManagerImpl::Save(const std::wstring& filename) const
	{
		// HACK: ύXtOɂ炸ۑ

#ifdef DATAMANAGER_USE_MESSAGE_PACK
		bool succeeded = false;
		if (IsBinaryFile(filename)) {
			succeeded = SaveBinary(filename, *this);
		}
		else {
			succeeded = GenericUtility::SaveObject(filename.c_str(), *this);
		}
#else	// #ifdef DATAMANAGER_USE_MESSAGE_PACK
		bool succeeded = GenericUtility::SaveObject(filename.c_str(), *this);
#endif	// #ifdef DATAMANAGER_USE_MESSAGE_PACK

		// HACK: ۑɐAύXtONA
		if (succeeded) {
			SetDirty(false);
		}

		return succeeded;
	}

	bool CDataManagerImpl::Load(const std::wstring& filename)
	{
#ifdef DATAMANAGER_USE_MESSAGE_PACK
		bool succeeded = false;
		if (IsBinaryFile(filename)) {
			succeeded = LoadBinary(filename, *this);
		}
		else {
			succeeded = GenericUtility::LoadObject(filename.c_str(), *this);
		}
#else	// #ifdef DATAMANAGER_USE_MESSAGE_PACK
		bool succeeded = GenericUtility::LoadObject(filename.c_str(), *this);
#endif	// #ifdef DATAMANAGER_USE_MESSAGE_PACK

		// HACK: [hƂɗĂꂽύXtOׂăNA
		SetDirty(false);

		// HACK: O[vȂ΁AftHgO[vǉ
		SetupDefaultGroup();
		assert(!m_groups.empty());

		// HACK: ǂݍݓrŎsꍇt@CҏWꂽꍇAJgO[vsɂȂ\
		// HACK: JgO[vmFĂȂ̂ŁANȂ悤ɂŃ`FbNĂ
		if (m_currentGroup >= m_groups.size()) {
			m_currentGroup = m_groups.size() - 1;

			// HACK: iȂ̂ŁAύXtO͗ĂȂ
		}

		return succeeded;
	}

	bool CDataManagerImpl::Copy(const CDataManagerPtr& manager)
	{
		if (!manager) {
			return false;
		}
		boost::shared_ptr<CDataManagerImpl> impl = boost::dynamic_pointer_cast<CDataManagerImpl>(manager);
		if (!impl) {
			return false;
		}

		*this = *impl;
		return true;
	}

	bool CDataManagerImpl::Add(const std::wstring& name)
	{
		m_groups.push_back(CDataGroupImpl(name));
		// HACK: ǉ̂ɕύXtOĂ̂ŁAł͐ݒ肷KvȂ
		return true;
	}

	bool CDataManagerImpl::Delete(size_t index)
	{
		if (index >= m_groups.size()) {
			return false;
		}
		// HACK: ݂̃O[v͍폜łȂ
		if (index == m_currentGroup) {
			return false;
		}
#if 0	// HACK: Ō̈݂͌̃JeS[̂͂Ȃ̂ŁA`FbNKvȂ
		// HACK: Ō̂ЂƂ͍폜łȂ
		if (m_groups.size() <= 1) {
			return false;
		}
#endif

		m_groups.erase(m_groups.begin() + index);
		// JgÔ̂폜ꂽÂw悤ɃJgCfbNX𒲐߂
		if (m_currentGroup > index) {
			--m_currentGroup;
		}
#if 0	// HACK: ݂̃JeS[͍폜łȂ̂ŁA`FbNKvȂ
		// Jg폜ꂽ牽Ȃ (̂̂JgɂȂ) AŌ̗vfꍇ͈Ow悤ɂ
		else if (m_currentGroup >= m_groups.size()) {
			m_currentGroup = m_groups.size() - 1;
		}
#endif

		// HACK: Add()  Move()  CDataGroup ̕ύXtOADelete() ł͖IɕύXtO𗧂ĂKv
		SetDirty(true);
		return true;
	}

	bool CDataManagerImpl::Move(size_t index, INT_PTR nMove)
	{
		if (index >= m_groups.size()) {
			return false;
		}
		if (nMove == 0) {
			return false;
		}

		// Vʒu (nMove 𑫂ƕɂȂ\̂ INT_PTR ɂ)
		INT_PTR nNewPos = index + nMove;
		if (nNewPos < 0) {
			nNewPos = 0;
		}
		// (m_groups.size() - 1 ͍Ō)
		else if ((INT_PTR)m_groups.size() <= nNewPos) {
			nNewPos = m_groups.size() - 1;
		}
		// VʒuςȂΉȂ
		if (index == nNewPos) {
			return false;
		}

		// 
		int step = nMove > 0 ? 1 : -1;
		// 炵Ă
		for (size_t i = index; i != nNewPos; i += step) {
			// OƓւ
			m_groups[i].Swap(m_groups[i + step]);
		}

		// JgړAɂ킹
		if (index == m_currentGroup) {
			m_currentGroup = nNewPos;
		}
		// JgÔ̂JgɂAJgOɂ炷
		else if (index < m_currentGroup && m_currentGroup <= (size_t)nNewPos) {
			--m_currentGroup;
		}
		// Jĝ̂JgOɂAJgɂ炷
		else if ((size_t)nNewPos <= m_currentGroup && m_currentGroup < index) {
			++m_currentGroup;
		}

		// HACK: Swap() ŕύXtO̂ŁAł͐ݒ肷KvȂ
		return true;
	}

	size_t CDataManagerImpl::Find(const std::wstring& name) const
	{
		for (CDataGroupVec::const_iterator it = m_groups.begin(); it != m_groups.end(); ++it) {
			if (it->GetName() == name) {
				return std::distance(m_groups.begin(), it);
			}
		}
		// Ȃ
		return ~0;
	}

	size_t CDataManagerImpl::GetGroupCount() const
	{
		return m_groups.size();
	}

	CDataGroup& CDataManagerImpl::GetGroup(size_t index)
	{
		assert(index < m_groups.size());
		return m_groups[index];
	}

	const CDataGroup& CDataManagerImpl::GetGroup(size_t index) const
	{
		assert(index < m_groups.size());
		return m_groups[index];
	}

	size_t CDataManagerImpl::GetCurrentGroupIndex() const
	{
		return m_currentGroup;
	}

	bool CDataManagerImpl::SetCurrentGroup(size_t index, bool setDirtyFlag)
	{
		if (index >= m_groups.size()) {
			return false;
		}

		m_currentGroup = index;

		// HACK: w肳ꂽƂύXtO𗧂Ă
		// HACK: Jg̕ύX̂тɗĂĂAقƂǏɗ悤ɂȂĂ܂
		// HACK: Jg̕ύXȂAƂĂقǑ債ł͂Ȃ͂
		// HACK: łAύXtO𗧂ĂȂꍇ́AAvP[VŕۑĂׂ
		if (setDirtyFlag) {
			SetDirty(true);
		}
		return true;
	}

	CDataGroup& CDataManagerImpl::GetCurrentGroup()
	{
		assert(m_currentGroup < m_groups.size());
		return m_groups[m_currentGroup];
	}

	const CDataGroup& CDataManagerImpl::GetCurrentGroup() const
	{
		assert(m_currentGroup < m_groups.size());
		return m_groups[m_currentGroup];
	}

	bool CDataManagerImpl::IsDirty() const
	{
		// (ꂩύXĂ true)

		// ύXĂAύXĂ
		if (CDirtyFlag::IsDirty()) {
			return true;
		}

		// O[v̂ꂩύXĂAύXĂ
		for (CDataGroupVec::const_iterator it = m_groups.begin(); it != m_groups.end(); ++it) {
			if (it->IsDirty()) {
				return true;
			}
		}

		return false;
	}

	void CDataManagerImpl::SetDirty(bool dirty) const
	{
		// ɐݒ肷
		CDirtyFlag::SetDirty(dirty);

		// HACK: ύXtONAƂ́AׂẴO[v̕ύXtONA
		// HACK: (ύXtO || ]̂)
		if (!dirty) {
			for (CDataGroupVec::const_iterator it = m_groups.begin(); it != m_groups.end(); ++it) {
				it->SetDirty(false);
			}
		}
	}

	std::wstring CDataManagerImpl::s_defaultGroupName = L"Default";

	void CDataManagerImpl::SetDefaultGroupName(const std::wstring& name)
	{
		s_defaultGroupName = name;
	}

	////////////////////////////////////////////////////////////////////////////////

	CDataManagerPtr CDataManager::CreateInstance()
	{
		return CDataManagerPtr(new CDataManagerImpl);
	}

	void CDataManager::SetDefaultGroupName(const std::wstring& name)
	{
		CDataManagerImpl::SetDefaultGroupName(name);
	}

}	// namespace ListPasteLib
