import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Polygon;
import java.io.IOException;

import javax.swing.JFrame;
import javax.swing.UIManager;

import controller.Controller;

import view.MapPanel;
import view.StatusBar;

import map.data.CityMap;
import map.ksj.KsjFactory;
import map.map25000.Map25000Factory;
import map.map25000.Map25000Storage;
import map.map25000.cell.CellSearch;

/**
 * 地図描画アプリケーション起動のためのクラス
 * 
 * @author ma38su
 */
public class Main {

	/**
	 * フレーム高さ
	 */
	private static final int F_WIDTH = 800;

	/**
	 * フレーム幅
	 */
	private static final int F_HEIGHT = 600;

	/**
	 * args[0]には地図データを格納するディレクトリのルートディレクトリの位置を指定できます。
	 * 引数によって指定されない場合は、"./"（同一ディレクトリ）をルートディレクトリとしてします。
	 * 
	 * @param args 引数
	 */
	public static void main(String[] args) {
		if(args.length == 0) {
			args = new String[]{"./"};
		}

		// 数値地図25000のパス
		final String gsi = "http://sdf.gsi.go.jp/data25k/";

		// 都道府県データの相対位置リスト
		final String list = "/data/prefecture.csv";
		
		// 都道府県界のデータ
		final String prefecture = "/data/prefecture.dat";

		// 都道府県のバウンディングボックスのデータリスト
		final String cell = "/data/prefecture.cell";

		new Main(args[0], gsi, list, prefecture, cell);
	}

	/**
	 * アプリケーション起動のための初期設定
	 * 
	 * @param dir 地図データディレクトリのルート
	 * @param gsi
	 * @param list
	 * @param prefData
	 * @param cellData
	 */
	public Main(final String dir, final String gsi, final String list, final String prefData, final String cellData) {
		final JFrame frame = new JFrame("Digital Map");

		// Close Setting
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) {
			e.printStackTrace();
		}

		final MapPanel panel = new MapPanel();
		
		final StatusBar statusbar = new StatusBar(" ");

		final Container container = frame.getContentPane();
		container.add(statusbar, BorderLayout.SOUTH);
		container.add(panel, BorderLayout.CENTER);

		// Listener
		final Controller listener = new Controller(panel);
		
		// Frame に メインメニューバーを設定
		frame.setJMenuBar(listener.getMenuBar());

		frame.setPreferredSize(new Dimension(Main.F_WIDTH, Main.F_HEIGHT));
		frame.pack();

		frame.setVisible(true);
		
		try {	
			statusbar.setReading("初期設定");
			final CityMap maps = new CityMap(panel, new CellSearch(cellData), new Map25000Factory(new Map25000Storage(dir + "data25k/", gsi, this.getClass().getResourceAsStream(list), statusbar)), statusbar);
			statusbar.clearReading();

			statusbar.setReading("国土数値情報 都道府県");
			final Polygon[] prefecture = KsjFactory.readPolygon(this.getClass().getResourceAsStream(prefData));
//			int[] coast = new int[]{82, 83, 84, 143, 288, 457, 505, 582, 725, 878, 941, 967, 1007, 1008, 1032, 1592, 1593, 1595, 1694, 1828, 1909, 1910, 1911, 1912, 1913, 1914, 1915, 2073, 2253, 2345, 2641, 3335, 3466, 3503, 3595, 3705, 4191, 4418, 4531, 4600, 5193};

			// 湖などを繰りぬく
			final int[] coast = new int[]{2345};
			final Polygon[] prefectureCoast = new Polygon[coast.length];
			int count = 0;
			for (final int i : coast) {
				prefectureCoast[count++] = prefecture[i];
			}
			statusbar.clearReading();

			panel.init(maps, prefecture, prefectureCoast);
			maps.start();

		} catch (IOException e) {
			statusbar.setReading(e.getMessage());
			e.printStackTrace();
		}
		statusbar.clearReading();

		frame.addKeyListener(listener);
		panel.addKeyListener(listener);
		panel.addMouseListener(listener);
		panel.addMouseMotionListener(listener);
		panel.addMouseWheelListener(listener);

		statusbar.setThreadPriority(Thread.MIN_PRIORITY);
	}
}
