package map.data;

import java.awt.Polygon;
import java.awt.Rectangle;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Collection;
import java.util.Map;

/**
 * 市区町村クラス
 * 
 * @author ma38su
 *
 */
public class City {
	/**
	 * 市町村番号フォーマット
	 */
	private final static NumberFormat CODE_FORMAT = new DecimalFormat("00000");
	
	/**
	 * 都道府県番号フォーマット
	 */
	private final static NumberFormat PREFECTURE_FORMAT = new DecimalFormat("00");

	/**
	 * 市区町村番号(int)を(String)に変換
	 * @param code 市区町村番号
	 * @return 市区町村番号
	 */
	public static String cityFormat(int code) {
		return City.CODE_FORMAT.format(code);
	}

	/**
	 * 都道府県番号(int)を(String)に変換
	 * @param code 都道府県番号
	 * @return 都道府県番号
	 */
	public static String prefectureFormat(int code) {
		return City.PREFECTURE_FORMAT.format(code);
	}
	
	/**
	 * 外接長方形
	 */
	private Rectangle area;

	/**
	 * 行政界
	 */
	private Curve[] border;
	/**
	 * 市町村名
	 */
	private Facility[] city;

	/**
	 * 海岸線
	 */
	private Map<Curve, Boolean> coast;

	/**
	 * 地域番号
	 */
	private final int code;
	
	/**
	 * 施設配列
	 */
	private Facility[] facility;
	
	/**
	 * 詳細データを保持しているかどうか
	 */
	private boolean isLoaded;
	
	/**
	 * 標高メッシュ配列
	 */
	private Mesh[] mesh;
	
	/**
	 * 頂点配列
	 */
	private Map<Integer, Node> node;

	/**
	 * 国土数値情報の市町村界
	 */
	private Polygon[] ksj;

	/**
	 * 鉄道
	 */
	private Curve[][] rail;
	
	/**
	 * 河川
	 */
	private Curve[] river;
	
	/**
	 * 道路
	 */
	private Road[][] road;

	/**
	 * 駅
	 */
	private Station[] station;
	
	/**
	 * 市区町村データのコンストラクタ
	 * @param code 市区町村番号
	 * @param polygon 行政界
	 */
	public City(int code, Polygon[] polygon) {
		this.code = code;
		this.ksj = polygon;
		this.isLoaded = false;
	}

	/**
	 * 市区町村データのコンストラクタ
	 * @param code 市町村番号
	 * @param area バウンディングボックス
	 */
	public City(int code, Rectangle area) {
		this.code = code;
		this.area = area;
		this.isLoaded = false;
	}

	/**
	 * 市区町村データのコンストラクタ
	 * @param code 市区町村番号
	 * @param area バウンディングボックス
	 * @param border 行政界
	 * @param coast
	 * @param river
	 * @param rail
	 * @param station
	 * @param node
	 * @param road
	 * @param city
	 * @param facility
	 * @param mesh
	 */
	public City(String code, Rectangle area, Curve[] border, Map<Curve, Boolean> coast, Curve[] river, Curve[][] rail, Station[] station, Map<Integer, Node> node, Road[][] road, Facility[] city, Facility[] facility, Mesh[] mesh) {
		this.code = Integer.parseInt(code);
		this.area = area;
		this.rail = rail;
		this.station = station;
		this.node = node;
		this.road = road;
		this.river = river;
		this.border = border;
		this.coast = coast;
		this.city = city;
		this.facility = facility;
		this.mesh = mesh;
		this.isLoaded = true;
	}
	/**
	 *　データの破棄
	 */
	public void dump() {
		this.rail = null;
		this.station = null;
		this.node = null;
		this.road = null;
		this.river = null;
		this.border = null;
		this.coast = null;
		this.facility = null;
		this.mesh = null;
		this.isLoaded = false;
	}
	public void dumpKsj() {
		this.ksj = null;
	}
	/**
	 * エリアを返す
	 * @return エリア
	 */
	public Rectangle getArea() {
		return this.area;
	}
	/**
	 * 市町村の高さを取得
	 * @return 市町村の高さ
	 */
	public int getAreaHeight() {
		return this.area.height;
	}
	/**
	 * 市町村の幅を取得
	 * @return 市町村の幅
	 */
	public int getAreaWidth() {
		return this.area.width;
	}
	/**
	 * 行政界の取得
	 * @return 行政界
	 */
	public Curve[] getBorder() {
		return this.border;
	}
	/**
	 * 市町村名の取得
	 * @return 市町村ラベル
	 */
	public Facility[] getCity() {
		return this.city;
	}
	public Map<Curve, Boolean> getCoast() {
		return this.coast;
	}
	/**
	 * 市町村番号の取得
	 * @return 市町村番号
	 */
	public int getCode() {
		return this.code;
	}
	public Facility[] getFacility() {
		return this.facility;
	}
	public Mesh[] getMesh() {
		return this.mesh;
	}
	/**
	 * 道路頂点の取得
	 * @return 道路頂点
	 */
	public Collection<Node> getNode() {
		return this.node.values();
	}
	/**
	 * 道路頂点の取得
	 * @param id
	 * @return 道路頂点
	 */
	public Node getNode(Integer id) {
		return this.node.get(id);
	}

	/**
	 * 原点のX座標を取得
	 * @return 原点のX座標
	 */
	public int getOriginX() {
		return this.area.x;
	}
	
	/**
	 * 原点のY座標を取得
	 * @return 原点のY座標
	 */
	public int getOriginY() {
		return this.area.y;
	}

	/**
	 * 行政界ポリゴンの取得
	 * @return 行政界ポリゴン(国土数値情報)
	 */
	public Polygon[] getPolygon() {
		return this.ksj;
	}

	/**
	 * 都道府県番号の取得
	 * @return 都道府県番号
	 */
	public String getPrefectureCode() {
		return City.CODE_FORMAT.format(this.code).substring(0, 2);
	}

	/**
	 * JRの鉄道区間の取得
	 * @return 鉄道区間
	 */
	public Curve[] getJRRailway() {
		return this.rail[0];
	}

	/**
	 * JR以外の鉄道区間の取得
	 * @return 鉄道区間
	 */
	public Curve[] getOtherRailway() {
		return this.rail[1];
	}
	
	/**
	 * 河川の取得
	 * @return 河川
	 */
	public Curve[] getRiver() {
		return this.river;
	}
	
	/**
	 * 道路の取得
	 * @return 道路
	 */
	public Road[][] getRoad() {
		return this.road;
	}
	
	/**
	 * 駅区間の取得
	 * @return 駅区間
	 */
	public Station[] getStation() {
		return this.station;
	}
	
	/**
	 * データを読み込んでいるかどうか
	 * @return データを読み込んでいればtrue
	 */
	public boolean hasData() {
		return this.isLoaded;
	}
	
	public boolean hasNode() {
		return this.node != null;
	}
	
	public boolean isLoaded() {
		return this.area != null;
	}
	
	/**
	 * データの読み込みが行われていなければ指定した番号の頂点を削除する
	 * @param id 頂点番号
	 */
	public void removeNode (Integer id) {
		this.node.remove(id);
	}
	
	public void setData(Curve[] border, Map<Curve, Boolean> coast, Curve[] river, Curve[][] rail, Station[] station, Map<Integer, Node> node, Road[][] road, Facility[] city, Facility[] facility, Mesh[] mesh) {
		this.rail = rail;
		this.station = station;
		this.node = node;
		this.road = road;
		this.river = river;
		this.border = border;
		this.coast = coast;
		this.city = city;
		this.facility = facility;
		this.mesh = mesh;
		this.isLoaded = true;
	}
	
	/**
	 * 頂点データを追加
	 * @param node 頂点データを追加する。
	 */
	public void setData(Map<Integer, Node> node) {
		this.node = node;
		this.isLoaded = false;
	}
	
	public void setScreen(Rectangle area) {
		this.area = area;
	}
}
