package view;


import java.io.File;

import javax.swing.JLabel;

import map.route.SearchThread;

/**
 * JLabelを継承したステータスバー
 * 
 * @author ma38su
 *
 */
public class StatusBar extends JLabel implements Runnable {
	
	/**
	 * インターネットから取得するファイルのサイズ
	 */
	private float fileSize;

	/**
	 * インターネットから取得しているファイル
	 */
	private File file;
	
	/**
	 * インターネットから取得している市区町村番号
	 */
	private String downloadCode;

	/**
	 * 補助記憶装置（ハードディスク）を読み込んでいる市区町村番号
	 */
	private String readCode;
	
	/**
	 * 表示範囲内の地図データであるか確認している市区町村番号
	 */
	private String checkCode;

	/**
	 * 使用メモリ量（MB）
	 */
	private float memory;

	/**
	 * ステータスバーの更新のためのスレッド
	 */
	private final Thread thread;

	/**
	 * 経路探索を行うスレッド
	 */
	private SearchThread search;

	/**
	 * ファイル入出力エラーの有無
	 */
	private boolean ioException;

	/**
	 * メモリ容量オーバーのフラグ
	 */
	private boolean outOfMemory;

	/**
	 * コンストラクタ
	 */
	public StatusBar() {
		super(" ");
		this.thread = new Thread(this);
		this.thread.start();
		this.fileSize = 0;
	}

	/**
	 * コンストラクタ
	 * @param msg 初期メッセージ
	 */
	public StatusBar(String msg) {
		super(msg);
		this.thread = new Thread(this);
		this.thread.start();
		this.fileSize = 0;
	}

	/**
	 * ファイル入出力エラーを表示
	 */
	public synchronized void ioException() {
		this.ioException = true;
	}

	/**
	 * OutOfMemoryを表示
	 */
	public synchronized void memoryError() {
		this.outOfMemory = true;
	}
	
	/**
	 * ステータスバーの更新スレッドの優先度を設定します。
	 * @param priority 優先度（0から5）
	 */
	public void setThreadPriority(int priority) {
		this.thread.setPriority(priority);
	}
	
	/**
	 * インターネット上から取得しているファイルを設定します。
	 * 
	 * @param code 市区町村番号
	 * @param length ファイル長
	 * @param file ダウンロード中のファイル
	 */
	public synchronized void setDLFile(String code, int length, File file) {
		this.downloadCode = code;
		this.fileSize = length;
		this.file = file;
	}

	/**
	 * 経路探索のためのスレッドを設定します。
	 * @param thread 経路探索のスレッド
	 */
	public synchronized void setThread(SearchThread thread) {
		this.search = thread;
	}

	/**
	 * ローカルの補助記憶装置（ハードディスク）からの読み込んでいる市区町村番号を設定します。
	 * @param code 市区町村番号
	 */
	public synchronized void setReading(String code) {
		this.readCode = code;
	}
	
	/**
	 * 読み込みが終了したことを設定します。
	 *
	 */
	public synchronized void clearReading() {
		this.readCode = null;
	}

	/**
	 * 表示範囲内であるかどうかチェックしている市区町村番号を設定します。
	 * @param code 市区町村番号
	 */
	public synchronized void setCheckCode(String code) {
		this.checkCode = code;
	}

	/**
	 * 表示範囲内かどうかのチェックが終了したことを設定します。
	 *
	 */
	public synchronized void clearCheckCode() {
		this.checkCode = null;
	}

	public void run() {
		while(true) {
			this.memory = (float)(int)((Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) / 1000) / 1000;
			this.loop();
		}
	}

	private synchronized void loop () {
		boolean flag = false;
		StringBuilder sb = new StringBuilder();
		sb.append("MEMORY : " + this.memory + "MB");
		if (this.outOfMemory) {
			sb.append(" OUT OF MEMORY");
		}
		if (this.search != null) {
			sb.append(this.search);
			flag = true;
		}
		if (this.readCode != null) {
			flag = true;
			sb.append(" / READ " + this.readCode);
		} else if(this.checkCode != null) {
			flag = true;
			sb.append(" / CHECK " + this.checkCode);
		}
		if(this.file != null) {
			sb.append(" / DOWNLOAD "+ this.downloadCode);
			if (this.ioException) {
				sb.append(" : I/O ERROR");
				this.file = null;
			} else if (this.fileSize > 0){
				sb.append(" : ");
				if (this.file.length() < this.fileSize) {
					sb.append((int)(this.file.length() * 100 / this.fileSize + 0.5f));
				} else {
					sb.append(100);
					this.file = null;
					this.fileSize = 0;
				}
				sb.append('%');
			}
			flag = true;
		}
		if (sb.length() == 0) {
			super.setText(" ");
		} else {
			super.setText(sb.toString());
		}
		try {
			if(!flag || this.outOfMemory || this.ioException) {
				Thread.sleep(2000L);
				this.outOfMemory = false;
				this.ioException = false;
			} else {
				Thread.sleep(250L);
			}
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	}
}
