package map;

import java.awt.Polygon;
import java.awt.geom.Point2D;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import database.SdfDatabase;

import map.model.Arc;
import map.model.City;
import map.model.Curve;

import util.Common;
import util.JasmineUtil;
import util.Log;

/**
 * 数値地図2500を読み込むためのクラス
 * @author ma38su
 */
public class Sdf2500Reader {
		
	private SdfDatabase storage;
	
	public Sdf2500Reader(SdfDatabase storage) {
		this.storage = storage;
	}
	
	private static final String KEY_MIZU = "MIZU";
	private static final String KEY_MIZU_ARC = "MIZU_ARC";
	
	public void productMizu(City city) {
		Polygon[] polygon = null;
		Curve[] curve = null;
		int code = city.getCode();
		if (this.storage.hasSerializable(code, Sdf2500Reader.KEY_MIZU) && this.storage.hasSerializable(code, Sdf2500Reader.KEY_MIZU_ARC)) {
			polygon = (Polygon[]) this.storage.readSerializable(code, Sdf2500Reader.KEY_MIZU);
			curve = (Curve[]) this.storage.readSerializable(code, Sdf2500Reader.KEY_MIZU_ARC);
		}
		if (polygon == null || curve == null) {
			if (city.getCode() == 5303) {
				System.out.println("mizu read");
			}
			List<Curve> list = new ArrayList<Curve>();
			try {
				File dir = this.storage.getSdf2500(code);
				if (dir != null && dir.isDirectory()) {
					Collection<Polygon> mizu = new ArrayList<Polygon>();
					for (File file : dir.listFiles()) {
						if (file.isDirectory()) {
							String path = file.getCanonicalPath() + File.separatorChar + "mizu" + File.separatorChar;
							File dataFile = new File(path + "mizu.arc");
							if (dataFile.isFile()) {
								Map<Integer, Arc> arcs = this.readArc(dataFile);
								Log.out(this, "read "+ dataFile);
								dataFile = new File(path + "mizu.pgn");
								this.readPolygon(mizu, dataFile, arcs);
								Log.out(this, "read "+ dataFile);
								for (Arc arc : arcs.values()) {
									if (arc.getTag() == 0) {
										list.add(arc.transformCurve());
									}
								}
							}
						}
					}
					if (mizu.size() == 0) {
						dir.delete();
					}
					if (mizu.size() > 0) {
						polygon = mizu.toArray(new Polygon[]{});
						curve = list.toArray(new Curve[]{});
						this.storage.writeSerializable(code, Sdf2500Reader.KEY_MIZU, polygon);
						this.storage.writeSerializable(code, Sdf2500Reader.KEY_MIZU_ARC, curve);
					}
				}
			} catch (IOException e) {
				Log.err(this, e);
			}
		}
		if (polygon != null && curve != null) {
			if (city.getCode() == 5303) {
				System.out.println("mizu readed: "+ polygon.length);
			}
			city.setSdk2500(polygon, curve);
		}
		if (city.getCode() == 5303) {
			System.out.println("mizu fin");
		}
	}

	/**
	 * ポリゴンデータを読み込みます。
	 * @param polygon 読み込むポリゴンデータ
	 * @param file 
	 * @param arcs アークデータ
	 * @throws IOException 入出力エラー
	 */
	private void readPolygon(Collection<Polygon> polygon, File file, Map<Integer, Arc> arcs) throws IOException {
		BufferedReader bi = null;
		try {
			bi = new BufferedReader(new FileReader(file));
			// ヘッダを読み飛ばす
			bi.readLine();
			int length = 0;
			int count = 0;
			Integer id = null;
			Arc[] aryArc = null;
			while (bi.ready()) {
				String line = bi.readLine();
				if (length == count) {
					if (count > 0 && id != null) {
						polygon.add(Arc.transformPolygon(aryArc));
					}
					String[] param = Common.PATTERN_CSV.split(line);
					count = 0;
					id = Integer.valueOf(param[1]);
					length = Integer.parseInt(param[2]);
					aryArc = new Arc[length];
				} else {
					int key = Integer.parseInt(line);
					Arc arc;
					if (key < 0) {
						arc = arcs.get(- key).reverse();
					} else {
						arc = arcs.get(key);
					}
					if (arc == null) {
						throw new IllegalArgumentException("エラー");
					}
					aryArc[count] = arc;
					count++;
				}
			}
			if (count > 0 && length == count && id != null) {
				polygon.add(Arc.transformPolygon(aryArc));
			}
		} catch (IOException e) {
			polygon = null;
			throw e;
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
	}

	/**
	 * アークデータを読み込みます。
	 * @param file ファイル
	 * @return アークデータ
	 * @throws IOException 入出力エラー
	 */
	private Map<Integer, Arc> readArc(File file) throws IOException {
		BufferedReader bi = null;
		Map<Integer, Arc> map = new HashMap<Integer, Arc>();
		try {
			bi = new BufferedReader(new FileReader(file));
			String[] param = Common.PATTERN_CSV.split(bi.readLine());
//			int flag = Integer.parseInt(param[5]);
			int kei = Integer.parseInt(param[6]);
			double x0 = Double.parseDouble(param[8]); // 8
			double y0 = Double.parseDouble(param[9]); // 9
			int length = 0;
			double[] aryX = null;
			double[] aryY = null;
			int count = 0;
			int type = 0;
			int tag = 0;
			Integer id = null;
			while (bi.ready()) {
				String line = bi.readLine();
				param = Common.PATTERN_CSV.split(line);
				if (length == count) {
					if (count > 0 && id != null) {
						map.put(id, new Arc(aryX, aryY, type, tag));
					}
					count = 0;
					String code = param[0];
					if (code.equals("L5101")) {
						// ArcData.TYPE_MIZU_INSIDE;
						type = 1;
					} else if (code.equals("L5106")) {
						// ArcData.TYPE_MIZU_SEASHORE
						type = 2;
					}
					tag = Integer.parseInt(param[1]); // 線種タグ
					id = Integer.valueOf(param[2]); // 個別番号
					length = Integer.parseInt(param[3]);
					aryX = new double[length];
					aryY = new double[length];
				} else {
					double x = x0 + Double.parseDouble(param[0]);
					double y = y0 + Double.parseDouble(param[1]);
					Point2D point = JasmineUtil.toLatLong(x, y, kei);
					aryX[count] = point.getX();
					aryY[count] = point.getY();
					count++;
				}
			}
			// 最後のひとつを追加
			if (id != null && count == length && count > 0) {
				map.put(id, new Arc(aryX, aryY, type, tag));
			}
		} catch (IOException e) {
			map = null;
			throw e;
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return map;
	}
}
