package map.route;

import java.awt.Graphics2D;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import util.Log;


import map.CityMap;
import map.model.Node;
import map.model.Road;

/**
 * 単方向探索による最短経路探索のためのアルゴリズム
 * 
 * ヒューリスティック関数を設定すればA*アルゴリズム
 * 設定しなければDijkstraのアルゴリズムにより最短経路を求める。
 * 
 * @author ma38su
 */
public class UnidirectSearchThread extends Thread implements SearchThread {

	/**
	 * 地図データ
	 */
	private final CityMap maps;
	
	/**
	 * 探索の始点
	 */
	private final Node start;
	
	/**
	 * 探索の終点
	 */
	private final Node terminal;

	/**
	 * 経路
	 */
	private final Set<Road> route;
	
	/**
	 * 最短経路
	 */
	private Map<Long, Long> path;
	
	/**
	 * 探索済みであればtrue
	 */
	private boolean isFinish;

	/**
	 * 最短経路探索のためのヒープ
	 */
	private Heap heap;
	
	/**
	 * スレッドが生きているかどうか
	 */
	private boolean isAlive;

	/**
	 * 探索を行っている頂点
	 */
	private Node p;

	/**
	 * 最短経路におけるコスト
	 */
	private float cost;

	/**
	 * 訪問頂点数
	 */
	private int vp;

	/**
	 * アルゴリズム名
	 */
	private String name;
	
	/**
	 * A* algorithm
	 * @param name アルゴリズム名
	 * @param start 始点
	 * @param terminal 終点
	 * @param maps 地図データ
	 * @param heap ヒープ
	 */
	public UnidirectSearchThread(String name, Node start, Node terminal, CityMap maps, Heap heap) {
		this.start = start;
		this.terminal = terminal;
		this.maps = maps;
		this.heap = heap;
		this.route = new HashSet<Road>();
		this.path = new HashMap<Long, Long>();
		this.isFinish = false;
		this.isAlive = true;
		this.name = name;
		this.maps.setSearchThread(this);

		// これ以上にすると満足に操作できなくなる。
		this.setPriority(Thread.MIN_PRIORITY);

		this.start();
	}
	
	/**
	 * 道路のコストを求める
	 * @param road 道路
	 * @return 道路のコスト
	 */
	private float getCost(Road road) {
		if (road == null) {
			return 0;
		}
		float cost = road.getCost();
		switch (road.getType()) {
			case 3 :
				if(Road.useHighway()) {
					return cost;
				}
				return Float.POSITIVE_INFINITY;
			default :
				float rate = 1 + (5 - road.getWidth()) * 0.5f;
				if (Road.useHighway()) {
					rate *= Road.GENERAL_RATE;
				}
				return cost * rate;
		}
	}
	
	/**
	 * 探索済みなら最短経路を返す
	 * @return 最短経路
	 */
	public Set<Road> getRoute() {
		if (this.isFinish && this.isAlive) {
			return this.route;
		} else {
			return null;
		}
	}
	
	/**
	 * 訪問頂点数を返します。
	 * 
	 * @return 訪問頂点数
	 */
	public int getVP () {
		if (this.isFinish && this.isAlive) {
			return this.vp;
		} else {
			return 0;
		}
	}
	
	/**
	 * このスレッドを殺す
	 *
	 */
	public void kill() {
		this.isAlive = false;
	}
	
	@Override
	public void run() {
		try {
			long t = this.terminal.getID();
			// 初期化
			Map<Long, Float> vp = new HashMap<Long, Float>();
			this.heap.put(this.start, 0);
			do {
				Heap.Entry entry = this.heap.poll();
				this.p = entry.getKey();
				float d = entry.getValue();
				if (this.p.getID() == t) {
					this.cost = d;
					this.maps.clearNode();
					this.vp = vp.size();
					this.traceback(this.terminal.getID());
					break;
				}
				// p を訪問済み頂点とする
				vp.put(this.p.getID(), d);
//				this.maps.removeNode(this.p.getID());

				// 隣接頂点までの距離を設定
				for (Map.Entry<Long, Road> edge : this.p.getEdge().entrySet()) {
					Long key = edge.getKey();
					if (vp.containsKey(key)) {
						continue;
					}
					final Node v = this.maps.getNode(key, false);
					final float cost = this.getCost(edge.getValue());

					if(cost == Float.POSITIVE_INFINITY) {
						continue;
					}

					if(this.heap.put(v, d + cost)) {
						this.path.put(v.getID(), this.p.getID());
					}
				}
			} while (this.isAlive && !this.heap.isEmpty());
		} catch (IOException e) {
			Log.err(this, e);
			this.cost = -1;
			this.route.clear();
		} catch (OutOfMemoryError e) {
			this.cost = -2;
			this.route.clear();
		} finally {
			this.isFinish = true;
			this.p = null;
			this.path = null;
			this.heap = null;
			this.maps.searchedRecover();
		}
	}
	
	@Override
	public String toString() {
		if (this.isAlive) {
			StringBuilder sb = new StringBuilder(" / SEARCH : ");
			sb.append(this.start.getID());
			sb.append(" => ");
			sb.append(this.terminal.getID());
			if (!this.isFinish) {
				sb.append(" : ");
				if(this.p != null) {
					double b = this.start.distance(this.p);
					double a = this.terminal.distance(this.p);
					sb.append((int)(b * 100 / (a + b)));
				} else {
					sb.append(0);
				}
				sb.append('%');
			} else if (this.cost >= 0) {
				sb.append(" : COST = ");
				sb.append(this.cost);
			} else {
				if (this.cost == -1) {
					sb.append(" : I/O ERROR");
				} else if (this.cost == -2) {
					sb.append(" : OutOfMemory");
				}
			}
			return sb.toString();
		} else {
			return "";
		}
	}

	/**
	 * 求めた最短経路をトレースバックします。
	 * @param r
	 * @throws IOException
	 */
	private void traceback(Long r) throws IOException {
		while (this.isAlive && this.path.containsKey(r)) {
			Long next = this.path.get(r);
			if (next == null) {
				break;
			}
			Node n = this.maps.getNode(r, true);
			if (n == null) {
				Log.out(this, r + " > "+ next);
				break;
			}
			Road road = n.getEdge().get(next);
			if(road != null) {
				if (road.getArrayX() == null) {
					throw new IOException();
				}
				this.route.add(road);
			}
			r = next;
		}
	}
	
	/**
	 * アルゴリズムと訪問頂点数の表示
	 * @param g
	 */
	public void draw(Graphics2D g) {
		if (this.isFinish) {
			g.drawString(this.name, 5, 35);
		} else {
			g.drawString(this.name + " : " + this.vp, 5, 35);
		}
	}
}