package database;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import jp.gr.java_conf.dangan.util.lha.LhaHeader;
import jp.gr.java_conf.dangan.util.lha.LhaInputStream;
import map.model.City;
import util.Common;
import util.GeneralFileFilter;
import util.Log;
import view.StatusBar;

/**
 * 数値地図のデータ管理
 * @author ma38su
 */
public class FileDatabase {

	/**
	 * 保存フォルダ
	 */
	private final String CACHE_DIR;

	/**
	 * 数値地図のベースとなるURL
	 */
	private static final String SDF_URL = "http://sdf.gsi.go.jp/";

	/**
	 * 国土数値情報のベースとなるURL
	 */
	private static final String KSJ_URL = "http://nlftp.mlit.go.jp/ksj/dls/data/";

	/**
	 * 都道府県のURL補助
	 */
	private final String[] prefecture;
	
	/**
	 * ステータスバー
	 */
	private final StatusBar status;

	/**
	 * 直列化したファイルの保存先ディレクトリ
	 */
	private String SERIALIZE_DIR;

	public static void main(String[] args) throws IOException {
		FileDatabase storage = new FileDatabase("d:/java/.digital_map/", "/data/prefecture.csv", null);
		for (File file : storage.getSdf25k(13429)) {
			System.out.println(file);
		}
	}
	
	/**
	 * コンストラクタ
	 * @param cacheDir データ格納ディレクトリ
	 * @param list リスト
	 * @param status ステータスバー
	 * @throws IOException 入出力エラー
	 */
	public FileDatabase(String cacheDir, String list, StatusBar status) throws IOException {
		this.CACHE_DIR = cacheDir;

		File dirSdf25k = new File(cacheDir + "sdf25k");
		File dirSdf2500 = new File(cacheDir + "sdf2500");
		File dirKsj = new File(cacheDir + "ksj");
		File dirSerializable = new File(cacheDir + "serialize");
		this.SERIALIZE_DIR = dirSerializable.getCanonicalPath() + File.separatorChar;
		
		Log.out(this, "init Cache Directory "+ dirSdf25k.getCanonicalPath());
		if(!dirSdf25k.isDirectory()) {
			dirSdf25k.mkdirs();
		}
		Log.out(this, "init Cache Directory "+ dirSdf2500.getCanonicalPath());
		if(!dirSdf2500.isDirectory()) {
			dirSdf2500.mkdirs();
		}
		Log.out(this, "init Cache Directory "+ dirKsj.getCanonicalPath());
		if(!dirKsj.isDirectory()) {
			dirKsj.mkdirs();
		}
		Log.out(this, "init Cache Serialize Directory "+ dirSerializable.getCanonicalPath());
		if(!dirSerializable.isDirectory()) {
			dirSerializable.mkdirs();
		}

		// 都道府県番号（都道府県数47だが、北海道は1のため）
		this.prefecture = new String[48];
		this.status = status;
		BufferedReader out = null;
		try {
			out = new BufferedReader(new InputStreamReader(FileDatabase.class.getResourceAsStream(list), "SJIS"));
			int i = 1;
			while(out.ready()) {
				String line = out.readLine();
				this.prefecture[i++] = Common.PATTERN_CSV.split(line)[1];
			}
		} finally {
			if(out != null) {
				out.close();
			}
		}
	}
	
	/**
	 * ファイルのコピーを行います。
	 * 入出力のストリームは閉じないので注意が必要です。
	 * 
	 * @param in 入力ストリーム
	 * @param out 出力ストリーム
	 * @throws IOException 入出力エラー
	 */
	private void copy(InputStream in, OutputStream out) throws IOException {
		final byte buf[] = new byte[1024];
		int size;
		while ((size = in.read(buf)) != -1) {
			out.write(buf, 0, size);
			out.flush();
		}
	}
	
	/**
	 * ファイルをダウンロードします。
	 * @param url URL
	 * @param file ダウンロード先のファイル
	 * @return ダウンロードできればtrue
	 * @throws IOException 入出力エラー
	 */
	private boolean download(URL url, File file) throws IOException {
		boolean ret = true;
		InputStream in = null;
		OutputStream out = null;
		try {
			URLConnection connect = url.openConnection();
			// ファイルのチェック（ファイルサイズの確認）
			int contentLength = connect.getContentLength();
			if (contentLength != file.length()) {
				if (!file.getParentFile().isDirectory()) {
					file.getParentFile().mkdirs();
				}
				final long start = System.currentTimeMillis();
				if (this.status != null) {
					this.status.setDownloadFile(file, contentLength);
				}
				// ダウンロード
				in = connect.getInputStream();
				out = new FileOutputStream(file);
				this.copy(in, out);
				Log.out(this, "download "+ url + " / " + (System.currentTimeMillis() - start) + "ms");
			}
		} catch (Exception e) {
			ret = false;
		} finally {
			if(in != null) {
				in.close();
			}
			if(out != null) {
				out.close();
			}
		}
		return ret;
	}

	/**
	 * 圧縮ファイルを展開します。
	 * @param lzh 展開するファイル
	 * @param dir 展開するディレクトリ
	 * @return 展開したファイル配列
	 * @throws IOException 入出力エラー
	 */
	private File[] extractLzh(File lzh, File dir) throws IOException {
		long start = System.currentTimeMillis();
		boolean isExtracted = false;
		LhaInputStream in = null;
		List<File> extracted = new ArrayList<File>();
		try {
			in = new LhaInputStream(new FileInputStream(lzh));
			LhaHeader entry;
			while ((entry = in.getNextEntry()) != null) {
				String entryPath = entry.getPath();
				/* 出力先ファイル */
				File outFile = new File(dir.getCanonicalPath() + File.separatorChar + entryPath);
				if (!outFile.exists() || entry.getOriginalSize() != outFile.length()) {
					/* entryPathにディレクトリを含む場合があるので */
					File dirParent = outFile.getParentFile();
					if(!dirParent.isDirectory()) {
						if (!dirParent.mkdir()) {
							throw new IOException("extract mkdir failed");
						}
					}
					/* ディレクトリはmkdirで作成する必要がある */
					if (entryPath.endsWith(File.separator)) {
						if (!outFile.mkdirs()) {
							throw new IOException("extract mkdir failed");
						}
					} else {
						FileOutputStream out = null;
						try {
							out = new FileOutputStream(outFile);
							this.copy(in, out);
						} finally {
							if (out != null) {
								out.close();
							}
						}
					}
					isExtracted = true;
				}
				extracted.add(outFile);
			}
		} finally {
			if (in != null) {
				in.close();
			}
		}
		long end = System.currentTimeMillis();
		if (isExtracted) {
			Log.out(this, "extract " + lzh + " / " + (end - start) + "ms");
		}
		return extracted.toArray(new File[]{});
	}

	/**
	 * 圧縮ファイルを展開します。
	 * @param zip 展開するファイル
	 * @param dir 展開するディレクトリ
	 * @param filter ファイルフィルター
	 * @return 展開したファイル配列
	 * @throws IOException 入出力エラー
	 */
	private File[] extractZip(File zip, File dir, FileFilter filter) throws IOException {
		long start = System.currentTimeMillis();
		boolean isExtracted = false;
		ZipInputStream in = null;
		List<File> extracted = new ArrayList<File>();
		try {
			in = new ZipInputStream(new FileInputStream(zip));
			ZipEntry entry;
			while ((entry = in.getNextEntry()) != null) {
				String entryPath = entry.getName();
				/* 出力先ファイル */
				File outFile = new File(dir.getCanonicalPath() + File.separatorChar + entryPath);
				if (filter == null || filter.accept(outFile)) {
					if (!outFile.exists() || entry.getSize() != outFile.length()) {
						/* entryPathにディレクトリを含む場合があるので */
						File dirParent = outFile.getParentFile();
						if(!dirParent.isDirectory()) {
							if (!dirParent.mkdir()) {
								throw new IOException("extract mkdir failed");
							}
						}
						/* ディレクトリはmkdirで作成する必要がある */
						if (entryPath.endsWith(File.separator)) {
							if (!outFile.mkdirs()) {
								throw new IOException("extract mkdir failed");
							}
						} else {
							FileOutputStream out = null;
							try {
								out = new FileOutputStream(outFile);
								this.copy(in, out);
							} finally {
								if (out != null) {
									out.close();
								}
							}
						}
						isExtracted = true;
					}
					extracted.add(outFile);
				}
			}
		} finally {
			if (in != null) {
				in.close();
			}
		}
		long end = System.currentTimeMillis();
		if (isExtracted) {
			Log.out(this, "extract " + zip + " / " + (end - start) + "ms");
		}
		return extracted.toArray(new File[]{});
	}

	
	/**
	 * 国土数値情報の行政界を取得します。
	 * @param code 都道府県番号
	 * @return 国土数値情報の行政界のファイル
	 * @throws IOException 入出力エラー
	 */
	public File getKsjBoder(int code) throws IOException {
		String stringCode = City.prefectureFormat(code);
		File dir = new File(this.CACHE_DIR + "ksj" + File.separatorChar + stringCode);
		File zip = new File(dir.getParent() + File.separatorChar + stringCode + ".zip");
		if (!dir.isDirectory()) {
			dir.mkdirs();
		}
		if (zip.exists() || !dir.isDirectory() || dir.list().length == 0) {
			/* 圧縮ファイルが残っている
			 * or ディレクトリが存在しない
			 * or ディレクトリ内のファイルが存在しない
			 * or ディレクトリの内容が正確でない（チェックできてない）
			 */
			URL url = new URL(FileDatabase.KSJ_URL + "N03-11A-" + stringCode + "-01.0a.zip");
			if (this.status != null) {
				this.status.startDownload(code, StatusBar.KSJ);
			}
			if (!this.download(url, zip)) {
				throw new IOException("download failed: "+ code);
			}
			if (this.status != null) {
				this.status.finishDownload();
			}
		}
		File[] extracts = null;
		FileFilter filter = new GeneralFileFilter("txt");
		if(zip.exists()) {
			// ファイルの展開
			extracts = this.extractZip(zip, dir, filter);
			if(!zip.delete()){
				throw new IOException("delete failure: "+ zip.getCanonicalPath());
			}
		} else if(dir.isDirectory()) {
			extracts = dir.listFiles(filter);
		}
		if(extracts == null || extracts.length == 0) {
			throw new IOException("file not found: "+ code);
		} else if (extracts.length != 1) {
			throw new IOException("files found: "+ code);
		}
		return extracts[0];
	}

	
	/**
	 * 市区町村番号に対応するファイル配列を取得します。
	 * 小笠原諸島に対応
	 * @param code 市区町村番号
	 * @return 市区町村番号に対応したファイル配列
	 * @throws IOException 入出力エラー
	 */
	public synchronized File[] getSdf25k(int code) throws IOException {
		if (code == 13421) {
			throw new IllegalArgumentException("小笠原諸島: "+ code);
		}
		String stringCode = City.cityFormat(code);
		int prefCode = code / 1000;
		// 保存先ディレクトリ
		File dir = new File(this.CACHE_DIR + "sdf25k" + File.separatorChar + City.prefectureFormat(prefCode) + File.separatorChar + stringCode);
		if (!dir.isDirectory()) {
			dir.mkdirs();
		}
		File lzh = new File(dir.getParent() + File.separatorChar + stringCode + ".lzh");
		if (lzh.exists() || !dir.isDirectory() || dir.list().length == 0) {
			/* 圧縮ファイルが残っている
			 * or ディレクトリが存在しない
			 * or ディレクトリ内のファイルが存在しない
			 * or ディレクトリの内容が正確でない（チェックできてない）
			 */
			URL url = new URL(FileDatabase.SDF_URL + "data25k/" + this.prefecture[prefCode] +  stringCode + ".lzh");
			if (this.status != null) {
				this.status.startDownload(code, StatusBar.SDF25000);
			}
			this.download(url, lzh);
			if (this.status != null) {
				this.status.finishDownload();
			}
		}
		File[] file = null;
		if(lzh.exists()) {
			// ファイルの展開
			file = this.extractLzh(lzh, dir.getParentFile());
			if(!lzh.delete()){
				throw new IOException("delete failure: "+ lzh.getCanonicalPath());
			}
		} else if(dir.isDirectory()) {
			file = dir.listFiles();
		}
		if(file == null || file.length == 0) {
			throw new IOException("file not found: "+ code);
		}
		return file;
	}
	
	/**
	 * 市区町村番号に対応するファイル配列を取得します。
	 * @param code 市区町村番号
	 * @return ディレクトリ
	 * @throws IOException 入出力エラー
	 */
	public synchronized File getSdf2500(int code) throws IOException {
		String stringCode = City.cityFormat(code);
		// 保存先ディレクトリ
		File dir = new File(this.CACHE_DIR + "sdf2500" + File.separatorChar + City.prefectureFormat(code / 1000) + File.separatorChar + stringCode + File.separatorChar);
		// 圧縮ファイル
		final File lzh = new File(dir.getCanonicalPath() + File.separatorChar + stringCode + ".lzh");
		if (lzh.exists() || !dir.isDirectory() || dir.list().length == 0) {
			URL url = new URL(FileDatabase.SDF_URL + "data2500/" +  this.prefecture[code / 1000] + City.cityFormat(code) + ".lzh");
			this.status.startDownload(code, StatusBar.SDF2500);
			if(!this.download(url, lzh)) {
				/**
				 * ダウンロードできなければ、ディレクトリは存在しない。
				 * ディレクトリの有無で数値地図2500の扱いを決める。
				 */
				dir = null;
			}
			this.status.finishDownload();
		}
		if(dir != null && lzh.exists()) {
			// ファイルの展開
			this.extractLzh(lzh, dir);
			if(!lzh.delete()){
				throw new IOException("delete failure: "+ lzh.getCanonicalPath());
			}
		}
		return dir;
	}
	
	/**
	 * 頂点の外部への接続情報の取得
	 * @param code 市町村番号
	 * @return 市町村番号に対応する頂点の外部への接続情報
	 */
	public InputStream getBoundaryNode(int code) {
		String codeStr = City.cityFormat(code);
		return this.getClass().getResourceAsStream("/.data/" + codeStr.substring(0, 2) + "/" + codeStr +".nod");
	}

	/**
	 * 直列化されたメッシュ標高が存在するか確認します。
	 * @param code 市区町村番号
	 * @param label ラベル
	 * @return 直列化されたメッシュ標高が存在すればtrue
	 */
	public boolean hasSerializable(int code, String label) {
		File file = new File(this.SERIALIZE_DIR + City.prefectureFormat(code / 1000) + File.separatorChar + City.cityFormat(code) + File.separatorChar + label +".obj");
		return file.exists();
	}

	/**
	 * 直列化されたメッシュ標高を読み込みます。
	 * synchronizedは経路探索処理と、地図データ表示のスレッドの衝突を防ぐため。
	 * @param code 市区町村番号
	 * @param label ラベル
	 * @return メッシュ標高
	 */
	public Object readSerializable(int code, String label) {
		synchronized (label) {
			Log.out(this, "read Serialize "+ label);
			Object obj = null;
			File file = new File(this.SERIALIZE_DIR + City.prefectureFormat(code / 1000) + File.separatorChar + City.cityFormat(code) + File.separatorChar + label +".obj");
			try {
				ObjectInputStream in = null;
				try {
					in = new ObjectInputStream(new FileInputStream(file));
					obj = in.readObject();
				} finally {
					if (in != null) {
						in.close();
					}
				}
			} catch (Exception e) {
				obj = null;
				Log.err(this, e);
				if (file.exists()) {
					file.delete();
				}
			}
			return obj;
		}
	}
	
	/**
	 * オブジェクトを直列化して保存します。
	 * synchronizedは経路探索処理と、地図データ表示の２つのスレッドの衝突を防ぐため。
	 * @param code 市区町村番号
	 * @param label ラベル
	 * @param obj 直列化可能なオブジェクト
	 */
	public void writeSerializable(int code, String label, Object obj) {
		synchronized (label) {
			Log.out(this, "save "+ code +" Serialize "+ label);
			File file = new File(this.SERIALIZE_DIR + City.prefectureFormat(code / 1000) + File.separatorChar + City.cityFormat(code) + File.separatorChar + label + ".obj");
			if (!file.getParentFile().isDirectory()) {
				file.getParentFile().mkdirs();
			}
			try {
				ObjectOutputStream out = null;
				try {
					out = new ObjectOutputStream(new FileOutputStream(file));
					out.writeObject(obj);
					out.flush();
				} finally {
					if (out != null) {
						out.close();
					}
				}
			} catch (Exception e) {
				Log.err(this, e);
				if (file.exists()) {
					file.delete();
				}
			}
		}
	}
}
