package database;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.RandomAccessFile;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class NameDatabase {

	public static final int MATCH_CONTAINS = 1;

	private final String file;
	
	private HashMap<Pointer, Pointer> map;

	private final Pointer p;

	private RandomAccessFile random;

	/**
	 * 
	 * @param args
	 * @throws IOException
	 */
	public static void main(String[] args) throws IOException {
		String mapDir = ".digital_map" + File.separatorChar;
		NameDatabase cityNameDB = new NameDatabase("/.data/city.csv", mapDir + "city.idx", mapDir + "city.dat");
		for (Map.Entry<Integer, String> entry : cityNameDB.get("城崎", MATCH_CONTAINS).entrySet()) {
			System.out.println(entry.getKey() + ": "+ entry.getValue());;
		}
	}
	
	/**
	 * データベース
	 * @param from
	 * @param db
	 * @param index
	 * @throws IOException
	 */
	public NameDatabase(String from, String db, String index) throws IOException {
		this.file = from;
		this.p = new Pointer();
		File disc = new File(db);
		BufferedReader bi = null;
		ObjectOutputStream out = null;
		if (disc.exists()) {
			this.map = this.readIndexMap(new File(index));
		} else {
			if (!disc.getParentFile().isDirectory()) {
				disc.getParentFile().mkdirs();
			}
		}
		if (this.map != null) {
			this.random = new RandomAccessFile(disc, "r");
		} else {
			this.map = new HashMap<Pointer, Pointer>();
			try {
				bi = new BufferedReader(new InputStreamReader(NameDatabase.class.getResourceAsStream(from), "SJIS"));
				this.random = new RandomAccessFile(disc, "rw");
				this.random.seek(0);
				this.random.setLength(0);
				String line;
				while ((line = bi.readLine()) != null) {
					String[] param = line.split(",");
					Pointer entry = new Pointer(this.random.getFilePointer());
					this.map.put(new Pointer(param[1]), entry);
					this.random.writeUTF(param[0]);
				}
				File file = new File(index);
				if (file.getParentFile().isDirectory()) {
					file.getParentFile().mkdirs();
				}
				out = new ObjectOutputStream(new FileOutputStream(file));
				out.writeObject(this.map);
				out.flush();
			} finally {
				if (bi != null) {
					bi.close();
				}
				if (out != null) {
					out.close();
				}
			}
		}
	}
	
	/**
	 * @param code 市区町村番号
	 * @return 文字列
	 */
	public String get(int code) {
		this.p.setPointer(code);
		Pointer entry = this.map.get(this.p);
		if (entry == null) {
			return null;
		}
		String name = null;
		try {
			this.random.seek(entry.pointer);
			name = this.random.readUTF();
		} catch (IOException e) {
			e.printStackTrace();
			name = null;
		}
		return name;
	}

	/**
	 * 市区町村名から市区町村名を検索します。
	 * @param name 市区町村名
	 * @param type 検索するタイプ
	 * @return 市区町村番号
	 * @throws IOException 
	 */
	public int getFirst(String name, int type) throws IOException {
		StringBuilder sb = new StringBuilder();
		switch (type) {
			case MATCH_CONTAINS :
				sb.append(".*");
				sb.append(name);
				sb.append(".*");
		}
		BufferedReader bi = null;
		Pattern pattern = Pattern.compile(sb.toString());
		Pattern split = Pattern.compile(",");
		Map<Integer, String> pref = new HashMap<Integer, String>();
		Set<Integer> set = new HashSet<Integer>();
		try {
			bi = new BufferedReader(new InputStreamReader(NameDatabase.class.getResourceAsStream(this.file), "SJIS"));
			String line;
			while ((line = bi.readLine()) != null) {
				String[] param = split.split(line);
				int code = Integer.parseInt(param[1]);
				int prefCode = code / 1000;
				Matcher matcher = pattern.matcher(param[0]);
				if (set.contains(prefCode) || matcher.matches()) {
					if (prefCode == 0) {
						set.add(code);
					} else {
						StringBuilder s = new StringBuilder();
						s.append(pref.get(prefCode));
						s.append(param[0]);
						return code;
					}
				}
				if (prefCode == 0) {
					pref.put(code, param[0]);
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return 0;
	}
	
	/**
	 * 市区町村名から市区町村名を検索します。
	 * @param name 市区町村名
	 * @param type 検索するタイプ
	 * @return 市区町村番号
	 * @throws IOException 
	 */
	public Map<Integer, String> get(String name, int type) throws IOException {
		StringBuilder sb = new StringBuilder();
		switch (type) {
			case MATCH_CONTAINS :
				sb.append(".*");
				sb.append(name);
				sb.append(".*");
		}
		BufferedReader bi = null;
		Pattern pattern = Pattern.compile(sb.toString());
		Pattern split = Pattern.compile(",");
		Map<Integer, String> map = new TreeMap<Integer, String>();
		Map<Integer, String> pref = new HashMap<Integer, String>();
		Set<Integer> set = new HashSet<Integer>();
		try {
			bi = new BufferedReader(new InputStreamReader(NameDatabase.class.getResourceAsStream(this.file), "SJIS"));
			String line;
			while ((line = bi.readLine()) != null) {
				String[] param = split.split(line);
				int code = Integer.parseInt(param[1]);
				int prefCode = code / 1000;
				Matcher matcher = pattern.matcher(param[0]);
				if (set.contains(prefCode) || matcher.matches()) {
					if (prefCode == 0) {
						set.add(code);
					} else {
						StringBuilder s = new StringBuilder();
						s.append(pref.get(prefCode));
						s.append(param[0]);
						map.put(code, s.toString());
					}
				}
				if (prefCode == 0) {
					pref.put(code, param[0]);
				}
			}
		} finally {
			if (bi != null) {
				bi.close();
			}
		}
		return map;
	}

	@SuppressWarnings("unchecked")
	private HashMap<Pointer, Pointer> readIndexMap(File index) {
		ObjectInputStream in = null;
		HashMap<Pointer, Pointer> map = null;
		if (index.isFile()) {
			try {
				try {
					in = new ObjectInputStream(new FileInputStream(index));
					map = (HashMap<Pointer, Pointer>) in.readObject();
				} catch (Exception e) {
					this.map = null;
					index.delete();
				} finally {
					if (in != null) {
						in.close();
					}
				}
			} catch (Exception e) {
				map = null;
			}
		}
		return map;
	}
}
