/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/*
#define DB(x) (x)
*/
#define DB(x)

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <gtk/gtkgl.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glext.h>

#include "glarea.h"
#include "window.h"
#include "camera.h"
#include "simplemath.h"
#include "util.h"
#include "info.h"
#include "mmap.h"
#include "mesh.h"
#include "saveimage.h"
#include "mmapthread.h"
#include "mark.h"

/********************************************/

static GtkWidget *hbox_all;
static GtkWidget *draw_dummy;
static GdkGLContext *glcontext_dummy;
static GdkGLDrawable *gldrawable_dummy;
static GdkGLContext *glcontext_main;
static GdkGLDrawable *gldrawable_main;

static guint t_dummy;
static guchar dummy[] = {
	255, 255, 255, 255, 255, 255, 208, 208, 208, 208, 208, 208,
	255, 255, 255, 255, 255, 255, 208, 208, 208, 208, 208, 208,
	208, 208, 208, 208, 208, 208, 255, 255, 255, 255, 255, 255,
	208, 208, 208, 208, 208, 208, 255, 255, 255, 255, 255, 255
};

static gint max_texture_size = 0;
static gint texture_format;
static gboolean support_compress = FALSE;
static gboolean support_vbo = FALSE;
static gboolean is_direct = FALSE;

gint
glarea_get_max_texture_size (void)
{
	return max_texture_size;
}

gint
glarea_get_texture_format (void)
{
	return texture_format;
}

gboolean
glarea_is_support_compress (void)
{
	return support_compress;
}

gboolean
glarea_is_support_vbo (void)
{
	return support_vbo;
}

gboolean
glarea_is_direct (void)
{
	return is_direct;
}


/********************************************/

static gboolean tilt_lock = FALSE;

void
glarea_tilt_lock (gboolean bool)
{
	tilt_lock = bool;
	if (tilt_lock == TRUE) {
		camera_set_tilt (0.0);
		glarea_force_render ();
	}
}

/********************************************/

static gboolean inclick_left = FALSE;
static gboolean inclick_right = FALSE;
static gboolean indrag = FALSE;

typedef struct {
	gint x;						/* ドラッグ開始位置 */
	gint y;
} DragState;

static DragState ds;

/* 視野の縦の角度 (度) */
static gdouble fovy;

typedef struct {
	GtkWidget *glarea;

	gint w;						/* 画面の幅の画素数 */
	gint h;						/* 画面の高さの画素数 */

	GdkGLContext *glcontext;
	GdkGLDrawable *gldrawable;

} GLArea;

static GLArea *area = NULL;

static void
area_init (GtkWidget * glarea)
{
	area = g_new (GLArea, 1);
	area->glarea = glarea;
	area->w = glarea->allocation.width;
	area->h = glarea->allocation.height;
}

void
glarea_get_size (gint * w, gint * h)
{
	*w = area->w;
	*h = area->h;
}

#include <errno.h>

static gboolean
glarea_gl_begin (void)
{
	gdk_threads_enter ();
	/*
	   g_print ("glarea_gl_begin:drawable:%p context:%p\n", gldrawable_main, glcontext_main);
	 */
	if (gdk_gl_drawable_gl_begin (gldrawable_main, glcontext_main) == FALSE) {
		gint err = errno;

		g_print ("####################################################################gl_begin:false\n");
		g_print ("###errno:%d\n", err);
		gl_error ();

		return FALSE;
	} else {
		return TRUE;
	}
}

static void
glarea_gl_end (void)
{
	gdk_gl_drawable_gl_end (gldrawable_main);
	gdk_flush ();

	gdk_threads_leave ();

	/*
	   g_print ("glarea_gl_end\n");
	 */
}

gboolean
glarea_dummy_gl_begin (void)
{
	gdk_threads_enter ();

	/*
	   g_print ("glarea_dummy_gl_begin:drawable:%p context:%p\n", gldrawable_dummy, glcontext_dummy);
	 */

	if (gdk_gl_drawable_gl_begin (gldrawable_dummy, glcontext_dummy) == FALSE) {
		g_print ("####################################################################dummy_gl_begin:false\n");
		gl_error ();
		return FALSE;
	} else {
		return TRUE;
	}
}

void
glarea_dummy_gl_end (void)
{
	gdk_gl_drawable_gl_end (gldrawable_dummy);
	gdk_flush ();

	gdk_threads_leave ();

	/*
	   g_print ("glarea_dummy_gl_end\n");
	 */
}

/******************************************/

static gboolean
idle_func_texture_free (gpointer data)
{
	guint *t_num = (guint *) data;

	/*
	   g_print ("idle_func_texture_free\n");
	 */

	glarea_gl_begin ();
	glDeleteTextures (1, t_num);
	glarea_gl_end ();

	g_free (t_num);

	return FALSE;				/* FALSE を返すと単発で終了 */
}

/* テクスチャーの開放を色々なスレッドでしようとすると面倒なので、idle ループ内でする。  */
void
glarea_util_texture_free (guint n)
{
	guint *t_num;

	t_num = g_new (guint, 1);
	*t_num = n;

	g_idle_add (idle_func_texture_free, t_num);
}

/******************************************/

static gboolean update_process_start = FALSE;
static GStaticMutex mutex_update = G_STATIC_MUTEX_INIT;
static gint n_update_thread = 0;

static void
thread_func_update (GObject * obj)
{

	if (n_update_thread == 1) {
		/*
		   g_print ("thread_func_update:%d:start\n", n_update_thread);
		 */
		mmap_update ();
		/*
		   g_print ("thread_func_update:%d:end\n", n_update_thread);
		 */
	} else {
	}
	--n_update_thread;


	return;
}

static void
glarea_update (void)
{
	if (update_process_start == FALSE) {
		return;
	}

	++n_update_thread;
	mmapthread_add_power (thread_func_update, NULL);
}

/******************************************/

static gint frame_count = 0;
static gdouble frame_render = 0.0;
static gdouble frame_all = 0.0;
static gdouble info_fps;
static gdouble info_render_time;
static gchar text_info[1000];
static gchar text_status[100];

void
frame_counter (gdouble sec_render, gdouble sec_all)
{
	frame_count += 1;
	frame_render += sec_render;
	frame_all += sec_all;

	if (frame_count == 30) {
		info_render_time = frame_render / frame_count;
		info_fps = frame_count / frame_all;


		frame_count = 0;
		frame_render = 0.0;
		frame_all = 0.0;
	}
}

void
create_info(void)
{
    gint n;

    n = mmapthread_get_n_download();

    if(n == 0){
	    text_status[0] = '\0';
    } else if(n == 1){
	    sprintf (text_status, "Downloading ....  ( %d file )", n);
    } else if(n > 1){
	    sprintf (text_status, "Downloading ....  ( %d files )", n);
    } else {
        /* ありえん */
	    text_status[0] = '\0';
    }

	sprintf (text_info, "fps : %6.2f   render : %6.2f (msec)       %s", info_fps, info_render_time * 1000.0, text_status);
}

/*
 * ダウンロード時のステータス表示に中身を表示しようとしたが、ダウンロード待ちの数だけ表示するようにしたので、保留。
void
glarea_status_push(const gchar *status)
{
    gint n;

    n = sizeof text_status / sizeof(gchar);
    g_print("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!%d\n", n);
    g_strlcpy(text_status, status, n);
}

void
glarea_status_pop(void)
{
    text_status[0] = '\0';
}
*/

/******************************************/

void
glarea_saveimage (void)
{
	gdouble lon, lat;
	gchar name[50];
	gchar w_or_e;
	gchar n_or_s;

	lon = camera_get_lon ();
	lat = camera_get_lat ();


	if (lon < 0.0) {
		lon *= -1.0;
		w_or_e = 'w';
	} else {
		w_or_e = 'e';
	}

	if (lat < 0.0) {
		lat *= -1.0;
		n_or_s = 's';
	} else {
		n_or_s = 'n';
	}

	sprintf (name, "%c%7.7d_%c%7.7d.png", w_or_e, (gint) (lon * 10000), n_or_s, (gint) (lat * 10000));

	saveimage_dialog (area->glarea, name);
}

/******************************************/

static guint idle_id;
static gboolean idle_render = FALSE;

static void
render_test (void)
{
	gdouble r = 100000.0 + mmap_get_world_radius ();

	glPushMatrix ();
	{
		gdouble sphere_color[] = { 0.0, 0.0, 0.0, 1.0 };

		glRotated (180.0 / 5.0 / 2.0, 0.0, 0.0, 1.0);
		glColor3dv (sphere_color);
		gdk_gl_draw_sphere (FALSE, r, 30, 15);
		glColor3d (1.0, 1.0, 1.0);
	}
	glPopMatrix ();
}

static void
cross_render (void)
{
	gdouble a0[] = { 5.0, 0.0 };
	gdouble a1[] = { 15.0, 0.0 };
	gdouble a2[] = { -5.0, 0.0 };
	gdouble a3[] = { -15.0, 0.0 };
	gdouble a4[] = { 0.0, 5.0 };
	gdouble a5[] = { 0.0, 15.0 };
	gdouble a6[] = { 0.0, -5.0 };
	gdouble a7[] = { 0.0, -15.0 };
	gdouble color[] = { 1.0, 0.0, 0.0 };

	glBegin (GL_LINES);
	glColor3dv (color);
	glVertex2dv (a0);
	glVertex2dv (a1);
	glVertex2dv (a2);
	glVertex2dv (a3);
	glVertex2dv (a4);
	glVertex2dv (a5);
	glVertex2dv (a6);
	glVertex2dv (a7);
	glEnd ();
}

static void
set_projection_matrix (void)
{
	gdouble l = 1000.0;			/* 適当に選んだ値 */
	gdouble near, far;
	gdouble alt;
	gdouble radius = mmap_get_world_radius ();

	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();

	alt = camera_get_alt ();

	fovy = 360.0 / G_PI * atan ((area->h / 2.0) / l);
	near = alt * 0.5;
	far = (radius + alt) * 2.0;
	/*
	   far = sqrt ((radius + alt) * (radius + alt) - radius * radius);
	   g_print("fovy:%.2f  near:%.2f  far:%.2f\n", fovy, near, far);
	 */
	gluPerspective (fovy, (gdouble) area->w / area->h, near, far * 2.0);
}

static void
set_modelview_matrix (void)
{
	camera_matrix_prepare ();
}

void
glarea_render_file (void)
{
	set_projection_matrix ();
	set_modelview_matrix ();
	glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	mmap_render ();

#if 0
	glDisable (GL_TEXTURE_2D);
	render_test ();				/* 地球の粗い外形 */
	camera_debug_view_render ();	/* 視野範囲 */
	camera_render_info ();		/* lon lat alt */
	glEnable (GL_TEXTURE_2D);
#endif
}

static void
render (void)
{
	/*
	   g_print ("glarea_render_all\n");
	 */

	glarea_gl_begin ();
	{
		glViewport (0, 0, area->w, area->h);
		glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		set_projection_matrix ();
		set_modelview_matrix ();

		mmap_render ();

		glDisable (GL_TEXTURE_2D);

		mark_render ();
#if 0
		render_test ();			/* 地球の粗い外形 */
		camera_debug_view_render ();	/* 視野範囲 */
#endif
		camera_render_info ();	/* lon lat alt */
		info_render (text_info, 1, 1);	/* fps & download info */


		{
			/* 投影方法を変えてるので最後に描画する */
			gdouble x, y;

			x = area->w / 2.0;
			y = area->h / 2.0;

			glDisable (GL_DEPTH_TEST);

			glMatrixMode (GL_PROJECTION);
			glLoadIdentity ();
			glMatrixMode (GL_MODELVIEW);
			glLoadIdentity ();

			gluOrtho2D (-x, x, -y, y);
			cross_render ();

			glEnable (GL_DEPTH_TEST);
		}

		glEnable (GL_TEXTURE_2D);

		if (gdk_gl_drawable_is_double_buffered (gldrawable_main)) {
			gdk_gl_drawable_swap_buffers (gldrawable_main);
		} else {
			glFlush ();
		}
	}
	glarea_gl_end ();
}

static void
render_single (void)
{
	gdouble sec_interval = 1.0 / 20.0;	/* 周期の目標（秒） */
	GTimeVal start;
	GTimeVal stop_render;
	GTimeVal stop_all;
	gdouble sec_render;
	gdouble sec_all;

	idle_render = camera_next_step ();
    create_info();

	g_get_current_time (&start);
	render ();
	g_get_current_time (&stop_render);
	sec_render = util_diff_gtimeval (&start, &stop_render);

	if (sec_interval > sec_render) {
		gdouble time;

		time = (sec_interval - sec_render) * 1000000.0;
		g_usleep ((gulong) time);
	}
	g_get_current_time (&stop_all);
	sec_all = util_diff_gtimeval (&start, &stop_all);
	frame_counter (sec_render, sec_all);
}


static gboolean
idle_func_render (gpointer data)
{
	gdouble sec_interval = 1.0 / 20.0;	/* 周期の目標（秒） */

	if (idle_render == FALSE) {
		g_usleep ((gulong) sec_interval);

	} else {
		/*
		   g_print ("idle_func_render\n");
		 */

		render_single ();

	}
	return TRUE;
}

static GStaticMutex mutex_render_single = G_STATIC_MUTEX_INIT;
static gint render_single_request = 0;

static gboolean
idle_func_render_single (gpointer data)
{
	/*
	   g_print ("idle_func_render_single\n");
	 */

	render_single ();

	g_static_mutex_lock (&mutex_render_single);
	--render_single_request;
	g_static_mutex_unlock (&mutex_render_single);

	/*
	   g_print ("idle_func_render_single:end\n");
	 */

	return FALSE;
}

void
glarea_force_render (void)
{
	g_static_mutex_lock (&mutex_render_single);
	if (render_single_request > 1) {
		;
	} else {
		++render_single_request;
		g_idle_add (idle_func_render_single, NULL);
	}
	g_static_mutex_unlock (&mutex_render_single);
}

void
glarea_force_update_and_render (void)
{
	glarea_update ();

	glarea_force_render ();
}

/******************************************/

static void
glarea_mouse_motion_cb (GtkWidget * glarea, GdkEventMotion * event, gpointer user_data)
{
	/*
	   DB(g_print ("glarea_mouse_motion_cb:x:%f y:%f\n",event->x,event->y));
	 */
	if (event == NULL) {
		g_print ("event is null\n");
		return;
	}

	if (inclick_left || inclick_right) {
		if (!indrag) {
			indrag = TRUE;
			ds.x = event->x;
			ds.y = event->y;

		} else {
			gint dx, dy;		/* マウス位置の前回値の差 */
			gdouble event_x, event_y;
			gdouble l;

			event_x = event->x;
			event_y = event->y;

			dx = event_x - ds.x;
			dy = event_y - ds.y;
			l = sqrt (dx * dx + dy * dy);

			if (inclick_left) {
				if (l != 0.0) {
					gdouble dist = camera_get_dist ();
					gdouble radius = mmap_get_world_radius ();

					/* ar:回転角度 */
					double ar = l * 360.0 / 2.0 / G_PI * atan (dist / radius * tan (RAD (fovy / 2.0))) / 15000.0;
					gdouble a = dy / l;
					gdouble b = dx / l;

					camera_rotate (a * sin (ar / 2.0), b * sin (ar / 2.0), 0.0);
				}
			} else {
				if (dx != 0) {
					gdouble ar = dx / 300.0;
					gdouble c = fabs (dx / l);

					camera_rotate (0.0, 0.0, -c * sin (ar / 2.0));
				}
				if (dy != 0 && tilt_lock == FALSE) {
					camera_tilt_set_diff (dy * 0.25);
				}
			}

			/* 前回値メモ */
			ds.x = event_x;
			ds.y = event_y;

			idle_render = TRUE;
			glarea_update ();
		}
	}
}

static void
glarea_mouse_press_cb (GtkWidget * glarea, GdkEventButton * event, gpointer data)
{
	DB (g_print ("glarea_mouse_press_cb:x:%f y:%f\n", event->x, event->y));

	if (event->button == 1) {
		inclick_left = TRUE;
	} else if (event->button == 3) {
		inclick_right = TRUE;
	}
}

static void
glarea_mouse_release_cb (GtkWidget * glarea, GdkEventButton * event, gpointer data)
{
	DB (g_print ("glarea_mouse_release_cb:x:%f y:%f\n", event->x, event->y));

	if (event->button == 1) {
		inclick_left = FALSE;
		indrag = FALSE;
	} else if (event->button == 3) {
		inclick_right = FALSE;
		indrag = FALSE;
	} else {
		;
	}
}

/* マウスホイールによる上昇下降 */
static gboolean
glarea_scroll_event_cb (GtkWidget * glarea, GdkEvent * event, gpointer user_data)
{
	GdkEventScroll *scroll = (GdkEventScroll *) event;

	DB (g_print ("glarea_scroll_event_cb\n"));

	if (scroll->direction == GDK_SCROLL_UP) {
		camera_dist_up ();
	} else if (scroll->direction == GDK_SCROLL_DOWN) {
		camera_dist_down ();
	}

	idle_render = TRUE;
	glarea_update ();

	return FALSE;
}

/*********************************/

static gboolean
glarea_configure_event_cb (GtkWidget * glarea, GdkEventConfigure * event, gpointer user_data)
{
	area->w = glarea->allocation.width;
	area->h = glarea->allocation.height;

	g_print ("glarea_configure_event_cb:w:%d h:%d\n", area->w, area->h);

	return FALSE;
}

static gboolean
glarea_expose_event_cb (GtkWidget * glarea, GdkEventExpose * event, gpointer data)
{
	g_print ("glarea_expose_event_cb\n");

	idle_render = TRUE;

	return FALSE;
}

static void
draw_dummy_realize_cb (GtkWidget * widget, gpointer user_data)
{
	g_print ("draw_dummy_realize\n");

	glcontext_dummy = gtk_widget_get_gl_context (draw_dummy);
	gldrawable_dummy = gtk_widget_get_gl_drawable (draw_dummy);
}

static void
create_draw_dummy (GtkWidget * draw_main)
{
	GdkGLContext *glcontext;
	GdkGLConfig *glconfig;

	glconfig = gdk_gl_config_new_by_mode (GDK_GL_MODE_RGB | GDK_GL_MODE_DEPTH | GDK_GL_MODE_DOUBLE);
	if (glconfig == NULL) {
		g_print ("gl_config_new_by_mode:err\n");
		exit (1);
	}

	/* メイン画面のコンテキストを取得 */
	glcontext = gtk_widget_get_gl_context (draw_main);

	draw_dummy = gtk_drawing_area_new ();
	g_signal_connect_after (draw_dummy, "realize", G_CALLBACK (draw_dummy_realize_cb), NULL);

	/* メイン画面のコンテキストとディスプレイリストとテクスチャーオブジェクトを共有 */
	gtk_widget_set_gl_capability (draw_dummy, glconfig, glcontext, TRUE, GDK_GL_RGBA_TYPE);

	g_object_unref (glconfig);

	gtk_box_pack_start (GTK_BOX (hbox_all), draw_dummy, FALSE, FALSE, 0);
	gtk_widget_show (draw_dummy);
}

static void
glarea_realize_cb (GtkWidget * glarea, gpointer user_data)
{
	DB (g_print ("glarea_realize_cb\n"));

	create_draw_dummy (glarea);

	area_init (glarea);

	glcontext_main = gtk_widget_get_gl_context (glarea);
	gldrawable_main = gtk_widget_get_gl_drawable (glarea);
	is_direct = gdk_gl_context_is_direct (glcontext_main);

	if (!gdk_gl_drawable_gl_begin (gldrawable_main, glcontext_main)) {
		g_print ("glarea_realize_cb:gl_begin:false\n");
		exit (1);
	}
	{
		gchar compress[] = "GL_ARB_texture_compression";
		gchar vbo[] = "GL_ARB_vertex_buffer_object";
		gint bytes;

		glClearColor (0.4, 0.4, 0.4, 1.0);
		glClearDepth (1.0);

		glEnable (GL_DEPTH_TEST);

		glEnable (GL_BLEND);
		glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		/*
		   glBlendFunc (GL_ONE, GL_ZERO);
		 */

		/*
		   glEnable (GL_CULL_FACE);
		   glCullFace (GL_FRONT);
		 */

		glDisable (GL_LIGHTING);

		glEnable (GL_TEXTURE_2D);

		glEnableClientState (GL_VERTEX_ARRAY);
		/*
		   glEnableClientState (GL_COLOR_ARRAY);
		   glEnableClientState (GL_NORMAL_ARRAY);
		 */
		glEnableClientState (GL_TEXTURE_COORD_ARRAY);

		glColorMaterial (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
		glEnable (GL_COLOR_MATERIAL);

		glMatrixMode (GL_PROJECTION);
		glLoadIdentity ();
		glMatrixMode (GL_MODELVIEW);
		glLoadIdentity ();

		if (gdk_gl_query_gl_extension (vbo) == TRUE) {
			g_print ("%s ------------ supported\n", vbo);
			support_vbo = TRUE;
			mesh_mmap_gl_init ();
		} else {
			g_print ("%s ------------ not supported\n", vbo);
			support_vbo = FALSE;
		}

		/* 圧縮テクスチャーのサポートの有無 */
		if (gdk_gl_query_gl_extension (compress) == TRUE) {
			g_print ("%s ------------ supported\n", compress);
			support_compress = TRUE;
		} else {
			g_print ("%s ------------ not supported\n", compress);
			support_compress = FALSE;
		}

		glGetIntegerv (GL_MAX_TEXTURE_SIZE, &max_texture_size);
		g_print ("max_texture_size:%d\n", max_texture_size);
		if (max_texture_size >= 2048) {
			max_texture_size = 2048;
		} else {
			g_print ("??? max_texture_size:%d\n", max_texture_size);
			exit (1);
		}

		glGenTextures (1, &t_dummy);
		glBindTexture (GL_TEXTURE_2D, t_dummy);
		glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexImage2D (GL_TEXTURE_2D, 0, GL_COMPRESSED_RGB_ARB, 4, 4, 0, GL_RGB, GL_UNSIGNED_BYTE, dummy);
		glGetTexLevelParameteriv (GL_TEXTURE_2D, 0, GL_TEXTURE_COMPRESSED_IMAGE_SIZE, &bytes);
		/* こんな所で得た texture_format を全てのテクスチャーのフォーマットだとしてるが、いいのか？ */
		glGetTexLevelParameteriv (GL_TEXTURE_2D, 0, GL_TEXTURE_INTERNAL_FORMAT, &texture_format);

		g_print ("dummy:bytes:%d format:%x\n", bytes, texture_format);

		info_init ();
	}
	gdk_gl_drawable_gl_end (gldrawable_main);

	g_signal_connect (area->glarea, "configure_event", G_CALLBACK (glarea_configure_event_cb), NULL);
	g_signal_connect (area->glarea, "expose_event", G_CALLBACK (glarea_expose_event_cb), NULL);
	g_signal_connect (area->glarea, "motion_notify_event", G_CALLBACK (glarea_mouse_motion_cb), NULL);
	g_signal_connect (area->glarea, "scroll_event", G_CALLBACK (glarea_scroll_event_cb), NULL);
	g_signal_connect (area->glarea, "button_release_event", G_CALLBACK (glarea_mouse_release_cb), NULL);
	g_signal_connect (area->glarea, "button_press_event", G_CALLBACK (glarea_mouse_press_cb), NULL);

	if (saveimage_check_glx (glarea) == FALSE) {
        /*
		window_menu_saveimage_sensitive (FALSE);
        */
	}

	g_print ("glarea_realize_cb:end\n");
}

/*********************************/
void
glarea_process_start (void)
{
	g_print ("glarea_process_start\n");

	update_process_start = TRUE;
	idle_id = g_idle_add (idle_func_render, NULL);

	glarea_force_update_and_render ();
}

GtkWidget *
glarea_create (void)
{
	GtkWidget *glarea;
	GdkGLConfig *glconfig;

	DB (g_print ("glarea_create\n"));

	glconfig = gdk_gl_config_new_by_mode (GDK_GL_MODE_RGB | GDK_GL_MODE_DEPTH | GDK_GL_MODE_DOUBLE);
	if (glconfig == NULL) {
		g_print ("gl_config_new_by_mode:err\n");
		exit (1);
	}

	glarea = gtk_drawing_area_new ();
	gtk_widget_set_events (glarea, GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK |
						   GDK_BUTTON_PRESS_MASK | GDK_EXPOSURE_MASK);
	gtk_widget_set_gl_capability (glarea, glconfig, NULL, TRUE, GDK_GL_RGBA_TYPE);
	g_signal_connect (glarea, "realize", G_CALLBACK (glarea_realize_cb), NULL);

	g_object_unref (glconfig);

	hbox_all = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_all), glarea, TRUE, TRUE, 2);

	return hbox_all;
}
