/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include <GL/gl.h>
#include <GL/glu.h>

#include "mmap.h"
#include "util.h"
#include "gsi.h"

gchar *
util_to_utf (const gchar * s, FROM from)
{
	gchar *t;
	GError *err = NULL;
	gsize byte_read;
	gsize byte_write;
	const char *from_codeset;
	const char euc[] = "EUC-JP";
	const char sjis[] = "SHIFT_JIS";

	if (s == NULL) {
		return NULL;
	}
	/*
	   if (g_utf8_validate (s, -1, NULL) == TRUE) {
	   return g_strdup (s);
	   }
	 */

	if (from == EUC) {
		from_codeset = euc;
	} else if (from == SJIS) {
		from_codeset = sjis;
	} else {
		g_get_charset (&from_codeset);
	}

	DB (g_print ("to_utf:from_charset:%s\n", from_codeset));

	t = g_convert (s, strlen (s), "UTF-8", from_codeset, &byte_read, &byte_write, &err);
	if (err != NULL) {
		g_print ("to_utf:g_convert:%s:%d:%d:%s\n", err->message, byte_read, byte_write, s);
		g_error_free (err);
		return NULL;
	}

	if (g_utf8_validate (t, -1, NULL) != TRUE) {
		exit (1);
	}
	return t;
}

gchar *
util_from_utf (const gchar * t)
{
	gchar *s;
	const gchar *end;
	GError *err = NULL;
	gsize byte_read;
	gsize byte_write;
	const char *to_codeset;

	if (t == NULL)
		return NULL;
	if (g_utf8_validate (t, -1, &end) != TRUE) {
		g_print ("from_utf:%s:is not utf8:end:%s:\n", t, end);
		printf ("from_utf:%s:is not utf8:end:%s:\n", t, end);
		return NULL;
	}
	g_get_charset (&to_codeset);

	DB (g_print ("codeset:%s\n", to_codeset));

	s = g_convert (t, strlen (t), to_codeset, "UTF-8", &byte_read, &byte_write, &err);
	if (err != NULL) {
		g_print ("from_utf:g_convert:%s:%s\n", t, err->message);
		g_error_free (err);
		return NULL;
	}

	return s;
}

/*************************************************/

/* 秒を返す */
gdouble
util_diff_gtimeval (GTimeVal * start, GTimeVal * stop)
{
	gdouble diff;

	diff = (stop->tv_sec + stop->tv_usec / 1000000.0) - (start->tv_sec + start->tv_usec / 1000000.0);
	/*
	   g_print ("## timer ## %.5f sec\n", diff);
	 */
	return diff;
}

/*************************************************/

gdouble
util_get_l (gdouble x0, gdouble y0, gdouble z0, gdouble x1, gdouble y1, gdouble z1)
{
	gdouble l;

	l = sqrt ((x1 - x0) * (x1 - x0) + (y1 - y0) * (y1 - y0) + (z1 - z0) * (z1 - z0));

	return l;
}

gdouble
util_get_degree_from_lonlat (gdouble lon0, gdouble lat0, gdouble lon1, gdouble lat1)
{
	gdouble deg;
	gdouble lon0_rad = 2.0 * G_PI * lon0 / 360.0;
	gdouble lat0_rad = 2.0 * G_PI * lat0 / 360.0;
	gdouble lon1_rad = 2.0 * G_PI * lon1 / 360.0;
	gdouble lat1_rad = 2.0 * G_PI * lat1 / 360.0;

	deg = acos (cos (lat0_rad) * cos (lat1_rad) * cos (lon1_rad - lon0_rad) + sin (lat0_rad) * sin (lat1_rad));

	return deg;
}

void
util_mkdir_p (const gchar * fullpath)
{
	GError *err = NULL;
	gchar *command_line;

	/*
	 * 外部のコマンドに頼らない方がいいかも
	 * ****************** 要修正 ****************
	 */

	command_line = g_strconcat ("mkdir -p \'", fullpath, "\'", NULL);

	/*
	   g_print("mics_mkdir_p:%s\n", command_line);
	 */
	if (g_spawn_command_line_sync (command_line, NULL, NULL, NULL, &err) == FALSE) {
		g_print ("mkdir_p error:%s:%s\n", fullpath, err->message);
		exit (1);
	}

	g_free (command_line);
}

void
util_print_bool (const gchar * text, gboolean bool)
{
	if (bool == TRUE) {
		g_print ("%s:TRUE\n", text);
	} else {
		g_print ("%s:FALSE\n", text);
	}
}

void
util_separator_dos_to_unix (gchar * path)
{
	gchar *p = path;

	/*
	   g_print ("unix_separator_dos_to_unix:%s:\n", path);
	 */

	while (*p != '\0') {
		/*
		   g_print ("unix_separator_dos_to_unix:%c:\n", *p);
		 */
		if (*p == '\\') {
			*p = '/';
		}
		++p;
	}
	/*
	   g_print ("unix_separator_dos_to_unix:%s:\n", path);
	 */
}

gdouble
util_char_to_double (const gchar * s)
{
	gdouble val;

	sscanf (s, "%lf", &val);

	return val;
}

guint
util_char_to_uint (const gchar * s)
{
	guint val;

	sscanf (s, "%u", &val);

	return val;
}

gint
util_char_to_int (const gchar * s)
{
	gint val;

	sscanf (s, "%d", &val);

	return val;
}

gboolean
util_char_to_boolean (const gchar * s)
{
	gboolean val;

	if (g_ascii_strcasecmp (s, "true") == 0) {
		val = TRUE;
	} else {
		val = FALSE;
	}

	return val;
}

/* 二つの四角い範囲が重なっているかをチェックする。
 * x, y は視界。xx,yy はもう一方の範囲。
 * x, xx は経度、y, yy は緯度。*/
gboolean
util_check_overlap (gdouble x0, gdouble y0, gdouble x1, gdouble y1, gdouble xx0, gdouble yy0, gdouble xx1, gdouble yy1)
{
	if (x1 > x0) {
		x1 += 360.0;
	}
	if (xx1 > xx0) {
		xx1 += 360.0;
	}

	if ((yy0 >= y0 && yy0 <= y1 && xx0 <= x1 && xx1 >= x0) ||
		(yy1 >= y0 && yy1 <= y1 && xx0 <= x1 && xx1 >= x0) ||
		(xx0 >= x0 && xx0 <= x1 && yy0 <= y1 && yy1 >= y0) || (xx1 >= x0 && xx1 <= x1 && yy0 <= y1 && yy1 >= y0) ||
		/* すっぽり入ってるパターン */
		(xx0 <= x0 && xx1 >= x1 && yy0 <= y0 && yy1 >= y1)) {
		return TRUE;
	} else {
		return FALSE;
	}
}

/**********************************************/


/* 経度緯度とあるレベルのタイル間隔から、それに相当するタイル座標 (x, y) をもとめる */
static void
util_tile_get_x_y (gdouble lon, gdouble lat, gdouble tile_deg, gint * x, gint * y)
{
	gdouble nx, ny;
	gdouble xd, yd;

	if (lon >= 360.0) {
		*x -= 360.0;
	}
	if (lat >= 180.0) {
		*y -= 180.0;
	}
	/*
	   g_print("get_x_y:lon:%.2f lat:%.2f, tile_deg:%.2f\n",lon, lat, tile_deg);
	 */
	lon = lon + 180.0;			/* 経度を西経 180 をゼロに変換 */
	lat = lat + 90.0;			/* 緯度を南緯 90 をゼロに、北緯 90 を 180 とする値に変換 */

	nx = round (360.0 / tile_deg);	/* -std=c99 必須 360.0 / 0.8 が 224 になりはまった。floor でもいいかもしれないが未調査 */
	ny = round (180.0 / tile_deg);
	/*
	   g_print("get_x_y:nx:%.30f ny:%.30f\n",360.0/deg, 180.0/tile_deg);
	   g_print("get_x_y:nx:%.30f ny:%.30f\n",nx, ny);
	 */
	xd = nx * lon / 360.0;
	yd = ny * lat / 180.0;

	*x = (gint) xd;
	*y = (gint) yd;

	if (lon == 360.0) {
		*x = nx - 1;
	}
	if (lat == 180.0) {
		*y = ny - 1;
	}
}

/* あたえられた経度緯度の範囲に相当するタイルの範囲をもとめる */
void
util_tile_get_view (gdouble west, gdouble south, gdouble east, gdouble north, gdouble tile_deg,
					gint * x0, gint * y0, gint * x1, gint * y1)
{
	gdouble deg_lon;
	/*
	   g_print("util_tile_get_view:n:%.2f s:%.2f w:%.2f e:%.2f\n",north,south, west,east);
	 */
	util_tile_get_x_y (west, south, tile_deg, x0, y0);
	util_tile_get_x_y (east, north, tile_deg, x1, y1);

	if (east > west) {
		deg_lon = east - west;
	} else {
		deg_lon = east + 360.0 - west;
	}

	/*
	   g_print("get_view:x0:%d x1:%d  y0:%d y1:%d  deg_lon:%.2f\n", *x0, *x1, *y0, *y1, deg_lon);
	   g_print("        :w:%.2f e:%.2f e-d:%.2f\n", west, east, east - tile_deg);
	 */

	if (*x0 == *x1 && deg_lon > 180.0) {
		util_tile_get_x_y (east - tile_deg, north, tile_deg, x1, y1);
		/*
		   g_print("get_view:new:x0:%d x1:%d\n", *x0, *x1);
		 */
	}
}

void
gl_error (void)
{
	GLenum err = glGetError ();

	if (err != GL_NO_ERROR) {
		g_print ("gl_err:%s\n", gluErrorString (err));
	} else {
		g_print ("gl_err:no error\n");
	}
}

/* 画像のバイナリデータから pixbuf を作って返す。失敗すると NULL を返す。返る pixbuf のリファレンスカウントは１。 */
GdkPixbuf *
util_bin_to_pixbuf (const gchar * data, gint size)
{
	GdkPixbufLoader *loader;
	GdkPixbuf *pixbuf = NULL;
	GError *err = NULL;

	if (data == NULL || size == 0) {
		return NULL;
	}

	loader = gdk_pixbuf_loader_new ();
	if (gdk_pixbuf_loader_write (loader, (guchar *) data, size, &err) == FALSE) {
		g_print ("error:gdk_pixbuf_loader_write:%s\n", err->message);

		g_object_unref (loader);

		return NULL;
	}

	/* gdk_pixbuf_loader_close() はこれ以上書き込まないことを loader に知らせてる */
	if (gdk_pixbuf_loader_close (loader, &err) == FALSE) {
		g_print ("error:gdk_pixbuf_loader_close:%s\n", err->message);

		g_object_unref (loader);

		return NULL;
	}

	pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
	g_object_ref (pixbuf);		/* gdk_pixbuf_loader_get_pixbuf() で得られる pixbuf は、
								   リファレンスカウントがゼロ */

	g_object_unref (loader);

	return pixbuf;
}



/* 地図画像を受け取り、周辺１８秒を除いた範囲を 2^n*2^nに縮小する */
/* mmap 用 */
GdkPixbuf *
util_pixbuf_scale_down (const GdkPixbuf * buf, gboolean left, gint size)
{
	GdkPixbuf *buf_new;
	gint w, h;
	gint w_left;				/* 横方向不要部分のピクセル数 */
	gint w_right;
	gint h_top;					/* 縦方向不要部分のピクセル数 */
	gint h_bottom;
	gdouble scale_x, scale_y;

	w = gdk_pixbuf_get_width (buf);
	h = gdk_pixbuf_get_height (buf);

	/*
	   g_print("util_pixbuf_scale_down:w:%d h:%d\n", w, h);
	 */

	if (left == TRUE) {
		w_left = w * (MARGIN - 1) / (W_SECOND + 2 * MARGIN);
		w_right = w * (MARGIN + 0) / (W_SECOND + 2 * MARGIN);

	} else {
		w_left = w * (MARGIN + 0) / (W_SECOND + 2 * MARGIN);
		w_right = w * (MARGIN - 1) / (W_SECOND + 2 * MARGIN);

	}
	h_top = h * (MARGIN - 1) / (H_SECOND + 2 * MARGIN);
	h_bottom = h * (MARGIN + 1) / (H_SECOND + 2 * MARGIN);

	scale_x = (gdouble) size / (w - w_left - w_right);
	scale_y = (gdouble) size / (h - h_top - h_bottom);

	buf_new = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, size, size);

	DB (g_print ("before_scale\n"));
	gdk_pixbuf_scale (buf, buf_new, 0, 0, size, size, -w_left * scale_x, -h_top * scale_y,
					  scale_x, scale_y, GDK_INTERP_BILINEAR);
	DB (g_print ("after_scale\n"));

	return buf_new;
}

void
util_sec_to_deg_min_sec (gdouble s_sec, gint * deg, gint * min, gdouble * sec)
{
	*deg = s_sec / 3600;
	*min = (s_sec - *deg * 3600) / 60;
	*sec = (s_sec - *deg * 3600 - *min * 60);
}

/**************************************************/

gint
util_get_cache_size(void)
{
    gchar *command;
    gchar *output = NULL;
    gint size;
    GError *err = NULL;

    /*
    command = g_strdup_printf("du -s %s %s", mmap_dir_cache, mmap_dir_gsi);
    */
    command = g_strdup_printf("du -s %s", mmap_dir);

    g_print("%s\n", command);

    if(g_spawn_command_line_sync (command, &output, NULL, NULL, &err) == FALSE){
        g_print("util_get_cache_size:spawn error!\n");
    }

    sscanf(output, "%d", &size);
    /*
    g_print("output:%s:\n", output);
    g_print("size:%d:\n", size);
    */

    g_free(command);
    g_free(output);

    return size;
}


