/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <math.h>

#include "config.h"

#include "mmap.h"
#include "ww_data.h"
#include "ww_object.h"
#include "ww_placename.h"
#include "ww_layerset.h"
#include "util.h"
#include "disk.h"
#include "camera.h"
#include "info.h"
#include "search.h"

typedef struct {
	gchar *placename;
	gdouble lon;
	gdouble lat;
	gchar *meta;

	gdouble xyz[3];

	gboolean visible;
} WwpPlace;

typedef struct {
	gchar *filename;
	gdouble west;
	gdouble east;
	gdouble north;
	gdouble south;

	gint n;
	WwpPlace *wwp_place;
} WplFile;

typedef struct _WwPlacenamePrivate WwPlacenamePrivate;
struct _WwPlacenamePrivate {
	gboolean dispose_has_run;

	/*****/

	gchar *name;
	gdouble distance_above_surface;
	gdouble minimum_display_altitude;
	gdouble maximum_display_altitude;

	gchar *placename_list_file_path;

	gchar *family;
	gint size;
	gboolean is_bold;
	gboolean is_italic;
	guint font_list_base;

	gdouble color[3];

	gint n;
	WplFile *wpl_file;
	GMutex *mutex_wpl_file;
};

#define WW_PLACENAME_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_PLACENAME, WwPlacenamePrivate))

static WwObjectClass *parent_class = NULL;

static void ww_placename_class_init (WwPlacenameClass * klass);
static void ww_placename_init (WwPlacename * object);
static void ww_placename_dispose (GObject * object);
static void ww_placename_finalize (GObject * object);
static GObject *ww_placename_constructor (GType type, guint n_props, GObjectConstructParam * props);
static void ww_placename_interface_init (gpointer g_iface, gpointer iface_data);

/*****************************************************************************/

static void ww_placename_set_attribute (WwData * data, const gchar ** name, const gchar ** value);
static void ww_placename_set_element (WwData * data, const gchar * element0, const gchar * element1,
									  const gchar * value);
static void ww_placename_set_parent (WwData * data, WwData * parent);
static void ww_placename_set_child (WwData * data, WwData * child);

static void ww_placename_update (WwObject * self);
static void ww_placename_render (WwObject * self);
static void ww_placename_set_on (WwObject * self, gboolean);
static void ww_placename_debug_print (WwObject * self);

/*****************************************************************************/

GType
ww_placename_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwPlacenameClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_placename_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwPlacename),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_placename_init
		};

		static const GInterfaceInfo ww_data_info = {
			(GInterfaceInitFunc) ww_placename_interface_init,	/* interface_init */
			NULL,				/* interface_finalize */
			NULL				/* interface_data */
		};

		type = g_type_register_static (WW_TYPE_OBJECT, "WwPlacename", &info, 0);

		g_type_add_interface_static (type, WW_TYPE_DATA, &ww_data_info);

		g_print ("ww_placename_get_type\n");
	}

	return type;
}

static void
ww_placename_class_init (WwPlacenameClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	WwObjectClass *wwobject_class = WW_OBJECT_CLASS (klass);
	/*
	   g_print ("ww_placename_class_init:c:%p:\n", klass);
	 */
	g_type_class_add_private (klass, sizeof (WwPlacenamePrivate));

	parent_class = g_type_class_peek_parent (klass);
	object_class->constructor = ww_placename_constructor;
	object_class->dispose = ww_placename_dispose;
	object_class->finalize = ww_placename_finalize;

	wwobject_class->update = ww_placename_update;
	wwobject_class->render = ww_placename_render;
	wwobject_class->set_on = ww_placename_set_on;
	wwobject_class->debug_print = ww_placename_debug_print;
}

static void
ww_placename_init (WwPlacename * self)
{
	WwPlacenamePrivate *priv = WW_PLACENAME_GET_PRIVATE (self);
	/*
	   g_print ("ww_placename_init:o:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;

	priv->name = NULL;
	priv->distance_above_surface = 0.0;
	priv->minimum_display_altitude = 0.0;
	priv->maximum_display_altitude = 0.0;

	priv->placename_list_file_path = NULL;

	priv->family = NULL;
	priv->size = 0;
	priv->is_bold = FALSE;
	priv->is_italic = FALSE;
	priv->font_list_base = 0;

	priv->color[0] = 0.0;
	priv->color[1] = 0.0;
	priv->color[2] = 0.0;

	priv->n = 0;
	priv->wpl_file = NULL;
	priv->mutex_wpl_file = g_mutex_new ();

	search_add_placename (WW_PLACENAME (self));
}

static void
ww_placename_dispose (GObject * obj)
{
	WwPlacename *self = WW_PLACENAME (obj);
	WwPlacenamePrivate *priv = WW_PLACENAME_GET_PRIVATE (self);

	/*
	   g_print ("ww_placename_dispose\n");
	 */

	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	/*
	   g_object_unref(priv->image_accessor);
	 */

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void free_wpl_file (WplFile * wpl, gint n);

static void
ww_placename_finalize (GObject * obj)
{
	WwPlacename *self = WW_PLACENAME (obj);
	WwPlacenamePrivate *priv = WW_PLACENAME_GET_PRIVATE (self);
	/*
	   g_print ("finalize\n");
	 */

	priv->placename_list_file_path = NULL;

	g_free (priv->name);
	g_free (priv->placename_list_file_path);
	g_free (priv->family);
	free_wpl_file (priv->wpl_file, priv->n);
	g_free (priv->wpl_file);
	g_mutex_free (priv->mutex_wpl_file);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_placename_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);

	/*
	   g_print ("constructor\n");
	 */

	object = object_class->constructor (type, n_props, props);

	return object;
}

static void
ww_placename_interface_init (gpointer g_iface, gpointer iface_data)
{
	WwDataInterface *iface = (WwDataInterface *) g_iface;

	iface->set_attribute = ww_placename_set_attribute;
	iface->set_element = ww_placename_set_element;
	iface->set_parent = ww_placename_set_parent;
	iface->set_child = ww_placename_set_child;
}

WwObject *
ww_placename_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_PLACENAME, NULL);


	/*
	   g_print ("ww_placename_new:o:%p\n", object);
	 */

	return WW_OBJECT (object);
}

/********************************************************************/

static void
free_wwp_place (WwpPlace * wwp_place, gint n)
{
	gint i;

	for (i = 0; i < n; ++i) {
		WwpPlace *wp = &(wwp_place[i]);
		g_free (wp->placename);
		g_free (wp->meta);
	}

	g_free (wwp_place);
}

static void
skip_meta_data (GIOChannel * ch)
{
	gint n;
	gint i;

	n = disk_read_int (ch);
	for (i = 0; i < n; ++i) {
		gchar *str0;
		gchar *str1;

		str0 = disk_read_string (ch);
		str1 = disk_read_string (ch);

		g_free (str0);
		g_free (str1);
	}
}

static gchar *
get_meta_data (GIOChannel * ch)
{
	gchar *meta_all = NULL;
	gint n;
	gint i;

	n = disk_read_int (ch);
	for (i = 0; i < n; ++i) {
		gchar *str0;
		gchar *str1;

		str0 = disk_read_string (ch);
		str1 = disk_read_string (ch);

		if (meta_all == NULL) {
			meta_all = g_strconcat ("(", str0, ",", str1, ")", NULL);
		} else {
			gchar *tmp;

			tmp = g_strconcat (meta_all, ",(", str0, ",", str1, ")", NULL);
			g_free (meta_all);
			meta_all = tmp;
		}

		g_free (str0);
		g_free (str1);
	}

	return meta_all;
}


static WwpPlace *
create_wwp_place (const gchar * path, const gchar * filename, gint * n)
{
	WwpPlace *wwp_place;
	gchar *dirname;
	gchar *fullpath;
	GIOChannel *ch;
	gint i;
	gint num;

	dirname = g_path_get_dirname (path);
	fullpath = g_strconcat (mmap_dir, dirname, "/", filename, NULL);
	ch = disk_channel_open (fullpath, READ_BIN);
	g_free (dirname);
	g_free (fullpath);

	if (ch == NULL) {

		*n = 0;
		return NULL;
	}


	num = disk_read_int (ch);
	wwp_place = g_new (WwpPlace, num);

	for (i = 0; i < num; ++i) {
		WwpPlace *wp = &(wwp_place[i]);
		gchar *placename;
		gdouble lon;
		gdouble lat;
		gchar *meta;
		gdouble x, y, z;

		placename = disk_read_string (ch);
		lat = disk_read_float (ch);
		lon = disk_read_float (ch);
		meta = get_meta_data (ch);

		mmap_deg_to_xyz (lon, lat, 0.0, &x, &y, &z);

		wp->placename = placename;
		wp->lat = lat;
		wp->lon = lon;
		wp->meta = meta;
		wp->xyz[0] = x;
		wp->xyz[1] = y;
		wp->xyz[2] = z;
		wp->visible = FALSE;
	}
	g_io_channel_unref (ch);

	*n = num;
	return wwp_place;
}

static void
free_wpl_file (WplFile * wpl, gint n)
{
	gint i;

	for (i = 0; i < n; ++i) {
		WplFile *wf = &(wpl[i]);

		g_free (wf->filename);
		free_wwp_place (wf->wwp_place, wf->n);
	}
}

static WplFile *
create_wpl_file (const gchar * path, gint * n)
{
	gchar *fullpath;
	GIOChannel *ch;
	gint i;
	gint num;
	WplFile *wpl_file;
	/*
	   g_print("create_wpl_file:%s:\n", path);
	 */
	fullpath = g_strconcat (mmap_dir, path, NULL);
	ch = disk_channel_open (fullpath, READ_BIN);
	g_free (fullpath);
	if (ch == NULL) {
		*n = 0;
		return NULL;
	}

	num = disk_read_int (ch);
	wpl_file = g_new (WplFile, num);

	for (i = 0; i < num; ++i) {
		WplFile *wf = &(wpl_file[i]);

		wf->filename = disk_read_string (ch);
		wf->west = disk_read_float (ch);
		wf->south = disk_read_float (ch);
		wf->east = disk_read_float (ch);
		wf->north = disk_read_float (ch);
		wf->n = 0;
		wf->wwp_place = NULL;
		/*
		   g_print("  wf:filename:%s w:%.2f s:%.2f e:%.2f n:%.2f\n", wf->filename, wf->west, wf->south, wf->east, wf->north);
		 */
	}

	g_io_channel_close (ch);
	g_io_channel_unref (ch);

	*n = num;
	return wpl_file;
}



static void
ww_placename_update (WwObject * self)
{
	WwPlacename *pl;
	WwPlacenamePrivate *priv;
	gdouble dist = camera_get_dist_target ();
	gint i;

	if (self == NULL) {
		return;
	}

	if (self->is_on == FALSE) {
		return;
	}

	pl = WW_PLACENAME (self);
	priv = WW_PLACENAME_GET_PRIVATE (pl);

	/*
	   g_print ("ww_placename_update:%s\n", self->name);
	 */

	if (dist > priv->maximum_display_altitude || dist < priv->minimum_display_altitude) {
		return;
	}

	for (i = 0; i < priv->n; ++i) {
		WplFile *wf = &(priv->wpl_file[i]);
		gdouble x0, y0, x1, y1;
		gdouble xx0, yy0, xx1, yy1;

		xx0 = wf->west;
		yy0 = wf->south;
		xx1 = wf->east;
		yy1 = wf->north;

		camera_get_view (&x0, &y0, &x1, &y1);

		/* ４つの辺それぞれに重なりがあるかどうか見ている */
		if ((yy0 >= y0 && yy0 <= y1 && xx0 <= x1 && xx1 >= x0) || (yy1 >= y0 && yy1 <= y1 && xx0 <= x1 && xx1 >= x0) ||
			(xx0 >= x0 && xx0 <= x1 && yy0 <= y1 && yy1 >= y0) || (xx1 >= x0 && xx1 <= x1 && yy0 <= y1 && yy1 >= y0) ||
			/* すっぽり入ってるパターン */
			(xx0 <= x0 && xx1 >= x1 && yy0 <= y0 && yy1 >= y1)) {

			gint j;

			if (wf->wwp_place == NULL) {
				gint n;

				wf->wwp_place = create_wwp_place (priv->placename_list_file_path, wf->filename, &n);
				wf->n = n;
			}

			for (j = 0; j < wf->n; ++j) {
				WwpPlace *wp = &(wf->wwp_place[j]);

				if (wp->lon > x0 && wp->lon < x1 && wp->lat > y0 && wp->lat < y1) {
					wp->visible = TRUE;
				} else {
					wp->visible = FALSE;
				}
			}



		} else {
			if (wf->wwp_place != NULL) {
				free_wwp_place (wf->wwp_place, wf->n);
				wf->wwp_place = NULL;
				wf->n = 0;
			}
		}
	}

	/*
	   g_print ("ww_placename_update::end\n");
	 */
}

static void
ww_placename_render (WwObject * obj)
{
	gint i;
	WwPlacenamePrivate *priv;
	gdouble dist = camera_get_dist_target ();

	if (obj == NULL) {
		return;
	}

	priv = WW_PLACENAME_GET_PRIVATE (obj);

	if (obj->is_on == FALSE) {
		return;
	}

	if (dist > priv->maximum_display_altitude || dist < priv->minimum_display_altitude) {
		return;
	}

	for (i = 0; i < priv->n; ++i) {
		gint j;
		WplFile *wf = &(priv->wpl_file[i]);

		if (wf->wwp_place == NULL) {
			continue;
		}

		for (j = 0; j < wf->n; ++j) {
			WwpPlace *wp = &(wf->wwp_place[j]);

			if (wp->visible == TRUE) {
				/*
				   g_print ("ww_placename_render:%s: %.2f %.2f %.2f:%d\n",
				   wp->placename, priv->color[0], priv->color[1], priv->color[2], priv->font_list_base);
				 */
				info_placename_render (wp->placename, priv->color, wp->xyz, priv->font_list_base);
			}
		}
	}

}

static void
ww_placename_set_on (WwObject * obj, gboolean is_on)
{
	WwObject *obj_parent = obj->wwparent;

	if (is_on == TRUE && WW_IS_LAYERSET (obj_parent)) {
		/* is_on == TRUE の時のみでないと、ループする */
		ww_layerset_check_show_only_one_layer (WW_LAYERSET (obj_parent));
	}

	parent_class->set_on (obj, is_on);
}

/**************/

static gboolean
wp_search (const gchar * place, const gchar * text)
{
	gchar *text_l;
	gchar *place_l;
	gboolean ret_val = FALSE;

	text_l = g_ascii_strdown (text, -1);
	place_l = g_ascii_strdown (place, -1);

	if (g_strrstr (place_l, text_l) != NULL) {
		ret_val = TRUE;
	} else {
		ret_val = FALSE;
	}

	g_free (text_l);
	g_free (place_l);

	return ret_val;
}


static void
wwp_place_search (WwpPlace * wwp_place, gint n, const gchar * text)
{
	gint i;

	for (i = 0; i < n; ++i) {
		WwpPlace *wp = &(wwp_place[i]);

		if (wp_search (wp->placename, text) == TRUE) {
			search_add_list_data (wp->placename, wp->lon, wp->lat, wp->meta);
			/*
			   g_print ("wwp_place_search:place:%s lon:%.2f lat:%.2f  meta:%s\n", wp->placename, wp->lon, wp->lat, wp->meta);
			 */
		}
	}
}

static void
wwp_place_search_file (const gchar * path, const gchar * filename, const gchar * text)
{
	gchar *dirname;
	gchar *fullpath;
	GIOChannel *ch;
	gint i;
	gint n_all;

	dirname = g_path_get_dirname (path);
	fullpath = g_strconcat (mmap_dir, dirname, "/", filename, NULL);
	ch = disk_channel_open (fullpath, READ_BIN);

	g_free (dirname);
	g_free (fullpath);

	n_all = disk_read_int (ch);

	for (i = 0; i < n_all; ++i) {
		gchar *name;
		gdouble lon, lat;
		gchar *meta = NULL;

		name = disk_read_string (ch);
		lat = disk_read_float (ch);
		lon = disk_read_float (ch);

		if (wp_search (name, text) == TRUE) {
			meta = get_meta_data (ch);

			search_add_list_data (name, lon, lat, meta);
			/*
			   g_print ("wwp_place_search:place:%s lon:%.2f lat:%.2f  meta:%s\n", name, lon, lat, meta);
			 */
		} else {
			skip_meta_data (ch);
		}

		g_free (name);
		g_free (meta);
	}
}

gboolean
ww_placename_search (WwPlacename * wp, const gchar * text)
{
	WwPlacenamePrivate *priv = WW_PLACENAME_GET_PRIVATE (wp);
	gint i;

	g_print ("ww_placename_search:%s:%d\n", text, priv->n);

	for (i = 0; i < priv->n; ++i) {
		WplFile *wf = &(priv->wpl_file[i]);

		g_print ("wf:%s:\n", wf->filename);


		if (search_thread_is_canceled () == TRUE) {
			return FALSE;
		}

		if (wf->wwp_place == NULL) {
			wwp_place_search_file (priv->placename_list_file_path, wf->filename, text);
		} else {
			wwp_place_search (wf->wwp_place, wf->n, text);
		}
	}

	return TRUE;
}

/**************/

typedef struct {
	gchar *name;
	guint8 r;
	guint8 g;
	guint8 b;
} WinColor;

static WinColor wincolor[] = {
	{"white", 0xff, 0xff, 0xff},
	{"yellow", 0xff, 0xff, 0x00},
	{"cyan", 0x00, 0xff, 0xff},
	{"brown", 0xa5, 0x2a, 0x2a},
	{"ivory", 0xff, 0xff, 0xf0},
	{"pink", 0xff, 0xc0, 0xcb},
	{"orange", 0xff, 0xa5, 0x00},
	{"blue", 0x00, 0x00, 0xff},	/* moon */
	{"red", 0xff, 0x00, 0x00},	/* moon */
	{"green", 0x00, 0xff, 0x00},	/* moon */
	{"lightblue", 0xad, 0xd8, 0xe6},	/* moon */
	{"darkblue", 0x00, 0x00, 0x8b},	/* moon */
	{"lightgreen", 0x90, 0xee, 0x90},	/* moon */
	{"lightgray", 0xd3, 0xd3, 0xd3},	/* moon */
	{"gold", 0xff, 0xd7, 0x00},	/* moon */
};

static void
wincolor_to_rgb (const gchar * colorname, guint8 * r, guint8 * g, guint8 * b)
{
	gint i;
	gint n;
	gchar *tmp = g_ascii_strdown (colorname, -1);

	n = sizeof wincolor / sizeof (WinColor);
	for (i = 0; i < n; ++i) {
		if (g_ascii_strcasecmp (colorname, wincolor[i].name) == 0) {
			*r = wincolor[i].r;
			*g = wincolor[i].g;
			*b = wincolor[i].b;

			return;
		}
	}

	g_free (tmp);

	*r = 0xff;
	*g = 0xff;
	*b = 0xff;
}

static void
ww_placename_set_attribute (WwData * data, const gchar ** name, const gchar ** value)
{
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);
	/*
	   g_print ("ww_placename_set_attribute:o:%p\n", obj);
	 */

	/* 専用の attribute はない */

	parent_wwdata_iface->set_attribute (data, name, value);
}

static void
ww_placename_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value)
{
	WwPlacename *obj = WW_PLACENAME (data);
	WwPlacenamePrivate *priv = WW_PLACENAME_GET_PRIVATE (obj);
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);

	/*
	   g_print ("ww_placename_set_element:o:%p e0:%s e1:%s value:%s\n", obj, element0, element1, value);
	 */

	if (g_ascii_strcasecmp (element0, "Name") == 0) {
		priv->name = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "DistanceAboveSurface") == 0) {
		priv->distance_above_surface = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "MinimumDisplayAltitude") == 0) {
		priv->minimum_display_altitude = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "MaximumDisplayAltitude") == 0) {
		priv->maximum_display_altitude = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "PlacenameListFilePath") == 0) {
		gchar *tmp = g_strdup (value);
		gint n;

		util_separator_dos_to_unix (tmp);
		priv->placename_list_file_path = tmp;

		priv->wpl_file = create_wpl_file (priv->placename_list_file_path, &n);
		priv->n = n;

	} else if (g_ascii_strcasecmp (element0, "Family") == 0) {
		priv->family = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "Size") == 0) {
		priv->size = util_char_to_int (value);

	} else if (g_ascii_strcasecmp (element0, "IsBold") == 0) {
		priv->is_bold = util_char_to_boolean (value);

	} else if (g_ascii_strcasecmp (element0, "IsItalic") == 0) {
		priv->is_italic = util_char_to_boolean (value);

	} else if (g_ascii_strcasecmp (element0, "WinColorName") == 0) {
		guint8 r, g, b;

		wincolor_to_rgb (value, &r, &g, &b);

		priv->color[0] = (gdouble) r / 0xff;
		priv->color[1] = (gdouble) g / 0xff;
		priv->color[2] = (gdouble) b / 0xff;

		/* これをするいいところがないのでこんな所でする */
		priv->font_list_base = info_placename_get_font_list_base (priv->size, priv->is_bold, priv->is_italic);

	} else if (g_ascii_strcasecmp (element0, "Red") == 0) {
		priv->color[0] = (gdouble) util_char_to_int (value) / 0xff;

		/* これをするいいところがないのでこんな所でする */
		priv->font_list_base = info_placename_get_font_list_base (priv->size, priv->is_bold, priv->is_italic);

	} else if (g_ascii_strcasecmp (element0, "Green") == 0) {
		priv->color[1] = (gdouble) util_char_to_int (value) / 0xff;

	} else if (g_ascii_strcasecmp (element0, "Blue") == 0) {
		priv->color[2] = (gdouble) util_char_to_int (value) / 0xff;

	} else {
		;
	}

	parent_wwdata_iface->set_element (data, element0, element1, value);
}

static void
ww_placename_set_parent (WwData * data, WwData * parent)
{
	WwDataInterface *parent_wwdata_iface = g_type_interface_peek (parent_class, WW_TYPE_DATA);

	/* 親のままでいい */
	parent_wwdata_iface->set_parent (data, parent);
}

static void
ww_placename_set_child (WwData * data, WwData * child)
{
	/* なにもしない */
}

static void
ww_placename_debug_print (WwObject * obj)
{
	WwPlacenamePrivate *priv = WW_PLACENAME_GET_PRIVATE (obj);

	g_print ("ww_placename_debug_print##########:o:%p\n", obj);

	parent_class->debug_print (obj);

	g_print ("\t name:%s\n", priv->name);
	g_print ("\tdistance_above_surface:%.2f\n", priv->distance_above_surface);
	g_print ("\tminimun_display_altitude:%.2f\n", priv->minimum_display_altitude);
	g_print ("\tmaximun_display_altitude:%.2f\n", priv->maximum_display_altitude);
	g_print ("\tplacename_list_file_path:%s\n", priv->placename_list_file_path);
	g_print ("\tfamily:%s\n", priv->family);
	g_print ("\tsize:%d\n", priv->size);
	util_print_bool ("\tis_bold", priv->is_bold);
	util_print_bool ("\tis_italic", priv->is_italic);
	g_print ("\tred:%.2f  green:%.2f  blue:%.2f\n", priv->color[0], priv->color[1], priv->color[2]);
	g_print ("\tn:%d wpl_file(p)::%p\n", priv->n, priv->wpl_file);

}
