/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*
#define DB(x) (x)
*/
#define DB(x)

#include <math.h>

#include "config.h"

#include "ww_terrainaccessor.h"
#include "mmap.h"
#include "util.h"
#include "disk.h"
#include "net.h"

/********************************/

typedef struct {
	gint level;
	gdouble north;
	gdouble south;
	gdouble west;
	gdouble east;

	gint16 *tile;
} OneTile;


static void onetile_free (WwTerrainaccessor * ta);

/********************************/

typedef struct _WwTerrainaccessorPrivate WwTerrainaccessorPrivate;
struct _WwTerrainaccessorPrivate {
	gboolean dispose_has_run;

	gchar *server_url;
	gchar *data_set_name;
	gdouble level_zero_tile_size_degrees;
	guint number_levels;
	guint samples_per_tile;
	gchar *data_format;
	gchar *file_extension;
	gchar *compression_type;

	gdouble north;
	gdouble south;
	gdouble west;
	gdouble east;

	struct _WwTerrainaccessor *higher_resolution_subsets;

	GSList *sl_set;				/* OneTile */

	/* attribute */
	gchar *name;
};

#define WW_TERRAINACCESSOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_TERRAINACCESSOR, WwTerrainaccessorPrivate))

static GObjectClass *parent_class = NULL;

static void ww_terrainaccessor_class_init (WwTerrainaccessorClass * klass);
static void ww_terrainaccessor_init (WwTerrainaccessor * object);
static void ww_terrainaccessor_finalize (GObject * object);
static void ww_terrainaccessor_dispose (GObject * object);
static GObject *ww_terrainaccessor_constructor (GType type, guint n_props, GObjectConstructParam * props);
static void ww_terrainaccessor_interface_init (gpointer g_iface, gpointer iface_data);

/*****************************************************************************/

static void ww_terrainaccessor_set_attribute (WwData * data, const gchar ** name, const gchar ** value);
static void ww_terrainaccessor_set_element (WwData * data, const gchar * element0, const gchar * element1,
											const gchar * value);
static void ww_terrainaccessor_set_parent (WwData * data, WwData * parent);
static void ww_terrainaccessor_set_child (WwData * data, WwData * child);

/*****************************************************************************/

GType
ww_terrainaccessor_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwTerrainaccessorClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_terrainaccessor_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwTerrainaccessor),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_terrainaccessor_init
		};

		static const GInterfaceInfo ww_data_info = {
			(GInterfaceInitFunc) ww_terrainaccessor_interface_init,	/* interface_init */
			NULL,				/* interface_finalize */
			NULL				/* interface_data */
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WwTerrainaccessor", &info, 0);

		g_type_add_interface_static (type, WW_TYPE_DATA, &ww_data_info);
	}

	return type;
}

static void
ww_terrainaccessor_class_init (WwTerrainaccessorClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	/*
	   g_print ("ww_terrainaccessor_class_init:c:%p:\n", klass);
	 */
	g_type_class_add_private (klass, sizeof (WwTerrainaccessorPrivate));

	parent_class = g_type_class_peek_parent (klass);

	object_class->constructor = ww_terrainaccessor_constructor;
	object_class->finalize = ww_terrainaccessor_finalize;
	object_class->dispose = ww_terrainaccessor_dispose;
}

static void
ww_terrainaccessor_init (WwTerrainaccessor * self)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (self);
	/*
	   g_print ("ww_terrainaccessor_init:o:%p:\n", self);
	 */
	priv->dispose_has_run = FALSE;

	priv->server_url = NULL;
	priv->data_set_name = NULL;
	priv->level_zero_tile_size_degrees = 0.0;
	priv->number_levels = 0;
	priv->samples_per_tile = 0;
	priv->data_format = NULL;
	priv->file_extension = NULL;
	priv->compression_type = NULL;

	priv->north = 0.0;
	priv->south = 0.0;
	priv->west = 0.0;
	priv->east = 0.0;

	priv->higher_resolution_subsets = NULL;

	priv->sl_set = NULL;

	/* attribute */
	priv->name = NULL;
}
static void
ww_terrainaccessor_dispose (GObject * obj)
{
	WwTerrainaccessor *self = WW_TERRAINACCESSOR (obj);
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (self);
	/*
	   g_print ("dispose\n");
	 */
	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	g_object_unref (priv->higher_resolution_subsets);
	priv->higher_resolution_subsets = NULL;

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_terrainaccessor_finalize (GObject * obj)
{
	WwTerrainaccessor *self = WW_TERRAINACCESSOR (obj);
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (self);

	/*
	   g_print ("finalize\n");
	 */
	g_free (priv->server_url);
	g_free (priv->data_set_name);
	g_free (priv->data_format);
	g_free (priv->file_extension);
	g_free (priv->compression_type);

	onetile_free (self);
	priv->sl_set = NULL;

	g_free (priv->name);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_terrainaccessor_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);

	/*
	   g_print ("ww_terrainaccessor_constructor\n");
	 */

	object = object_class->constructor (type, n_props, props);

	return object;
}

static void
ww_terrainaccessor_interface_init (gpointer g_iface, gpointer iface_data)
{
	WwDataInterface *iface = (WwDataInterface *) g_iface;
	iface->set_attribute = ww_terrainaccessor_set_attribute;
	iface->set_element = ww_terrainaccessor_set_element;
	iface->set_parent = ww_terrainaccessor_set_parent;
	iface->set_child = ww_terrainaccessor_set_child;
}

WwTerrainaccessor *
ww_terrainaccessor_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_TERRAINACCESSOR, NULL);
	/*
	   g_print ("ww_terrainaccessor_new:o:%p\n", object);
	 */
	return WW_TERRAINACCESSOR (object);
}

/********************************************************************/

static void
ww_terrainaccessor_set_attribute (WwData * data, const gchar ** name, const gchar ** value)
{
	WwTerrainaccessor *obj = WW_TERRAINACCESSOR (data);
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);
	gint i;
	/*
	   g_print("ww_terrainaccessor_set_attribute:o:%p\n", obj);
	 */
	for (i = 0; name[i] != NULL; ++i) {
		DB (g_print ("ww_terrainaccessor_set_attribute:name:%s: value:%s:\n", name[i], value[i]));

		if (g_ascii_strcasecmp (name[i], "Name") == 0) {
			priv->name = g_strdup (value[i]);

		} else {
			;
		}
	}
}

static void
ww_terrainaccessor_set_element (WwData * data, const gchar * element0, const gchar * element1, const gchar * value)
{
	WwTerrainaccessor *obj = WW_TERRAINACCESSOR (data);
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);
	/*
	   g_print("ww_terrainaccessor_set_element:o:%p  e0:%s e1:%s value:%s\n", obj, element0, element1, value);
	 */
	if (g_ascii_strcasecmp (element0, "ServerUrl") == 0) {
		priv->server_url = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "DataSetName") == 0) {
		priv->data_set_name = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "LevelZeroTileSizeDegrees") == 0) {
		priv->level_zero_tile_size_degrees = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "NumberLevels") == 0) {
		priv->number_levels = util_char_to_uint (value);

	} else if (g_ascii_strcasecmp (element0, "SamplesPerTile") == 0) {
		priv->samples_per_tile = util_char_to_uint (value);

	} else if (g_ascii_strcasecmp (element0, "DataFormat") == 0) {
		priv->data_format = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "FileExtension") == 0) {
		priv->file_extension = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "CompressonType") == 0) {	/* typo ? */
		priv->compression_type = g_strdup (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "North") == 0) {
		priv->north = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "South") == 0) {
		priv->south = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "West") == 0) {
		priv->west = util_char_to_double (value);

	} else if (g_ascii_strcasecmp (element0, "Value") == 0 && g_ascii_strcasecmp (element1, "East") == 0) {
		priv->east = util_char_to_double (value);

	} else {
		;
	}
}

static void
ww_terrainaccessor_set_parent (WwData * data, WwData * parent)
{
	/*
	   WwTerrainaccessor *obj = WW_TERRAINACCESSOR (data);
	   WwTerrainaccessor *obj_parent = WW_TERRAINACCESSOR (parent);

	   obj->wwparent = obj_parent;
	   g_object_ref (obj_parent);
	 */
}

static void
ww_terrainaccessor_set_child (WwData * data, WwData * child)
{
	WwTerrainaccessor *obj = WW_TERRAINACCESSOR (data);
	WwTerrainaccessor *obj_child = WW_TERRAINACCESSOR (child);
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);

	priv->higher_resolution_subsets = obj_child;
}

/******************************/

void
ww_terrainaccessor_debug_print (WwTerrainaccessor * obj)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);

	g_print ("terrainaccessor:name\t:%s:##########################\n", priv->name);
	g_print ("terrainaccessor:server_url\t:%s:\n", priv->server_url);
	g_print ("terrainaccessor:data_set_name\t:%s:\n", priv->data_set_name);
	g_print ("terrainaccessor:level_zero_tile_size_degrees\t:%.2f:\n", priv->level_zero_tile_size_degrees);
	g_print ("terrainaccessor:number_levels\t:%d:\n", priv->number_levels);
	g_print ("terrainaccessor:samples_per_tile\t:%d:\n", priv->samples_per_tile);
	g_print ("terrainaccessor:data_format\t:%s:\n", priv->data_format);
	g_print ("terrainaccessor:file_extension\t:%s:\n", priv->file_extension);
	g_print ("terrainaccessor:compression_type\t:%s:\n", priv->compression_type);
	g_print ("terrainaccessor:north\t:%.2f:\n", priv->north);
	g_print ("terrainaccessor:south\t:%.2f:\n", priv->south);
	g_print ("terrainaccessor:west\t:%.2f:\n", priv->west);
	g_print ("terrainaccessor:east\t:%.2f:\n", priv->east);
	if (priv->higher_resolution_subsets != NULL) {
		ww_terrainaccessor_debug_print (priv->higher_resolution_subsets);
	}
}


static void
onetile_free (WwTerrainaccessor * ta)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (ta);
	GSList *sl;

	for (sl = priv->sl_set; sl != NULL; sl = sl->next) {
		OneTile *one = sl->data;

		g_free (one->tile);
		g_free (one);
		sl->data = NULL;
	}

	g_slist_free (priv->sl_set);
	priv->sl_set = NULL;
}

static void
get_x_y (gdouble lon, gdouble lat, gint level, gdouble tile_deg, gint * x, gint * y)
{
	gint nx;
	gint ny;

	lon = lon + 180.0;			/* 経度を西経 180 をゼロに変換 */
	lat = lat + 90.0;			/* 緯度を南緯 90 をゼロに、北緯 90 を 180 とする値に変換 */

	nx = 360.0 / (tile_deg / pow (2.0, level));
	ny = 180.0 / (tile_deg / pow (2.0, level));
	*x = nx * lon / 360.0;
	*y = ny * lat / 180.0;

	if (lon == 360.0) {
		*x = nx - 1;
	}
	if (lat == 180.0) {
		*y = ny - 1;
	}
}

static gint
suitable_level (WwTerrainaccessor * obj, gdouble x0, gdouble y0, gdouble x1, gdouble y1, gint nx, gint ny)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);
	gdouble dlon, dlat, dmesh;	/* 隣り合う２点間の度数 */
	gint i;
	gint level = -1;

	dlon = (x1 - x0) / nx;
	dlat = (y1 - y0) / ny;
	if (dlon < dlat) {
		dmesh = dlon;
	} else {
		dmesh = dlat;
	}

	for (i = 0; i < priv->number_levels; ++i) {
		gdouble dtile;

		dtile = priv->level_zero_tile_size_degrees / pow (2.0, i) / priv->samples_per_tile;

		if (dtile < dmesh) {
			level = i;
			/*
			   g_print ("terrain_tile_set_prepare:dmesh:%.2f  dtile:%.2f  level:%d\n", dmesh, dtile, level);
			 */
			break;
		}
	}

	return level;
}

/* キャッシュディレクトリの path_bil_7z で示される 7z ファイルをキャッシュディレクトリの
 * odir 以下に展開する */
static void
extract_7z (const gchar * path_bil_7z, const gchar * odir)
{
	gchar *argv[] = { "7za", "e", NULL, NULL, NULL };
	GError *err = NULL;

	argv[2] = (gchar *) odir;
	argv[3] = (gchar *) path_bil_7z;

	/*
	   g_print ("%s %s %s %s\n", argv[0], argv[1], argv[2], argv[3]);
	 */
	g_spawn_sync (NULL, argv, NULL, G_SPAWN_SEARCH_PATH, NULL, NULL, NULL, NULL, NULL, &err);
	if (err != NULL) {
		g_print ("spawn error:%s:\n", err->message);
	}
}

gdouble
ww_terrainaccessor_get_alt (WwTerrainaccessor * obj, gdouble lon, gdouble lat)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);
	GSList *sl;
	gdouble alt = 0.0;

	if (obj == NULL) {
		return 0.0;
	}
	/*
	   g_print("####terrain_tile_set_get_alt:lon:%.2f lat:%.2f\n", lon, lat);
	 */
	for (sl = priv->sl_set; sl != NULL; sl = sl->next) {
		OneTile *one = sl->data;
		/*
		   g_print("w:%.2f  e:%.2f  n:%.2f  s:%.2f\n", one->west, one->east, one->north, one->south);
		 */
		if (lon >= one->west && lon < one->east && lat >= one->south && lat < one->north) {
			gint16 *tile = one->tile;
			gdouble deg = priv->level_zero_tile_size_degrees / pow (2.0, one->level);
			gdouble dlon, dlat;
			gdouble x, y;
			gint xi, yi;
			gint k;

			dlon = lon - one->west;
			dlat = one->north - lat;

			x = dlon / deg * priv->samples_per_tile;
			y = dlat / deg * priv->samples_per_tile;

			/*
			   g_print("terrain_tile_set_get_alt:dlon:%.2f dlat:%.2f  x:%.2f  y:%.2f\n", dlon, dlat, x, y);
			   g_print("terrain_tile_set_get_alt:deg:%.2f samples_per_tile:%d\n", deg, priv->samples_per_tile);
			 */

			xi = (gint) x;
			yi = (gint) y;
			if (yi == priv->samples_per_tile) {
				yi = priv->samples_per_tile - 1;
			}

			k = xi + yi * priv->samples_per_tile;
			alt = (gdouble) tile[k];

			/*
			   g_print("terrain_tile_set_get_alt:xi:%d  yi:%d  k:%d alt:%.2f    x:%.2f y:%.2f\n", xi, yi, k, alt, x,y);

			   if(alt == 0.0){
			   g_print("alt:::::000000000\n");
			   }
			 */
			break;
		}
	}
	return alt;
}


static void
data_download (WwTerrainaccessor * obj, gint level, gint x, gint y)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);
	gchar *path_bil;
	gchar *path_bil_7z;
	gchar *odir;				/* 展開先ディレクトリ */
	const gchar *world_name;

	/*
	   g_print("data_download:L %d :x %d :y %d\n", level, x, y);
	 */

	world_name = mmap_get_world_name ();

	path_bil = g_strdup_printf ("%s/%s/TerrainAccessor/%s/%d/%04d/%04d_%04d.bil",
								mmap_dir_cache, world_name, priv->name, level, y, y, x);
	path_bil_7z = g_strdup_printf ("%s/%s/TerrainAccessor/%s/%d/%04d/%04d_%04d.bil.7z",
								   mmap_dir_cache, world_name, priv->name, level, y, y, x);
	odir = g_strdup_printf ("-o%s/%s/TerrainAccessor/%s/%d/%04d", mmap_dir_cache, world_name, priv->name, level, y);

	if (filetest (path_bil) == TRUE) {	/* bil があった */
		;

	} else if (filetest (path_bil_7z) == TRUE) {	/* bil.7z があった */
		/* ダウンロードスレッドで解凍はしない
		   extract_7z (path_bil_7z, odir);
		 */

	} else {
		gchar *data;
		gint size;
		gchar *uri;

		uri = g_strdup_printf ("%s?T=%s&L=%d&X=%d&Y=%d", priv->server_url, priv->data_set_name, level, x, y);

		data = net_download (uri, &size);
		if (data == NULL || size == 0) {
			g_print ("???get_tile:%s x:%d y:%d\n", priv->name, x, y);
			g_free (uri);
			g_free (path_bil);
			g_free (path_bil_7z);
			g_free (odir);
			return;
		}

		disk_save_bin (path_bil_7z, data, size);
		/*
		   extract_7z (path_bil_7z, odir);
		 */

		g_free (data);
		g_free (uri);
	}

	g_free (path_bil);
	g_free (path_bil_7z);
	g_free (odir);
}

/* あたえられた範囲のデータが全てディスク上にある状態を用意する */
/* nx と ny は分割数。ということは、頂点数は nx+1 と ny+1 になることに注意 */
void
ww_terrainaccessor_data_prepare (WwTerrainaccessor * obj, gdouble x0, gdouble y0, gdouble x1, gdouble y1, gint nx,
								 gint ny)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);
	gint level = -1;
	gint i, j;
	gint nx0, ny0;				/* 範囲の左下座標 */
	gint nx1, ny1;				/* 範囲の右上座標 */
	gdouble deg = priv->level_zero_tile_size_degrees;

	/*
	   g_print ("terrainaccessor_data_prepare:%.2f %.2f %.2f %.2f %d %d\n", x0, y0, x1, y1, nx, ny);
	 */

	level = suitable_level (obj, x0, y0, x1, y1, nx, ny);
	if (level == -1) {
		return;
	}

	get_x_y (x0, y0, level, deg, &nx0, &ny0);
	get_x_y (x1, y1, level, deg, &nx1, &ny1);

	for (j = ny0; j <= ny1; ++j) {
		for (i = nx0; i <= nx1; ++i) {
			data_download (obj, level, i, j);
		}
	}
}

static gint16 *
get_tile (WwTerrainaccessor * obj, gint level, gint x, gint y)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);
	gint16 *tile;
	gchar *path_bil;
	gchar *path_bil_7z;
	gint size;
	gchar *odir;				/* 展開先ディレクトリ */
	const gchar *world_name;

	/*
	   g_print("get_tile:L %d :x %d :y %d\n", level, x, y);
	 */

	world_name = mmap_get_world_name ();

	path_bil = g_strdup_printf ("%s/%s/TerrainAccessor/%s/%d/%04d/%04d_%04d.bil",
								mmap_dir_cache, world_name, priv->name, level, y, y, x);
	path_bil_7z = g_strdup_printf ("%s/%s/TerrainAccessor/%s/%d/%04d/%04d_%04d.bil.7z",
								   mmap_dir_cache, world_name, priv->name, level, y, y, x);
	odir = g_strdup_printf ("-o%s/%s/TerrainAccessor/%s/%d/%04d", mmap_dir_cache, world_name, priv->name, level, y);

	if (filetest (path_bil) == TRUE) {	/* bil があった */
		/*
		   g_print("get_tile:%s\n", path_bil);
		 */
		;

	} else if (filetest (path_bil_7z) == TRUE) {	/* bil.7z があった */
		/*
		   g_print("get_tile:%s\n", path_bil_7z);
		 */
		extract_7z (path_bil_7z, odir);

	} else {
		/*
		   g_print("get_tile:??????????\n");
		 */
		/* ダウンロードは終わっているはず */
		;
	}

	tile = (gint16 *) disk_load_bin (path_bil, &size);

	g_free (path_bil);
	g_free (path_bil_7z);
	g_free (odir);

	return tile;
}

/* あたえられた範囲のデータが全てメモリー上にある状態を用意する */
/* nx と ny は分割数。ということは、頂点数は nx+1 と ny+1 になることに注意 */
void
ww_terrainaccessor_read_data (WwTerrainaccessor * obj, gdouble x0, gdouble y0, gdouble x1, gdouble y1, gint nx, gint ny)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);
	gint level = -1;
	gint i, j;
	gint nx0, ny0;				/* 範囲の左下座標 */
	gint nx1, ny1;				/* 範囲の右上座標 */
	gdouble deg = priv->level_zero_tile_size_degrees;

	/*
	   g_print ("terrainaccessor_read_data:%.2f %.2f %.2f %.2f %d %d\n", x0, y0, x1, y1, nx, ny);
	 */

	level = suitable_level (obj, x0, y0, x1, y1, nx, ny);
	if (level == -1) {
		g_print ("#######################terrainaccessor_read_data:return\n");
		return;
	}

	get_x_y (x0, y0, level, deg, &nx0, &ny0);
	get_x_y (x1, y1, level, deg, &nx1, &ny1);

	onetile_free (obj);

	for (j = ny0; j <= ny1; ++j) {
		for (i = nx0; i <= nx1; ++i) {
			OneTile *one;
			gint16 *tile;

			tile = get_tile (obj, level, i, j);
			if (tile == NULL) {
				continue;
			}

			one = g_new (OneTile, 1);
			one->level = level;
			one->north = -90.0 + (j + 1.0) * deg / pow (2.0, level);
			one->south = -90.0 + (j + 0.0) * deg / pow (2.0, level);
			one->west = -180.0 + (i + 0.0) * deg / pow (2.0, level);
			one->east = -180.0 + (i + 1.0) * deg / pow (2.0, level);
			one->tile = tile;

			priv->sl_set = g_slist_prepend (priv->sl_set, one);
		}
	}
}

/*###########################
 * 標高データが２階層であることを決め打ち
 * 要修正
 *
 * と思っていたら、いつの間にか１階層しかないようになっている(2006/5/7)
 *########################## */
WwTerrainaccessor *
ww_terrainaccessor_select (WwTerrainaccessor * obj, gdouble x0, gdouble y0, gdouble x1, gdouble y1)
{
	WwTerrainaccessorPrivate *priv = WW_TERRAINACCESSOR_GET_PRIVATE (obj);

	if (priv->higher_resolution_subsets != NULL) {
		WwTerrainaccessor *t = priv->higher_resolution_subsets;
		WwTerrainaccessorPrivate *p = WW_TERRAINACCESSOR_GET_PRIVATE (t);

		if (x0 >= p->west && y0 >= p->south && x1 <= p->east && y1 <= p->north) {
			return t;
		}
	}

	return obj;
}
